﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/Service/ldn_LocalCommunicationServiceBase.h>
#include <nn/ldn/detail/Utility/ldn_Crypto.h>
#include <nn/ldn/detail/Utility/ldn_SecureRandom.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Ldn.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    int NormalizeChannel(int channel) NN_NOEXCEPT
    {
        int normalizedChannel;

        // 製品機では常に無線チャンネルを自動選択します。
        if (IsProductionEnvironment())
        {
            normalizedChannel = AutoChannel;
        }

        // 開発機ではマクロの指定により挙動を切り替えます。
        else
        {
            #if defined(NN_LDN_BUILD_CONFIG_CHANNEL_AUTO)
            // 上位層からの指定を無視して無線チャンネルを自動選択します。
            NN_UNUSED(channel);
            normalizedChannel = AutoChannel;
            #elif defined(NN_LDN_BUILD_CONFIG_CHANNEL_SETTINGS)
            // Settings に 0 以上の値が設定されている場合はそのチャンネルを使用し、
            // -1 が設定されている場合には上位層からの指示に従います。
            int settings = nn::settings::system::GetLdnChannel();
            normalizedChannel = 0 <= settings ? settings : channel;
            #elif defined(NN_LDN_BUILD_CONFIG_CHANNEL_RAW)
            // 上位層から指定された無線チャンネルをそのまま使用します。
            normalizedChannel = channel;
            #elif defined(NN_LDN_BUILD_CONFIG_CHANNEL_STATIC)
            // 上位層からの指定を無視してビルド時に指定した無線チャンネルを使用します。
            NN_UNUSED(channel);
            normalizedChannel = NN_LDN_BUILD_CONFIG_CHANNEL;
            #endif
        }
        NN_SDK_ASSERT(0 <= normalizedChannel);
        return normalizedChannel;
    }

    SecurityMode NormalizeSecurityMode(SecurityMode securityMode) NN_NOEXCEPT
    {
        SecurityMode normalizeSecurityMode;

        // 製品機では常に SecurityMode_Product を使用します。
        if (IsProductionEnvironment())
        {
            normalizeSecurityMode = SecurityMode_Product;
        }

        // 開発機ではマクロの指定により挙動を切り替えます。
        else
        {
            #if defined(NN_LDN_BUILD_CONFIG_SECURITY_MODE_PRODUCT)
            // 上位層からの指定を無視して SecurityMode_Product を使用します。
            NN_UNUSED(securityMode);
            normalizeSecurityMode = SecurityMode_Product;
            #elif defined(NN_LDN_BUILD_CONFIG_SECURITY_MODE_DEBUG)
            // 上位層からの指定を無視して SecurityMode_Debug を使用します。
            NN_UNUSED(securityMode);
            normalizeSecurityMode = SecurityMode_Debug;
            #elif defined(NN_LDN_BUILD_CONFIG_SECURITY_MODE_SYSTEM_DEBUG)
            // 上位層からの指定を無視して SecurityMode_SystemDebug を使用します。
            NN_UNUSED(securityMode);
            normalizeSecurityMode = SecurityMode_SystemDebug;
            #elif defined(NN_LDN_BUILD_CONFIG_SECURITY_MODE_MANUAL)
            // 上位層からの指定を尊重し、そのまま使用します。
            normalizeSecurityMode = securityMode;
            #endif
        }
        return normalizeSecurityMode;
    }

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    LocalCommunicationServiceBase::LocalCommunicationServiceBase(
        LocalCommunicationServiceManager* pManager) NN_NOEXCEPT
        : m_pManager(pManager),
          m_Context(pManager->CreateContext())
    {
        NN_SDK_ASSERT_NOT_NULL(pManager);
        NN_SDK_ASSERT_GREATER(m_Context, 0);
        const bool isInterProcess = true;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(
            &m_StateChangeEvent, nn::os::EventClearMode_AutoClear, isInterProcess));
    }

    LocalCommunicationServiceBase::~LocalCommunicationServiceBase() NN_NOEXCEPT
    {
        m_pManager->DestroyContext(m_Context);
        nn::os::DestroySystemEvent(&m_StateChangeEvent);
        NN_LDN_LOG_INFO("context %d destroyed\n", m_Context);
    }

    Result LocalCommunicationServiceBase::GetStateChangeEvent(
        nn::sf::Out<nn::sf::NativeHandle> pOutReadableHandle) const NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        nn::os::NativeHandle handle = nn::os::GetReadableHandleOfSystemEvent(&m_StateChangeEvent);
        const bool isManaged = false;
        pOutReadableHandle.Set(nn::sf::NativeHandle(handle, isManaged));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::GetState(nn::sf::Out<Bit32> pOutState) const NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        *pOutState = static_cast<Bit32>(m_pManager->GetState(m_Context));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::Scan(
        nn::sf::OutArray<NetworkInfo> pOutScanResultArray,
        nn::sf::Out<int16_t> pOutCount,
        const ScanFilter& filter,
        int16_t channel) NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        int count;
        Result result = m_pManager->Scan(
            m_Context,
            pOutScanResultArray.GetLength() == 0 ? nullptr : pOutScanResultArray.GetData(),
            &count,
            static_cast<int>(pOutScanResultArray.GetLength()),
            filter,
            NormalizeChannel(channel));
        *pOutCount = static_cast<int16_t>(count);
        return result;
    }

    Result LocalCommunicationServiceBase::ScanPrivate(
        nn::sf::OutArray<NetworkInfo> pOutScanResultArray,
        nn::sf::Out<int16_t> pOutCount,
        const ScanFilter& filter,
        int16_t channel) NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        int count;
        Result result = m_pManager->Scan(
            m_Context,
            pOutScanResultArray.GetLength() == 0 ? nullptr : pOutScanResultArray.GetData(),
            &count,
            static_cast<int>(pOutScanResultArray.GetLength()),
            filter,
            channel);
        *pOutCount = static_cast<int16_t>(count);
        return result;
    }

    Result LocalCommunicationServiceBase::GetNetworkInfoAndHistory(
        nn::sf::Out<NetworkInfo> pOutNetwork,
        nn::sf::OutArray<NodeLatestUpdate> outUpdates) NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        int bufferCount = static_cast<int>(outUpdates.GetLength());
        NN_RESULT_DO(m_pManager->GetNetworkInfo(
            m_Context, pOutNetwork.GetPointer(),
            outUpdates.GetLength() != NodeCountMax ? nullptr : outUpdates.GetData(), bufferCount));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::GetNetworkInfo(
        nn::sf::Out<NetworkInfo> pOutNetwork) const NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->GetNetworkInfo(m_Context, pOutNetwork.GetPointer()));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::GetIpv4Address(
        nn::sf::Out<Ipv4Address> pOutAddress,
        nn::sf::Out<SubnetMask> pOutMask) const NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->GetIpv4Address(
            m_Context, pOutAddress.GetPointer(), pOutMask.GetPointer()));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::GetSecurityParameter(
        nn::sf::Out<SecurityParameter> pOutSecurityParameter) const NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->GetSecurityParameter(
            m_Context, pOutSecurityParameter.GetPointer()));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::GetNetworkConfig(
        nn::sf::Out<NetworkConfig> pOutNetworkConfig) const NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->GetNetworkConfig(
            m_Context, pOutNetworkConfig.GetPointer()));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::OpenAccessPoint() NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->OpenAccessPoint(m_Context, &m_StateChangeEvent));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::CloseAccessPoint() NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->CloseAccessPoint(m_Context));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::CreateNetwork(
        const NetworkConfig& network,
        const SecurityConfig& security,
        const UserConfig& user) NN_NOEXCEPT
    {
        NN_LDN_TRACE;

        // 無線チャンネルを矯正します。
        NetworkConfig normalizedNetworkConfig = network;
        normalizedNetworkConfig.channel = static_cast<int16_t>(NormalizeChannel(network.channel));

        // SecurityMode を矯正します。
        SecurityConfig normalizedSecurityConfig = security;
        normalizedSecurityConfig.securityMode = static_cast<Bit16>(NormalizeSecurityMode(
            static_cast<SecurityMode>(security.securityMode)));

        // SecurityParameter をランダムに生成します。
        SecurityParameter securityParam;
        FillSecureRandom(&securityParam, sizeof(SecurityParameter));

        // ネットワークを構築します。
        NN_RESULT_DO(m_pManager->CreateNetwork(
            m_Context, normalizedNetworkConfig, normalizedSecurityConfig,
            securityParam, user, 0, nullptr));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::CreateNetworkPrivate(
        const NetworkConfig& network,
        const SecurityConfig& securityConfig,
        const SecurityParameter& securityParam,
        const UserConfig& user,
        const nn::sf::InArray<AddressEntry>& entries) NN_NOEXCEPT
    {
        NN_LDN_TRACE;

        // SecurityMode を矯正します。
        SecurityConfig normalizedSecurityConfig = securityConfig;
        normalizedSecurityConfig.securityMode = static_cast<Bit16>(NormalizeSecurityMode(
            static_cast<SecurityMode>(securityConfig.securityMode)));

        // ネットワークを構築します。
        NN_RESULT_DO(m_pManager->CreateNetwork(
            m_Context, network, normalizedSecurityConfig,
            securityParam, user, static_cast<int>(entries.GetLength()),
            entries.GetLength() == 0U ? nullptr : entries.GetData()));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::DestroyNetwork() NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->DestroyNetwork(m_Context));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::Reject(Ipv4Address address) NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->Reject(m_Context, address));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::SetAdvertiseData(
    const nn::sf::InBuffer& data) NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->SetAdvertiseData(
            m_Context, data.GetPointerUnsafe(), data.GetSize()));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::SetStationAcceptPolicy(Bit8 policy) NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->SetStationAcceptPolicy(
            m_Context, static_cast<AcceptPolicy>(policy)));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::AddAcceptFilterEntry(MacAddress station) NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->AddAcceptFilterEntry(m_Context, station));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::ClearAcceptFilter() NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->ClearAcceptFilter(m_Context));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::OpenStation() NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->OpenStation(m_Context, &m_StateChangeEvent));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::CloseStation() NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->CloseStation(m_Context));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::Connect(
        const NetworkInfo& network,
        const SecurityConfig& security,
        const UserConfig& user,
        int version,
        Bit32 option) NN_NOEXCEPT
    {
        NN_LDN_TRACE;

        // Any を指定された場合はビーコンに含まれる SecurityMode を使用します。
        // ただし、製品機では SecurityMode_Product 固定です。
        SecurityConfig normalizedSecurityConfig = security;
        normalizedSecurityConfig.securityMode = static_cast<Bit16>(NormalizeSecurityMode(
            static_cast<SecurityMode>(security.securityMode == SecurityMode_Any ?
            network.ldn.securityMode : security.securityMode)));

        // NetworkInfo を分解します。
        NetworkConfig networkConfig;
        networkConfig.intentId = network.networkId.intentId;
        networkConfig.nodeCountMax = network.ldn.nodeCountMax;
        networkConfig.channel = network.common.channel;
        networkConfig.localCommunicationVersion = network.ldn.nodes[0].localCommunicationVersion;
        SecurityParameter securityParam;
        std::memcpy(securityParam._serverRandom, network.ldn.serverRandom, RandomSize);
        securityParam._sessionId = network.networkId.sessionId;

        // ネットワークに接続します。
        NN_RESULT_DO(m_pManager->Connect(
            m_Context,  networkConfig, normalizedSecurityConfig, securityParam,
            user, version, static_cast<ConnectOption>(option)));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::ConnectPrivate(
        const NetworkConfig& network,
        const SecurityConfig& securityConfig,
        const SecurityParameter& securityParam,
        const UserConfig& user,
        int version,
        Bit32 option) NN_NOEXCEPT
    {
        NN_LDN_TRACE;

        // 製品機では SecurityMode_Product 固定です。
        SecurityConfig normalizedSecurityConfig = securityConfig;
        normalizedSecurityConfig.securityMode = static_cast<Bit16>(NormalizeSecurityMode(
            static_cast<SecurityMode>(securityConfig.securityMode)));

        // ネットワークに接続します。
        NN_RESULT_DO(m_pManager->Connect(
            m_Context,  network, normalizedSecurityConfig, securityParam,
            user, version, static_cast<ConnectOption>(option)));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::Disconnect() NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->Disconnect(m_Context));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceBase::GetDisconnectReason(
        nn::sf::Out<int16_t> pOutReason) const NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        DisconnectReason reason;
        Result result = m_pManager->GetDisconnectReason(m_Context, &reason);
        *pOutReason = static_cast<int16_t>(reason);
        return result;
    }

    Result LocalCommunicationServiceBase::SetWirelessControllerRestriction(
        Bit32 restriction) NN_NOEXCEPT
    {
        NN_LDN_TRACE;
        NN_RESULT_DO(m_pManager->SetWirelessControllerRestriction(
            m_Context, static_cast<WirelessControllerRestriction>(restriction)));
        NN_RESULT_SUCCESS;
    }

}}} // namespace nn::ldn::detail
