﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/btm/btm_Api.h>
#include <nn/btm/btm_Result.h>
#include <nn/ldn/ldn_PrivateResult.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/Debug/ldn_Assert.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NintendoEthernet.h>
#include <nn/ldn/detail/NetworkInterface/ldn_Wlan.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/wlan/wlan_ScanResultReader.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    // サポートする無線チャンネルです。
    const int SupportedChannels[] = { 1, 6, 11, 36, 40, 44, 48 };
    const int SupportedChannelCount = sizeof(SupportedChannels) / sizeof(int);
    const int AutoSelectableChannels[] = { 1, 6, 11 };
    const int AutoSelectableChannelCount = sizeof(AutoSelectableChannels) / sizeof(int);
    NN_STATIC_ASSERT(SupportedChannelCount <= SupportedChannelCountMax);
    NN_STATIC_ASSERT(AutoSelectableChannelCount <= AutoSelectableChannelCountMax);

    // 無線通信に関連するパラメータ (単位は ms) です。
    const int BeaconInterval = 100;
    const int ChannelScanTime = 110;
    const int HomeChannelTime = 0;
    const int InactivePeriodMax = 4000;
    NN_STATIC_ASSERT(BeaconInterval < ChannelScanTime);
    NN_STATIC_ASSERT(3000 < InactivePeriodMax && InactivePeriodMax <= 180000);

    // セキュリティに関連するパラメータです。
    const size_t KeySizeMin = 16;
    const size_t KeySizeMax = 64;

    // 通信用のキューの設定です。
    const int ReceiveBufferCountMax = 16;
    const int ActionFrameReceiveBufferCountMax = 8;

    // Advertise 用のプロトコルです。
    const ProtocolId AdvertiseProtocol = MakeProtocolId(ProtocolCode_Advertise);

    // Advertise 用の Action Frame のヘッダです。
    struct AdvertiseActionFrameHeader
    {
        uint16_t offset;
        Bit16 _reserved;
    };

    // Action Frame で配信できる Advertise のサイズです。
    const size_t AdvertiseActionFramePayloadSizeMax =
        NintendoActionFramePayloadSizeMax - sizeof(AdvertiseActionFrameHeader);

    // Advertise 用の Action Frame のペイロードとして載せるデータです。
    struct AdvertiseActionFrame
    {
        ActionFrameHeader actionFrameHeader;
        NintendoActionFrameHeader nintendoActionFrameHeader;
        AdvertiseActionFrameHeader advertiseActionFrameHeader;
        Bit8 advertise[AdvertiseActionFramePayloadSizeMax];
    };
    NN_STATIC_ASSERT(sizeof(AdvertiseActionFrame) == sizeof(ActionFrame));

    // AdvertiseActionFrame の最小サイズです。
    const size_t AdvertiseActionFrameSizeMin =
        sizeof(AdvertiseActionFrame) - AdvertiseActionFramePayloadSizeMax;

    // BTM 関連の変数です。
    bool g_IsBtmInitialized = false;
    nn::os::SystemEvent g_BtmEvent;

    // BTM は 1 回しか初期化できません。
    void InitializeBtm() NN_NOEXCEPT
    {
        if (!g_IsBtmInitialized)
        {
            nn::btm::InitializeBtmInterface();
            nn::btm::RegisterSystemEventForConnectedDeviceCondition(g_BtmEvent.GetBase());
            g_IsBtmInitialized = true;
        }
    }

    // 有効なチャンネルであることを検証します。
    inline bool IsSupportedChannel(int channel)
    {
        return std::any_of(
            std::begin(SupportedChannels), std::end(SupportedChannels),
            [channel](int supportedChannel) { return channel == supportedChannel; });
    }

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    WlanActionFrameMonitor::WlanActionFrameMonitor(
        WlanActionFrameMonitorBuffer* buffer) NN_NOEXCEPT
        : m_ReceivedEvent(nn::os::EventClearMode_AutoClear, false),
          m_Mutex(false),
          m_Buffer(buffer),
          m_ScanBuffer(nullptr),
          m_ScanBufferCount(0),
          m_ScanResultCount(0),
          m_RxEntry(0),
          m_IsMonitoring(false)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::os::ThreadStackAlignment);
    }

    WlanActionFrameMonitor::~WlanActionFrameMonitor() NN_NOEXCEPT
    {
        if (m_IsMonitoring)
        {
            StopMonitoring();
        }
    }

    void WlanActionFrameMonitor::StartMonitoring(
        L2ScanResult* scanBuffer, int bufferCount) NN_NOEXCEPT
    {
        StartMonitoring(scanBuffer, bufferCount, BroadcastMacAddress);
    }

    void WlanActionFrameMonitor::StartMonitoring(
        L2ScanResult* scanBuffer, int bufferCount, MacAddress bssid) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsMonitoring);
        NN_SDK_ASSERT_NOT_NULL(scanBuffer);
        NN_SDK_ASSERT(0 < bufferCount);
        NN_SDK_ASSERT_NOT_EQUAL(bssid, ZeroMacAddress);

        // バッファは 0 で埋めておきます。
        {
            std::lock_guard<nn::os::Mutex> lock(m_Mutex);
            std::memset(m_Buffer, 0, sizeof(WlanActionFrameMonitorBuffer));
            std::memset(scanBuffer, 0, sizeof(L2ScanResult) * bufferCount);
            m_ScanBuffer = scanBuffer;
            m_ScanBufferCount = bufferCount;
            m_ScanResultCount = 0;
        }
        m_ReceivedEvent.Clear();

        // Action Frame の受信に使用する RX エントリを作成します。
        auto subType = static_cast<Bit16>(ActionFrameSubType_Advertise);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(
            &m_RxEntry, &subType, 1, ActionFrameReceiveBufferCountMax));

        // Action Frame を監視するスレッドを生成します。
        m_IsMonitoring = true;
        m_Bssid = bssid;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &m_MonitorThread, MonitorThread, this,
            m_Buffer->monitorThreadStack, sizeof(m_Buffer->monitorThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(ldn, ActionFrameMonitor)));
        nn::os::SetThreadNamePointer(
            &m_MonitorThread, NN_SYSTEM_THREAD_NAME(ldn, ActionFrameMonitor));
        nn::os::StartThread(&m_MonitorThread);
    }

    void WlanActionFrameMonitor::StopMonitoring() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsMonitoring);

        // Action Frame の監視を停止します。
        m_IsMonitoring = false;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::CancelGetActionFrame(m_RxEntry));

        // Action Frame を監視するスレッドの終了まで待機します。
        nn::os::WaitThread(&m_MonitorThread);
        nn::os::DestroyThread(&m_MonitorThread);

        // RX エントリを削除します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(m_RxEntry));
    }

    nn::os::SystemEvent& WlanActionFrameMonitor::GetReceivedEvent() NN_NOEXCEPT
    {
        return m_ReceivedEvent;
    }

    int WlanActionFrameMonitor::GetScanResultCount() const NN_NOEXCEPT
    {
        return m_ScanResultCount;
    }

    bool WlanActionFrameMonitor::GetScanResult(
        L2ScanResult* pOutResult, int index) const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT_NOT_NULL(pOutResult);
        NN_SDK_ASSERT(0 <= index);
        if (0 <= index && index < m_ScanResultCount)
        {
            *pOutResult = m_ScanBuffer[index];
            return true;
        }
        else
        {
            return false;
        }
    }

    bool WlanActionFrameMonitor::GetScanResult(
        L2ScanResult* pOutResult, MacAddress bssid) const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT_NOT_NULL(pOutResult);
        NN_SDK_ASSERT_NOT_EQUAL(bssid, ZeroMacAddress);
        NN_SDK_ASSERT_NOT_EQUAL(bssid, BroadcastMacAddress);
        for (int i = 0; i < m_ScanResultCount; ++i)
        {
            if (bssid == m_ScanBuffer[i].bssid)
            {
                *pOutResult = m_ScanBuffer[i];
                return true;
            }
        }
        return false;
    }

    void WlanActionFrameMonitor::Merge(const void* wlanScanBuffer) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT_NOT_NULL(m_ScanBuffer);
        NN_SDK_ASSERT_NOT_NULL(wlanScanBuffer);

        // L2 のスキャン結果を 1 つずつ順番に加工します。
        nn::wlan::BeaconScanResultReader beaconScanResultReader(wlanScanBuffer);
        int scanResultCount = static_cast<int>(beaconScanResultReader.GetCount());
        for (int i = 0; i < scanResultCount; ++i)
        {
            // WLAN からスキャン結果を取得します。
            auto  bssReader = beaconScanResultReader.GetNextDescription();
            auto  bssid     = MakeMacAddress(bssReader.GetBssid().GetMacAddressData());;
            auto  ssid      = MakeSsid(bssReader.GetSsid().GetSsidData(),
                                       bssReader.GetSsid().GetLength());
            auto  rssi      = static_cast<int8_t>(bssReader.GetRssi());
            auto  channel   = bssReader.GetChannel();

            // Action Frame を受信できた LDN の AP に対応するビーコンは無視します。
            int j;
            for (j = 0; j < m_ScanResultCount && m_ScanBuffer[j].bssid != bssid; ++j)
            {
            }
            if (j < m_ScanResultCount)
            {
                continue;
            }

            // それ以外のビーコンは末尾に追加します。
            for (j = m_ScanResultCount; j < m_ScanBufferCount; ++j)
            {
                if (m_ScanBuffer[j].bssid == bssid || m_ScanBuffer[j].bssid == ZeroMacAddress)
                {
                    break;
                }
            }
            if (j < m_ScanBufferCount)
            {
                m_ScanBuffer[j].bssid   = bssid;
                m_ScanBuffer[j].ssid    = ssid;
                m_ScanBuffer[j].channel = channel;
                m_ScanBuffer[j].rssi    = rssi;
                continue;
            }
        }

        // スキャン結果を前方に詰めます。
        int left  = 0;
        int right = m_ScanBufferCount - 1;
        while (left <= right)
        {
            while (left < m_ScanBufferCount && IsSupportedChannel(m_ScanBuffer[left].channel))
            {
                ++left;
            }
            while (0 <= right && !IsSupportedChannel(m_ScanBuffer[right].channel))
            {
                --right;
            }
            if (left < right)
            {
                std::swap(m_ScanBuffer[left], m_ScanBuffer[right]);
                ++left;
                --right;
            }
        }

        NN_SDK_ASSERT_MINMAX(left, 0, m_ScanBufferCount);
        m_ScanResultCount = left;
    }

    void WlanActionFrameMonitor::SortByRssi() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT_NOT_NULL(m_ScanBuffer);
        for (int i = 0; i < m_ScanResultCount - 1; ++i)
        {
            int maxIndex = i;
            for (int j = i + 1; j < m_ScanResultCount; ++j)
            {
                if (m_ScanBuffer[maxIndex].rssi < m_ScanBuffer[j].rssi)
                {
                    maxIndex = j;
                }
            }
            if (i != maxIndex)
            {
                std::swap(m_ScanBuffer[i], m_ScanBuffer[maxIndex]);
            }
        }
    }

    void WlanActionFrameMonitor::MonitorThread(void* pArg) NN_NOEXCEPT
    {
        NN_LDN_LOG_DEBUG("ActionFrameMonitorThread: started\n");
        auto& monitor = *static_cast<WlanActionFrameMonitor*>(pArg);

        // StopMonitoring() を呼ばれるまで Action Frame の受信を継続します。
        while (monitor.m_IsMonitoring)
        {
            auto &frame = *reinterpret_cast<AdvertiseActionFrame*>(monitor.m_Buffer->receive);
            auto receiveBuffer = reinterpret_cast<uint8_t*>(&frame.nintendoActionFrameHeader);
            auto receiveBufferSize = sizeof(monitor.m_Buffer->receive) - sizeof(ActionFrameHeader);

            // Action Frame の受信まで待機します。
            nn::wlan::MacAddress src;
            size_t size;
            uint16_t channel;
            int16_t rssi;
            Result result = nn::wlan::Local::GetActionFrameEx(
                &src, receiveBuffer, receiveBufferSize, &size, monitor.m_RxEntry, &channel, &rssi);
            if (nn::wlan::ResultGetFrameCancelled::Includes(result) ||
                nn::wlan::ResultInvalidState::Includes(result))
            {
                // Action Frame の受信を継続できる状態ではないので終了します。
                break;
            }
            else if (nn::wlan::ResultBufferTooShort::Includes(result))
            {
                // バッファよりも大きなデータなので受信できませんでした。
                // wlan 内部でそのフレームは削除されます。
                continue;
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            // 解釈できないデータは無視します。
            size_t receivedSize = size + sizeof(ActionFrameHeader);
            if (receivedSize < AdvertiseActionFrameSizeMin ||
                sizeof(AdvertiseActionFrame) < receivedSize ||
                frame.advertiseActionFrameHeader.offset != 0 ||
                frame.nintendoActionFrameHeader.protocol != AdvertiseProtocol)
            {
                NN_LDN_LOG_WARN("received invalid action frame\n");
                continue;
            }

            // 対象のネットワークから受信した Action Frame でなければ無視します。
            // bssid に FF:FF:FF:FF:FF:FF を与えた場合はあらゆる入力を受け入れます。
            const auto bssid = MakeMacAddress(src.GetMacAddressData());
            if (monitor.m_Bssid != BroadcastMacAddress && monitor.m_Bssid != bssid)
            {
                continue;
            }

            // 既存のスキャン結果と BSSID が一致する物がないか確認します。
            int i;
            for (i = 0; i < monitor.m_ScanResultCount; ++i)
            {
                if (monitor.m_ScanBuffer[i].bssid == bssid)
                {
                    break;
                }
            }

            // スキャン結果を保存し、イベントをシグナルします。
            if (i < monitor.m_ScanResultCount)
            {
                // 既存のスキャン結果を最新の受信データで上書きします。
                std::lock_guard<nn::os::Mutex> lock(monitor.m_Mutex);
                auto& out = monitor.m_ScanBuffer[i];
                out.channel = static_cast<int16_t>(channel);
                out.rssi = static_cast<int8_t>(rssi);
                out.dataSize = static_cast<uint16_t>(receivedSize - AdvertiseActionFrameSizeMin);
                std::memcpy(out.data, frame.advertise, out.dataSize);
            }
            else
            {
                // 新しい BSSID なので末尾に追加します。
                std::lock_guard<nn::os::Mutex> lock(monitor.m_Mutex);
                auto& out = monitor.m_ScanBuffer[monitor.m_ScanResultCount];
                out.bssid = bssid;
                out.channel = static_cast<int16_t>(channel);
                out.rssi = static_cast<int8_t>(rssi);
                out.dataSize = static_cast<uint16_t>(receivedSize - AdvertiseActionFrameSizeMin);
                std::memcpy(out.data, frame.advertise, out.dataSize);
                ++monitor.m_ScanResultCount;
            }

            // Action Frame の受信を通知するためシグナルします。
            monitor.m_ReceivedEvent.Signal();
        }
        NN_LDN_LOG_DEBUG("ActionFrameMonitorThread: finished\n");
    }

}}}} // namespace nn::ldn::detail::impl

namespace nn { namespace ldn { namespace detail
{
    size_t Wlan::GetRequiredBufferSize() NN_NOEXCEPT
    {
        return sizeof(impl::WlanBuffer);
    }

    Wlan::Wlan(void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_WlanBuffer(static_cast<impl::WlanBuffer*>(buffer)),
          m_Monitor(&m_WlanBuffer->actionFrameMonitor),
          m_Status(L2State_None),
          m_MacAddress(ZeroMacAddress),
          m_OperationMode(OperationMode_Stable),
          m_WirelessControllerRestriction(WirelessControllerRestriction_Enabled),
          m_RxIndex(0),
          m_IsDistributing(false)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::os::ThreadStackAlignment);
        NN_SDK_ASSERT(GetRequiredBufferSize() <= bufferSize);

        // WLAN ライブラリを初期化し、状態変化を通知するイベントを取得します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
        nn::wlan::Local::GetConnectionEvent(m_StateChangeEvent.GetBase());

        // BTM ライブラリを初期化します。
        InitializeBtm();

        // MAC アドレスを取得します。
        nn::wlan::MacAddress macAddress;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));
        std::memcpy(&m_MacAddress, macAddress.GetMacAddressData(), sizeof(MacAddress));
    }

    Wlan::~Wlan() NN_NOEXCEPT
    {
        // ネットワークから離脱します。
        if (m_Status == L2State_AccessPoint || m_Status == L2State_AccessPointCreated)
        {
            CloseAccessPoint();
        }
        else if (m_Status == L2State_Station || m_Status == L2State_StationConnected)
        {
            CloseStation();
        }

        // WLAN ライブラリを終了します。
        nn::wlan::FinalizeLocalManager();
    }

    nn::os::SystemEvent& Wlan::GetStateChangeEvent() NN_NOEXCEPT
    {
        return m_StateChangeEvent;
    }

    nn::os::SystemEvent& Wlan::GetBeaconReceivedEvent() NN_NOEXCEPT
    {
        return m_Monitor.GetReceivedEvent();
    }

    Result Wlan::GetNetworkProfile(NetworkProfile* pOutProfile) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutProfile);

        // 現在の状態を確認します。
        if (m_Status != L2State_AccessPointCreated && m_Status != L2State_StationConnected)
        {
            return ResultInvalidState();
        }

        // WLAN から現在の接続状態を取得します。
        nn::wlan::ConnectionStatus status;
        Result result = nn::wlan::Local::GetConnectionStatus(&status);
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // ネットワークに接続できていない場合は失敗します。
        if (!IsSupportedChannel(status.channel))
        {
            return ResultL2NotSupportedChannel();
        }

        // BSSID, SSID, 無線チャンネルを LDN ライブラリの形式に合わせて変換します。
        pOutProfile->bssid   = MakeMacAddress(status.bssid.GetMacAddressData());
        pOutProfile->ssid    = MakeSsid(status.ssid.GetSsidData(), status.ssid.GetLength());
        pOutProfile->channel = static_cast<int16_t>(status.channel);
        NN_RESULT_SUCCESS;
    }

    void Wlan::GetNetworkInterfaceProfile(NetworkInterfaceProfile* pOutProfile) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutProfile);
        std::memset(pOutProfile, 0, sizeof(NetworkInterfaceProfile));
        pOutProfile->mtu = 1500;
        pOutProfile->capability = NetworkInterfaceCapability_HardwareEncryption |
                                  NetworkInterfaceCapability_AutoChannelSelection;
        pOutProfile->nodeCountMax = nn::wlan::ConnectableClientsCountMax + 1;
        pOutProfile->type = NetworkInterfaceType_Wireless80211;
        pOutProfile->physicalAddress = m_MacAddress;
        pOutProfile->beaconDataSizeMax = AdvertiseActionFramePayloadSizeMax;
        pOutProfile->supportedChannelCount = static_cast<int8_t>(SupportedChannelCount);
        pOutProfile->autoSelectableChannelCount = static_cast<int8_t>(AutoSelectableChannelCount);
        for (int i = 0; i < SupportedChannelCount; ++i)
        {
            pOutProfile->supportedChannels[i] =
                static_cast<int16_t>(SupportedChannels[i]);
        }
        for (int i = 0; i < AutoSelectableChannelCount; ++i)
        {
            pOutProfile->autoSelectableChannels[i] =
                static_cast<int16_t>(AutoSelectableChannels[i]);
        }
        std::strncpy(pOutProfile->name, "wl0", NetworkInterfaceNameLengthMax);
    }

    Result Wlan::Scan(
        L2ScanResult* outBuffer, int* pOutCount, int bufferCount, int channel) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outBuffer);
        NN_SDK_ASSERT_NOT_NULL(pOutCount);
        NN_SDK_ASSERT(0 < bufferCount);
        NN_SDK_ASSERT(channel == AutoChannel || IsSupportedChannel(channel));

        // 必要な項目だけ設定するため ScanParameters は 0 初期化します。
        nn::wlan::ScanParameters param;
        std::memset(&param, 0, sizeof(param));
        *pOutCount = 0;

        // 無線チャンネルの自動設定です。
        param.scanType = nn::wlan::ScanType_Passive;
        if (channel == AutoChannel)
        {
            for (int i = 0; i < AutoSelectableChannelCount; ++i)
            {
                param.channelList[i] = AutoSelectableChannels[i];
            }
            param.channelCount = static_cast<uint8_t>(AutoSelectableChannelCount);
        }
        else if (IsSupportedChannel(channel))
        {
            param.channelList[0] = static_cast<uint16_t>(channel);
            param.channelCount = 1;
        }
        else
        {
            return ResultBadRequest();
        }

        // 全ての AP をスキャン対象にします。
        param.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

        // チャンネル毎のスキャン時間設定です。
        param.channelScanTime = ChannelScanTime;
        param.homeChannelTime = HomeChannelTime;

        // ステーションではアクションフレームの監視を一旦停止します。
        if (m_Status == L2State_StationConnected)
        {
            m_Monitor.StopMonitoring();
        }

        // アクションフレームの監視を開始します。
        m_Monitor.StartMonitoring(outBuffer, bufferCount);

        // スキャンを実行します。
        Result result = nn::wlan::Local::StartScan(
            m_WlanBuffer->scan.wlan, sizeof(m_WlanBuffer->scan.wlan), param);

        // スキャン結果と Action Frame の受信結果をマージ・ソートします。
        if (result.IsSuccess())
        {
            m_Monitor.Merge(m_WlanBuffer->scan.wlan);
            m_Monitor.SortByRssi();
            *pOutCount = m_Monitor.GetScanResultCount();
        }

        // アクションフレームの監視を停止します。
        m_Monitor.StopMonitoring();

        // ステーション用のアクションフレームの監視を再開します。
        if (m_Status == L2State_StationConnected)
        {
            m_Monitor.StartMonitoring(&m_WlanBuffer->scan.result, 1, m_Bssid);
        }

        // Result の変換です。
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_SDK_ASSERT_MINMAX(*pOutCount, 0, bufferCount);
        NN_RESULT_SUCCESS;
    }

    Result Wlan::OpenAccessPoint() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_None);

        // アクセスポイントとしての動作を開始します。
        Result result;
        if (m_OperationMode == OperationMode_HighSpeed)
        {
            result = nn::wlan::Local::OpenLcsMasterMode();
        }
        else
        {
            result = nn::wlan::Local::OpenMasterMode();
        }
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        else if (nn::wlan::ResultWlanDeviceAbnormal::Includes(result))
        {
            // TODO: SIGLO-48865 のワークアラウンドなので解決後に削除します。
            return ResultWifiOff();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        m_Status = L2State_AccessPoint;
        NN_RESULT_SUCCESS;
    }

    Result Wlan::CreateNetwork(
        const Ssid& ssid, int channel, int nodeCountMax,
        const void* key, size_t keySize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPoint);

        // 最大接続台数に問題が無いか確認します。
        NN_RESULT_DO(SetWlanMode(nodeCountMax));

        // L2 のネットワークパラメータを初期化します。
        nn::wlan::MasterBssParameters params;
        std::memset(&params, 0, sizeof(params));

        // SSID はステルスモードで運用します。
        params.hiddenSsid = true;
        params.ssid.Set(ssid.raw);

        // 無線チャンネルの設定です。 AutoChannel を指定すると自動選択です。
        if (channel == AutoChannel)
        {
            params.channel = -1;
        }
        else
        {
            NN_SDK_ASSERT(IsSupportedChannel(channel));
            params.channel = static_cast<int16_t>(channel);
        }

        // STA の KeepAlive を有効化し、切断までの無通信時間を設定します。
        params.inactivePeriod = static_cast<uint32_t>(InactivePeriodMax / 1000);
        params.autoKeepAlive  = true;

        // ビーコン間隔をミリ秒単位で設定します。
        params.beaconInterval = static_cast<uint16_t>(BeaconInterval);

        // 実機では RateSet は無関係ですが機器連携に備えて適当なパラメータを設定します。
        params.supportedRates = nn::wlan::RateSetLegacy_11gMask;
        params.basicRates     = nn::wlan::RateSetLegacy_11bMask;

        // セキュリティの設定です。
        nn::wlan::Security& security = params.security;
        if (key == nullptr || keySize == 0U)
        {
            security.privacyMode = nn::wlan::SecurityMode_Open;
            security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
        }
        else
        {
            NN_SDK_ASSERT_MINMAX(keySize, KeySizeMin, KeySizeMax);
            security.privacyMode = nn::wlan::SecurityMode_StaticAes;
            security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
            std::memcpy(security.key, key, std::min(KeySizeMax, keySize));
        }

        // L2 ネットワークを生成します。
        Result result = nn::wlan::Local::CreateBss(params);
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            SetWlanMode(0);
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // データ通信で使用する RX エントリを生成します。
        m_RxIndex = CreateRxEntry();

        m_Status = L2State_AccessPointCreated;
        NN_RESULT_SUCCESS;
    }


    Result Wlan::SetBeaconData(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(dataSize <= AdvertiseActionFramePayloadSizeMax);
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPointCreated);

        // 既に配信中の場合には一旦配信を停止します。
        Result result;
        if (m_IsDistributing)
        {
            m_IsDistributing = false;
            result = nn::wlan::Local::CancelActionFrameWithBeacon();
            if (nn::wlan::ResultInvalidState::Includes(result))
            {
                return ResultL2InvalidState();
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        // データサイズが 0 の場合には配信しません。
        if (data == nullptr || dataSize == 0)
        {
            NN_RESULT_SUCCESS;
        }

        // 送信バッファを 0 で初期化します。
        auto& frame = *reinterpret_cast<AdvertiseActionFrame*>(m_WlanBuffer->send);
        std::memset(&frame, 0, sizeof(frame));

        // アクションフレームの Header を設定します。
        frame.actionFrameHeader.dst = BroadcastMacAddress;
        frame.actionFrameHeader.src = m_MacAddress;
        frame.actionFrameHeader.bssid = BroadcastMacAddress;
        frame.nintendoActionFrameHeader.category = ActionFrameCategory_VendorSpecific;
        frame.nintendoActionFrameHeader.oui = NintendoOui;
        frame.nintendoActionFrameHeader.subType = ActionFrameSubType_Advertise;
        frame.nintendoActionFrameHeader.protocol = MakeProtocolId(ProtocolCode_Advertise);

        // アクションフレームのペイロードを設定します。
        size_t copySize = std::min(dataSize, sizeof(frame.advertise));
        std::memcpy(frame.advertise, data, copySize);

        // フレームのサイズを計算します。
        size_t frameSize = sizeof(ActionFrameHeader) +
                           sizeof(NintendoActionFrameHeader) +
                           sizeof(AdvertiseActionFrameHeader) +
                           copySize;

        // WLAN での送信に適した形式に変換します。
        auto wlanData     = reinterpret_cast<const uint8_t*>(&frame.nintendoActionFrameHeader);
        auto wlanDataSize = static_cast<uint32_t>(frameSize - sizeof(ActionFrameHeader));

        // Advertise を ActionFrame に載せて周期的に配信します。
        result = nn::wlan::Local::SetActionFrameWithBeacon(wlanData, wlanDataSize);
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        m_IsDistributing = true;
        NN_RESULT_SUCCESS;
    }

    Result Wlan::Reject(MacAddress address) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPointCreated);

        // ステーションを切断します。
        nn::wlan::DisconnectClient disconnect;
        disconnect.clientMacAddress = nn::wlan::MacAddress(address.raw);
        disconnect.reasonCode = nn::wlan::Dot11ReasonCode_AuthInvalid;
        Result result = nn::wlan::Local::Disconnect(
            nn::wlan::LocalCommunicationMode_Master, &disconnect);

        // LDN の Result に変換します。
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        else if (nn::wlan::ResultNoClients::Includes(result) ||
                 nn::wlan::ResultClientNotFound::Includes(result))
        {
            return ResultNodeNotFound();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        NN_RESULT_SUCCESS;
    }

    Result Wlan::GetStations(
        L2StationInfo* outStations, int* pOutCount, int bufferCount) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPointCreated);
        NN_SDK_ASSERT_NOT_NULL(outStations);
        NN_SDK_ASSERT_NOT_NULL(pOutCount);
        NN_SDK_ASSERT(0 < bufferCount);

        // 出力を初期化します。
        *pOutCount = 0;
        std::memset(outStations, 0, sizeof(L2StationInfo) * bufferCount);

        // WLAN からステーションの状態を取得します。
        nn::wlan::ClientStatus clients[nn::wlan::ConnectableClientsCountMax];
        Bit32 map;
        Result result = nn::wlan::Local::GetClientStatus(clients, &map);
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // ステーションの状態を出力します。
        int count = 0;
        for (uint8_t i = 0; i < nn::wlan::ConnectableClientsCountMax && count < bufferCount; ++i)
        {
            auto& station = outStations[count];
            auto  client = clients[i];
            station.aid = static_cast<int8_t>(i + 1);
            if (client.state == nn::wlan::ConnectionState_Connected)
            {
                // NN_SDK_ASSERT_RANGE(client.rssi, RssiMin, 0);
                station.status = static_cast<Bit8>(L2StationState_Connected);
                station.rssi = static_cast<int8_t>(client.rssi);
                station.macAddress = MakeMacAddress(client.clientMacAddress.GetMacAddressData());
                station.connectedAt = client.updateTick.GetInt64Value();
            }
            else
            {
                station.status = static_cast<Bit8>(L2StationState_Disconnected);
                station.rssi = RssiMin;
            }
            ++count;
        }
        *pOutCount = count;

        NN_SDK_ASSERT_MINMAX(*pOutCount, 0, bufferCount);
        NN_RESULT_SUCCESS;
    }

    Result Wlan::DestroyNetwork() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPointCreated);

        // Action Frame の配信を停止します。
        if (m_IsDistributing)
        {
            SetBeaconData(nullptr, 0);
        }

        // RX エントリを削除します。
        DeleteRxEntry(m_RxIndex);

        // L2 ネットワークを破棄します。
        Result result = nn::wlan::Local::DestroyBss();
        NN_ABORT_UNLESS_RESULT_SUCCESS(SetWlanMode(0));
        m_Status = L2State_AccessPoint;

        // LDN の Result に変換します。
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_RESULT_SUCCESS;
    }

    Result Wlan::CloseAccessPoint() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_AccessPoint || m_Status == L2State_AccessPointCreated);

        // ネットワークを破棄します。
        if (m_Status == L2State_AccessPointCreated)
        {
            DestroyNetwork();
        }
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPoint);

        // アクセスポイントとしての動作を停止します。
        Result result;
        if (m_OperationMode == OperationMode_HighSpeed)
        {
            result = nn::wlan::Local::CloseLcsMasterMode();
        }
        else
        {
            result = nn::wlan::Local::CloseMasterMode();
        }
        m_Status = L2State_None;
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_RESULT_SUCCESS;
    }

    Result Wlan::OpenStation() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_None);

        // ステーションとしての動作を開始します。
        Result result;
        if (m_OperationMode == OperationMode_HighSpeed)
        {
            result = nn::wlan::Local::OpenLcsClientMode();
        }
        else
        {
            result = nn::wlan::Local::OpenClientMode();
        }
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        else if (nn::wlan::ResultWlanDeviceAbnormal::Includes(result))
        {
            // TODO: SIGLO-48865 のワークアラウンドなので解決後に削除します。
            return ResultWifiOff();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        m_Status = L2State_Station;
        NN_RESULT_SUCCESS;
    }

    Result Wlan::Connect(
        const Ssid& ssid, int channel, int nodeCountMax,
        const void* key, size_t keySize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_Station);
        NN_SDK_ASSERT(IsSupportedChannel(channel));
        NN_SDK_ASSERT_NOT_NULL(key);
        NN_UNUSED(KeySizeMin);

        // 接続不可能な無線チャンネルを指定された場合には失敗します。
        if (!IsSupportedChannel(channel))
        {
            return ResultNetworkNotFound();
        }

        // 最大接続台数に問題が無いか確認します。
        NN_RESULT_DO(SetWlanMode(nodeCountMax));

        // ネットワークに接続します。
        Result result = ConnectWithRetry(&m_Bssid, ssid, channel, key, keySize, 3);
        if (result.IsFailure())
        {
            NN_RESULT_DO(SetWlanMode(0));
            return result;
        }

        // ビーコンの監視を開始します。
        m_Monitor.StartMonitoring(&m_WlanBuffer->scan.result, 1, m_Bssid);

        // RX エントリを生成します。
        m_RxIndex = CreateRxEntry();
        m_Status = L2State_StationConnected;
        NN_RESULT_SUCCESS;
    }

    bool Wlan::GetBeaconData(L2ScanResult* pOutBeacon) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_StationConnected);
        NN_SDK_ASSERT_NOT_NULL(pOutBeacon);
        return m_Monitor.GetScanResult(pOutBeacon, m_Bssid);
    }

    Result Wlan::Disconnect() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_StationConnected);

        // ビーコンの監視を停止します。
        m_Monitor.StopMonitoring();

        // RX エントリを削除します。
        DeleteRxEntry(m_RxIndex);

        // L2 ネットワークから切断します。
        Result result;
        result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator);
        NN_ABORT_UNLESS_RESULT_SUCCESS(SetWlanMode(0));
        m_Status = L2State_Station;

        // WLAN のエラーを LDN のエラーに変換します。
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_RESULT_SUCCESS;
    }

    Result Wlan::GetConnectionStatus(L2StationInfo* pOut) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_Station || m_Status == L2State_StationConnected);
        NN_SDK_ASSERT_NOT_NULL(pOut);

        // 0 埋めします。
        std::memset(pOut, 0, sizeof(L2StationInfo));

        // WLAN から現在の接続状態を取得します。
        nn::wlan::ConnectionStatus wlanStatus;
        Result result = nn::wlan::Local::GetConnectionStatus(&wlanStatus);
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            pOut->status = L2StationState_Disconnected;
            pOut->rssi = RssiMin;
            pOut->disconnectReason = DisconnectReason_Unknown;
            pOut->macAddress = m_MacAddress;
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // LDN ライブラリ用のデータ構造に変換します。
        if (wlanStatus.state == nn::wlan::ConnectionState_Connected)
        {
            pOut->status = L2StationState_Connected;

            // 通信品質を取得します。
            int32_t rssi;
            if (nn::wlan::Local::GetRssi(&rssi).IsSuccess())
            {
                NN_SDK_ASSERT_MINMAX(rssi, RssiMin, 0);
                pOut->rssi = static_cast<int8_t>(rssi);
            }
            else
            {
                rssi = RssiMin;
            }
        }
        else
        {
            pOut->status = L2StationState_Disconnected;
            pOut->rssi = RssiMin;

            // 切断理由を取得します。
            switch (wlanStatus.cause)
            {
            case nn::wlan::CauseOfInfo_BeaconLost:
                pOut->disconnectReason = L2DisconnectReason_BeaconLost;
                break;
            case nn::wlan::CauseOfInfo_DisconnectReq:
                pOut->disconnectReason = L2DisconnectReason_DisconnectCommand;
                break;
            case nn::wlan::CauseOfInfo_RecieveDisconnect:
                pOut->disconnectReason = L2DisconnectReason_DisconnectedByAccessPoint;
                break;
            default:
                pOut->disconnectReason = L2DisconnectReason_Unknown;
                break;
            }
        }
        pOut->macAddress = m_MacAddress;
        NN_RESULT_SUCCESS;
    }

    Result Wlan::CloseStation() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_Station || m_Status == L2State_StationConnected);

        // ネットワークから切断します。
        if (m_Status == L2State_StationConnected)
        {
            Disconnect();
        }

        // ステーションとしての動作を停止します。
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_Station);
        Result result;
        if (m_OperationMode == OperationMode_HighSpeed)
        {
            result = nn::wlan::Local::CloseLcsClientMode();
        }
        else
        {
            result = nn::wlan::Local::CloseClientMode();
        }
        m_Status = L2State_None;
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_RESULT_SUCCESS;
    }

    Result Wlan::Send(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_AccessPointCreated ||
                      m_Status == L2State_StationConnected);
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT(dataSize <= sizeof(NintendoEthernetFrame));

        // 指定されたフレームを送信します。
        Result result = nn::wlan::Local::PutFrameRaw(
            reinterpret_cast<const uint8_t*>(data), static_cast<uint32_t>(dataSize));
        if (nn::wlan::ResultGetFrameCancelled::Includes(result))
        {
            NN_LDN_LOG_DEBUG("failed to send: cancelled\n");
            return ResultCancelled();
        }
        else if (nn::wlan::ResultInvalidState::Includes(result))
        {
            NN_LDN_LOG_WARN("failed to send: invalid state\n");
            return ResultL2InvalidState();
        }
        else if (nn::wlan::ResultTxQueueIsFull::Includes(result))
        {
            NN_LDN_LOG_WARN("failed to send: tx queue is full\n");
            return ResultTxQueueIsFull();
        }
        else if (nn::wlan::ResultCommandFailure::Includes(result))
        {
            NN_LDN_LOG_WARN("failed to send: command failure\n");
            return ResultSendFailed();
        }
        else if (nn::wlan::ResultNotAuthorized::Includes(result))
        {
            NN_LDN_LOG_WARN("failed to send: not authorized\n");
            return ResultSendFailed();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_RESULT_SUCCESS;
    }

    Result Wlan::Receive(
        void* buffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_AccessPointCreated ||
                      m_Status == L2State_StationConnected);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT(sizeof(EthernetHeader) <= bufferSize);

        Result result;
        do
        {
            result = nn::wlan::Local::GetFrameRaw(
                static_cast<uint8_t*>(buffer), bufferSize, pOutSize, m_RxIndex);
            if (nn::wlan::ResultGetFrameCancelled::Includes(result))
            {
                NN_LDN_LOG_DEBUG("failed to receive: cancelled\n");
                return ResultCancelled();
            }
            else if (nn::wlan::ResultInvalidState::Includes(result))
            {
                NN_LDN_LOG_WARN("failed to receive: invalid state\n");
                return ResultL2InvalidState();
            }
            else if (nn::wlan::ResultNotAuthorized::Includes(result))
            {
                NN_LDN_LOG_WARN("failed to receive: not authorized\n");
                return ResultL2InvalidState();
            }
            else if (nn::wlan::ResultBufferTooShort::Includes(result))
            {
                NN_LDN_LOG_WARN("failed to receive: insufficient buffer\n");
                NN_LDN_ASSERT_RESULT_SUCCESS(result);
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        } while (result.IsFailure());
        NN_RESULT_SUCCESS;
    }

    Result Wlan::Cancel() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_AccessPointCreated ||
                      m_Status == L2State_StationConnected);

        // 受信処理をキャンセルします。
        Result result = nn::wlan::Local::CancelGetFrame(m_RxIndex);
        if (nn::wlan::ResultInvalidState::Includes(result))
        {
            return ResultL2InvalidState();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        NN_RESULT_SUCCESS;
    }

    L2State Wlan::GetState() const NN_NOEXCEPT
    {
        return m_Status;
    }

    void Wlan::SetOperationMode(OperationMode mode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_None);
        m_OperationMode = mode;
    }

    void Wlan::SetWirelessControllerRestriction(
        WirelessControllerRestriction restriction) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_None);
        m_WirelessControllerRestriction = restriction;
    }

    int Wlan::ConvertRssiToLinkLevel(int rssi) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(rssi <= 0);
        return static_cast<int>(nn::wlan::Local::ConvertRssiToLinkLevel(rssi));
    }

    Result Wlan::SetWlanMode(int nodeCountMax) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_MINMAX(nodeCountMax, 0, NodeCountMax);

        // 最大接続ノード数から wlanmMode を選択します。
        nn::btm::WlanMode wlanMode;
        if (nodeCountMax == 0)
        {
            wlanMode = nn::btm::WlanMode_None;
        }
        else if (m_WirelessControllerRestriction == WirelessControllerRestriction_Disabled)
        {
            wlanMode = nn::btm::WlanMode_User8;
        }
        else if (4 < nodeCountMax)
        {
            wlanMode = nn::btm::WlanMode_Local8;
        }
        else
        {
            wlanMode = nn::btm::WlanMode_Local4;
        }
        NN_LDN_LOG_DEBUG("set wlan mode to %d\n", wlanMode);

        // WlanMode の変更を要求します。
        Result result;
        const int retryCountMax = 384;
        for (int retryCount = 0; retryCount < retryCountMax; ++retryCount)
        {
            result = nn::btm::SetWlanMode(wlanMode);
            if (nn::btm::ResultInvalidUsecase::Includes(result))
            {
                // コントローラの接続台数が多いため、ローカル通信を利用できません。
                return ResultNodeCountLimitation();
            }
            else if (nn::btm::ResultBusy::Includes(result))
            {
                // 少し待ってからリトライします。
                NN_LDN_LOG_WARN("btm is busy.\n");
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }
            else
            {
                break;
            }
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // 状態変更が完了するまで待機します。
        nn::btm::DeviceConditionList deviceCondition;
        do
        {
            const auto timeout = nn::TimeSpan::FromMilliSeconds(32000);
            if (!g_BtmEvent.TimedWait(timeout))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(ResultBtmTimeout());
            }
            nn::btm::GetConnectedDeviceCondition(&deviceCondition);
        } while (deviceCondition.wlanMode != wlanMode);

        NN_RESULT_SUCCESS;
    }

    uint32_t Wlan::CreateRxEntry() const NN_NOEXCEPT
    {
        // OUI Extended Ethertype を RX エントリに登録します。
        // 上記以外の Ethertype のフレームを送受信することは想定していません。
        uint32_t rxIndex;
        Bit16 ethertype = (OuiExtendedEthertype.raw[0] << 8) | OuiExtendedEthertype.raw[1];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(
            &rxIndex, &ethertype, 1, static_cast<uint32_t>(ReceiveBufferCountMax)));

        // LDN 関連のフレームだけを受信するように設定します。
        nn::wlan::ReceivedDataMatchInfo match;
        std::memcpy(&match.matchData[0], &NintendoOui, sizeof(Oui));
        match.matchData[sizeof(Oui)] = static_cast<Bit8>(ProtocolModule_Ldn);
        match.matchLength = sizeof(Oui) + 1;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxIndex, match));
        return rxIndex;
    }

    void Wlan::DeleteRxEntry(uint32_t rxIndex) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxIndex));
    }

    Result Wlan::ConnectWithRetry(
        MacAddress* pOutBssid, const Ssid& ssid, int channel, const void* key,
        size_t keySize, int retryCountMax) NN_NOEXCEPT
    {
        // 接続先を設定します。
        nn::wlan::Ssid l2Ssid(ssid.raw);
        nn::wlan::MacAddress l2Bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        uint16_t l2Channel = static_cast<uint16_t>(channel);

        // 通信路の暗号化の設定です。
        nn::wlan::Security security;
        std::memset(&security, 0, sizeof(nn::wlan::Security));
        if (key == nullptr || keySize == 0U)
        {
            security.privacyMode = nn::wlan::SecurityMode_Open;
            security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
        }
        else
        {
            NN_SDK_ASSERT_MINMAX(keySize, KeySizeMin, KeySizeMax);
            security.privacyMode = nn::wlan::SecurityMode_StaticAes;
            security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
            std::memcpy(security.key, key, std::min(KeySizeMax, keySize));
        }

        // 自動接続切断を有効化し、ビーコン通知イベントは使いません。
        bool isEnabledKeepAlive = true;
        auto indication = nn::wlan::BeaconIndication_Disable;

        // 接続の成功率を上げるため、リトライします。
        int retryCount = 0;
        nn::wlan::ConnectionStatus status;
        do
        {
            // リトライ回数の上限に達した場合は失敗です。
            if (retryCountMax <= retryCount)
            {
                NN_LDN_LOG_INFO("connection failed: network not found\n");
                return ResultNetworkNotFound();
            }
            ++retryCount;

            // L2 ネットワークに接続します。
            NN_LDN_LOG_DEBUG("connecting...\n");
            Result result = nn::wlan::Local::Connect(
                l2Ssid, l2Bssid, l2Channel, security, isEnabledKeepAlive, indication);
            if (nn::wlan::ResultInvalidState::Includes(result))
            {
                NN_LDN_LOG_WARN("connection failed: invalid state\n");
                return ResultL2InvalidState();
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            // 接続処理の成否を取得します。
            result = nn::wlan::Local::GetConnectionStatus(&status);
            if (nn::wlan::ResultInvalidState::Includes(result))
            {
                NN_LDN_LOG_WARN("connection failed: cannot get connection status\n");
                return ResultL2InvalidState();
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        } while (status.state != nn::wlan::ConnectionState_Connected);

        *pOutBssid = MakeMacAddress(status.bssid.GetMacAddressData());
        NN_RESULT_SUCCESS;
    }

}}} // namespace nn::ldn::detail
