﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/ldn/detail/Advertise/ldn_Aes128CtrSha256Advertise.h>

namespace nn { namespace ldn { namespace detail
{
    size_t Aes128CtrSha256AdvertiseBuilder::GetRequiredBufferSize() NN_NOEXCEPT
    {
        return sizeof(Advertise);
    }

    Aes128CtrSha256AdvertiseBuilder::Aes128CtrSha256AdvertiseBuilder(
        void* buffer, size_t bufferSize, const Bit8 (&key)[16]) NN_NOEXCEPT
        : m_Impl(AdvertiseFormat_Aes128CtrSha256, CalculateSha256),
          m_Buffer(*static_cast<Advertise*>(buffer))
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, NN_ALIGNOF(Advertise));
        NN_SDK_ASSERT(GetRequiredBufferSize() <= bufferSize);
        NN_SDK_ASSERT_NOT_NULL(key);
        NN_UNUSED(bufferSize);
        std::memcpy(m_AesKey, key, sizeof(key));
    }

    Aes128CtrSha256AdvertiseBuilder::~Aes128CtrSha256AdvertiseBuilder() NN_NOEXCEPT
    {
        // 一応メモリ上から鍵を消しておきます。
        std::memset(m_AesKey, 0, sizeof(m_AesKey));
    }

    void Aes128CtrSha256AdvertiseBuilder::SetNetworkId(const NetworkId& networkId) NN_NOEXCEPT
    {
        m_Impl.SetNetworkId(networkId);
    }

    void Aes128CtrSha256AdvertiseBuilder::SetVersion(Version version) NN_NOEXCEPT
    {
        m_Impl.SetVersion(version);
    }

    void Aes128CtrSha256AdvertiseBuilder::Build(
        void* pOutAdvertise, size_t* pOutSize, size_t bufferSize,
        const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutAdvertise);
        NN_SDK_ASSERT_ALIGNED(pOutAdvertise, NN_ALIGNOF(Advertise));
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT(AdvertiseSizeMin + dataSize <= bufferSize);

        // 平文のアドバータイズを構築します。
        size_t advertiseSize;
        m_Impl.Build(&m_Buffer, &advertiseSize, sizeof(m_Buffer), data, dataSize);
        const auto& plainAdvertise = m_Buffer;

        // ヘッダ部分をコピーします。
        auto& encryptedAdvertise = *static_cast<Advertise*>(pOutAdvertise);
        encryptedAdvertise.header = plainAdvertise.header;

        // カウンタを暗号化の IV として使用します。
        Bit8 iv[16];
        std::memset(iv, 0, sizeof(iv));
        std::memcpy(iv, &plainAdvertise.header.counter, sizeof(plainAdvertise.header.counter));

        // 署名とペイロード部分を暗号化します。
        const size_t bodySize = advertiseSize - AdvertiseSizeMin;
        const size_t encryptSize = AdvertiseDigestSize + bodySize;
        const size_t ciphertextLength = nn::crypto::EncryptAes128Ctr(
            &encryptedAdvertise.digest, bufferSize - AdvertiseHeaderSize,
            m_AesKey, sizeof(m_AesKey), iv, sizeof(iv),
            plainAdvertise.digest, encryptSize);

        // 暗号化後のペイロードサイズが変化することは想定外です。
        NN_SDK_ASSERT_EQUAL(ciphertextLength, encryptSize);
        NN_UNUSED(ciphertextLength);
        *pOutSize = advertiseSize;
    }

    size_t AesCtr128Sha256AdvertiseParser::GetRequiredBufferSize() NN_NOEXCEPT
    {
        return  impl::PlainAdvertiseParserImpl::GetRequiredBufferSize() + sizeof(Advertise);
    }

    AesCtr128Sha256AdvertiseParser::AesCtr128Sha256AdvertiseParser(
        void* buffer, size_t bufferSize, const Bit8 (&key)[16]) NN_NOEXCEPT
        : m_Impl(buffer, impl::PlainAdvertiseParserImpl::GetRequiredBufferSize(),
            AdvertiseFormat_Aes128CtrSha256, CalculateSha256),
          m_DecryptedAdvertise(*reinterpret_cast<Advertise*>(
            static_cast<Bit8*>(buffer) + impl::PlainAdvertiseParserImpl::GetRequiredBufferSize()))
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(&m_DecryptedAdvertise, NN_ALIGNOF(Advertise));
        NN_SDK_ASSERT(GetRequiredBufferSize() <= bufferSize);
        NN_SDK_ASSERT_NOT_NULL(key);
        NN_UNUSED(bufferSize);
        std::memcpy(m_AesKey, key, sizeof(key));
    }

    AesCtr128Sha256AdvertiseParser::~AesCtr128Sha256AdvertiseParser() NN_NOEXCEPT
    {
        // 一応メモリ上から鍵を消しておきます。
        std::memset(m_AesKey, 0, sizeof(m_AesKey));
    }

    AdvertiseParserResult AesCtr128Sha256AdvertiseParser::Parse(
        const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(data);

        // 処理負荷を抑えるため復号前に前回からの差分の有無を確認します。
        // ヘッダから不正な Advertise とみなされたデータもここで除外されます。
        auto result = m_Impl.VerifyHeader(data, dataSize);
        if (result != AdvertiseParserResult_Success)
        {
            return result;
        }

        // ヘッダ部分をコピーします。
        auto& encryptedAdvertise = *static_cast<const Advertise*>(data);
        auto& header = encryptedAdvertise.header;
        std::memcpy(&m_DecryptedAdvertise, &header, sizeof(AdvertiseHeader));

        // 復号化に使用する IV を生成します。
        Bit8 iv[16];
        std::memset(iv, 0, sizeof(iv));
        std::memcpy(iv, &header.counter, sizeof(header.counter));

        // 復号化します。
        const Bit8*  cipherText  = encryptedAdvertise.digest;
        const size_t cipherSize  = dataSize - AdvertiseHeaderSize;
        const size_t bufferSize  = AdvertiseDigestSize + AdvertiseBodySizeMax;
        Bit8* plainText = m_DecryptedAdvertise.digest;
        const size_t plainSize = nn::crypto::DecryptAes128Ctr(
            plainText, bufferSize,
            m_AesKey, sizeof(m_AesKey), iv, sizeof(iv),
            cipherText, cipherSize);
        NN_SDK_ASSERT_EQUAL(plainSize, cipherSize);
        NN_UNUSED(plainSize);

        // 署名を検証します。
        result = m_Impl.VerifySign(&m_DecryptedAdvertise, dataSize);
        if (result != AdvertiseParserResult_Success)
        {
            return result;
        }

        // アドバータイズを解析します。
        m_Impl.Parse(&m_DecryptedAdvertise, dataSize);
        return AdvertiseParserResult_Success;
    }

    const NetworkId AesCtr128Sha256AdvertiseParser::GetNetworkId() const NN_NOEXCEPT
    {
        return m_Impl.GetNetworkId();
    }

    const Version AesCtr128Sha256AdvertiseParser::GetVersion() const NN_NOEXCEPT
    {
        return m_Impl.GetVersion();
    }

    size_t AesCtr128Sha256AdvertiseParser::GetDataSize() const NN_NOEXCEPT
    {
        return m_Impl.GetDataSize();
    }

    const void* AesCtr128Sha256AdvertiseParser::GetData() const NN_NOEXCEPT
    {
        return m_Impl.GetData();
    }

}}} // namespace nn::ldn::detail
