﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/ldn/detail/Advertise/ldn_AdvertiseDirector.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/Utility/ldn_Math.h>
#include <nn/ldn/detail/Utility/ldn_ReverseByteOrder.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    enum AdvertiseNodeState
    {
        //! 接続済です。
        AdvertiseNodeState_Connected = 1 << 0,
    };

    struct AdvertiseNodeData
    {
        //! アクセスポイントから割り当てられた IPv4 アドレスです。
        Ipv4Address         ipv4Address;

        //! MAC アドレスです。
        MacAddress          macAddress;

        //! ノードの状態です。
        Bit8                status;
        NN_PADDING1;

        //! ユーザ名です。
        char                userName[UserNameBytesMax];

        //! ローカル通信バージョンです。
        int16_t             localCommunicationVersion;

        //  予約領域です。
        Bit8                _reserved[10];
    };
    NN_STATIC_ASSERT(sizeof(AdvertiseNodeData) == 56);

    struct AdvertiseBody
    {
        //! 内部処理用のパラメータです。
        Bit8                serverRandom[RandomSize];

        //! アクセスポイントの @ref SecurityMode です。
        Bit16               securityMode;

        //! ステーションの接続受付ポリシー (@ref AcceptPolicy) です。
        Bit8                stationAcceptPolicy;
        NN_PADDING3;

        //! ネットワークに参加可能なノードの最大台数です。アクセスポイントを含みます。
        int8_t              nodeCountMax;

        //! 現時点でネットワークに接続しているノードの台数です。
        int8_t              nodeCount;

        //! ネットワークに参加しているノードの情報です。
        AdvertiseNodeData   nodes[NodeCountMax];
        NN_PADDING2;

        //! スキャン結果に含まれる任意データのバイトサイズです。
        uint16_t            advertiseDataSize;

        //! スキャン結果に含まれる任意データです。
        Bit8                advertiseData[AdvertiseDataSizeMax];

        //  予約領域です。
        Bit8                _reserved[420];
    };
    NN_STATIC_ASSERT(sizeof(AdvertiseBody) == 1280);

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    void CreateAdvertiseBody(
        void* buffer, size_t* pOutSize, size_t bufferSize, const LdnNetworkInfo& ldn) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, NN_ALIGNOF(AdvertiseBody));
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT(sizeof(AdvertiseBody) <= bufferSize);
        NN_UNUSED(bufferSize);
        auto& data = *static_cast<AdvertiseBody*>(buffer);
        std::memset(&data, 0, sizeof(AdvertiseBody));
        std::memcpy(data.serverRandom, ldn.serverRandom, RandomSize);
        data.securityMode = ConvertToNetworkByteOrder(ldn.securityMode);
        data.stationAcceptPolicy = ldn.stationAcceptPolicy;
        data.nodeCountMax = ldn.nodeCountMax;
        data.nodeCount = ldn.nodeCount;
        for (int i = 0; i < NodeCountMax; ++i)
        {
            const auto& src = ldn.nodes[i];
            auto& dst = data.nodes[i];
            dst.ipv4Address = ConvertToNetworkByteOrder(src.ipv4Address);
            dst.macAddress = src.macAddress;
            dst.status = static_cast<Bit8>(src.isConnected ? AdvertiseNodeState_Connected : 0);
            std::memcpy(dst.userName, src.userName, UserNameBytesMax);
            dst.localCommunicationVersion = ConvertToNetworkByteOrder(
                src.localCommunicationVersion);
        }
        data.advertiseDataSize = ConvertToNetworkByteOrder(ldn.advertiseDataSize);
        std::memcpy(data.advertiseData, ldn.advertiseData, AdvertiseDataSizeMax);
        *pOutSize = sizeof(AdvertiseBody);
    }

    bool AnalyzeAdvertiseBody(
        LdnNetworkInfo* pOut, const void* data, size_t dataSize) NN_NOEXCEPT
    {
        // Advertise データが想定と異なる場合には失敗します。
        if (dataSize < sizeof(AdvertiseBody))
        {
            return false;
        }
        const auto& body = *reinterpret_cast<const AdvertiseBody*>(data);

        // 出力全体を 0 で初期化します。
        auto& ldn = *pOut;
        std::memset(&ldn, 0, sizeof(ldn));

        // Advertise を解析します。
        std::memcpy(ldn.serverRandom, body.serverRandom, RandomSize);
        ldn.securityMode = ConvertToHostByteOrder(body.securityMode);
        ldn.stationAcceptPolicy = body.stationAcceptPolicy;
        ldn.nodeCountMax = Clamp<int8_t>(body.nodeCountMax, 1, NodeCountMax);
        ldn.nodeCount = Clamp<int8_t>(body.nodeCount, 1, ldn.nodeCountMax);
        for (int i = 0; i < NodeCountMax; ++i)
        {
            const auto& src = body.nodes[i];
            auto& dst = ldn.nodes[i];
            dst.ipv4Address.raw = ConvertToHostByteOrder(src.ipv4Address.raw);
            dst.macAddress = src.macAddress;
            dst.nodeId = static_cast<int8_t>(i);
            dst.localCommunicationVersion = ConvertToHostByteOrder(src.localCommunicationVersion);
            dst.isConnected = ((src.status & AdvertiseNodeState_Connected) != 0);
            std::memcpy(dst.userName, src.userName, UserNameBytesMax);
        }
        uint16_t advertiseDataSize = ConvertToHostByteOrder(body.advertiseDataSize);
        ldn.advertiseDataSize = std::min<uint16_t>(advertiseDataSize, AdvertiseDataSizeMax);
        std::memcpy(ldn.advertiseData, body.advertiseData, ldn.advertiseDataSize);
        return true;
    }

}}} // namespace nn::ldn::detail
