﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <new>
#include <nn/nn_SdkAbort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/err/err_Api.h>
#include <nn/err/err_ShowErrorApiForSystem.h>
#include <nn/lcs/lcs_Api.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/lcs/lcs_PrivateDebugApi.h>
#include <nn/lcs/lcs_PrivateResult.h>
#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/lcs_Types.h>
#include <nn/lcs/detail/lcs_ApplicationShareApi.h>
#include <nn/lcs/detail/lcs_Client.h>
#include <nn/lcs/detail/lcs_Context.h>
#include <nn/lcs/detail/lcs_Definition.h>
#include <nn/lcs/detail/lcs_Host.h>
#include <nn/lcs/detail/lcs_NetworkApi.h>
#include <nn/lcs/detail/lcs_NetworkInfo.h>
#include <nn/lcs/detail/lcs_Resource.h>
#include <nn/lcs/detail/lcs_Socket.h>
#include <nn/lcs/detail/lcs_State.h>
#include <nn/lcs/detail/lcs_Util.h>
#include <nn/lcs/detail/lcs_Version.h>
#include <nn/lcs/detail/Advertise/lcs_AdvertiseUtility.h>
#include <nn/lcs/detail/Debug/lcs_Assert.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/lcs/detail/Packet/lcs_Packet.h>
#include <nn/lcs/detail/Packet/lcs_PacketManager.h>
#include <nn/ldn.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace lcs { namespace
{
    nn::os::ThreadType g_MigrationThread;
    detail::Host g_Host;
    detail::Client g_Client;
    detail::LcsResources* g_Resource = nullptr;
    nn::os::Mutex g_Mutex(false);

    void ChangeState(State state, bool isForce) NN_NOEXCEPT
    {
        if (!isForce && g_Resource->state.GetState() == State_Completed)
        {
            return;
        }
        g_Resource->state.SetState(state);
        g_Resource->stateEvent.SignalEvent();
    }

    void ClearResource() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(g_Resource);

        ::std::memset(&g_Resource->migrationInfo, 0, sizeof(detail::MigrationInfo));
        g_Resource->contentsShareFailureReason = ContentsShareFailureReason_None;
        g_Resource->suspendedReason = SuspendedReason_None;
        ::std::memset(&g_Resource->currentNetworkSetting, 0, sizeof(detail::NetworkSetting));
        g_Resource->myIndex = 0;
        g_Resource->hostAddress = 0;
        ::std::memset(&g_Resource->advertiseData, 0, sizeof(detail::AdvertiseData));
        ::std::memset(&g_Resource->sessionInfo, 0, sizeof(SessionInfo));
        g_Resource->nodeInfoCount = 0;
        ::std::memset(&g_Resource->nodeInfo, 0, sizeof(NodeInfo) * NodeCountMax);
        g_Resource->requiredStorageSize = 0;
        g_Resource->shareAppInfoCount = 0;
        ::std::memset(&g_Resource->shareAppInfo, 0, sizeof(detail::ApplicationInfo) * SharableContentsCountMax);
        g_Resource->downloadTaskInfoCount = 0;
        ::std::memset(&g_Resource->downloadTaskInfo, 0, sizeof(detail::DownloadTaskInfo) * SharableContentsCountMax);
        g_Resource->contentInfoCount = 0;
        ::std::memset(&g_Resource->contentInfo, 0, sizeof(ContentsInfo) * DownloadableContentsCountMax);
        g_Resource->nodeDetailProgressCount = 0;
        ::std::memset(&g_Resource->nodeDetailProgress, 0, sizeof(detail::NodeDetailProgress) * NodeCountMax);
        ::std::memset(&g_Resource->shareComponent, 0, sizeof(detail::ShareComponent));
        ::std::memset(&g_Resource->context, 0, sizeof(detail::ResumeContext));
    }

    void NetworkErrorHandling(Result result)
    {
        if (result.IsFailure())
        {
            if (ResultWifiOff::Includes(result))
            {
                g_Resource->contentsShareFailureReason = ContentsShareFailureReason_FlightMode;
            }
            else if (ResultSleep::Includes(result))
            {
                g_Resource->contentsShareFailureReason = ContentsShareFailureReason_Sleep;
            }
            else if (ResultCommunicationError::Includes(result))
            {
                g_Resource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            ChangeState(State_ContentsShareFailed, true);
        }
    }

    Result MigrationRole() NN_NOEXCEPT
    {
        // 親交代の初期設定が終わるまでロック
        ::std::lock_guard<nn::os::Mutex> guard(g_Mutex);

        // これまでのロールを終了
        if (g_Host.IsInitialized())
        {
            NN_LCS_LOG_DEBUG("g_Host.Close()\n");
            g_Host.Close();
            NN_RESULT_DO(detail::CloseAccessPoint());
            NN_RESULT_DO(detail::OpenStation());
        }
        else if (g_Client.IsInitialized())
        {
            NN_LCS_LOG_DEBUG("g_Client.Leave\n");
            g_Client.Leave(detail::LeaveType_Permanent);
        }

        // 新しいロールの設定
        if (g_Resource->migrationInfo.role == detail::LcsRole_Host)
        {
            NN_LCS_LOG_DEBUG("g_Host.Reopen\n");
            NN_RESULT_DO(detail::CloseStation());
            NN_RESULT_DO(detail::OpenAccessPoint());
            Result result = g_Host.Reopen(g_Resource);
            if (result.IsFailure())
            {
                NN_RESULT_DO(detail::CloseAccessPoint());
                NN_RESULT_DO(detail::OpenStation());
            }
            // デバッグ用の値を設定
            SetInnerRole(InnerRole_Master);
        }
        else if (g_Resource->migrationInfo.role == detail::LcsRole_Client)
        {
            NN_LCS_LOG_DEBUG("g_Client.Rejoin\n");
            NN_RESULT_DO(g_Client.Rejoin(g_Resource));
            // デバッグ用の値を設定
            SetInnerRole(InnerRole_Slave);
        }
        NN_RESULT_SUCCESS;
    }

    void MigrationThread(void *arg) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("MigrationThread\n");

        // リクエストを止める
        nn::ns::RequestServerStopper requestServerStopper;
        detail::GetRequestServerStopper(&requestServerStopper);

        while (g_Resource->isRunning)
        {
            nn::os::WaitEvent(&(g_Resource->migrationEvent));
            nn::os::ClearEvent(&(g_Resource->migrationEvent));

            if (!g_Resource->isRunning)
            {
                break;
            }

            Result result = MigrationRole();
            if (result.IsFailure())
            {
                NetworkErrorHandling(result);
                continue;
            }

            // 新しいロールでの動作
            if (g_Host.IsInitialized())
            {
                NN_LCS_LOG_DEBUG("g_Host.WaitMigrationClients\n");
                g_Host.WaitMigrationClients();
                if (!g_Host.IsInitialized() && detail::IsAccessPointOpened())
                {
                    result = detail::CloseAccessPoint();
                    if (result.IsFailure())
                    {
                        NetworkErrorHandling(result);
                        continue;
                    }
                    result = detail::OpenStation();
                    if (result.IsFailure())
                    {
                        NetworkErrorHandling(result);
                    }
                }
            }
            else if (g_Client.IsInitialized())
            {
                NN_LCS_LOG_DEBUG("g_Client.ConnectMigrationHost\n");
                result = g_Client.ConnectMigrationHost();
                if (result.IsFailure())
                {
                    NetworkErrorHandling(result);
                }
            }
            else
            {
                continue;
            }
        }
        NN_LCS_LOG_DEBUG("MigrationThread End\n");
    }

}}} // namespace nn::lcs::<unnamed>

namespace nn { namespace lcs
{
    //! @name 共通の API
    //! @{

    Result Initialize(void* buffer, size_t bufferSize, const Config& config) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_NULL(buffer);
        NN_LCS_REQUIRES_ALIGNED(buffer, RequiredBufferAlignment);
        NN_LCS_REQUIRES_GREATER_EQUAL(bufferSize, RequiredBufferSize);
        NN_RESULT_DO(detail::OpenStation());

        g_Resource = new(buffer) detail::LcsResources;
        ::std::memset(g_Resource, 0x00, bufferSize);

        // ErrorContext を初期化
        nn::err::ErrorContext errorContex = {};
        SetErrorContext(errorContex);

        // デバッグ用の値を設定
        SetInnerRole(InnerRole_None);

        size_t size;
        config.GetName(g_Resource->userName, &size, UserNameBytesMax + 1);
        g_Resource->contentsShareFailureReason = ContentsShareFailureReason_None;
        g_Resource->suspendedReason = SuspendedReason_None;
        g_Resource->localCommunicationId = GetLocalCommunicationId();
        g_Resource->version = detail::MakeVersion(GetMajorVersion(), GetMinorVersion());
        g_Resource->monitorThreadPriority = nn::os::DefaultThreadPriority - 1;
        g_Resource->comThreadPriority = nn::os::DefaultThreadPriority;
        g_Resource->shareThreadPriority = nn::os::DefaultThreadPriority + 1;
        g_Resource->timeoutThreadPriority = nn::os::DefaultThreadPriority - 2;

        NN_LCS_LOG_DEBUG("LocalCommunicationId : %llx\n", g_Resource->localCommunicationId);
        NN_LCS_LOG_DEBUG("LcsVersion : %x\n", g_Resource->version._raw);

        nn::ns::Initialize();

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Resource->systemUpdateControl.Occupy());

        nn::os::InitializeEvent(&(g_Resource->migrationEvent), false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMutex(&g_Resource->mutex, false, 0);

        g_Resource->isRunning = true;
        int migrationThreadPriority = nn::os::DefaultThreadPriority + 1;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_MigrationThread, MigrationThread, NULL,
                g_Resource->migrationThreadStack, sizeof(g_Resource->migrationThreadStack), migrationThreadPriority));

        nn::os::StartThread(&g_MigrationThread);

        g_Resource->state.Initialize();

        NN_RESULT_SUCCESS;
    }

    void Finalize() NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);

        LeaveSession();

        detail::CloseStation();

        g_Resource->systemUpdateControl.Relieve();
        g_Resource->systemUpdateControl.~SystemUpdateControl();

        g_Resource->stateEvent.Finalize();
        g_Resource->state.Finalize();

        g_Resource->isRunning = false;
        nn::os::SignalEvent(&(g_Resource->migrationEvent));
        nn::os::WaitThread(&g_MigrationThread);
        nn::os::DestroyThread(&g_MigrationThread);

        nn::os::FinalizeEvent(&(g_Resource->migrationEvent));
        nn::os::FinalizeMutex(&g_Resource->mutex);

        g_Resource = nullptr;

        nn::ns::Finalize();
    }

    void AttachStateChangeEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pEvent);

        g_Resource->stateEvent.Initialize(pEvent);
    }

    State GetState() NN_NOEXCEPT
    {
        if (g_Resource == nullptr)
        {
            return State_None;
        }

        return g_Resource->state.GetState();
    }

    Result GetSessionInfo(SessionInfo* pOutInfo) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutInfo);

        const auto state = g_Resource->state.GetState();
        NN_RESULT_THROW_UNLESS(
            state == State_Opened || state == State_Joined || state == State_Transferring ||
            state == State_Suspended || state == State_Completed, ResultInvalidState());

        if (g_Host.IsInitialized())
        {
            g_Host.UpdateSessionInfo();
        }
        else if (g_Client.IsInitialized())
        {
            g_Client.UpdateSessionInfo();
        }

        ::std::memcpy(pOutInfo, &g_Resource->sessionInfo, sizeof(SessionInfo));

        NN_RESULT_SUCCESS;
    }

    uint32_t GetMyIndex() NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        return g_Resource->myIndex;
    }

    Result GetNodes(NodeInfo* pOutNodes, int* pOutCount, int bufferCount) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutNodes);
        NN_LCS_REQUIRES_NOT_NULL(pOutCount);
        NN_LCS_REQUIRES_GREATER(bufferCount, 0);

        const auto state = g_Resource->state.GetState();
        NN_RESULT_THROW_UNLESS(
            state == State_Opened || state == State_Joined || state == State_Transferring ||
            state == State_Suspended || state == State_Completed, ResultInvalidState());

        nn::os::LockMutex(&g_Resource->mutex);
        *pOutCount = ::std::min(bufferCount, g_Resource->nodeInfoCount);
        NN_SDK_ASSERT_RANGE(*pOutCount, 0, nn::lcs::NodeCountMax);
        ::std::memcpy(pOutNodes, g_Resource->nodeInfo, sizeof(NodeInfo) * (*pOutCount));
        nn::os::UnlockMutex(&g_Resource->mutex);

        NN_RESULT_SUCCESS;
    }

    Result GetRequiredStorageSize(size_t* pOutSize) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutSize);

        const auto state = g_Resource->state.GetState();
        NN_RESULT_THROW_UNLESS(
            state == State_Opened || state == State_Joined || state == State_Transferring ||
            state == State_Suspended || state == State_Completed, ResultInvalidState());

        if (g_Host.IsInitialized())
        {
            g_Host.UpdateRequiredStorageSize();
        }
        else if (g_Client.IsInitialized())
        {
            g_Client.UpdateRequiredStorageSize();
        }
        *pOutSize = g_Resource->requiredStorageSize;

        NN_RESULT_SUCCESS;
    }

    Result GetSessionState(SessionState* pOutState) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutState);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Transferring, ResultInvalidState());

        if (g_Host.IsInitialized())
        {
            g_Host.GetSessionState(pOutState);
        }
        else if (g_Client.IsInitialized())
        {
            g_Client.GetSessionState(pOutState);
        }
        else
        {
            *pOutState = SessionState_None;
        }

        NN_RESULT_SUCCESS;
    }

    Result GetProgress(Progress* pOutProgress) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutProgress);

        NN_RESULT_THROW_UNLESS(
            (g_Resource->state.GetState() == State_Transferring ||
             g_Resource->state.GetState() == State_ContentsShareFailed ||
             g_Resource->state.GetState() == State_Suspended ||
             g_Resource->state.GetState() == State_Completed), ResultInvalidState());

        if (g_Host.IsInitialized())
        {
            g_Host.GetProgress(pOutProgress);
        }
        else if (g_Client.IsInitialized())
        {
            g_Client.GetProgress(pOutProgress);
        }
        else
        {
            pOutProgress->transferRole = TransferRole_None;
            pOutProgress->downloadedSize = 0;
            pOutProgress->size = 0;
            pOutProgress->transferIndex = 0;
            ::std::memset(&pOutProgress->info, 0, sizeof(ContentsInfo));
        }

        NN_RESULT_SUCCESS;
    }

    Result GetNodeProgress(NodeProgress* pOutNodeProgress, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutNodeProgress);

        NN_RESULT_THROW_UNLESS(
            (g_Resource->state.GetState() == State_Transferring ||
                g_Resource->state.GetState() == State_ContentsShareFailed ||
                g_Resource->state.GetState() == State_Suspended ||
                g_Resource->state.GetState() == State_Completed), ResultInvalidState());
        NN_RESULT_THROW_UNLESS(index != 0, ResultNodeNotFound());

        Result result;

        if (g_Host.IsInitialized())
        {
            result = g_Host.GetNodeProgress(pOutNodeProgress, index);
        }
        else if (g_Client.IsInitialized())
        {
            result = g_Client.GetNodeProgress(pOutNodeProgress, index);
        }
        else
        {
            for (int i = 0; i < g_Resource->nodeInfoCount; ++i)
            {
                if (g_Resource->nodeInfo[i].index == index)
                {
                    pOutNodeProgress->contentCount = 0;
                    pOutNodeProgress->downloadedContentCount = 0;
                    NN_RESULT_SUCCESS;
                }
            }
            return ResultNodeNotFound();
        }

        return result;
    }

    Result GetDownloadedContents(
        ContentsInfo* outContents, int* pOutCount, int bufferSize) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(outContents);
        NN_LCS_REQUIRES_NOT_NULL(pOutCount);
        NN_LCS_REQUIRES_GREATER(bufferSize, 0);

        NN_RESULT_THROW_UNLESS(
            (g_Resource->state.GetState() == State_Transferring ||
             g_Resource->state.GetState() == State_ContentsShareFailed ||
             g_Resource->state.GetState() == State_Suspended ||
             g_Resource->state.GetState() == State_Completed), ResultInvalidState());

        *pOutCount = ::std::min(bufferSize, g_Resource->contentInfoCount);
        ::std::memcpy(outContents, g_Resource->contentInfo, sizeof(ContentsInfo) * (*pOutCount));

        NN_RESULT_SUCCESS;
    }

    Result GetDownloadedEulaDataSize(size_t* pOutSize, const char* eulaDataPath) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutSize);
        NN_LCS_REQUIRES_NOT_NULL(eulaDataPath);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Suspended, ResultInvalidState());

        if (g_Host.IsInitialized())
        {
            return g_Host.GetEulaDataSize(pOutSize, eulaDataPath);
        }
        else if (g_Client.IsInitialized())
        {
            return g_Client.GetEulaDataSize(pOutSize, eulaDataPath);
        }
        else
        {
            NN_SDK_ASSERT(false, "This node is not host nor client");
            *pOutSize = 0U;
            return nn::lcs::ResultInvalidState();
        }
    }


    Result GetDownloadedEulaData(
        size_t* pOutSize, void* outBuffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutSize);
        NN_LCS_REQUIRES_NOT_NULL(outBuffer);
        NN_LCS_REQUIRES_NOT_NULL(eulaDataPath);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Suspended, ResultInvalidState());

        if (g_Host.IsInitialized())
        {
            return g_Host.GetEulaData(pOutSize, outBuffer, bufferSize, eulaDataPath);
        }
        else if (g_Client.IsInitialized())
        {
            return g_Client.GetEulaData(pOutSize, outBuffer, bufferSize, eulaDataPath);
        }
        else
        {
            NN_SDK_ASSERT(false, "This node is not host nor client");
            *pOutSize = 0U;
            return nn::lcs::ResultInvalidState();
        }
    }


    Result GetSessionContext(SessionContext* pOutSessionContext) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutSessionContext);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Suspended, ResultInvalidState());

        if (g_Host.IsInitialized())
        {
            return g_Host.GetResumeContext(pOutSessionContext);
        }
        else if (g_Client.IsInitialized())
        {
            return g_Client.GetResumeContext(pOutSessionContext);
        }
        else
        {
            NN_SDK_ASSERT(false, "This node is not host nor client");
            std::memset(pOutSessionContext, 0, sizeof(SessionContext));
            return nn::lcs::ResultInvalidState();
        }
    }

    Result ResumeContentsShare() NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Suspended, ResultInvalidState());

        if (g_Host.IsInitialized())
        {
            return g_Host.ResumeShare();
        }
        else if (g_Client.IsInitialized())
        {
            return g_Client.ResumeShare();
        }
        else
        {
            NN_SDK_ASSERT(false, "This node is not host nor client");
            return nn::lcs::ResultInvalidState();
        }
    }

    Result ResumeSession(const SessionContext& sessionContext) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Initialized, ResultInvalidState());

        Result result;
        detail::ResumeContext resumeContext = {};
        ::std::memcpy(&resumeContext, &sessionContext, sizeof(detail::ResumeContext));

        NN_LCS_LOG_DEBUG("ResumeContext Role : %d\n", resumeContext.lcsRole);
        NN_LCS_LOG_DEBUG("ResumeContext HostIP : %x\n", resumeContext.hostAddress);
        NN_LCS_LOG_DEBUG("ResumeContext SuspendedReason: %d\n", resumeContext.suspendedReason);

        // Host は再起動することがないので、Host を指定されることはない。
        if (resumeContext.lcsRole == detail::LcsRole_Client)
        {
            result = g_Client.Resume(sessionContext, g_Resource);
            if (result.IsSuccess())
            {
                // 適用待ちタスクがあったら退避
                for (int i = 0; i < resumeContext.sessionInfo.contentsCount; ++i)
                {
                    if (EvacuateDownloadTask(
                        &g_Resource->downloadTaskInfo[g_Resource->downloadTaskInfoCount],
                        resumeContext.sessionInfo.contents[i].applicationId.value).IsSuccess())
                    {
                        ++g_Resource->downloadTaskInfoCount;
                        NN_LCS_LOG_DEBUG("EvacuateDownloadTask Success[count:%d]\n", g_Resource->downloadTaskInfoCount);
                    }
                }
                // 自身のコンテンツの情報を保存
                for (int i = 0; i < resumeContext.sessionInfo.contentsCount; ++i)
                {
                    detail::GetApplicationDetailInfo(&g_Resource->ownAppInfo[i].detailInfo,
                        g_Resource->appControlDataBuffer, detail::AppControlDataSize,
                        resumeContext.sessionInfo.contents[i].applicationId.value);
                    detail::GetApplicationDeliveryInfo(&g_Resource->ownAppInfo[i].deliveryInfoCount,
                        g_Resource->ownAppInfo[i].deliveryInfo, detail::ApplicationDeliveryInfoCountMax,
                        resumeContext.sessionInfo.contents[i].applicationId.value);
                }
                g_Resource->ownAppInfoCount = resumeContext.sessionInfo.contentsCount;

                // デバッグ用の値を設定
                SetInnerRole(InnerRole_Slave);
            }
        }
        else
        {
            result = ResultInvalidContext();
        }

        return result;
    }

    Result LeaveSession() NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);

        const auto state = g_Resource->state.GetState();
        NN_RESULT_THROW_UNLESS(
            state == State_Opened || state == State_Joined || state == State_Transferring ||
            state == State_ContentsShareFailed || state == State_Suspended || state == State_Completed,
            ResultInvalidState());

        // Migration と衝突しないようにロック
        ::std::lock_guard<nn::os::Mutex> guard(g_Mutex);

        // デバッグ用の値を設定
        SetInnerRole(InnerRole_None);

        if (g_Host.IsInitialized())
        {
            NN_LCS_LOG_DEBUG("g_Host.Close\n");
            g_Host.Close();
            detail::CloseAccessPoint();
            detail::OpenStation();
        }
        else if (g_Client.IsInitialized())
        {
            NN_LCS_LOG_DEBUG("g_Client.Leave\n");
            g_Client.Leave(detail::LeaveType_Permanent);
        }

        // タスクの復旧
        for (int i = 0; i < g_Resource->downloadTaskInfoCount; ++i)
        {
            NN_LCS_LOG_DEBUG("RecreateDownloadTask[%d]\n", i);
            auto result = RecreateDownloadTask(g_Resource->downloadTaskInfo[i]);
            if (result.IsFailure())
            {
                NN_LCS_LOG_DEBUG("Fail to RecreateDownloadTask : %08x\n", result.GetInnerValueForDebug());
            }
        }

        ClearResource();

        ChangeState(State_Initialized, true);

        NN_RESULT_SUCCESS;
    }

    Result SuspendSession() NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Suspended, ResultInvalidState());

        // デバッグ用の値を設定
        SetInnerRole(InnerRole_None);

        if (g_Host.IsInitialized())
        {
            g_Host.Close();
            detail::CloseAccessPoint();
            detail::OpenStation();
        }
        else if (g_Client.IsInitialized())
        {
            g_Client.Leave(detail::LeaveType_Temporary);
        }

        // タスクの復旧
        for (int i = 0; i < g_Resource->downloadTaskInfoCount; ++i)
        {
            NN_LCS_LOG_DEBUG("RecreateDownloadTask[%d]\n", i);
            auto result = RecreateDownloadTask(g_Resource->downloadTaskInfo[i]);
            if (result.IsFailure())
            {
                NN_LCS_LOG_DEBUG("Fail to RecreateDownloadTask : %08x\n", result.GetInnerValueForDebug());
            }
        }

        ClearResource();

        ChangeState(State_Initialized, true);

        NN_RESULT_SUCCESS;
    }

    ContentsShareFailureReason GetContentsShareFailureReason() NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);

        if (g_Resource->state.GetState() != State_ContentsShareFailed)
        {
            return ContentsShareFailureReason_None;
        }

        return g_Resource->contentsShareFailureReason;
    }

    SuspendedReason GetSuspendedReason() NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);

        if (g_Resource->state.GetState() != State_Suspended)
        {
            return SuspendedReason_None;
        }

        return g_Resource->suspendedReason;
    }

    Result ConvertFailureReasonToResult(ContentsShareFailureReason reason) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);

        switch (reason)
        {
        case nn::lcs::ContentsShareFailureReason_FlightMode:
            return ResultFailureFlightMode();
        case nn::lcs::ContentsShareFailureReason_Sleep:
            return ResultFailureSleep();
        case nn::lcs::ContentsShareFailureReason_CommunicationError:
            return ResultFailureCommunicationError();
        case nn::lcs::ContentsShareFailureReason_NodeLeaved:
            return ResultFailureNodeLeaved();
        case nn::lcs::ContentsShareFailureReason_CannotUpdateSystem:
            return ResultFailureCannotUpdateSystem();
        case nn::lcs::ContentsShareFailureReason_ContentsCorrupted:
            return ResultFailureContentsCorrupted();
        case nn::lcs::ContentsShareFailureReason_NotExistDownloadContents:
            return ResultFailureNotExistDownloadContents();
        case nn::lcs::ContentsShareFailureReason_BuiltInStorageError:
            return ResultFailureBuiltInStorageError();
        case nn::lcs::ContentsShareFailureReason_GameCardError:
            return ResultFailureGameCardError();
        case nn::lcs::ContentsShareFailureReason_SdCardError:
            return ResultFailureSdCardError();
        case nn::lcs::ContentsShareFailureReason_StorageError:
            return ResultFailureStorageError();
        case nn::lcs::ContentsShareFailureReason_UnknownError:
            return ResultFailureUnknownError();
        case nn::lcs::ContentsShareFailureReason_UnshareableContents:
            return ResultFailureUnshareableContents();
        case nn::lcs::ContentsShareFailureReason_NotEnoughStorageSpace:
            return ResultFailureNotEnoughStorageSpace();
        default:
            break;
        }

        NN_RESULT_SUCCESS;
    }

    void ShowError(Result result) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(result.GetModule(), 211);

        nn::err::ErrorContext errorContext = {};
        GetErrorContext(&errorContext);

        if (errorContext.type == err::ErrorContextType::LocalContentShare)
        {
            nn::err::ShowError(result, errorContext);
        }
        else
        {
            nn::err::ShowError(result);
        }
    }

    Result IsWaitingPatchInstallTaskDiscard(bool* isDiscard) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(isDiscard);

        const auto state = g_Resource->state.GetState();
        NN_RESULT_THROW_UNLESS(
            state == State_Opened || state == State_Joined || state == State_Transferring ||
            state == State_ContentsShareFailed || state == State_Suspended || state == State_Completed,
            ResultInvalidState());

        if (g_Resource->downloadTaskInfoCount > 0)
        {
            *isDiscard = true;
        }
        else
        {
            *isDiscard = false;
        }

        NN_RESULT_SUCCESS;
    }


    //! @}
    //! @name ホスト専用の API
    //! @{


    Result OpenSession(const SessionSettings& settings) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_MINMAX(settings.applicationCount, 1, nn::lcs::SharableContentsCountMax);
        NN_LCS_REQUIRES_MINMAX(settings.nodeCountMax, 1, nn::lcs::NodeCountMax);
        NN_LCS_REQUIRES_EQUAL(settings.contentsShareVersionPolicy,
                              nn::lcs::ContentsShareVersionPolicy_Latest);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Initialized, ResultInvalidState());

        ClearResource();
        ::std::memcpy(&g_Resource->sessionSettings, &settings, sizeof(SessionSettings));

        //Init で Open した STA を Close
        NN_RESULT_DO(detail::CloseStation());
        NN_RESULT_DO(detail::OpenAccessPoint());
        Result result = g_Host.Open(g_Resource);
        if (result.IsFailure())
        {
            g_Host.Close();
            detail::CloseAccessPoint();
            detail::OpenStation();
            return result;
        }

        // 適用待ちタスクがあったら退避
        for (int i = 0; i < settings.applicationCount; ++i)
        {
            if (EvacuateDownloadTask(
                &g_Resource->downloadTaskInfo[g_Resource->downloadTaskInfoCount],
                settings.applications[i].value).IsSuccess())
            {
                ++g_Resource->downloadTaskInfoCount;
                NN_LCS_LOG_DEBUG("EvacuateDownloadTask Success[count:%d]\n", g_Resource->downloadTaskInfoCount);
            }
        }

        // 自身のコンテンツの情報を保存
        for (int i = 0; i < settings.applicationCount; ++i)
        {
            detail::GetApplicationDetailInfo(&g_Resource->ownAppInfo[i].detailInfo,
                g_Resource->appControlDataBuffer, detail::AppControlDataSize, settings.applications[i].value);
            detail::GetApplicationDeliveryInfo(&g_Resource->ownAppInfo[i].deliveryInfoCount,
                g_Resource->ownAppInfo[i].deliveryInfo, detail::ApplicationDeliveryInfoCountMax,
                settings.applications[i].value);
        }
        g_Resource->ownAppInfoCount = settings.applicationCount;

        ChangeState(State_Opened, false);

        // デバッグ用の値を設定
        SetInnerRole(InnerRole_Master);

        NN_RESULT_SUCCESS;
    }

    Result SetClientAcceptPolicy(AcceptPolicy policy) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_INCLUDE(policy,
            nn::lcs::AcceptPolicy_AlwaysAccept,
            nn::lcs::AcceptPolicy_AlwaysReject);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Opened, ResultInvalidState());

        switch (policy)
        {
        case nn::lcs::AcceptPolicy_AlwaysAccept:
        {
            g_Host.SetAcceptance(true);
            break;
        }
        case nn::lcs::AcceptPolicy_AlwaysReject:
        {
            g_Host.SetAcceptance(false);
            break;
        }
        default:
            break;
        }

        NN_RESULT_SUCCESS;
    }

    Result StartContentsShare() NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Opened, ResultInvalidState());

        ChangeState(State_Transferring, false);
        g_Host.Start();

        NN_RESULT_SUCCESS;
    }


    //! @}
    //! @name クライアント専用の API
    //! @{


    Result Scan(SessionInfo* pOutInfo, int* pOutCount, int bufferCount) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_LCS_REQUIRES_NOT_NULL(pOutInfo);
        NN_LCS_REQUIRES_NOT_NULL(pOutCount);
        NN_LCS_REQUIRES_GREATER(bufferCount, 0);

        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Initialized, ResultInvalidState());

        NN_RESULT_DO(detail::Scan(
            pOutInfo, pOutCount, bufferCount, g_Resource->localCommunicationId));

        NN_RESULT_SUCCESS;
    }

    Result JoinSession(const SessionInfo& session) NN_NOEXCEPT
    {
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource, nullptr);
        NN_LCS_REQUIRES_NOT_EQUAL(g_Resource->state.GetState(), State_None);
        NN_RESULT_THROW_UNLESS(g_Resource->state.GetState() == State_Initialized, ResultInvalidState());

        ClearResource();

        NN_RESULT_DO(g_Client.Join(session, g_Resource));

        // 適用待ちタスクがあったら退避
        for (int i = 0; i < session.contentsCount; ++i)
        {
            if (EvacuateDownloadTask(
                &g_Resource->downloadTaskInfo[g_Resource->downloadTaskInfoCount],
                session.contents[i].applicationId.value).IsSuccess())
            {
                ++g_Resource->downloadTaskInfoCount;
                NN_LCS_LOG_DEBUG("EvacuateDownloadTask Success[count:%d]\n", g_Resource->downloadTaskInfoCount);
            }
        }

        // 自身のコンテンツの情報を保存
        for (int i = 0; i < session.contentsCount; ++i)
        {
            detail::GetApplicationDetailInfo(&g_Resource->ownAppInfo[i].detailInfo,
                g_Resource->appControlDataBuffer, detail::AppControlDataSize,
                session.contents[i].applicationId.value);
            detail::GetApplicationDeliveryInfo(&g_Resource->ownAppInfo[i].deliveryInfoCount,
                g_Resource->ownAppInfo[i].deliveryInfo, detail::ApplicationDeliveryInfoCountMax,
                session.contents[i].applicationId.value);
        }
        g_Resource->ownAppInfoCount = session.contentsCount;

        ChangeState(State_Joined, false);

        // デバッグ用の値を設定
        SetInnerRole(InnerRole_Slave);

        NN_RESULT_SUCCESS;
    }

    //! @}

}} // end of namespace nn::lcs

