﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/lcs_PrivateResult.h>
#include <nn/lcs/detail/lcs_ApplicationShareApi.h>
#include <nn/lcs/detail/lcs_NodeManager.h>
#include <nn/lcs/detail/lcs_Socket.h>
#include <nn/lcs/detail/lcs_SystemShareApi.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/os/os_Random.h>
#include <nn/ns/ns_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace lcs { namespace detail { namespace {

    nn::os::Mutex m_Mutex(false);

}}}}

namespace nn { namespace lcs { namespace detail {

    int GetNodeNumberFromIndex(NodeDetailInfo* info, int infoCount, uint32_t index)
    {
        for (int i = 0; i < infoCount; ++i)
        {
            if (info[i].nodeInfo.index == index)
            {
                return i;
            }
        }
        return -1;
    }

    int GetNodeNumberFromIpv4Address(NodeDetailInfo* info, int infoCount, Ipv4Address address)
    {
        for (int i = 0; i < infoCount; ++i)
        {
            if (info[i].ipAddress == address)
            {
                return i;
            }
        }
        return -1;
    }


    NodeManager::NodeManager()  NN_NOEXCEPT :
        m_NodeCountMax(0),
        m_NodeCount(0)
    {
    }

    void NodeManager::Initialize(int nodeCountMax) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_MINMAX(nodeCountMax, 0, NodeCountMax);

        m_NodeCountMax = nodeCountMax;
        m_NodeCount = 0;

        do
        {
            nn::os::GenerateRandomBytes(&m_Index, sizeof(m_Index));
        } while (m_Index == 0);

        ::std::memset(m_NodeInfo, 0, (sizeof(NodeDetailInfo) * m_NodeCountMax));
    }

    void NodeManager::Finalize() NN_NOEXCEPT
    {
        for (int i = 0; i < m_NodeCount; ++i)
        {
            SocketClose(&m_NodeInfo[i].socket);
        }
        ClearNodes();
    }

    uint32_t NodeManager::AddNode(Ipv4Address address, const MacAddress& macAddress, uint32_t index) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);
        if (m_NodeCount == NodeCountMax)
        {
            NN_LCS_LOG_DEBUG("AddNode : ResultNodeCountLimitation\n");
            return 0;
        }
        if (GetNodeNumberFromIpv4Address(m_NodeInfo, m_NodeCount, address) != -1)
        {
            NN_LCS_LOG_DEBUG("AddNode : Already Added\n");
            return 0;
        }
        if (index == 0)
        {
            if (m_Index == 0)
            {
                ++m_Index;
            }
            m_NodeInfo[m_NodeCount].nodeInfo.index = m_Index;
            ++m_Index;
        }
        else
        {
            m_NodeInfo[m_NodeCount].nodeInfo.index = index;
        }
        uint32_t retIndex = m_NodeInfo[m_NodeCount].nodeInfo.index;
        m_NodeInfo[m_NodeCount].ipAddress = address;
        ::std::memcpy(&m_NodeInfo[m_NodeCount].macAddress, &macAddress, sizeof(MacAddress));
        m_NodeInfo[m_NodeCount].isReceive = false;
        m_NodeInfo[m_NodeCount].state = NodeInnerState_Temporary;
        m_NodeInfo[m_NodeCount].socket = -1;

#ifdef NN_LCS_BUILD_CONFIG_DEBUG_LOG_LEVEL_DEBUG
        NN_LCS_LOG_DEBUG("/// Add Info[%d] ////////////////////\n", m_NodeCount);
        NN_LCS_LOG_DEBUG("Index : %x\n", m_NodeInfo[m_NodeCount].nodeInfo.index);
        NN_LCS_LOG_DEBUG("IpAddress : %x\n", m_NodeInfo[m_NodeCount].ipAddress);
        //for (int i = 0; i < 6; ++i)
        //{
        //    NN_LCS_LOG_DEBUG("MacAddress[%d] : %x\n", i, m_NodeInfo[m_NodeCount].macAddress.raw[i]);
        //}
#endif // NN_LCS_BUILD_CONFIG_DEBUG_LOG_LEVEL_DEBUG
        NN_LCS_LOG_DEBUG("///////////////////////////////\n");
        ++m_NodeCount;

        return retIndex;
    }

    bool NodeManager::AddNodeSocket(int socket, Ipv4Address address) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);
        int num = GetNodeNumberFromIpv4Address(m_NodeInfo, m_NodeCountMax, address);
        if (num == -1)
        {
            NN_LCS_LOG_DEBUG("AddNodeSocket : Node Not Found\n");
            return false;
        }
        m_NodeInfo[num].socket = socket;
        NN_LCS_LOG_DEBUG("AddNodeSocket : %x\n", address);

        return true;
    }

    bool NodeManager::DeleteNodeSocket(uint32_t index) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);
        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCountMax, index);
        if (num == -1)
        {
            NN_LCS_LOG_DEBUG("AddNodeDetail : Node Not Found\n");
            return false;
        }
        SocketClose(&m_NodeInfo[num].socket);
        m_NodeInfo[num].socket = -1;

        return true;
    }

    bool NodeManager::AddNodeDetail(const NodeDetailInfo& detail) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);
        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCountMax, detail.nodeInfo.index);
        if (num == -1)
        {
            NN_LCS_LOG_DEBUG("AddNodeDetail : Node Not Found\n");
            return false;
        }
        ::std::strncpy(
            m_NodeInfo[num].nodeInfo.userName, detail.nodeInfo.userName, UserNameBytesMax);
        m_NodeInfo[num].nodeInfo.userName[UserNameBytesMax] = 0;
        //Index, ipAddress, macAddress, socket は更新しない
        m_NodeInfo[num].version = detail.version;
        ::std::memcpy(&m_NodeInfo[num].sysInfo, &detail.sysInfo, sizeof(SystemDeliveryInfo));
        for (int i = 0; i < SharableContentsCountMax; ++i)
        {
            m_NodeInfo[num].shareApp[i].hasApp = detail.shareApp[i].hasApp;
            m_NodeInfo[num].shareApp[i].info.deliveryInfoCount = detail.shareApp[i].info.deliveryInfoCount;
            ::std::memcpy(&m_NodeInfo[num].shareApp[i].info.detailInfo,
                &detail.shareApp[i].info.detailInfo, sizeof(ApplicationDetailInfo));
            ::std::memcpy(&m_NodeInfo[num].shareApp[i].info.deliveryInfo,
                &detail.shareApp[i].info.deliveryInfo,
                detail.shareApp[i].info.deliveryInfoCount * sizeof(ApplicationDeliveryInfo));
        }
        m_NodeInfo[num].downloadedCount = detail.downloadedCount;
        NN_LCS_LOG_DEBUG("ChangeInnerState on AddNodeDetail : %s[%x] / %d -> %d\n",
            m_NodeInfo[num].nodeInfo.userName, m_NodeInfo[num].nodeInfo.index, m_NodeInfo[num].state, detail.state);
        m_NodeInfo[num].isCanUpdate = false;
        m_NodeInfo[num].state = detail.state;


#ifdef NN_LCS_BUILD_CONFIG_DEBUG_LOG_LEVEL_DEBUG
        NN_LCS_LOG_DEBUG("/// Add DetailInfo[%d] ////////////////////\n", num);
        NN_LCS_LOG_DEBUG("UserName : %s\n", m_NodeInfo[num].nodeInfo.userName);
        NN_LCS_LOG_DEBUG("Index : %x\n", m_NodeInfo[num].nodeInfo.index);
        //for (int i = 0; i < 6; ++i)
        //{
        //    NN_LCS_LOG_DEBUG("MacAddress[%d] : %x\n", i, m_NodeInfo[num].macAddress.raw[i]);
        //}
        NN_LCS_LOG_DEBUG("IpAddress : %x\n", m_NodeInfo[num].ipAddress);
        NN_LCS_LOG_DEBUG("LcsVersion : %x\n", m_NodeInfo[num].version._raw);
        ShowSystemDeliveryInfo(m_NodeInfo[num].sysInfo);
        for (int i = 0; i < SharableContentsCountMax; ++i)
        {
            if (m_NodeInfo[num].shareApp[i].info.detailInfo.id != 0)
            {
                NN_LCS_LOG_DEBUG("App%dId : %llx\n", i, m_NodeInfo[num].shareApp[i].info.detailInfo.id);
                NN_LCS_LOG_DEBUG("App%dDispVersion : %s\n", i, m_NodeInfo[num].shareApp[i].info.detailInfo.displayVersion);
                NN_LCS_LOG_DEBUG("App%dInstallSize : %lld\n", i, m_NodeInfo[num].shareApp[i].info.detailInfo.requiredSize);
                for (int j = 0; j < m_NodeInfo[num].shareApp[i].info.deliveryInfoCount; ++j)
                {
                    ShowApplicationDeliveryInfo(m_NodeInfo[num].shareApp[i].info.deliveryInfo[j]);
                }
            }
        }
        NN_LCS_LOG_DEBUG("DownloadedCount : %d\n", m_NodeInfo[num].downloadedCount);
        NN_LCS_LOG_DEBUG("InnerState : %d\n", m_NodeInfo[num].state);
        NN_LCS_LOG_DEBUG("///////////////////////////////\n");
#endif // NN_LCS_BUILD_CONFIG_DEBUG_LOG_LEVEL_DEBUG

        return true;
    }

    void NodeManager::GetNodeInfo(int* pOutNodeCount, NodeInfo* outInfo, int infoCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutNodeCount);
        NN_SDK_ASSERT_NOT_NULL(outInfo);

        *pOutNodeCount = 0;

        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (infoCount == i)
            {
                return;
            }
            ::std::memcpy(&outInfo[i], &m_NodeInfo[i].nodeInfo, sizeof(NodeInfo));
            (*pOutNodeCount)++;
        }
    }

    void NodeManager::GetConnectedNodeInfo(int* pOutNodeCount, NodeInfo* outInfo, int infoCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutNodeCount);
        NN_SDK_ASSERT_NOT_NULL(outInfo);

        *pOutNodeCount = 0;

        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (infoCount == i)
            {
                return;
            }
            if (m_NodeInfo[i].state != NodeInnerState_Temporary)
            {
                ::std::memcpy(&outInfo[*pOutNodeCount], &m_NodeInfo[i].nodeInfo, sizeof(NodeInfo));
                (*pOutNodeCount)++;
            }
        }
    }

    Result NodeManager::GetNodeDetailInfo(NodeDetailInfo* pOutInfo, uint32_t index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutInfo);

        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCount, index);
        if (num == -1)
        {
            return ResultNodeNotFound();
        }
        ::std::memcpy(pOutInfo, &m_NodeInfo[num], sizeof(NodeDetailInfo));
        NN_RESULT_SUCCESS;
    }

    uint32_t NodeManager::GetNodeIndex(Ipv4Address address) NN_NOEXCEPT
    {
        int num = GetNodeNumberFromIpv4Address(m_NodeInfo, m_NodeCount, address);
        if (num == -1)
        {
            return 0;
        }
        return m_NodeInfo[num].nodeInfo.index;
    }

    int NodeManager::GetNodeSocket(uint32_t index) NN_NOEXCEPT
    {
        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCount, index);
        if (num == -1)
        {
            return -1;
        }
        return m_NodeInfo[num].socket;
    }

    bool NodeManager::SetLeaveTime(int64_t time, uint32_t index) NN_NOEXCEPT
    {
        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCount, index);
        if (num == -1)
        {
            return false;
        }
        m_NodeInfo[num].leaveTime = time;
        return true;
    }

    int64_t NodeManager::GetLeaveTime(uint32_t index) NN_NOEXCEPT
    {
        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCount, index);
        if (num == -1)
        {
            return -1;
        }
        return m_NodeInfo[num].leaveTime;
    }

    Result NodeManager::RemoveNode(uint32_t index) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);
        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCount, index);
        if (num == -1)
        {
            return ResultNodeNotFound();
        }

        SocketClose(&m_NodeInfo[num].socket);
        --m_NodeCount;
        ::std::memmove(&m_NodeInfo[num],
            &m_NodeInfo[num + 1], sizeof(NodeDetailInfo) * (m_NodeCount - num));
        NN_LCS_LOG_DEBUG("RemoveClient : %x\n", index);
        NN_RESULT_SUCCESS;
    }

    void NodeManager::ClearNodes() NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);
        for (int i = 0; i < m_NodeCount; ++i)
        {
            SocketClose(&m_NodeInfo[i].socket);
        }
        ::std::memset(m_NodeInfo, 0, (sizeof(NodeDetailInfo) * m_NodeCountMax));
        m_NodeCount = 0;
    }

    int NodeManager::GetNodeCount() NN_NOEXCEPT
    {
        return m_NodeCount;
    }

    int NodeManager::GetConnectedNodeCount() NN_NOEXCEPT
    {
        int count = 0;

        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (m_NodeInfo[i].state != NodeInnerState_Temporary)
            {
                ++count;
            }
        }
        return count;
    }

    bool NodeManager::IsNodeCountLimited() NN_NOEXCEPT
    {
        if (m_NodeCount > m_NodeCountMax)
        {
            return true;
        }
        return false;
    }

    Result NodeManager::SetInnerState(NodeInnerState state, uint32_t index) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);
        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCount, index);
        if (num == -1)
        {
            return ResultNodeNotFound();
        }
        NN_LCS_LOG_DEBUG("ChangeInnerState : %s[%x] / %d -> %d\n",
            m_NodeInfo[num].nodeInfo.userName, m_NodeInfo[num].nodeInfo.index, m_NodeInfo[num].state, state);
        m_NodeInfo[num].state = state;
        NN_RESULT_SUCCESS;
    }

    bool NodeManager::IsInnerState(uint32_t index, NodeInnerState state) NN_NOEXCEPT
    {
        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCount, index);
        if (num == -1)
        {
            return false;
        }

        if (m_NodeInfo[num].state == state)
        {
            return true;
        }
        return false;
    }

    bool NodeManager::IsInnerStateAll(NodeInnerState state) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);
        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (m_NodeInfo[i].state != state)
            {
                return false;
            }
        }
        return true;
    }


    bool NodeManager::IsExsistNode(uint32_t index) NN_NOEXCEPT
    {
        if (GetNodeNumberFromIndex(m_NodeInfo, m_NodeCount, index) < 0)
        {
            return false;
        }
        return true;
    }

    void NodeManager::FindLatestSystemNode(uint32_t* pOutIndex, Ipv4Address myAddress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutIndex);

        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);

        int value = 0;
        SystemDeliveryInfo sysInfo = {};

        int myNum = GetNodeNumberFromIpv4Address(m_NodeInfo, m_NodeCount, myAddress);
        NN_LCS_LOG_DEBUG("FindLatestSystemNode / MyNum : %d\n", myNum);
        // 自分が見つからないことはないから、そんなことがあったらアボート
        NN_ABORT_UNLESS_GREATER_EQUAL(myNum, 0);
        *pOutIndex = m_NodeInfo[myNum].nodeInfo.index;
        ::std::memcpy(&sysInfo, &m_NodeInfo[myNum].sysInfo, sizeof(SystemDeliveryInfo));

        // 端末が自分以外いない場合は終了
        if (m_NodeCount < 2)
        {
            return;
        }

        for (int i = 0; i < m_NodeCount; ++i)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                CompareSystemDeliveryInfo(&value, sysInfo, m_NodeInfo[i].sysInfo));
            //NN_LCS_LOG_DEBUG("FindLatet value : %d\n", value);
            if (value < 0)
            {
                *pOutIndex = m_NodeInfo[i].nodeInfo.index;
                ::std::memcpy(&sysInfo, &m_NodeInfo[i].sysInfo, sizeof(SystemDeliveryInfo));
                NN_LCS_LOG_DEBUG("LatetSystem Index : %x\n", *pOutIndex);
            }
        }
    }

    void NodeManager::FindLatestAppDetail(ApplicationInfo* pOutAppInfo, int appIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutAppInfo);
        NN_SDK_ASSERT_LESS(appIndex, SharableContentsCountMax);

        int value = 0;
        bool hasContents = false;

        // いったん自分の情報を入れる
        ::std::memcpy(pOutAppInfo, &m_NodeInfo[0].shareApp[appIndex].info, sizeof(ApplicationInfo));

        // 自分がパッチの実体を持っていなければ情報を変更
        HasAllContentsToDeliver(&hasContents, m_NodeInfo[0].shareApp[appIndex].info.deliveryInfo,
            m_NodeInfo[0].shareApp[appIndex].info.deliveryInfoCount);
        if (!hasContents)
        {
            ::std::memset(pOutAppInfo->detailInfo.displayVersion, '\0', DisplayVersionSizeMax);
            pOutAppInfo->deliveryInfoCount = 0;
        }

        // 端末が自分以外いない場合は終了
        if (m_NodeCount < 2)
        {
            return;
        }

        for (int i = 1; i < m_NodeCount; ++i)
        {
            if (m_NodeInfo[i].shareApp[appIndex].info.deliveryInfoCount == 0)
            {
                continue;
            }

            Result result = HasAllContentsToDeliver(&hasContents, m_NodeInfo[i].shareApp[appIndex].info.deliveryInfo,
                m_NodeInfo[i].shareApp[appIndex].info.deliveryInfoCount);
            if (result.IsFailure())
            {
                continue;
            }
            if (!hasContents)
            {
                continue;
            }
            // shareApp に入ってるのが本体なしの情報だったらそのままコピー
            if (pOutAppInfo->deliveryInfoCount == 0)
            {
                ::std::memcpy(pOutAppInfo,
                    &m_NodeInfo[i].shareApp[appIndex].info, sizeof(ApplicationInfo));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(CompareApplicationDeliveryInfo(&value,
                    pOutAppInfo->deliveryInfo,
                    pOutAppInfo->deliveryInfoCount,
                    m_NodeInfo[i].shareApp[appIndex].info.deliveryInfo,
                    m_NodeInfo[i].shareApp[appIndex].info.deliveryInfoCount));
                if (value < 0)
                {
                    ::std::memcpy(pOutAppInfo,
                        &m_NodeInfo[i].shareApp[appIndex].info, sizeof(ApplicationInfo));
                }
            }
        }
    }

    Result NodeManager::ClearRequireDownload(uint32_t index) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);
        int num = GetNodeNumberFromIndex(m_NodeInfo, m_NodeCount, index);
        if (num == -1)
        {
            return ResultNodeNotFound();
        }
        m_NodeInfo[num].requireSystemDownload.isRequire = false;
        for (int i = 0; i < SharableContentsCountMax; ++i)
        {
            m_NodeInfo[num].shareApp[i].requireAppDownload.isRequire = false;
        }
        NN_RESULT_SUCCESS;
    }

    void NodeManager::ClearRequireDownload() NN_NOEXCEPT
    {
        for (int i = 0; i < m_NodeCount; ++i)
        {
            ::std::memset(&m_NodeInfo[i].requireSystemDownload, 0, sizeof(RequireDownload));
            for (int j = 0; j < SharableContentsCountMax; ++j)
            {
                ::std::memset(&m_NodeInfo[i].shareApp[j].requireAppDownload, 0, sizeof(RequireDownload));
            }
        }
    }

    Result NodeManager::CheckLatestApplication(const ApplicationInfo& appInfo, int appIndex) NN_NOEXCEPT
    {
        ShareApplicationInfo latestShareAppInfo = {};
        FindLatestAppDetail(&latestShareAppInfo.info, appIndex);
        NN_LCS_LOG_DEBUG("LatestVersion : %s\n", appInfo.detailInfo.displayVersion);
        NN_LCS_LOG_DEBUG("NodesVersion : %s\n", latestShareAppInfo.info.detailInfo.displayVersion);
        if (latestShareAppInfo.info.deliveryInfoCount == 0)
        {
            NN_LCS_LOG_DEBUG("Patch Entity Not Found!\n");
            return ResultFailureNodeLeaved();
        }
        int value = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CompareApplicationDeliveryInfo(&value,
            const_cast<ApplicationDeliveryInfo*>(appInfo.deliveryInfo), appInfo.deliveryInfoCount,
            latestShareAppInfo.info.deliveryInfo, latestShareAppInfo.info.deliveryInfoCount));
        if (value > 0)
        {
            NN_LCS_LOG_DEBUG("Needed Nodes are Left. Fail share!\n");
            return ResultFailureNodeLeaved();
        }
        else if (value < 0)
        {
            NN_LCS_LOG_DEBUG("AppVersion is Unexpected!\n");
            return ResultFailureStorageError();
        }

        NN_RESULT_SUCCESS;
    }

    void NodeManager::CheckNodeContents() NN_NOEXCEPT
    {
        bool hasContents = false;

        for (int index = 0; index < m_NodeCount; ++index)
        {
            for (int i = 0; i < SharableContentsCountMax; ++i)
            {
                if (HasAllContentsToDeliver(&hasContents,
                    m_NodeInfo[index].shareApp[i].info.deliveryInfo,
                    m_NodeInfo[index].shareApp[i].info.deliveryInfoCount).IsFailure())
                {
                    m_NodeInfo[index].requireSystemDownload.isRequire = true;
                }
                else
                {
                    if (hasContents)
                    {
                        m_NodeInfo[index].shareApp[i].hasApp = true;
                    }
                    else
                    {
                        m_NodeInfo[index].shareApp[i].hasApp = false;
                    }
                }
            }
        }
    }

    bool NodeManager::MakeSystemUpdateSchedule(int* pOutListCount, RejectNodeInfo* pOutList,
        int listCount, Ipv4Address hostAddress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutListCount);
        NN_SDK_ASSERT_NOT_NULL(pOutList);

        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);

        NN_LCS_LOG_DEBUG("MakeSystemUpdateSchedule\n");

        *pOutListCount = 0;
        ::std::memset(pOutList, 0, listCount * sizeof(RejectNodeInfo));

        int value = 0;
        int count = 0;
        Ipv4Address address[NodeCountMax] = {};
        SystemDeliveryInfo sysInfo[NodeCountMax] = {};

        // AppDeliveryInfo の互換性があるかを確認。なければ更新対象
        for (int i = 0; i < m_NodeCount; ++i)
        {
            m_NodeInfo[i].requireSystemDownload.isRequire = false;
            m_NodeInfo[i].requireSystemDownload.index = 0;
            m_NodeInfo[i].requireSystemDownload.formAddress = 0;

            Result result = VerifyDeliveryProtocolVersion(m_NodeInfo[i].sysInfo);
            if (result.IsFailure())
            {
                if (nn::ns::ResultInvalidSystemDeliveryProtocolVersion().Includes(result))
                {
                    // 普通はいないはず。もしいたら切断
                    pOutList[*pOutListCount].index = m_NodeInfo[i].nodeInfo.index;
                    pOutList[*pOutListCount].reason = ContentsShareFailureReason_CannotUpdateSystem;
                    ++(*pOutListCount);
                }
                else if (nn::ns::ResultInvalidApplicationDeliveryProtocolVersion().Includes(result))
                {
                    NN_LCS_LOG_DEBUG("%d : %s[%x] need update System\n",
                        i, m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index);
                    m_NodeInfo[i].requireSystemDownload.isRequire = true;
                }
                else if (nn::ns::ResultSystemUpdateRequiredForLocalContentDelivery().Includes(result))
                {
                    // LCSRequiredSystemVersion を満たしていないので、送信できるリストには入れない。
                    NN_LCS_LOG_DEBUG("%d : %s[%x] can not send System(LCSRequiredSystemVersion)\n",
                        i, m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index);
                }
                else
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
            }
            else
            {
                NN_LCS_LOG_DEBUG("%d : %s[%x] can send System\n",
                    i, m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index);

                // 更新がいらない端末なので、送信できるリスト入り
                address[count] = m_NodeInfo[i].ipAddress;
                ::std::memcpy(&sysInfo[count], &m_NodeInfo[i].sysInfo, sizeof(SystemDeliveryInfo));
                ++count;
            }
        }

        // 送信してもらえる端末探し
        // まず、ホストが配信できるかを確認
        SystemDeliveryInfo hostSysInfo = {};
        GetSystemDeliveryInfo(&hostSysInfo);
        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (m_NodeInfo[i].requireSystemDownload.isRequire)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    SelectLatestSystemDeliveryInfo(&value, &hostSysInfo, 1, m_NodeInfo[i].sysInfo));
                if (value < 0)
                {
                    continue;
                }
                NN_LCS_LOG_DEBUG("%d : %s[%x] can do SystemUpdate From Host\n",
                    i, m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index);
                m_NodeInfo[i].requireSystemDownload.formAddress = hostAddress;
                m_NodeInfo[i].requireSystemDownload.index = GetNodeIndex(hostAddress);
                m_NodeInfo[i].isCanUpdate = true;
            }
        }

        // 送信できるリストから探す
        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (m_NodeInfo[i].requireSystemDownload.isRequire && m_NodeInfo[i].requireSystemDownload.index == 0)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    SelectLatestSystemDeliveryInfo(&value, sysInfo, count, m_NodeInfo[i].sysInfo));
                if (value < 0)
                {
                    //もらえない端末は切断リスト入り
                    NN_LCS_LOG_DEBUG("%d : %s[%x] can not receive SystemUpdate\n",
                        i, m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index);
                    pOutList[*pOutListCount].index = m_NodeInfo[i].nodeInfo.index;
                    if (m_NodeInfo[i].isCanUpdate)
                    {
                        pOutList[*pOutListCount].reason = ContentsShareFailureReason_NodeLeaved;
                    }
                    else
                    {
                        pOutList[*pOutListCount].reason = ContentsShareFailureReason_CannotUpdateSystem;
                    }
                    ++(*pOutListCount);
                    continue;
                }
                // もらえるシステムで AppDeliveryInfo が読めるようになるか確認
                int num = GetNodeNumberFromIpv4Address(m_NodeInfo, m_NodeCount, address[value]);
                NN_ABORT_UNLESS_GREATER_EQUAL(num, 0);
                bool hasContent = false;
                if (HasAllContentsToDeliver(&hasContent, m_NodeInfo[num].shareApp[0].info.deliveryInfo,
                    m_NodeInfo[num].shareApp[0].info.deliveryInfoCount).IsSuccess())
                {
                    // 読めるなら配信する
                    NN_LCS_LOG_DEBUG("%d : %s[%x] can do SystemUpdate From Client\n",
                        i, m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index);
                    m_NodeInfo[i].requireSystemDownload.formAddress = address[value];
                    m_NodeInfo[i].requireSystemDownload.index = GetNodeIndex(address[value]);
                    m_NodeInfo[i].isCanUpdate = true;
                }
                else
                {
                    // 配信されるシステムでは AppDeliveryInfo が読めないので、切断リスト入り
                    NN_LCS_LOG_DEBUG("%d : %s[%x] can not receive latest SystemUpdate\n",
                        i, m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index);
                    pOutList[*pOutListCount].index = m_NodeInfo[i].nodeInfo.index;
                    if (m_NodeInfo[i].isCanUpdate)
                    {
                        pOutList[*pOutListCount].reason = ContentsShareFailureReason_NodeLeaved;
                    }
                    else
                    {
                        pOutList[*pOutListCount].reason = ContentsShareFailureReason_CannotUpdateSystem;
                    }
                    ++(*pOutListCount);
                }
            }
        }
        return true;
    } // NOLINT(impl/function_size)

    Result NodeManager::MakeContentShareSchedule(int* pOutListCount, RejectNodeInfo* pOutList,
        int listCount, const ApplicationInfo& appInfo, int appIndex, Ipv4Address hostAddress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutListCount);
        NN_SDK_ASSERT_NOT_NULL(pOutList);

        ::std::lock_guard<nn::os::Mutex> guard(m_Mutex);

        NN_LCS_LOG_DEBUG("MakeContentShareSchedule\n");

        *pOutListCount = 0;
        ::std::memset(pOutList, 0, listCount * sizeof(RejectNodeInfo));

        // 配るコンテンツがあるか確認。なければ終了
        if (appInfo.deliveryInfoCount == 0)
        {
            NN_RESULT_SUCCESS;
        }

        // 配りたいバージョンのコンテンツを持つ端末がいなければ終了
        NN_RESULT_DO(CheckLatestApplication(appInfo, appIndex));

        // 配信できるかを確認。
        CheckNodeContents();

        // 受信情報をクリア
        ClearRequireDownload();

        // パッチ探し
        for (int i = 0; i < m_NodeCount; ++i)
        {
            // ホストの情報を優先してみたいからホストの num を探してセット
            int hostNum = 0;
            for (int j = 0; j < m_NodeCount; ++j)
            {
                if (m_NodeInfo[j].ipAddress == hostAddress)
                {
                    hostNum = j;
                    break;
                }
            }

            bool hasPatch = false;
            // パッチを持っているか確認
            NN_ABORT_UNLESS_RESULT_SUCCESS(HasAllContentsToDeliver(&hasPatch,
                m_NodeInfo[i].shareApp[appIndex].info.deliveryInfo, m_NodeInfo[i].shareApp[appIndex].info.deliveryInfoCount));

            // 最新持ってるか確認
            int value = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(CompareApplicationDeliveryInfo(&value,
                const_cast<ApplicationDeliveryInfo*>(appInfo.deliveryInfo), appInfo.deliveryInfoCount,
                m_NodeInfo[i].shareApp[appIndex].info.deliveryInfo,
                m_NodeInfo[i].shareApp[appIndex].info.deliveryInfoCount));

            // パッチを持っていないか、最新じゃなければパッチをもらえるかを確認する
            if (!hasPatch || (value > 0))
            {
                NN_LCS_LOG_DEBUG("%s[%x] Need App%d Update\n",
                    m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index, appIndex);
                m_NodeInfo[i].shareApp[appIndex].requireAppDownload.isRequire = true;

                auto result = CanReceivePatch(i, appInfo, appIndex, hostAddress);
                if (result.IsFailure())
                {
                    pOutList[*pOutListCount].index = m_NodeInfo[i].nodeInfo.index;

                    if (ResultFailureNodeLeaved::Includes(result))
                    {
                        pOutList[*pOutListCount].reason = ContentsShareFailureReason_NodeLeaved;
                    }
                    else if (ResultFailureCannotUpdateSystem::Includes(result))
                    {
                        pOutList[*pOutListCount].reason = ContentsShareFailureReason_CannotUpdateSystem;
                    }
                    else if (ResultFailureNotExistDownloadContents::Includes(result))
                    {
                        pOutList[*pOutListCount].reason = ContentsShareFailureReason_NotExistDownloadContents;
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    }
                    ++(*pOutListCount);
                }

            }
            // 更新が必要ない端末はこちら
            else
            {
                m_NodeInfo[i].shareApp[appIndex].requireAppDownload.isRequire = false;
            }
        }
        NN_RESULT_SUCCESS;
    }

    bool NodeManager::IsCanUpdateSystem(SystemDeliveryInfo receiverSysInfo) NN_NOEXCEPT
    {
        int index = 0;
        int count = 0;
        Ipv4Address address[NodeCountMax] = {};
        SystemDeliveryInfo senderSysInfo[NodeCountMax] = {};

        // VerifyDeliveryProtocolVersion で互換があるシステムを持つ端末をリストアップ
        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (VerifyDeliveryProtocolVersion(m_NodeInfo[i].sysInfo).IsSuccess())
            {
                address[count] = m_NodeInfo[i].ipAddress;
                senderSysInfo[count] = m_NodeInfo[i].sysInfo;
                ++count;
            }
        }

        // Master 自身は必ずリストアップされるので、0 < count を満たす
        NN_SDK_ASSERT(0 < count);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            SelectLatestSystemDeliveryInfo(&index, senderSysInfo, count, receiverSysInfo));
        NN_LCS_LOG_DEBUG("Index : %d\n", index);

        return 0 <= index && VerifyDeliveryProtocolVersion(senderSysInfo[index]).IsSuccess();
    }

    bool NodeManager::IsCanUpdateSystem(SystemDeliveryInfo receiverSysInfo,
        const ApplicationInfo& appInfo) NN_NOEXCEPT
    {
        int index = 0;
        int count = 0;
        SystemDeliveryInfo senderSysInfo[NodeCountMax] = {};

        // VerifyDeliveryProtocolVersion で互換があるシステムを持つ端末をリストアップ
        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (VerifyDeliveryProtocolVersion(m_NodeInfo[i].sysInfo).IsSuccess())
            {
                senderSysInfo[count] = m_NodeInfo[i].sysInfo;
                ++count;
            }
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            SelectLatestSystemDeliveryInfo(&index, senderSysInfo, count, receiverSysInfo,
                appInfo.deliveryInfo, appInfo.deliveryInfoCount));
        NN_LCS_LOG_DEBUG("Index : %d\n", index);
        if (index < 0)
        {
            return false;
        }
        return true;
    }

    Result NodeManager::CanReceivePatch(int nodeIndex,
        const ApplicationInfo& appInfo, int appIndex, Ipv4Address hostAddress) NN_NOEXCEPT
    {
        // LCSRequiredSystemVersion を満たしているか確認。満たしていなければ LUP 対象
        Result result = VerifyDeliveryProtocolVersion(m_NodeInfo[nodeIndex].sysInfo);
        if (result.IsFailure())
        {
            if (nn::ns::ResultSystemUpdateRequiredForLocalContentDelivery().Includes(result))
            {
                NN_LCS_LOG_DEBUG("%s[%x] Need SystemUpdate For LCSRequiredVersion\n",
                    m_NodeInfo[nodeIndex].nodeInfo.userName, m_NodeInfo[nodeIndex].nodeInfo.index, appIndex);
                // LUP ができるか確認
                if (!IsCanUpdateSystem(m_NodeInfo[nodeIndex].sysInfo))
                {
                    // システム更新できずで切断
                    return ResultFailureCannotUpdateSystem();
                }
            }
            else
            {
                // ここでその他のエラーはあり得ないので、Abort させる
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }

        bool isCanRecv = FindPatchSender(nodeIndex, false, false, appInfo, appIndex, hostAddress);
        NN_LCS_LOG_DEBUG("FindPatchSender(false/false) : %d\n", isCanRecv);
        if (!isCanRecv)
        {
            isCanRecv = FindPatchSender(nodeIndex, true, false, appInfo, appIndex, hostAddress);
            NN_LCS_LOG_DEBUG("FindPatchSender(true/false) : %d\n", isCanRecv);
            if (!isCanRecv)
            {
                isCanRecv = FindPatchSender(nodeIndex, true, true, appInfo, appIndex, hostAddress);
                NN_LCS_LOG_DEBUG("FindPatchSender(true/true) : %d\n", isCanRecv);
                if (!isCanRecv)
                {
                    // パッチもらえない端末
                    if (m_NodeInfo[nodeIndex].isReceive)
                    {
                        // 一度でも受信をしたことがあったあとの失敗なら NodeLeaved
                        return ResultFailureNodeLeaved();
                    }
                    else if (IsCanUpdateSystem(m_NodeInfo[nodeIndex].sysInfo, appInfo))
                    {
                        if (m_NodeInfo[nodeIndex].isCanUpdate)
                        {
                            return ResultFailureNodeLeaved();
                        }
                        else
                        {
                            // その他はパッチなくてダメ
                            return ResultFailureNotExistDownloadContents();
                        }
                    }
                    else
                    {
                        // パッチはもらえそうだったのでシステム更新できずで切断
                        return ResultFailureCannotUpdateSystem();
                    }
                }
            }
        }
        NN_RESULT_SUCCESS;
    }

    bool NodeManager::FindSystemSender(int nodeNum, Ipv4Address hostAddress) NN_NOEXCEPT
    {
        int index = 0;
        NodeDetailInfo detailInfo = {};

        // まず Host からもらえるか確認
        if (GetNodeDetailInfo(&detailInfo, GetNodeIndex(hostAddress)).IsFailure())
        {
            return false;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            SelectLatestSystemDeliveryInfo(&index, &detailInfo.sysInfo, 1, m_NodeInfo[nodeNum].sysInfo));
        if (index == 0 && VerifyDeliveryProtocolVersion(detailInfo.sysInfo).IsSuccess())
        {
            NN_LCS_LOG_DEBUG("Update %x from %x\n", m_NodeInfo[nodeNum].ipAddress, hostAddress);
            m_NodeInfo[nodeNum].requireSystemDownload.isRequire = true;
            m_NodeInfo[nodeNum].requireSystemDownload.formAddress = hostAddress;
            m_NodeInfo[nodeNum].requireSystemDownload.index = GetNodeIndex(hostAddress);
            return true;
        }

        int count = 0;
        Ipv4Address address[NodeCountMax] = {};
        SystemDeliveryInfo senderSysInfo[NodeCountMax] = {};

        // VerifyDeliveryProtocolVersion で互換があるシステムを持つ端末をリストアップ
        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (VerifyDeliveryProtocolVersion(m_NodeInfo[i].sysInfo).IsSuccess())
            {
                address[count] = m_NodeInfo[i].ipAddress;
                senderSysInfo[count] = m_NodeInfo[i].sysInfo;
                ++count;
            }
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            SelectLatestSystemDeliveryInfo(&index, senderSysInfo, count, m_NodeInfo[nodeNum].sysInfo));
        if (index >=0 && VerifyDeliveryProtocolVersion(senderSysInfo[index]).IsSuccess())
        {
            NN_LCS_LOG_DEBUG("Update %x from %x\n", m_NodeInfo[nodeNum].ipAddress, address[index]);
            m_NodeInfo[nodeNum].requireSystemDownload.isRequire = true;
            m_NodeInfo[nodeNum].requireSystemDownload.formAddress = address[index];
            m_NodeInfo[nodeNum].requireSystemDownload.index = GetNodeIndex(address[index]);
            return true;
        }
        return false;
    }

    bool NodeManager::FindSystemSender(int nodeNum, Ipv4Address hostAddress,
        const ApplicationInfo &appInfo) NN_NOEXCEPT
    {
        int index = 0;
        NodeDetailInfo detailInfo = {};

        // まず Host からもらえるか確認
        if (GetNodeDetailInfo(&detailInfo, GetNodeIndex(hostAddress)).IsFailure())
        {
            return false;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            SelectLatestSystemDeliveryInfo(&index, &detailInfo.sysInfo, 1, m_NodeInfo[nodeNum].sysInfo,
                appInfo.deliveryInfo, appInfo.deliveryInfoCount));
        if (index == 0)
        {
            NN_LCS_LOG_DEBUG("Update %x from %x\n", m_NodeInfo[nodeNum].ipAddress, hostAddress);
            m_NodeInfo[nodeNum].requireSystemDownload.isRequire = true;
            m_NodeInfo[nodeNum].requireSystemDownload.formAddress = hostAddress;
            m_NodeInfo[nodeNum].requireSystemDownload.index = GetNodeIndex(hostAddress);
            m_NodeInfo[nodeNum].isReceive = true;
            return true;
        }

        int count = 0;
        Ipv4Address address[NodeCountMax] = {};
        SystemDeliveryInfo senderSysInfo[NodeCountMax] = {};

        // VerifyDeliveryProtocolVersion で互換があるシステムを持つ端末をリストアップ
        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (VerifyDeliveryProtocolVersion(m_NodeInfo[i].sysInfo).IsSuccess())
            {
                address[count] = m_NodeInfo[i].ipAddress;
                senderSysInfo[count] = m_NodeInfo[i].sysInfo;
                ++count;
            }
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            SelectLatestSystemDeliveryInfo(&index, senderSysInfo, count, m_NodeInfo[nodeNum].sysInfo,
                appInfo.deliveryInfo, appInfo.deliveryInfoCount));
        if (index >= 0)
        {
            NN_LCS_LOG_DEBUG("Update %x from %x\n", m_NodeInfo[nodeNum].ipAddress, address[index]);
            m_NodeInfo[nodeNum].requireSystemDownload.isRequire = true;
            m_NodeInfo[nodeNum].requireSystemDownload.formAddress = address[index];
            m_NodeInfo[nodeNum].requireSystemDownload.index = GetNodeIndex(address[index]);
            m_NodeInfo[nodeNum].isReceive = true;
            return true;
        }
        return false;
    }

    bool NodeManager::FindPatchSender(int nodeNum, bool allowReceiverSystemUpdate, bool allowSenderSystemUpdate,
        const ApplicationInfo& appInfo, int appIndex, Ipv4Address hostAddress) NN_NOEXCEPT
    {
        // ホストの情報を優先してみたいからホストの num を探してセット
        int hostNum = 0;
        for (int j = 0; j < m_NodeCount; ++j)
        {
            if (m_NodeInfo[j].ipAddress == hostAddress)
            {
                hostNum = j;
                break;
            }
        }
        // パッチを持っていないか、最新じゃなければパッチをもらえるかを確認する
        int value = 0;
        for (int j = hostNum; j < m_NodeCount; ++j)
        {
            NN_LCS_LOG_DEBUG("Check[%d] : %s[%x] \n", j, m_NodeInfo[j].nodeInfo.userName, m_NodeInfo[j].nodeInfo.index);
            // 最新持ってる端末探し
            if (!m_NodeInfo[j].shareApp[appIndex].hasApp)
            {
                if (j == hostNum)
                {
                    j = -1;
                    hostNum = -1;
                }
                continue;
            }
            NN_LCS_LOG_DEBUG("%s[%x] has patch\n", m_NodeInfo[j].nodeInfo.userName, m_NodeInfo[j].nodeInfo.index);

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                CompareApplicationDeliveryInfo(&value,
                    const_cast<ApplicationDeliveryInfo*>(appInfo.deliveryInfo),
                    appInfo.deliveryInfoCount,
                    m_NodeInfo[j].shareApp[appIndex].info.deliveryInfo,
                    m_NodeInfo[j].shareApp[appIndex].info.deliveryInfoCount));
            if (value > 0)
            {
                if (j == hostNum)
                {
                    j = -1;
                    hostNum = -1;
                }
                continue;
            }
            NN_LCS_LOG_DEBUG("%s[%x] has latest patch\n", m_NodeInfo[j].nodeInfo.userName, m_NodeInfo[j].nodeInfo.index);

            bool isNeedSenderSystemUpdate = false;
            // 最新持ちが LCSRequiredSystemVersion を満たしているか確認
            if (VerifyDeliveryProtocolVersion(m_NodeInfo[j].sysInfo).IsFailure())
            {
                if (allowSenderSystemUpdate && IsCanUpdateSystem(m_NodeInfo[j].sysInfo))
                {
                    // 継続
                    isNeedSenderSystemUpdate = true;
                }
                else
                {
                    if (j == hostNum)
                    {
                        j = -1;
                        hostNum = -1;
                    }
                    continue;
                }
            }

            // 最新持ちからもらえるか確認
            bool isDelivery;
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                CanDeliverApplication(&isDelivery,
                    m_NodeInfo[nodeNum].shareApp[appIndex].info.deliveryInfo,
                    m_NodeInfo[nodeNum].shareApp[appIndex].info.deliveryInfoCount,
                    m_NodeInfo[j].shareApp[appIndex].info.deliveryInfo,
                    m_NodeInfo[j].shareApp[appIndex].info.deliveryInfoCount));
            NN_LCS_LOG_DEBUG("%s[%x] can Delivery? : %d\n",
                m_NodeInfo[j].nodeInfo.userName, m_NodeInfo[j].nodeInfo.index, isDelivery);
            if (!isDelivery)
            {
                if (j == hostNum)
                {
                    j = -1;
                    hostNum = -1;
                }
                continue;
            }
            bool isNeedReceiverSystemUpdate = false;
            // 受信するためにシステム更新がいるか確認
            if (VerifyDeliveryProtocolVersion(m_NodeInfo[nodeNum].sysInfo).IsFailure())
            {
                isNeedReceiverSystemUpdate = true;
            }
            else
            {
                // 受信したアプリを起動するためにシステム更新がいるか確認
                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    NeedsSystemUpdateToDeliverApplication(&isNeedReceiverSystemUpdate,
                        m_NodeInfo[j].shareApp[appIndex].info.deliveryInfo,
                        m_NodeInfo[j].shareApp[appIndex].info.deliveryInfoCount,
                        m_NodeInfo[nodeNum].sysInfo));
            }
            NN_LCS_LOG_DEBUG("Receiver need SystemUpdate? : %d\n", isNeedReceiverSystemUpdate);

            if (allowSenderSystemUpdate)
            {
                if (allowReceiverSystemUpdate)
                {
                    if (isNeedSenderSystemUpdate)
                    {
                        if (IsCanUpdateSystem(m_NodeInfo[j].sysInfo))
                        {
                            if (isNeedReceiverSystemUpdate)
                            {
                                if (FindSystemSender(nodeNum, hostAddress, m_NodeInfo[j].shareApp[appIndex].info))
                                {
                                    if (FindSystemSender(j, hostAddress))
                                    {
                                        NN_LCS_LOG_DEBUG("Get Patch! - 1\n");
                                        m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.formAddress = m_NodeInfo[j].ipAddress;
                                        m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.index = m_NodeInfo[j].nodeInfo.index;
                                        m_NodeInfo[nodeNum].isCanUpdate = true;
                                        m_NodeInfo[nodeNum].isReceive = true;
                                        return true;
                                    }
                                }
                            }
                            else
                            {
                                if (FindSystemSender(j, hostAddress))
                                {
                                    NN_LCS_LOG_DEBUG("Get Patch - 2!\n");
                                    m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.formAddress = m_NodeInfo[j].ipAddress;
                                    m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.index = m_NodeInfo[j].nodeInfo.index;
                                    m_NodeInfo[nodeNum].isCanUpdate = true;
                                    m_NodeInfo[nodeNum].isReceive = true;
                                    return true;
                                }
                            }
                        }
                    }
                    else
                    {
                        if (isNeedReceiverSystemUpdate)
                        {
                            if (FindSystemSender(nodeNum, hostAddress, m_NodeInfo[j].shareApp[appIndex].info))
                            {
                                NN_LCS_LOG_DEBUG("Get Patch! - 3\n");
                                m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.formAddress = m_NodeInfo[j].ipAddress;
                                m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.index = m_NodeInfo[j].nodeInfo.index;
                                m_NodeInfo[nodeNum].isCanUpdate = true;
                                m_NodeInfo[nodeNum].isReceive = true;
                                return true;
                            }
                        }
                        else
                        {
                            NN_LCS_LOG_DEBUG("Get Patch! - 4\n");
                            m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.formAddress = m_NodeInfo[j].ipAddress;
                            m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.index = m_NodeInfo[j].nodeInfo.index;
                            m_NodeInfo[nodeNum].isCanUpdate = true;
                            m_NodeInfo[nodeNum].isReceive = true;
                            return true;
                        }
                    }
                }
                else
                {
                    if (!isNeedReceiverSystemUpdate)
                    {
                        if (isNeedSenderSystemUpdate)
                        {
                            if (FindSystemSender(j, hostAddress))
                            {
                                NN_LCS_LOG_DEBUG("Get Patch! - 5\n");
                                m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.formAddress = m_NodeInfo[j].ipAddress;
                                m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.index = m_NodeInfo[j].nodeInfo.index;
                                m_NodeInfo[nodeNum].isCanUpdate = true;
                                m_NodeInfo[nodeNum].isReceive = true;
                                return true;
                            }
                        }
                    }
                }
            }
            else
            {
                if (allowReceiverSystemUpdate)
                {
                    if (!isNeedSenderSystemUpdate)
                    {
                        if (isNeedReceiverSystemUpdate)
                        {
                            if (FindSystemSender(nodeNum, hostAddress, m_NodeInfo[j].shareApp[appIndex].info))
                            {
                                NN_LCS_LOG_DEBUG("Get Patch! - 6\n");
                                m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.formAddress = m_NodeInfo[j].ipAddress;
                                m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.index = m_NodeInfo[j].nodeInfo.index;
                                m_NodeInfo[nodeNum].isCanUpdate = true;
                                m_NodeInfo[nodeNum].isReceive = true;
                                return true;
                            }
                        }
                        else
                        {
                            NN_LCS_LOG_DEBUG("Get Patch! - 7\n");
                            m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.formAddress = m_NodeInfo[j].ipAddress;
                            m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.index = m_NodeInfo[j].nodeInfo.index;
                            m_NodeInfo[nodeNum].isCanUpdate = true;
                            m_NodeInfo[nodeNum].isReceive = true;
                            return true;
                        }
                    }
                }
                else
                {
                    if (!isNeedSenderSystemUpdate && !isNeedReceiverSystemUpdate)
                    {
                        NN_LCS_LOG_DEBUG("Get Patch! - 8\n");
                        m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.formAddress = m_NodeInfo[j].ipAddress;
                        m_NodeInfo[nodeNum].shareApp[appIndex].requireAppDownload.index = m_NodeInfo[j].nodeInfo.index;
                        m_NodeInfo[nodeNum].isCanUpdate = true;
                        m_NodeInfo[nodeNum].isReceive = true;
                        return true;
                    }
                }
            }
            // このパッチはダメだからパッチ探しへ戻る
            if (j == hostNum)
            {
                j = -1;
                hostNum = -1;
            }
            continue;
        }
        return false;
    } // NOLINT(impl/function_size)


    void NodeManager::ClearDownloadedCount() NN_NOEXCEPT
    {
        for (int i = 0; i < m_NodeCount; ++i)
        {
            m_NodeInfo[i].downloadedCount = 0;
        }
    }

    void NodeManager::GetNodeProgress(int* pOutProgressCount,
        NodeDetailProgress* outProgress, int progressCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutProgressCount);
        NN_SDK_ASSERT_NOT_NULL(outProgress);

        *pOutProgressCount = 0;

        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (progressCount == i)
            {
                return;
            }

            int count = 0;
            if (m_NodeInfo[i].requireSystemDownload.isRequire)
            {
                ++count;
            }
            for (int j = 0; j < SharableContentsCountMax; ++j)
            {
                if (m_NodeInfo[i].shareApp[j].requireAppDownload.isRequire)
                {
                    ++count;
                }
            }

            NN_LCS_LOG_DEBUG("GetNodeProgress[%d] downloadedCount : %d\n", i, m_NodeInfo[i].downloadedCount);
            outProgress[i].index = m_NodeInfo[i].nodeInfo.index;
            outProgress[i].progress.downloadedContentCount = static_cast<int8_t>(m_NodeInfo[i].downloadedCount);
            outProgress[i].progress.contentCount = static_cast<int8_t>(count + m_NodeInfo[i].downloadedCount);
            (*pOutProgressCount)++;
        }
    }

    bool NodeManager::IsCompletedSystemDownloadAll() NN_NOEXCEPT
    {
        for (int i = 0; i < m_NodeCount; ++i)
        {
            if (m_NodeInfo[i].requireSystemDownload.isRequire)
            {
                NN_LCS_LOG_DEBUG("%s[%x] require System\n",
                    m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index);
                return false;
            }
        }
        return true;
    }

    bool NodeManager::IsCompletedContentDownloadAll() NN_NOEXCEPT
    {
        for (int i = 0; i < m_NodeCount; ++i)
        {
            for (int j = 0; j < SharableContentsCountMax; ++j)
            {
                if (m_NodeInfo[i].shareApp[j].requireAppDownload.isRequire)
                {
                    NN_LCS_LOG_DEBUG("%s[%x] require App%d\n",
                        m_NodeInfo[i].nodeInfo.userName, m_NodeInfo[i].nodeInfo.index, j);
                    return false;
                }
            }
        }
        return true;
    }


}}} // namespace nn::lcs::detail
