﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/lcs_Result.private.h>
#include <nn/lcs/lcs_Types.h>
#include <nn/lcs/lcs_Config.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/lcs/detail/lcs_ApplicationShareApi.h>
#include <nn/lcs/detail/lcs_Definition.h>
#include <nn/lcs/detail/lcs_Client.h>
#include <nn/lcs/detail/lcs_NetworkApi.h>
#include <nn/lcs/detail/lcs_Socket.h>
#include <nn/lcs/detail/lcs_SystemShareApi.h>
#include <nn/lcs/detail/Advertise/lcs_AdvertiseUtility.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/lcs/detail/Packet/lcs_Packet.h>
#include <nn/lcs/detail/Packet/lcs_PacketManager.h>
#include <nn/lcs/detail/Packet/lcs_PacketReceiver.h>
#include <nn/lcs/detail/Packet/lcs_PacketUtilty.h>
#include <nn/ldn.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>

namespace nn { namespace lcs { namespace detail
{

    namespace
    {
        nn::os::Mutex g_Mutex(false);
    }

    Client::Client() NN_NOEXCEPT
    {
        ClearMemory();
    };

    void Client::ClearMemory() NN_NOEXCEPT
    {
        m_pResource = nullptr;

        m_IsConnected = false;
        m_IsInitializedClient = false;
        m_IsShareCompleted = false;
        m_IsAcceptSystemUpdate = false;
        m_IsNeedReboot = false;

        m_Socket = 0;
        m_MyAddress = 0;
        m_MasterAddress = 0;

        ::std::memset(&m_AdvertiseData, 0, sizeof(AdvertiseData));
        ::std::memset(m_NodeProgress, 0, sizeof(AdvertiseNodeProgress) * NodeCountMax);

        m_Role = TransferRole_None;
        m_NodeIndex = 0;

        m_RequiredSize = 0;
        m_IsWaitReceiver = false;
        m_TotalSendSize = 0;
    }

    void Client::ChangeState(State state, bool isDisconnect) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("ChangeState / %d -> %d(FailuerR : %d / SuspendR%d)\n",
            m_pResource->state.GetState(), state, m_pResource->contentsShareFailureReason, m_pResource->suspendedReason);

        if (m_pResource->state.GetState() == State_Completed || m_pResource->state.GetState() == state)
        {
            return;
        }

        if (state == State_ContentsShareFailed)
        {
            if (m_IsNeedReboot)
            {
                return;
            }
            SendLeaveSession(m_Socket, LeaveReason_None);
            FinalizeSocket();
            if (!isDisconnect)
            {
                nn::os::SignalEvent(&m_CancelEvent);
                nn::os::WaitThread(&m_pResource->monitorThread);
                GetAdvertiseData(&m_AdvertiseData);
                Disconnect();
            }
        }
        else if (state == State_Completed)
        {
            m_Role = TransferRole_None;
            FinalizeSocket();
            nn::os::SignalEvent(&m_CancelEvent);
            nn::os::WaitThread(&m_pResource->monitorThread);
            GetAdvertiseData(&m_AdvertiseData);
            Disconnect();
        }

        m_pResource->state.SetState(state);
        NN_LCS_LOG_DEBUG("State is Changed : SignalEvent\n");
        m_pResource->stateEvent.SignalEvent();
    }

    Result Client::InitializeClient() NN_NOEXCEPT
    {
        m_IsConnected = true;

        if (!m_IsInitializedClient)
        {
            nn::os::InitializeEvent(&m_SessionInfoEvent, false, nn::os::EventClearMode_ManualClear);
            nn::os::InitializeEvent(&m_StartTransferEvent, false, nn::os::EventClearMode_AutoClear);
            nn::os::InitializeEvent(&m_CancelEvent, false, nn::os::EventClearMode_ManualClear);
            nn::os::InitializeEvent(&m_ResumeEvent, false, nn::os::EventClearMode_ManualClear);

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::os::CreateThread(&m_pResource->monitorThread, this->MonitorThreadEntry, this,
                    m_pResource->monitorThreadStack, sizeof(m_pResource->monitorThreadStack),
                    m_pResource->monitorThreadPriority));
            nn::os::StartThread(&m_pResource->monitorThread);

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::os::CreateThread(&m_pResource->comThread, this->ComThreadEntry, this,
                    m_pResource->comThreadStack, sizeof(m_pResource->comThreadStack),
                    m_pResource->comThreadPriority));
            nn::os::StartThread(&m_pResource->comThread);

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::os::CreateThread(&m_pResource->shareThread, this->ShareThreadEntry, this,
                    m_pResource->shareThreadStack, sizeof(m_pResource->shareThreadStack),
                    m_pResource->shareThreadPriority));
            nn::os::StartThread(&m_pResource->shareThread);

            m_IsInitializedClient = true;
        }

        AdvertiseData advertiseData = {};

        if (GetAdvertiseData(&advertiseData).IsFailure())
        {
            return ResultCommunicationError();
        }
        if (!ConvertAdvertiseDataToSessionInfo(&m_pResource->sessionInfo, advertiseData))
        {
            return ResultCommunicationError();
        }

        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_None;
        m_pResource->suspendedReason = SuspendedReason_None;

        NN_RESULT_SUCCESS;
    }

    void Client::FinalizeClient() NN_NOEXCEPT
    {
        m_IsConnected = false;

        if (m_IsInitializedClient)
        {
            nn::os::SignalEvent(&m_SessionInfoEvent);
            nn::os::SignalEvent(&m_StartTransferEvent);
            nn::os::SignalEvent(&m_CancelEvent);
            nn::os::SignalEvent(&m_ResumeEvent);

            nn::os::WaitThread(&m_pResource->monitorThread);
            nn::os::WaitThread(&m_pResource->comThread);
            nn::os::WaitThread(&m_pResource->shareThread);

            nn::os::DestroyThread(&m_pResource->monitorThread);
            nn::os::DestroyThread(&m_pResource->comThread);
            nn::os::DestroyThread(&m_pResource->shareThread);

            nn::os::FinalizeEvent(&m_SessionInfoEvent);
            nn::os::FinalizeEvent(&m_StartTransferEvent);
            nn::os::FinalizeEvent(&m_CancelEvent);
            nn::os::FinalizeEvent(&m_ResumeEvent);

            m_IsInitializedClient = false;
        }

        ClearMemory();
    }

    Result Client::InitializeSocket(Ipv4Address address, int retryCountMax) NN_NOEXCEPT
    {
        int retryCount = 0;
        while (retryCount < retryCountMax)
        {
            if (SocketOpen(&m_Socket) == 0)
            {
                break;
            }
            retryCount++;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(SocketInitializeRetryIntervalMilliSecond));
        }
        if (retryCount == retryCountMax)
        {
            NN_LCS_LOG_DEBUG("SocketOpen Error!\n");
            return ResultSocketOpenError();
        }

        NN_LCS_LOG_INFO("Socket Connect\n");
        while (retryCount < retryCountMax)
        {
            if (SocketConnect(&m_Socket, address) == 0)
            {
                break;
            }
            retryCount++;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(SocketInitializeRetryIntervalMilliSecond));
        }
        if (retryCount == retryCountMax)
        {
            NN_LCS_LOG_DEBUG("SocketConnect Error!\n");
            SocketClose(&m_Socket);
            return ResultSocketConnectError();
        }
        NN_LCS_LOG_INFO("Socket Connect End\n");

        if (nn::socket::Fcntl(m_Socket, nn::socket::FcntlCommand::F_SetFl, static_cast<int>(nn::socket::FcntlFlag::O_NonBlock)) != 0)
        {
            SocketClose(&m_Socket);
            return ResultSocketFcntlError();
        }
        NN_RESULT_SUCCESS;
    }

    void Client::FinalizeSocket()  NN_NOEXCEPT
    {
        SocketClose(&m_Socket);
    }

    Result Client::Join(const SessionInfo& session, LcsResources* resource) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(resource);

        Result result;

        m_pResource = resource;

        // 本体を持っているかと、パッチが適用中で配信できないとかがないか確認
        for (int i = 0; i < session.contentsCount; ++i)
        {
            result = CanGetApplicationInfo(session.contents[i].applicationId.value);
            if (result.IsFailure())
            {
                if (ResultApplicationNotFound::Includes(result))
                {
                    return ResultNoSharableContentsSession();
                }
                else if (ResultApplyDeltaTaskExists::Includes(result))
                {
                    return ResultApplyDeltaTaskExists();
                }
            }
        }

        NetworkSetting setting = {};
        ::std::memcpy(&setting, session._internal, NetworkSettingSize);

        NN_RESULT_DO(Connect(setting, static_cast<int16_t>(GetMajorVersion(m_pResource->version))));
        NN_LCS_LOG_DEBUG("Connect Success\n\n");

        if (GetMyIpv4Address(&m_MyAddress).IsFailure())
        {
            Disconnect();
            return ResultCommunicationError();
        }
        if (GetAccessPointIpv4Address(&m_pResource->hostAddress).IsFailure())
        {
            Disconnect();
            return ResultCommunicationError();
        }
        m_MasterAddress = m_pResource->hostAddress;

        NN_LCS_LOG_DEBUG("My IpAddress : %x / Host IpAddress : %x\n", m_MyAddress, m_MasterAddress);

        result = InitializeSocket(m_MasterAddress, 1);
        if (result.IsFailure())
        {
            Disconnect();
            return result;
        }

        NN_LCS_LOG_DEBUG("Send JoinSession\n");
        SystemDeliveryInfo sysInfo;
        GetSystemDeliveryInfo(&sysInfo);
        result = SendNodeDetailInfo(
            m_Socket, m_pResource->userName, m_pResource->version, sysInfo,
            session, m_pResource->contentInfoCount, m_pResource->appControlDataBuffer, AppControlDataSize, false);
        if (result.IsFailure())
        {
            FinalizeSocket();
            Disconnect();
            return result;
        }
        result = WaitJoinSessionRes(&m_pResource->myIndex, m_Socket, m_pResource->packetBuffer, PacketSize);
        if (result.IsFailure())
        {
            FinalizeSocket();
            Disconnect();
            return result;
        }

        InitializeClient();

        // SessionInfo がもらえるまで待つ
        if (!WaitSessionInfo())
        {
            FinalizeSocket();
            FinalizeClient();
            Disconnect();
            return ResultNoResponse();
        }

        ::std::memcpy(&(m_pResource->currentNetworkSetting), &setting, NetworkSettingSize);

        return result;
    }

    Result Client::Rejoin(LcsResources* resource) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(resource);

        NN_LCS_LOG_DEBUG("Rejoin\n");

        m_pResource = resource;
        m_IsConnected = true;

        NN_RESULT_SUCCESS;
    }

    Result Client::ConnectMigrationHost() NN_NOEXCEPT
    {
        int count = static_cast<int>(Timeout.GetMilliSeconds() / RetryInterval.GetMilliSeconds());
        bool isSuccessed = false;

        for (int i = 0; i < count; ++i)
        {
            {
                // Leave されないようにロック
                ::std::lock_guard<nn::os::Mutex> guard(g_Mutex);

                if (!m_IsConnected)
                {
                    NN_LCS_LOG_DEBUG("Leave ConnectMigrationHost\n");
                    NN_RESULT_SUCCESS;
                }

                // LDN の Private 接続
                NN_LCS_LOG_DEBUG("TryConnect\n");
                if (Connect(m_pResource->currentNetworkSetting, static_cast<int16_t>(GetMajorVersion(m_pResource->version))).IsSuccess())
                {
                    Result result;

                    NN_UTIL_SCOPE_EXIT
                    {
                        if (!isSuccessed)
                        {
                            NN_LCS_LOG_DEBUG("ConnectMigrationHost Fail\n");
                            Disconnect();
                        }
                    };

                    NN_RESULT_DO(GetMyIpv4Address(&m_MyAddress));
                    NN_RESULT_DO(GetAccessPointIpv4Address(&m_MasterAddress));
                    NN_LCS_LOG_DEBUG("My IpAddress : %x / Host IpAddress : %x\n", m_MyAddress, m_MasterAddress);

                    NN_RESULT_DO(InitializeSocket(m_MasterAddress, SocketInitializeRetryCountMax));

                    InitializeClient();

                    NN_LCS_LOG_DEBUG("Send NodeDetailInfo\n");
                    SystemDeliveryInfo sysInfo;
                    GetSystemDeliveryInfo(&sysInfo);
                    NN_RESULT_DO(SendNodeDetailInfo(m_Socket, m_pResource->userName, m_pResource->version, sysInfo,
                        m_pResource->sessionInfo, m_pResource->contentInfoCount,
                        m_pResource->appControlDataBuffer, AppControlDataSize, true));
                    // SessionInfo がもらえるまで待つ
                    if (!WaitSessionInfo())
                    {
                        NN_LCS_LOG_DEBUG("WaitSessionInfo Error!\n");
                        FinalizeSocket();
                        FinalizeClient();
                        return ResultNoResponse();
                    }

                    NN_LCS_LOG_DEBUG("Rejoin Success\n");
                    isSuccessed = true;
                    NN_RESULT_SUCCESS;
                }
                NN_LCS_LOG_DEBUG("Connect Failed\n");
            }
            nn::os::SleepThread(RetryInterval);
        }
        return ResultConnectionFailed();
    }

    void Client::Leave(LeaveType type) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Leave\n");

        // Migration と衝突しないようにロック
        ::std::lock_guard<nn::os::Mutex> guard(g_Mutex);

        if (m_IsConnected)
        {
            LeaveReason reason = LeaveReason_None;
            m_Role = TransferRole_None;

            if (type == LeaveType_Temporary &&
                (m_pResource->suspendedReason == SuspendedReason_RebootRequired ||
                    m_pResource->suspendedReason == SuspendedReason_StorageSpaceNotEnough))
            {
                reason = LeaveReason_Reboot;
            }
            NN_LCS_LOG_DEBUG("Send Leave : %d\n", reason);
            SendLeaveSession(m_Socket, reason);
            // ToDo : Leaveを送った後、LDNの切断が速すぎて、Hostに受信されるまでに切断されてしまうのを回避
            //        そのうち、Leave 送ったら Host から Reject を受け取って、そこで切断するようにする
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            FinalizeSocket();
            FinalizeClient();
            Disconnect();
        }
    }

    Result Client::Resume(const SessionContext& context, LcsResources* resource) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(resource);

        NN_LCS_LOG_DEBUG("Resume\n");

        ResumeContext resumeContext;
        ::std::memcpy(&resumeContext, &context, sizeof(ResumeContext));

        // 再起動要因が正しいか確認
        if (resumeContext.suspendedReason != SuspendedReason_RebootRequired &&
            resumeContext.suspendedReason != SuspendedReason_StorageSpaceNotEnough)
        {
            return ResultInvalidContext();
        }

        m_pResource = resource;

        NetworkSetting setting = {};

        ::std::memcpy(&setting, resumeContext.sessionInfo._internal, NetworkSettingSize);

        // LDN の Private 接続
        NN_RESULT_DO(Connect(setting, static_cast<int16_t>(GetMajorVersion(m_pResource->version))));
        NN_LCS_LOG_DEBUG("Connect Success\n");

        m_MasterAddress = resumeContext.hostAddress;
        if (GetMyIpv4Address(&m_MyAddress).IsFailure())
        {
            Disconnect();
            return ResultCommunicationError();
        }
        NN_LCS_LOG_DEBUG("My IpAddress : %x\n\n", m_MyAddress);
        NN_LCS_LOG_DEBUG("Socket Init : %x\n", m_MasterAddress);

        if (InitializeSocket(m_MasterAddress, 1).IsFailure())
        {
            Disconnect();
            return ResultCommunicationError();
        }
        NN_LCS_LOG_DEBUG("Socket Init Success\n");

        InitializeClient();
        NN_LCS_LOG_DEBUG("Client Init Success\n");

        NN_ABORT_UNLESS_LESS_EQUAL(resumeContext.contentsCount, DownloadableContentsCountMax);
        m_pResource->contentInfoCount = resumeContext.contentsCount;
        NN_LCS_LOG_DEBUG("m_pResource->contentInfoCount : %d\n", m_pResource->contentInfoCount);
        for (int i = 0; i < DownloadableContentsCountMax; ++i)
        {
            ::std::memcpy(&m_pResource->contentInfo[i], &resumeContext.contentsInfo[i], sizeof(ContentsInfo));
        }
        ::std::memcpy(&m_pResource->sessionInfo, &resumeContext.sessionInfo, sizeof(SessionInfo));

        m_pResource->suspendedReason = resumeContext.suspendedReason;

        bool isSuccess = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                Leave(LeaveType_Permanent);
            }
        };

        // システム更新できているか確認
        if (m_pResource->suspendedReason == SuspendedReason_RebootRequired)
        {
            int value = 0;
            SystemDeliveryInfo sysInfo = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(GetSystemDeliveryInfo(&sysInfo));
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                CompareSystemDeliveryInfo(&value, sysInfo, resumeContext.systemDeliveryInfo));
            NN_LCS_LOG_DEBUG("value : %d\n", value);
            if (value < 0)
            {
                m_IsNeedReboot = true;
            }
            else if (value > 0)
            {
                return ResultUnexpectedSystemVersion();
            }
        }

        m_RequiredSize = resumeContext.requiredStorageSize;
        m_pResource->myIndex = resumeContext.myIndex;

        for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
        {
            // 適用タスクが実行されているかを確認
            auto result = CanGetApplicationInfo(m_pResource->sessionInfo.contents[i].applicationId.value);
            if (result.IsFailure())
            {
                if (ResultApplicationNotFound::Includes(result))
                {
                    return ResultNoSharableContentsSession();
                }
                else if (ResultApplyDeltaTaskExists::Includes(result))
                {
                    return ResultApplyDeltaTaskExists();
                }
                else
                {
                    NN_ABORT("CanGetApplicationInfo Error : %08x\n", result.GetInnerValueForDebug());
                }
            }
        }

        // ApplicationDeliveyInfo が変わっていないか確認
        for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
        {
            GetApplicationDeliveryInfo(&m_pResource->ownAppInfo[i].deliveryInfoCount,
                m_pResource->ownAppInfo[i].deliveryInfo, ApplicationDeliveryInfoCountMax,
                m_pResource->sessionInfo.contents[i].applicationId.value);

            NN_LCS_LOG_DEBUG("AppID %llx\n", m_pResource->sessionInfo.contents[i].applicationId.value);
            for (int j = 0; j < resumeContext.appHashCount; ++j)
            {
                NN_LCS_LOG_DEBUG("ResumeContext AppID %llx\n", resumeContext.appHash[j].id);
                if (m_pResource->sessionInfo.contents[i].applicationId.value == resumeContext.appHash[j].id &&
                    IsApplicationDeliveryInfoHash(resumeContext.appHash[j]))
                {
                    char hash[ApplicationDeliveryInfoHashSize] = {};
                    NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationDeliveryInfoHash(
                        hash, ApplicationDeliveryInfoHashSize, m_pResource->ownAppInfo[i]));
                    NN_RESULT_THROW_UNLESS(
                        std::memcmp(hash, resumeContext.appHash[j].hash, ApplicationDeliveryInfoHashSize) == 0,
                        ResultUnexpectedApplicationVersion());
                }
            }
        }

        isSuccess = true;

        // 自分の情報をホストに送ってアップデートしてもらう
        SystemDeliveryInfo sysInfo;
        GetSystemDeliveryInfo(&sysInfo);
        SendNodeDetailInfo(m_Socket, m_pResource->userName, m_pResource->version, sysInfo,
            m_pResource->sessionInfo, m_pResource->contentInfoCount,
            m_pResource->appControlDataBuffer, AppControlDataSize, true);

        // SessionInfo がもらえるまで待つ
        if (!WaitSessionInfo())
        {
            FinalizeSocket();
            FinalizeClient();
            Disconnect();
            return ResultNoResponse();
        }

        ResumeShare();

        NN_RESULT_SUCCESS;
    }// NOLINT(impl/function_size)

    bool Client::IsInitialized() NN_NOEXCEPT
    {
        return m_IsConnected;
    }

    void Client::UpdateSessionInfo() NN_NOEXCEPT
    {
        GetSessionInfoFromNetwork(&m_pResource->sessionInfo);
    }

    void Client::UpdateRequiredStorageSize() NN_NOEXCEPT
    {
        size_t size = 0;
        ApplicationDeliveryInfo appDeliveryInfo[ApplicationDeliveryInfoCountMax] = {};
        int appDeliveryInfoCount = 0;

        for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
        {
            // 自分の情報を取得
            if (GetApplicationDeliveryInfo(
                &appDeliveryInfoCount, appDeliveryInfo, ApplicationDeliveryInfoCountMax,
                m_pResource->sessionInfo.contents[i].applicationId.value).IsFailure())
            {
                size += static_cast<size_t>(m_pResource->shareAppInfo[i].detailInfo.requiredSize);
                continue;
            }
            // 自分がパッチを持っていなければ必要サイズをそのまま加える
            bool hasPatch = false;
            NN_ABORT_UNLESS_RESULT_SUCCESS(HasAllContentsToDeliver(&hasPatch, appDeliveryInfo, appDeliveryInfoCount));
            if (!hasPatch)
            {
                size += static_cast<size_t>(m_pResource->shareAppInfo[i].detailInfo.requiredSize);
                continue;
            }
            // 自分のパッチと配信しているパッチで比較する
            if (m_pResource->shareAppInfo[i].deliveryInfoCount != 0)
            {
                int value = 0;
                if (CompareApplicationDeliveryInfo(
                    &value,
                    appDeliveryInfo,
                    appDeliveryInfoCount,
                    m_pResource->shareAppInfo[i].deliveryInfo,
                    m_pResource->shareAppInfo[i].deliveryInfoCount).IsSuccess())
                {
                    if (value < 0)
                    {
                        size += static_cast<size_t>(m_pResource->shareAppInfo[i].detailInfo.requiredSize);
                        continue;
                    }
                }
            }
        }
        m_pResource->requiredStorageSize = size;
    }

    void Client::GetSessionState(SessionState* pOutState) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutState);

        if (GetAdvertiseData(&m_AdvertiseData).IsSuccess())
        {
            *pOutState = static_cast<SessionState>(m_AdvertiseData.sessionState);
        }
        else
        {
            *pOutState = SessionState_None;
        }
    }

    void Client::GetProgress(Progress* pOutProgress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutProgress);

        uint64_t downloadedSize = 0;
        uint64_t size = 0;
        uint8_t role = m_Role;
        uint32_t transferIndex = 0;

        if (m_Role == TransferRole_None)
        {
            // 送信完了後、相手を待っているときの処理
            if (m_IsWaitReceiver)
            {
                downloadedSize = m_TotalSendSize + 1;
                size = m_TotalSendSize + 1;
                role = TransferRole_Sender;
                transferIndex = m_NodeIndex;
            }
            // 通常の進捗処理
            else
            {
                downloadedSize = 0;
                size = 0;
                transferIndex = 0;
            }
        }
        else
        {
            if (m_pResource->contentInfo[m_pResource->contentInfoCount].contentsFlag.Test(ContentsType::System::Index))
            {
                if (m_Role == TransferRole_Sender)
                {
                    GetSendSystemUpdateProgress(&downloadedSize, &size);
                }
                else if (m_Role == TransferRole_Receiver)
                {
                    GetReceiveSystemUpdateProgress(&downloadedSize, &size, &(m_pResource->systemUpdateControl));
                }
            }
            else
            {
                if (m_Role == TransferRole_Sender)
                {
                    GetSendApplicationProgress(&downloadedSize, &size, m_pResource->contentInfo[m_pResource->contentInfoCount].applicationId.value);
                }
                else if (m_Role == TransferRole_Receiver)
                {
                    GetReceiveApplicationProgress(&downloadedSize, &size, m_pResource->contentInfo[m_pResource->contentInfoCount].applicationId.value);
                }
            }
            transferIndex = m_NodeIndex;

            // 取得した進捗で、SystemUpdate のサイズが total に反映されないようなので対処
            if (downloadedSize >= size)
            {
                downloadedSize = size;
            }
            // 相手の commit 待ちを表す進捗として +1 しておく
            // なので、常に実際に送受信するサイズより size(=送受信するサイズ)は 1Byte 多いものになる。
            size += 1;
        }

        pOutProgress->downloadedSize = downloadedSize;
        pOutProgress->size = size;
        pOutProgress->transferRole = role;
        pOutProgress->transferIndex = transferIndex;
        ::std::memcpy(&(pOutProgress->info), &m_pResource->contentInfo[m_pResource->contentInfoCount], sizeof(ContentsInfo));
    }

    Result Client::GetNodeProgress(NodeProgress* pOutNodeProgress, uint32_t index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutNodeProgress);

        for (int i = 0; i < NodeCountMax; ++i)
        {
            if (m_IsShareCompleted)
            {
                if (nn::socket::InetNtohl(m_NodeProgress[i].id) == index)
                {
                    pOutNodeProgress->contentCount = m_NodeProgress[i].contentCount;
                    pOutNodeProgress->downloadedContentCount = m_NodeProgress[i].downloadedContentCount;
                    NN_RESULT_SUCCESS;
                }
            }
            else
            {
                if (nn::socket::InetNtohl(m_AdvertiseData.progress[i].id) == index)
                {
                    pOutNodeProgress->contentCount = m_AdvertiseData.progress[i].contentCount;
                    pOutNodeProgress->downloadedContentCount = m_AdvertiseData.progress[i].downloadedContentCount;
                    NN_RESULT_SUCCESS;
                }
            }
        }
        return ResultNodeNotFound();
    }

    Result Client::GetEulaDataSize(size_t* pOutSize, const char* eulaDataPath) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(eulaDataPath);

        return GetDownloadedEulaDataSize(pOutSize, eulaDataPath, &(m_pResource->systemUpdateControl));
    }

    Result Client::GetEulaData(
        size_t* pOutSize, void* buffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_NOT_NULL(eulaDataPath);

        return GetDownloadedEulaData(
            pOutSize, buffer, bufferSize, eulaDataPath, &(m_pResource->systemUpdateControl));
    }

    Result Client::GetResumeContext(SessionContext* pOutSessionContext) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSessionContext);

        ::std::memcpy(pOutSessionContext->_context, &m_pResource->context, sizeof(ResumeContext));

        NN_RESULT_SUCCESS;
    }

    Result Client::ResumeShare() NN_NOEXCEPT
    {
        ChangeState(State_Transferring);

        if (m_pResource->suspendedReason == SuspendedReason_IncompatibleContentsInfo)
        {
            m_pResource->suspendedReason = SuspendedReason_None;
            m_IsAcceptSystemUpdate = true;
            NN_LCS_LOG_DEBUG("Send Resume\n");
            if (SendResumeSession(m_Socket).IsFailure())
            {
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                ChangeState(State_ContentsShareFailed);
            }
        }
        else if (m_pResource->suspendedReason == SuspendedReason_NeedTerminateApplication)
        {
            NN_LCS_LOG_DEBUG("ResumeShare - SuspendedReason_NeedTerminateApplication\n");
            m_pResource->suspendedReason = SuspendedReason_None;

            if (!m_pResource->shareComponent.contentsFlag.Test(ContentsType::System::Index))
            {
                NN_LCS_LOG_DEBUG("Send Resume\n");
                if (SendResumeSession(m_Socket).IsFailure())
                {
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                    ChangeState(State_ContentsShareFailed);
                    NN_RESULT_SUCCESS;
                }
            }
            nn::os::SignalEvent(&m_ResumeEvent);
        }
        else if (m_pResource->suspendedReason == SuspendedReason_EulaRequired)
        {
            NN_LCS_LOG_DEBUG("ResumeShare - SuspendedReason_EulaRequired\n");

            NN_LCS_LOG_DEBUG("Send Resume\n");
            if (SendResumeSession(m_Socket).IsFailure())
            {
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                ChangeState(State_ContentsShareFailed);
                NN_RESULT_SUCCESS;
            }
            nn::os::SignalEvent(&m_ResumeEvent);
        }
        else if (m_pResource->suspendedReason == SuspendedReason_RebootRequired)
        {
            NN_LCS_LOG_DEBUG("ResumeShare - SuspendedReason_RebootRequired\n");
            if (m_IsNeedReboot)
            {
                ChangeState(State_Suspended);
            }
            else
            {
                m_pResource->suspendedReason = SuspendedReason_None;
                NN_LCS_LOG_DEBUG("Send Resume\n");
                if (SendResumeSession(m_Socket).IsFailure())
                {
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                    ChangeState(State_ContentsShareFailed);
                }
            }
        }
        else if (m_pResource->suspendedReason == SuspendedReason_StorageSpaceNotEnough)
        {
            NN_LCS_LOG_DEBUG("m_RequiredSize : %lld\n", m_RequiredSize);
            if (HasFreeSpaceSize(m_RequiredSize))
            {
                m_pResource->suspendedReason = SuspendedReason_None;
                NN_LCS_LOG_DEBUG("Send Resume\n");
                if (SendResumeSession(m_Socket).IsFailure())
                {
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                    ChangeState(State_ContentsShareFailed);
                    NN_RESULT_SUCCESS;
                }
            }
            else
            {
                MakeResumeContext();
                ChangeState(State_Suspended);
            }
        }
        NN_RESULT_SUCCESS;
    }

    void Client::MonitorThread() NN_NOEXCEPT
    {
        nn::Result result;
        nn::os::SystemEventType stateChangeEvent;

        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType stateChangeHolder;
        nn::os::MultiWaitHolderType cancelHolder;
        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &stateChangeEvent);
        nn::os::InitializeMultiWaitHolder(&cancelHolder, &m_CancelEvent);
        nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

        while (nn::os::WaitAny(&multiWait) != &cancelHolder)
        {
            nn::os::TryWaitSystemEvent(&stateChangeEvent);

            NN_LCS_LOG_DEBUG("stateChangeEvent\n");

            nn::ldn::State ldnState = nn::ldn::GetState();
            if (ldnState != nn::ldn::State_StationConnected)
            {
                // Ldn の切断理由を取得するために一旦 API を呼ぶ
                nn::ldn::NetworkInfo info = {};
                result = nn::ldn::GetNetworkInfo(&info);
                if (nn::ldn::ResultWifiOff().Includes(result))
                {
                    NN_LCS_LOG_DEBUG("ContentsShareFailureReason_FlightMode\n");
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_FlightMode;
                }
                else if (nn::ldn::ResultSleep().Includes(result))
                {
                    NN_LCS_LOG_DEBUG("ContentsShareFailureReason_Sleep\n");
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_Sleep;
                }
                NN_LCS_LOG_DEBUG("contentsShareFailureReason : %d\n", m_pResource->contentsShareFailureReason);
                if (m_pResource->contentsShareFailureReason == ContentsShareFailureReason_None)
                {
                    NN_LCS_LOG_DEBUG("ContentsShareFailureReason_NodeLeaved\n");
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NodeLeaved;
                }
                ChangeState(State_ContentsShareFailed, true);
                break;
            }
        }
        nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
        nn::os::UnlinkMultiWaitHolder(&cancelHolder);
        nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
        nn::os::FinalizeMultiWaitHolder(&cancelHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        NN_LCS_LOG_DEBUG("End MonitorThread\n");
    }

    void Client::ComThread() NN_NOEXCEPT
    {
        while (m_IsConnected)
        {
            if (!RecvPacket())
            {
                break;
            }

            // アドバタイズデータを確認
            GetAdvertiseData(&m_AdvertiseData);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ReceiveIntervalMilliSeconds));
        }
        NN_LCS_LOG_DEBUG("End ComThread\n");
    }

    void Client::ShareThread() NN_NOEXCEPT
    {
        while (m_IsConnected)
        {
            NN_LCS_LOG_DEBUG("Share Wait\n");
            nn::os::WaitEvent(&m_StartTransferEvent);
            NN_LCS_LOG_DEBUG("Share Signal!\n");
            if (!m_IsConnected)
            {
                return;
            }

            if (m_pResource->state.GetState() == State_Transferring)
            {
                Result result;
                SystemDeliveryInfo sysInfo = {};

                NN_LCS_LOG_DEBUG("role : %d / index : %x / address : %x\n",
                    m_pResource->shareComponent.role, m_pResource->shareComponent.index, m_pResource->shareComponent.address);

                GetSystemDeliveryInfo(&sysInfo);

                // 進捗用の設定
                m_NodeIndex = m_pResource->shareComponent.index;

                if (m_pResource->shareComponent.contentsFlag.Test(ContentsType::System::Index))
                {
                    NN_LCS_LOG_DEBUG("SystemRequest\n");

                    // 進捗用の設定
                    ::std::memset(&m_pResource->contentInfo[m_pResource->contentInfoCount], 0, sizeof(ContentsInfo));
                    m_pResource->contentInfo[m_pResource->contentInfoCount].contentsFlag.Reset();
                    m_pResource->contentInfo[m_pResource->contentInfoCount].contentsFlag.Set(ContentsType::System::Index);

                    if (m_pResource->shareComponent.role == TransferRole_Sender)
                    {
                        NN_LCS_LOG_DEBUG("I'm System Sender\n");
                        // 送信タスクの生成
                        nn::ns::AsyncResult asyncResult;
                        RequestSendSystemUpdate(&asyncResult, m_pResource->shareComponent.address, SharePort,
                            m_pResource->shareComponent.systemDeliveryInfo);

                        // 進捗用の設定
                        m_Role = TransferRole_Sender;
                        m_IsWaitReceiver = true;

                        result = WaitTask(&asyncResult, m_NodeIndex);
                        if (result.IsFailure())
                        {
                            asyncResult.Cancel();
                            m_Role = TransferRole_None;
                            if (ResultAsyncResultError::Includes(result))
                            {
                                ChangeState(State_ContentsShareFailed);
                                return;
                            }
                            else if (ResultCommunicationError::Includes(result) ||
                                ResultRequestNodeLeaved::Includes(result) || ResultUserCancel::Includes(result))
                            {
                                NN_LCS_LOG_DEBUG("Send Failed : %08x\n", result.GetInnerValueForDebug());
                                m_IsWaitReceiver = false;
                            }
                            else
                            {
                                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                            }
                        }
                        uint64_t sendSize = 0;
                        GetSendSystemUpdateProgress(&sendSize, &m_TotalSendSize);

                    }
                    else if (m_pResource->shareComponent.role == TransferRole_Receiver)
                    {
                        NN_LCS_LOG_DEBUG("I'm System Receiver\n");
                        // NUP のタスクを破棄する。再生成の必要はない
                        NN_ABORT_UNLESS_RESULT_SUCCESS(
                            m_pResource->systemUpdateControl.SetupToReceiveSystemUpdate());

                        result = ReceiveSystemUpdate();
                        if (result.IsSuccess())
                        {
                            sysInfo = m_pResource->shareComponent.systemDeliveryInfo;
                        }
                        else if (ResultUserCancel::Includes(result) || ResultRequestNodeLeaved::Includes(result))
                        {
                            // 継続するので、そのまま。
                        }
                        else if (ResultCommunicationError::Includes(result))
                        {
                            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                            ChangeState(State_ContentsShareFailed);
                            return;
                        }
                        else if (ResultAsyncResultError::Includes(result))
                        {
                            ChangeState(State_ContentsShareFailed);
                            return;
                        }
                        else
                        {
                            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                        }
                    }
                }
                else
                {
                    NN_LCS_LOG_DEBUG("ID : %llx\n", m_pResource->shareComponent.id);
                    int count = 0;
                    NN_ABORT_UNLESS_LESS_EQUAL(m_pResource->sessionInfo.contentsCount, SharableContentsCountMax);
                    while (count < m_pResource->sessionInfo.contentsCount)
                    {
                        if (m_pResource->shareComponent.id == m_pResource->sessionInfo.contents[count].applicationId.value)
                        {
                            // 進捗用の設定
                            NN_LCS_LOG_DEBUG("Set m_pResource->contentInfo : %llx\n", m_pResource->shareComponent.id);
                            ::std::memcpy(&m_pResource->contentInfo[m_pResource->contentInfoCount],
                                &m_pResource->sessionInfo.contents[count], sizeof(ContentsInfo));
                            m_pResource->shareComponent.contentsFlag = m_pResource->sessionInfo.contents[count].contentsFlag;
                            break;
                        }
                        ++count;
                    }
                    NN_ABORT_UNLESS_LESS(count, SharableContentsCountMax);

                    if (m_pResource->shareComponent.role == TransferRole_Sender)
                    {
                        NN_LCS_LOG_DEBUG("I'm App Sender\n");
                        // 送信するアプリの状態が変化していないか確認
                        for (int i = 0; i < m_pResource->ownAppInfoCount; ++i)
                        {
                            if (m_pResource->ownAppInfo[i].detailInfo.id == m_pResource->shareComponent.id)
                            {
                                if (ConfirmSendApplication(m_pResource->ownAppInfo[i]).IsFailure())
                                {
                                    NN_LCS_LOG_DEBUG("ConfirmSendApplication Error\n");
                                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_StorageError;
                                    ChangeState(State_ContentsShareFailed);
                                    return;
                                }
                            }
                        }

                        // 送信者なので、ContentMetaKey を生成
                        ContentMetaKey contentMetaKey[ContentMetaKeyMax] = {};
                        int contentMetaKeyCount = 0;
                        if (ListContentMetaKeyToDeliverApplication(
                            &contentMetaKeyCount, contentMetaKey, ContentMetaKeyMax, m_pResource->shareComponent.id).IsFailure())
                        {
                            NN_LCS_LOG_DEBUG("ListContentMetaKeyToDeliverApplication Error\n");
                            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_StorageError;
                            ChangeState(State_ContentsShareFailed);
                            return;
                        }

                        // 返事をして ContentMetaKey を送信
                        NN_LCS_LOG_DEBUG("Send ShareContentSendRes\n");
                        if (SendShareContentSendRes(m_Socket, contentMetaKey, contentMetaKeyCount).IsFailure())
                        {
                            NN_LCS_LOG_DEBUG("Send ShareContentSendRes Error\n");
                            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                            ChangeState(State_ContentsShareFailed);
                            return;
                        }

                        // 送信タスクの生成
                        nn::ns::AsyncResult asyncResult;
                        if (RequestSendApplication(&asyncResult, m_pResource->shareComponent.address,
                            SharePort, m_pResource->shareComponent.id).IsFailure())
                        {
                            NN_LCS_LOG_DEBUG("RequestSendApplication Error\n");
                            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_StorageError;
                            ChangeState(State_ContentsShareFailed);
                            return;
                        }
                        // 進捗用の設定
                        m_Role = TransferRole_Sender;
                        m_IsWaitReceiver = true;

                        result = WaitTask(&asyncResult, m_NodeIndex);
                        if (result.IsFailure())
                        {
                            asyncResult.Cancel();
                            m_Role = TransferRole_None;
                            if (ResultAsyncResultError::Includes(result))
                            {
                                ChangeState(State_ContentsShareFailed);
                                return;
                            }
                            else if (ResultCommunicationError::Includes(result) ||
                                ResultRequestNodeLeaved::Includes(result) || ResultUserCancel::Includes(result))
                            {
                                NN_LCS_LOG_DEBUG("Send Failed : %08x\n", result.GetInnerValueForDebug());
                                m_IsWaitReceiver = false;
                            }
                            else
                            {
                                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                            }
                        }
                        uint64_t sendSize = 0;
                        GetSendApplicationProgress(&sendSize, &m_TotalSendSize,
                            m_pResource->contentInfo[m_pResource->contentInfoCount].applicationId.value);
                    }
                    else if (m_pResource->shareComponent.role == TransferRole_Receiver)
                    {
                        NN_LCS_LOG_DEBUG("I'm App Receiver\n");

                        Response res = Response_Reject;
                        for (int i = 0; i < m_pResource->shareAppInfoCount; ++i)
                        {
                            if (m_pResource->shareAppInfo[i].detailInfo.id == m_pResource->shareComponent.id)
                            {
                                m_RequiredSize = m_pResource->shareAppInfo[i].detailInfo.requiredSize;
                            }
                        }

                        if (HasFreeSpaceSize(m_RequiredSize))
                        {
                            res = Response_Accept;
                        }

                        NN_LCS_LOG_DEBUG("Send ShareContentRecvRes : %d\n", res);
                        if (SendShareContentRecvRes(m_Socket, res).IsFailure())
                        {
                            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                            ChangeState(State_ContentsShareFailed);
                            return;
                        }
                        if (res == Response_Reject)
                        {
                            m_pResource->suspendedReason = SuspendedReason_StorageSpaceNotEnough;
                            MakeResumeContext();
                            ChangeState(State_Suspended);
                            continue;
                        }

                        result = ReceiveApplication();
                        if (result.IsFailure())
                        {
                            if (ResultUserCancel::Includes(result) || ResultRequestNodeLeaved::Includes(result))
                            {
                                // 継続するので、そのまま
                            }
                            else if (ResultInvalidContentDeliveryRequest::Includes(result) ||
                                nn::ns::ResultReceiveApplicationTaskNotFound::Includes(result))
                            {
                                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NotExistDownloadContents;
                                ChangeState(State_ContentsShareFailed);
                                return;
                            }
                            else if (ResultCommunicationError::Includes(result) ||
                                nn::ns::ResultApplicationContentNotDownloaded::Includes(result))
                            {
                                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                                ChangeState(State_ContentsShareFailed);
                                return;
                            }
                            else if (ResultAsyncResultError::Includes(result))
                            {
                                ChangeState(State_ContentsShareFailed);
                                return;
                            }
                            else if (nn::ncm::ResultSdCardContentStorageNotActive::Includes(result))
                            {
                                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_SdCardError;
                                ChangeState(State_ContentsShareFailed);
                                return;
                            }
                            else
                            {
                                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                            }
                        }
                    }
                }

                m_Role = TransferRole_None;

                NN_LCS_LOG_DEBUG("m_IsWaitReceiver : %d\n", m_IsWaitReceiver);
                // 自分の情報をホストに送ってアップデートしてもらう(送信者も状態を変えてもらうために送る)
                if (!m_IsWaitReceiver)
                {
                    NN_LCS_LOG_DEBUG("Send NodeDetailInfo\n");
                    SendNodeDetailInfo(m_Socket, m_pResource->userName, m_pResource->version, sysInfo,
                        m_pResource->sessionInfo, m_pResource->contentInfoCount,
                        m_pResource->appControlDataBuffer, AppControlDataSize, true);
                }

                if (m_IsNeedReboot)
                {
                    m_pResource->suspendedReason = SuspendedReason_RebootRequired;
                    MakeResumeContext();
                    ChangeState(State_Suspended);
                }
            }
        }
        NN_LCS_LOG_DEBUG("End ShareThread\n");
    }// NOLINT(impl/function_size)

    bool Client::RecvPacket() NN_NOEXCEPT
    {
        CommonHeader header;
        ssize_t recvSize = RecvCommonHeaderWithPeek(m_Socket, &header);
        if (recvSize < 0)
        {
            nn::socket::Errno error = nn::socket::GetLastError();
            if (error == nn::socket::Errno::EAgain)
            {
            }
            else
            {
                NN_LCS_LOG_WARN("%s failed: errno = %d\n", NN_CURRENT_FUNCTION_NAME, error);
                return false;
            }
        }
        else if (sizeof(CommonHeader) <= static_cast<size_t>(recvSize))
        {
            switch (header.type)
            {
            case Type_Accept:
            {
                RecvAccept();
                break;
            }
            case Type_Reject:
            {
                RecvReject();
                break;
            }
            case Type_SessionInfo:
            {
                RecvSessionInfo();
                break;
            }
            case Type_StartContentsSare:
            {
                RecvStartContentsSare();
                break;
            }
            case Type_Migration:
            {
                RecvMigration();
                break;
            }
            case Type_ShareSystemClear:
            {
                RecvShareSystemClear();
                break;
            }
            case Type_ShareSystemSendReq:
            {
                RecvShareSystemSendReq();
                break;
            }
            case Type_ShareSystemRecvReq:
            {
                RecvShareSystemRecvReq();
                break;
            }
            case Type_ShareContentClear:
            {
                RecvShareContentClearReq();
                break;
            }
            case Type_ShareContentSendReq:
            {
                RecvShareContentSendReq();
                break;
            }
            case Type_ShareContentRecvReq:
            {
                RecvShareContentRecvReq();
                break;
            }
            case Type_EndContentShare:
            {
                RecvEndContentShare();
                break;
            }
            default:
                break;
            }
        }
        else if (0 < recvSize)
        {
            NN_LCS_LOG_WARN("%s failed: small packet %d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            NN_LCS_LOG_DEBUG("Socket Recv Error!\n");
            /* Ldn の切断より先に SocketError になって失敗理由が異なってしまうので現状お蔵入り
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
            */
            return false;
        }
        return true;
    }

    void Client::RecvAccept() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_Accept\n");

        ssize_t recvSize = RecvData(
            m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(AcceptPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else if (static_cast<size_t>(recvSize) < sizeof(AcceptPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
    }

    void Client::RecvReject() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_Reject\n");

        ssize_t recvSize = RecvData(
            m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(RejectPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else if (static_cast<size_t>(recvSize) < sizeof(RejectPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else
        {
            RejectPacket* packet = reinterpret_cast<RejectPacket*>(m_pResource->packetBuffer);
            RejectReason reason = static_cast<RejectReason>(packet->rejectReason);
            NN_LCS_LOG_DEBUG("RejectReason : %d\n", packet->rejectReason);

            switch (reason)
            {
            case RejectReason_IncompatibleSystemVersion:
                // 接続時しか発生しない
                break;
            case RejectReason_ConnectionFailed:
                // 接続時しか発生しない
                break;
            case RejectReason_CommunicationError:
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                break;
            case RejectReason_NodeLeaved:
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NodeLeaved;
                break;
            case RejectReason_CannotUpdateSystem:
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CannotUpdateSystem;
                break;
            case RejectReason_NotExistDownloadContents:
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NotExistDownloadContents;
                break;
            case RejectReason_UnshareableContents:
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_UnshareableContents;
                break;
            default:
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_UnknownError;
                break;
            }
            ChangeState(State_ContentsShareFailed);
        }
    }

    void Client::RecvSessionInfo() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_RecvSessionInfo\n");

        bool isCompleted = false;
        bool isChangeNodeCount = false;
        int appInfoCount = 0;
        int appDeliveryInfoCount = 0;
        int nodeCount = 0;
        int preNodeCount = 0;

        int recvNodeCount = 0;
        NodeInfo recvNodeInfo[NodeCountMax] = {};

        ssize_t expectedSize = sizeof(SessionInfoPacket);
        while (!isCompleted)
        {
            ssize_t recvSize = RecvData(
                m_Socket, m_pResource->packetBuffer, PacketSize, expectedSize);
            if (recvSize < 0)
            {
                nn::socket::Errno error = nn::socket::GetLastError();
                if (error == nn::socket::Errno::EAgain)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    continue;
                }
                else
                {
                    NN_LCS_LOG_WARN("%s failed: errno = %d\n", NN_CURRENT_FUNCTION_NAME, error);
                    break;
                }
            }
            else if (recvSize < expectedSize)
            {
                NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                    NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
                break;
            }
            NN_LCS_LOG_DEBUG("%s: Received %d Bytes (expected %d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME,
                static_cast<int>(recvSize), static_cast<int>(expectedSize));

            if (expectedSize == sizeof(SessionInfoPacket))
            {
                SessionInfoPacket* packet = reinterpret_cast<SessionInfoPacket*>(m_pResource->packetBuffer);
                if (SharableContentsCountMax < packet->appCount ||
                    NodeCountMax < packet->nodeCount)
                {
                    NN_LCS_LOG_WARN("Invalid Session Packet: app=%d node=%d\n",
                        packet->appCount, packet->nodeCount);
                    break;
                }
                m_pResource->shareAppInfoCount = packet->appCount;
                nodeCount = packet->nodeCount;
                NN_LCS_LOG_DEBUG("shareAppInfoCount : %d / nodeCount : %d\n",
                    m_pResource->shareAppInfoCount, nodeCount);
                expectedSize = sizeof(AppInfoPacket);
            }
            else if (expectedSize == sizeof(AppInfoPacket))
            {
                AppInfoPacket* appInfoPacket = reinterpret_cast<AppInfoPacket*>(m_pResource->packetBuffer);
                if (ApplicationDeliveryInfoCountMax < appInfoPacket->appDeliveryInfoCount)
                {
                    NN_LCS_LOG_WARN("Invalid Application Info Packet: count=%d\n",
                        appInfoPacket->appDeliveryInfoCount);
                    break;
                }
                ::std::memcpy(&m_pResource->shareAppInfo[appInfoCount].detailInfo, appInfoPacket->detailInfo, ApplicationDetailInfoSize);
                m_pResource->shareAppInfo[appInfoCount].deliveryInfoCount = appInfoPacket->appDeliveryInfoCount;
                if (m_pResource->shareAppInfo[appInfoCount].deliveryInfoCount > 0)
                {
                    expectedSize = sizeof(AppDeliveryInfoPacket);
                }
                else
                {
                    expectedSize = sizeof(AppInfoPacket);
                    ++appInfoCount;
                    if (appInfoCount == m_pResource->shareAppInfoCount)
                    {
                        expectedSize = sizeof(NodeInfoPacket);
                    }
                }
            }
            else if (expectedSize == sizeof(AppDeliveryInfoPacket))
            {
                AppDeliveryInfoPacket *appDeliveryInfoPacket =
                    reinterpret_cast<AppDeliveryInfoPacket*>(m_pResource->packetBuffer);
                ::std::memcpy(&m_pResource->shareAppInfo[appInfoCount].deliveryInfo[appDeliveryInfoCount],
                    appDeliveryInfoPacket->deliveryInfo, ApplicationDeliveryInfoSize);
                ++appDeliveryInfoCount;
                if (appDeliveryInfoCount == m_pResource->shareAppInfo[appInfoCount].deliveryInfoCount)
                {
                    expectedSize = sizeof(AppInfoPacket);
                    ++appInfoCount;
                    if (appInfoCount == m_pResource->shareAppInfoCount)
                    {
                        expectedSize = sizeof(NodeInfoPacket);
                    }
                }
            }
            else if (expectedSize == sizeof(NodeInfoPacket))
            {
                NodeInfoPacket *nodeInfoPacket =
                    reinterpret_cast<NodeInfoPacket*>(m_pResource->packetBuffer);
                if (nodeInfoPacket->index == 0)
                {
                    NN_LCS_LOG_WARN("Invalid Node Info Packet: index = 0\n");
                    break;
                }
                recvNodeInfo[recvNodeCount].index = nn::socket::InetNtohl(nodeInfoPacket->index);
                ::std::memcpy(recvNodeInfo[recvNodeCount].userName,
                    nodeInfoPacket->userName, UserNameBytesMax + 1);
                recvNodeInfo[recvNodeCount].userName[UserNameBytesMax] = 0;
                NN_LCS_LOG_DEBUG("%d : %s[%x]\n",
                    recvNodeCount, recvNodeInfo[recvNodeCount].userName, recvNodeInfo[recvNodeCount].index);
                ++recvNodeCount;
                if (nodeCount == recvNodeCount)
                {
                    preNodeCount = m_pResource->nodeInfoCount;
                    NN_LCS_LOG_DEBUG("NodeCount : %d -> %d\n", preNodeCount, nodeCount);
                    if (preNodeCount != nodeCount)
                    {
                        isChangeNodeCount = true;
                    }
                    // NodeInfo を更新
                    nn::os::LockMutex(&m_pResource->mutex);
                    ::std::memcpy(m_pResource->nodeInfo, recvNodeInfo, sizeof(NodeInfo) * nodeCount);
                    m_pResource->nodeInfoCount = nodeCount;
                    nn::os::UnlockMutex(&m_pResource->mutex);
                    isCompleted = true;
                }
            }
        }
        NN_LCS_LOG_DEBUG("End RecvSessionInfoRecv\n");

        if (!isCompleted && m_pResource->contentsShareFailureReason == ContentsShareFailureReason_None)
        {
            NN_LCS_LOG_DEBUG("RecvSessionInfoRecv Error\n");
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
            return;
        }

        UpdateSessionInfo();

        // 誰も抜けていないときと、送信先の端末が抜けたときに送信完了サイズをクリア
        if (m_IsWaitReceiver && (preNodeCount == nodeCount || !ExistsNode(m_NodeIndex)))
        {
            NN_LCS_LOG_DEBUG("Clear TotalSendSize\n");
            m_IsWaitReceiver = false;
            m_TotalSendSize = 0;
            if (m_Role == TransferRole_None)
            {
                SystemDeliveryInfo sysInfo = {};
                GetSystemDeliveryInfo(&sysInfo);
                SendNodeDetailInfo(m_Socket, m_pResource->userName, m_pResource->version, sysInfo,
                    m_pResource->sessionInfo, m_pResource->contentInfoCount,
                    m_pResource->appControlDataBuffer, AppControlDataSize, true);
            }
        }

        nn::os::SignalEvent(&m_SessionInfoEvent);
        if (isChangeNodeCount && m_pResource->state.GetState() != State_Initialized &&
            m_pResource->contentsShareFailureReason == ContentsShareFailureReason_None)
        {
            NN_LCS_LOG_DEBUG("NodeCount is Changed : SignalEvent\n");
            m_pResource->stateEvent.SignalEvent();
        }
    } // NOLINT(impl/function_size)

    void Client::RecvStartContentsSare() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_StartContentsSare\n");

        ssize_t recvSize = RecvData(
            m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(StartContentsSarePacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else if (static_cast<size_t>(recvSize) < sizeof(StartContentsSarePacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else
        {
            UpdateSessionInfo();

            // 端末の進捗を取得するためにインデックス保存
            m_pResource->nodeDetailProgressCount = m_pResource->nodeInfoCount;
            for (int i = 0; i < m_pResource->nodeDetailProgressCount; ++i)
            {
                m_pResource->nodeDetailProgress[i].index = m_pResource->nodeInfo[i].index;
                m_pResource->nodeDetailProgress[i].progress.contentCount = 0;
                m_pResource->nodeDetailProgress[i].progress.downloadedContentCount = 0;
            }

            // 自身のコンテンツの情報を更新
            m_pResource->context.appHashCount = 0;
            for (int i = 0; i < m_pResource->ownAppInfoCount; ++i)
            {
                if (m_pResource->ownAppInfo[i].detailInfo.id == m_pResource->shareComponent.id)
                {
                    detail::GetApplicationDetailInfo(&m_pResource->ownAppInfo[i].detailInfo,
                        m_pResource->appControlDataBuffer, detail::AppControlDataSize,
                        m_pResource->shareComponent.id);
                    detail::GetApplicationDeliveryInfo(&m_pResource->ownAppInfo[i].deliveryInfoCount,
                        m_pResource->ownAppInfo[i].deliveryInfo, detail::ApplicationDeliveryInfoCountMax,
                        m_pResource->shareComponent.id);
                }
                // ApplicationDeliveryInfo のハッシュ値を計算してコンテキストに保存
                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    GetApplicationDeliveryInfoHash(m_pResource->context.appHash[i].hash,
                    ApplicationDeliveryInfoHashSize, m_pResource->ownAppInfo[i]));
                m_pResource->context.appHash[i].id = m_pResource->ownAppInfo[i].detailInfo.id;
                ++m_pResource->context.appHashCount;
            }

            // 自身のコンテンツ情報を Host へ送信
            SystemDeliveryInfo sysInfo = {};
            GetSystemDeliveryInfo(&sysInfo);
            SendNodeDetailInfo(m_Socket, m_pResource->userName, m_pResource->version, sysInfo,
                m_pResource->sessionInfo, m_pResource->contentInfoCount,
                m_pResource->appControlDataBuffer, AppControlDataSize, true);

            ChangeState(State_Transferring);
        }
    }

    void Client::RecvMigration() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_Migration\n");

        // 各情報のアップデート
        UpdateSessionInfo();
        UpdateRequiredStorageSize();
        // Advertise データを保存
        if (GetAdvertiseData(&m_pResource->advertiseData).IsFailure())
        {
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
            return;
        }

        if (RecvData(m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(MigrationPacket)) == 0)
        {
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
            return;
        }

        MigrationPacket* packet = reinterpret_cast<MigrationPacket*>(m_pResource->packetBuffer);

        ::std::memcpy(&(m_pResource->migrationInfo), packet->migrationInfo, MigrationInfoSize);
        ::std::memcpy(&(m_pResource->currentNetworkSetting),
            &(m_pResource->migrationInfo.newNetworkSetting), NetworkSettingSize);

        // MigrationThread に連絡
        nn::os::SignalEvent(&(m_pResource->migrationEvent));
    }

    void Client::RecvShareSystemClear() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareSystemClear\n");

        ssize_t recvSize = RecvData(
            m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(SystemShareClearPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else if (static_cast<size_t>(recvSize) < sizeof(SystemShareClearPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else
        {
            SystemShareClearPacket* packet =
                reinterpret_cast<SystemShareClearPacket*>(m_pResource->packetBuffer);

            if (m_pResource->state.GetState() == State_Suspended)
            {
                NN_LCS_LOG_DEBUG("Send ShareSystemClearRes : Suspend\n");
                if (SendShareSystemClearRes(m_Socket, Response_Suspend).IsFailure())
                {
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                    ChangeState(State_ContentsShareFailed);
                    return;
                }
            }

            if (packet->systemUpdateReason == SystemUpdateReason_IncompatibleContentsInfo)
            {
                NN_LCS_LOG_DEBUG("Recv SystemUpdateReason_IncompatibleContentsInfo\n");
                if (m_IsAcceptSystemUpdate)
                {
                    NN_LCS_LOG_DEBUG("Send ShareSystemClearRes : Accept\n");
                    if (SendShareSystemClearRes(m_Socket, Response_Accept).IsFailure())
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                        ChangeState(State_ContentsShareFailed);
                        return;
                    }
                }
                else
                {
                    m_pResource->suspendedReason = SuspendedReason_IncompatibleContentsInfo;
                    NN_LCS_LOG_DEBUG("Send ShareSystemClearRes : Suspend\n");
                    if (SendShareSystemClearRes(m_Socket, Response_Suspend).IsFailure())
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                        ChangeState(State_ContentsShareFailed);
                        return;
                    }
                    ChangeState(State_Suspended);
                }
            }
            else if (packet->systemUpdateReason == SystemUpdateReason_NeedToRunApplication)
            {
                NN_LCS_LOG_DEBUG("Send ShareSystemClearRes\n");
                if (SendShareSystemClearRes(m_Socket, Response_Accept).IsFailure())
                {
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                    ChangeState(State_ContentsShareFailed);
                    return;
                }
            }
        }
    }

    void Client::RecvShareSystemSendReq() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareSystemSendReq\n");

        ssize_t recvSize = RecvData(
            m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(SystemShareInfoPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else if (static_cast<size_t>(recvSize) < sizeof(SystemShareInfoPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else
        {
            SystemShareInfoPacket* packet =
                reinterpret_cast<SystemShareInfoPacket*>(m_pResource->packetBuffer);

            m_pResource->shareComponent.role = packet->role;
            m_pResource->shareComponent.index = nn::socket::InetNtohl(packet->nodeIndex);
            m_pResource->shareComponent.address = nn::socket::InetNtohl(packet->address);
            m_pResource->shareComponent.contentsFlag.Set(ContentsType::System::Index);

            SystemDeliveryInfo sysInfo = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(GetSystemDeliveryInfo(&sysInfo));
            ::std::memcpy(&(m_pResource->shareComponent.systemDeliveryInfo), &sysInfo, sizeof(SystemDeliveryInfo));

            m_pResource->shareComponent.connectTimeout = packet->connectTimeout;
            m_pResource->shareComponent.channel = packet->channel;

            NN_LCS_LOG_DEBUG("Send ShareSystemSendRes\n");
            SendShareSystemSendRes(m_Socket, sysInfo);

            nn::os::SignalEvent(&m_StartTransferEvent);
        }
    }

    void Client::RecvShareSystemRecvReq() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareSystemRecvReq\n");

        ssize_t recvSize = RecvData(
            m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(SystemShareInfoPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else if (static_cast<size_t>(recvSize) < sizeof(SystemShareInfoPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else
        {
            SystemShareInfoPacket* packet =
                reinterpret_cast<SystemShareInfoPacket*>(m_pResource->packetBuffer);

            m_pResource->shareComponent.role = packet->role;
            m_pResource->shareComponent.index = nn::socket::InetNtohl(packet->nodeIndex);
            m_pResource->shareComponent.address = nn::socket::InetNtohl(packet->address);
            ::std::memcpy(m_pResource->shareComponent.systemDeliveryInfo.sysDeliveryInfo,
                packet->sysInfo, SystemDeliveryInfoSize);
            m_pResource->shareComponent.contentsFlag.Set(ContentsType::System::Index);
            m_pResource->shareComponent.connectTimeout = packet->connectTimeout;
            m_pResource->shareComponent.channel = packet->channel;

            NN_LCS_LOG_DEBUG("Send ShareSystemRecvRes\n");
            if (SendShareSystemRecvRes(m_Socket, Response_Accept).IsFailure())
            {
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                ChangeState(State_ContentsShareFailed);
                return;
            }

            nn::os::SignalEvent(&m_StartTransferEvent);
        }
    }

    void Client::RecvShareContentClearReq() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareContentClear\n");

        ssize_t recvSize = RecvData(
            m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(ContentShareClearPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else if (static_cast<size_t>(recvSize) < sizeof(ContentShareClearPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else
        {
            if (m_pResource->state.GetState() == State_Suspended)
            {
                NN_LCS_LOG_DEBUG("Recv Type_ShareContentClear : Response_Suspend\n");
                if (SendShareContentClearRes(m_Socket, Response_Suspend).IsFailure())
                {
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                    ChangeState(State_ContentsShareFailed);
                    return;
                }
            }

            ContentShareClearPacket* packet =
                reinterpret_cast<ContentShareClearPacket*>(m_pResource->packetBuffer);

            Response response = Response_Accept;

            m_pResource->shareComponent.id = GetMerged(nn::socket::InetNtohl(packet->appUpperId),
                nn::socket::InetNtohl(packet->appLowerId));
            for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
            {
                if (m_pResource->shareAppInfo[i].detailInfo.id == m_pResource->shareComponent.id)
                {
                    m_RequiredSize = m_pResource->shareAppInfo[i].detailInfo.requiredSize;
                }
            }

            if (!HasFreeSpaceSize(m_RequiredSize))
            {
                response = Response_Suspend;
                m_pResource->suspendedReason = SuspendedReason_StorageSpaceNotEnough;
                MakeResumeContext();
                ChangeState(State_Suspended);
            }

            NN_LCS_LOG_DEBUG("Send ShareContentClearRes\n");
            if (SendShareContentClearRes(m_Socket, response).IsFailure())
            {
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                ChangeState(State_ContentsShareFailed);
                return;
            }
        }
    }

    void Client::RecvShareContentSendReq() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareContentSendReq\n");

        ssize_t recvSize = RecvData(
            m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(ContentShareInfoPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else if (static_cast<size_t>(recvSize) < sizeof(ContentShareInfoPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else
        {
            ContentShareInfoPacket* packet =
                reinterpret_cast<ContentShareInfoPacket*>(m_pResource->packetBuffer);

            NN_ABORT_UNLESS_EQUAL(packet->contentMetaKeyCount, 0);

            m_pResource->shareComponent.role = packet->role;
            m_pResource->shareComponent.index = nn::socket::InetNtohl(packet->nodeIndex);
            m_pResource->shareComponent.address = nn::socket::InetNtohl(packet->address);
            m_pResource->shareComponent.id = GetMerged(nn::socket::InetNtohl(packet->appUpperId),
                nn::socket::InetNtohl(packet->appLowerId));
            m_pResource->shareComponent.contentsFlag.Reset();
            for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
            {
                if (m_pResource->sessionInfo.contents[i].applicationId.value == m_pResource->shareComponent.id)
                {
                    m_pResource->shareComponent.contentsFlag = m_pResource->sessionInfo.contents[i].contentsFlag;
                }
            }
            m_pResource->shareComponent.connectTimeout = packet->connectTimeout;
            m_pResource->shareComponent.channel = packet->channel;

            nn::os::SignalEvent(&m_StartTransferEvent);
        }
    }

    void Client::RecvShareContentRecvReq() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareContentRecvReq\n");

        bool isCompleted = false;
        uint8_t contentMetaKeyCountMax = 0;
        ssize_t expectedSize = sizeof(ContentShareInfoPacket);
        while (!isCompleted)
        {
            ssize_t recvSize = RecvData(
                m_Socket, m_pResource->packetBuffer, PacketSize, expectedSize);
            if (recvSize < 0)
            {
                nn::socket::Errno error = nn::socket::GetLastError();
                if (error == nn::socket::Errno::EAgain)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    continue;
                }
                else
                {
                    NN_LCS_LOG_WARN("%s failed: errno = %d\n", NN_CURRENT_FUNCTION_NAME, error);
                    break;
                }
            }
            else if (recvSize < expectedSize)
            {
                NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                    NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
                break;
            }

            if (expectedSize == sizeof(ContentShareInfoPacket))
            {
                NN_LCS_LOG_DEBUG("Recv ContentShareInfoPacket\n");
                ContentShareInfoPacket* packet =
                    reinterpret_cast<ContentShareInfoPacket*>(m_pResource->packetBuffer);
                m_pResource->shareComponent.role = packet->role;
                contentMetaKeyCountMax = packet->contentMetaKeyCount;
                m_pResource->shareComponent.contentMetaKeyCount = 0;
                m_pResource->shareComponent.index = nn::socket::InetNtohl(packet->nodeIndex);
                m_pResource->shareComponent.address = nn::socket::InetNtohl(packet->address);
                m_pResource->shareComponent.id = GetMerged(nn::socket::InetNtohl(packet->appUpperId),
                    nn::socket::InetNtohl(packet->appLowerId));
                m_pResource->shareComponent.contentsFlag.Reset();
                m_pResource->shareComponent.connectTimeout = packet->connectTimeout;
                m_pResource->shareComponent.channel = packet->channel;
                NN_LCS_LOG_DEBUG("contentMetaKeyCountMax : %d\n", contentMetaKeyCountMax);

                if (contentMetaKeyCountMax != 0)
                {
                    expectedSize = sizeof(ContentMetaKeyPacket);
                }
                else
                {
                    isCompleted = true;
                }
            }
            else if (expectedSize == sizeof(ContentMetaKeyPacket))
            {
                NN_LCS_LOG_DEBUG("Recv ContentMetaKeyPacket\n");
                ContentMetaKeyPacket *contentMetaKeyPacket =
                    reinterpret_cast<ContentMetaKeyPacket*>(m_pResource->packetBuffer);
                ::std::memcpy(
                    &m_pResource->shareComponent.contentMetaKey[m_pResource->shareComponent.contentMetaKeyCount],
                    contentMetaKeyPacket->contentMetaKey, ContentMetaKeySize);
                ++m_pResource->shareComponent.contentMetaKeyCount;
                if (m_pResource->shareComponent.contentMetaKeyCount == contentMetaKeyCountMax)
                {
                    isCompleted = true;
                }
            }
        }

        if (isCompleted)
        {
            nn::os::SignalEvent(&m_StartTransferEvent);
        }
        else
        {
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
    }

    void Client::RecvEndContentShare() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_EndContentShare\n");

        // まず端末の進捗(アドバタイズデータ)を更新
        GetAdvertiseData(&m_AdvertiseData);

        ssize_t recvSize = RecvData(
            m_Socket, m_pResource->packetBuffer, PacketSize, sizeof(EndContentsShareWithProgressPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else if (static_cast<size_t>(recvSize) < sizeof(EndContentsShareWithProgressPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
            ChangeState(State_ContentsShareFailed);
        }
        else
        {
            // 終了時の各端末の進捗を保存
            EndContentsShareWithProgressPacket* packet =
                reinterpret_cast<EndContentsShareWithProgressPacket*>(m_pResource->packetBuffer);
            ::std::memcpy(m_NodeProgress, packet->progress, sizeof(AdvertiseNodeProgress) * NodeCountMax);

            NN_LCS_LOG_DEBUG("--- GetFinalProgress ---\n");
            for (int i = 0; i < NodeCountMax; ++i)
            {
                NN_LCS_LOG_DEBUG("%x : %d / %d\n", m_NodeProgress[i].id,
                    m_NodeProgress[i].downloadedContentCount, m_NodeProgress[i].contentCount);
            }
            NN_LCS_LOG_DEBUG("------------------------\n");

            m_IsShareCompleted = true;

            if (m_pResource->state.GetState() != State_Suspended)
            {
                ChangeState(State_Completed);
            }
        }
    }

    Result Client::ReceiveSystemUpdate() NN_NOEXCEPT
    {
        int retryCount = 0;
        bool isSuccess = false;
        // 受信タスクを生成するとき、タイミング的に相手の準備ができていない場合があるので、リトライをする
        for (retryCount = 0; retryCount < ShareTaskRetryCountMax; ++retryCount)
        {
            nn::ns::AsyncResult asyncResult;

            NN_LCS_LOG_DEBUG("RequestReceiveApplication RetryCount : %d\n", retryCount);
            // 受信タスクの生成
            NN_RESULT_DO(RequestReceiveSystemUpdate(&asyncResult, m_pResource->shareComponent.address, SharePort,
                m_pResource->shareComponent.systemDeliveryInfo, &(m_pResource->systemUpdateControl)));

            if (!isSuccess)
            {
                // 進捗用の設定
                m_Role = TransferRole_Receiver;

                // 受信者になったので、アプリの終了を確認してもらうため Suspend する
                m_pResource->suspendedReason = SuspendedReason_NeedTerminateApplication;
                ChangeState(State_Suspended);
                isSuccess = true;
            }

            auto result = WaitTask(&asyncResult, m_NodeIndex);
            if (result.IsSuccess())
            {
                // Apply 処理
                result = ApplySystemUpdate();
                if (result.IsFailure())
                {
                    if (ResultUserCancel::Includes(result) || ResultNsError::Includes(result))
                    {
                        asyncResult.Cancel();
                        m_Role = TransferRole_None;
                        m_NodeIndex = 0;
                        return result;
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    }
                }
                break;
            }
            else if (ResultAsyncResultError::Includes(result) || ResultUserCancel::Includes(result) ||
                ResultRequestNodeLeaved::Includes(result))
            {
                asyncResult.Cancel();
                m_Role = TransferRole_None;
                m_NodeIndex = 0;
                return result;
            }
            else if (ResultCommunicationError::Includes(result))
            {
                // そのままリトライをするので何もしない。
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            // リトライするので、少し待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ShareTaskRetryIntervalMilliSeconds));
        }

        m_Role = TransferRole_None;
        m_NodeIndex = 0;

        if (retryCount == ShareTaskRetryCountMax)
        {
            return ResultCommunicationError();
        }

        NN_RESULT_SUCCESS;
    }

    Result Client::ApplySystemUpdate() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Complete\n");
        if (m_pResource->state.GetState() == State_Suspended &&
            m_pResource->suspendedReason == SuspendedReason_NeedTerminateApplication)
        {
            NN_LCS_LOG_DEBUG("WaitResume\n");
            nn::os::ClearEvent(&m_ResumeEvent);
            nn::os::WaitEvent(&m_ResumeEvent);
            if (!m_IsConnected)
            {
                NN_LCS_LOG_DEBUG("Discard\n");
                return ResultUserCancel();
            }
        }
        NN_LCS_LOG_DEBUG("System EULA\n");
        m_pResource->suspendedReason = SuspendedReason_EulaRequired;
        ChangeState(State_Suspended);

        NN_LCS_LOG_DEBUG("WaitResume\n");
        nn::os::ClearEvent(&m_ResumeEvent);
        nn::os::WaitEvent(&m_ResumeEvent);
        if (!m_IsConnected)
        {
            NN_LCS_LOG_DEBUG("Discard\n");
            return ResultUserCancel();
        }

        // システムの適用
        m_IsNeedReboot = true;
        NN_ABORT_UNLESS_EQUAL(
            ApplyReceivedUpdate(&(m_pResource->systemUpdateControl)), true);
        NN_LCS_LOG_DEBUG("ApplyReceivedUpdate Done\n");

        ++m_pResource->contentInfoCount;
        NN_ABORT_UNLESS_LESS_EQUAL(m_pResource->contentInfoCount, DownloadableContentsCountMax);
        NN_LCS_LOG_DEBUG("Download Complete[contentInfoCount : %d]\n", m_pResource->contentInfoCount);
        NN_RESULT_SUCCESS;
    }

    Result Client::ReceiveApplication() NN_NOEXCEPT
    {
        int retryCount = 0;
        bool isSuccess = false;
        // 受信タスクを生成するとき、タイミング的に相手の準備ができていない場合があるので、リトライをする
        for (retryCount = 0; retryCount < ShareTaskRetryCountMax; ++retryCount)
        {
            nn::ns::AsyncResult asyncResult;

            NN_LCS_LOG_DEBUG("RequestReceiveApplication RetryCount : %d\n", retryCount);
            // 受信タスクの生成
            NN_RESULT_DO(RequestReceiveApplication(&asyncResult,
                m_pResource->shareComponent.address, SharePort, m_pResource->shareComponent.id,
                m_pResource->shareComponent.contentMetaKey, m_pResource->shareComponent.contentMetaKeyCount));
            if (!isSuccess)
            {
                // 進捗用の設定
                m_Role = TransferRole_Receiver;

                // 受信者になったので、アプリの終了を確認してもらうため Suspend する
                m_pResource->suspendedReason = SuspendedReason_NeedTerminateApplication;
                ChangeState(State_Suspended);
                isSuccess = true;
            }

            // 受信完了待ち
            auto result = WaitTask(&asyncResult, m_NodeIndex);
            if (result.IsSuccess())
            {
                // commit 処理
                result = CommitApplication();
                if (result.IsFailure())
                {
                    if (ResultUserCancel::Includes(result) ||
                        nn::ns::ResultReceiveApplicationTaskNotFound::Includes(result) ||
                        nn::ns::ResultApplicationContentNotDownloaded::Includes(result) ||
                        nn::ncm::ResultSdCardContentStorageNotActive::Includes(result))
                    {
                        asyncResult.Cancel();
                        m_Role = TransferRole_None;
                        m_NodeIndex = 0;
                        return result;
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    }
                }
                break;
            }
            else if (ResultAsyncResultError::Includes(result) || ResultUserCancel::Includes(result) ||
                ResultRequestNodeLeaved::Includes(result))
            {
                m_Role = TransferRole_None;
                m_NodeIndex = 0;
                return result;
            }
            else if (ResultCommunicationError::Includes(result))
            {
                // そのままリトライをするので何もしない。
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            // リトライするので、少し待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ShareTaskRetryIntervalMilliSeconds));
        }

        m_Role = TransferRole_None;
        m_NodeIndex = 0;

        if (retryCount == ShareTaskRetryCountMax)
        {
            return ResultCommunicationError();
        }

        NN_RESULT_SUCCESS;
    }

    Result Client::CommitApplication() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Complete\n");
        if (m_pResource->state.GetState() == State_Suspended &&
            m_pResource->suspendedReason == SuspendedReason_NeedTerminateApplication)
        {
            NN_LCS_LOG_DEBUG("WaitResume\n");
            nn::os::WaitEvent(&m_ResumeEvent);
            nn::os::ClearEvent(&m_ResumeEvent);
            if (!m_IsConnected)
            {
                NN_LCS_LOG_DEBUG("Discard\n");
                return ResultUserCancel();
            }
        }
        NN_LCS_LOG_DEBUG("Commit : %llx\n", m_pResource->shareComponent.id);
        auto result = CommitReceiveApplication(m_pResource->shareComponent.id);
        if (result.IsSuccess())
        {
            if (m_IsShareCompleted)
            {
                NN_LCS_LOG_DEBUG("ContetsShare Completed\n");
                ChangeState(State_Completed);
            }
        }
        else
        {
            NN_LCS_LOG_DEBUG("Commit Error! : %08x\n", result.GetInnerValueForDebug());
            return result;
        }

        // 自身のコンテンツの情報を更新
        for (int i = 0; i < m_pResource->ownAppInfoCount; ++i)
        {
            if (m_pResource->ownAppInfo[i].detailInfo.id == m_pResource->shareComponent.id)
            {
                detail::GetApplicationDetailInfo(&m_pResource->ownAppInfo[i].detailInfo,
                    m_pResource->appControlDataBuffer, detail::AppControlDataSize,
                    m_pResource->shareComponent.id);
                detail::GetApplicationDeliveryInfo(&m_pResource->ownAppInfo[i].deliveryInfoCount,
                    m_pResource->ownAppInfo[i].deliveryInfo, detail::ApplicationDeliveryInfoCountMax,
                    m_pResource->shareComponent.id);
            }
        }

        ++m_pResource->contentInfoCount;
        NN_ABORT_UNLESS_LESS_EQUAL(m_pResource->contentInfoCount, DownloadableContentsCountMax);
        NN_LCS_LOG_DEBUG("Download Complete[contentInfoCount : %d]\n", m_pResource->contentInfoCount);
        NN_RESULT_SUCCESS;
    }

    void Client::MakeResumeContext() NN_NOEXCEPT
    {
        m_pResource->context.lcsRole = LcsRole_Client;
        m_pResource->context.hostAddress = m_MasterAddress;
        m_pResource->context.contentsCount = static_cast<uint8_t>(m_pResource->contentInfoCount);
        for (int i = 0; i < DownloadableContentsCountMax; ++i)
        {
            ::std::memcpy(&m_pResource->context.contentsInfo[i], &m_pResource->contentInfo[i], sizeof(ContentsInfo));
        }

        GetSessionInfoFromNetwork(&m_pResource->sessionInfo);
        ::std::memcpy(&m_pResource->context.sessionInfo, &m_pResource->sessionInfo, sizeof(SessionInfo));

        m_pResource->context.suspendedReason = m_pResource->suspendedReason;
        ::std::memcpy(&m_pResource->context.systemDeliveryInfo,
            &m_pResource->shareComponent.systemDeliveryInfo, SystemDeliveryInfoSize);
        m_pResource->context.requiredStorageSize = m_RequiredSize;
        m_pResource->context.myIndex = m_pResource->myIndex;
    }

    bool Client::ExistsNode(uint32_t index) NN_NOEXCEPT
    {
        for (int i = 0; i < m_pResource->nodeInfoCount; ++i)
        {
            if (m_pResource->nodeInfo[i].index == index)
            {
                return true;
            }
        }
        return false;
    }

    Result Client::WaitTask(nn::ns::AsyncResult* asyncResult, uint32_t index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(asyncResult);

        NN_LCS_LOG_DEBUG("WaitTask\n");

        // Leave されるまで
        while (m_IsConnected)
        {
            // タスクの監視
            if (asyncResult->TryWait())
            {
                NN_LCS_LOG_DEBUG("Task Finished\n");
                Result result = asyncResult->Get();

                if (result.IsFailure())
                {
                    NN_LCS_LOG_DEBUG("Get : %08x\n", result.GetInnerValueForDebug());

                    // ErrorContext を保存
                    nn::err::ErrorContext errorContex = {};
                    asyncResult->GetErrorContext(&errorContex);
                    SetErrorContext(errorContex);

                    // 失敗原因が FS 系だったら失敗理由を入れる
                    if (nn::fs::ResultSdCardAccessFailed::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_SdCardError;
                    }
                    else if (nn::fs::ResultGameCardAccessFailed::Includes(result) ||
                             nn::ncm::ResultInvalidContentStorage::Includes(result) ||
                             nn::ncm::ResultGameCardContentMetaDatabaseNotActive::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_GameCardError;
                    }
                    else if (nn::fs::ResultMmcAccessFailed::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_BuiltInStorageError;
                    }
                    else if (nn::fs::ResultDataCorrupted::Includes(result) ||
                             nn::ncm::ResultContentHashNotMatched::Includes(result))
                    {
                        // 壊れたコンテンツを受け取ったときにエラー
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_ContentsCorrupted;
                    }
                    // 容量が足りないときにエラー
                    else if (nn::ncm::ResultNotEnoughInstallSpace::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NotEnoughStorageSpace;
                    }
                    else if (nn::nim::ResultLocalCommunicationSessionClosed::Includes(result) ||
                             nn::nim::ResultLocalCommunicationSocketUnexpectedError::Includes(result) ||
                             nn::nim::ResultLocalCommunicationInvalidSignature::Includes(result) ||
                             nn::nim::ResultLocalCommunicationUnexpectedTag::Includes(result))
                    {
                        return ResultCommunicationError();
                    }
                    else if (nn::nim::ResultLocalCommunicationConnectionCanceled::Includes(result))
                    {
                        asyncResult->Cancel();
                    }
                    else
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_UnknownError;
                    }
                    return ResultAsyncResultError();
                }

                // 相手がいなくなってないか確認
                for (int i = 0; i < m_pResource->nodeInfoCount; ++i)
                {
                    // 相手がいたら成功
                    if (m_pResource->nodeInfo[i].index == index)
                    {
                        NN_RESULT_SUCCESS;
                    }
                }
                NN_LCS_LOG_DEBUG("Leave Index : %x\n", index);
                asyncResult->Cancel();
                return ResultRequestNodeLeaved();
            }
            // 相手がいなくなってないか確認
            bool isContinue = false;
            for (int i = 0; i < m_pResource->nodeInfoCount; ++i)
            {
                // 相手がいたら継続
                if (m_pResource->nodeInfo[i].index == index)
                {
                    isContinue = true;
                }
            }
            if (!isContinue)
            {
                NN_LCS_LOG_DEBUG("Leave Index : %x\n", index);
                asyncResult->Cancel();
                return ResultRequestNodeLeaved();
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(AsyncWaitIntervalMilliSeconds));
        }
        return ResultUserCancel();
    }

    bool Client::WaitSessionInfo() NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType sessionInfoHolder;
        nn::os::MultiWaitHolderType timeoutHolder;
        nn::os::TimerEventType timeoutEvent;

        nn::os::InitializeTimerEvent(&timeoutEvent, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&sessionInfoHolder, &m_SessionInfoEvent);
        nn::os::InitializeMultiWaitHolder(&timeoutHolder, &timeoutEvent);

        nn::os::LinkMultiWaitHolder(&multiWait, &sessionInfoHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &timeoutHolder);

        nn::os::ClearEvent(&m_SessionInfoEvent);
        nn::os::ClearTimerEvent(&timeoutEvent);

        nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(TimeoutMilliSeconds);
        nn::os::StartOneShotTimerEvent(&timeoutEvent, interval);

        nn::os::MultiWaitHolderType *holder = nn::os::WaitAny(&multiWait);

        if (holder == &timeoutHolder)
        {
            isSuccess = false;
        }
        nn::os::UnlinkMultiWaitHolder(&sessionInfoHolder);
        nn::os::UnlinkMultiWaitHolder(&timeoutHolder);
        nn::os::FinalizeMultiWaitHolder(&sessionInfoHolder);
        nn::os::FinalizeMultiWaitHolder(&timeoutHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        nn::os::FinalizeTimerEvent(&timeoutEvent);

        return isSuccess;
    }

}}} // namespace nn::lcs
