﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/lcs_PrivateResult.h>
#include <nn/lcs/detail/lcs_ApplicationShareApi.h>
#include <nn/lcs/detail/lcs_NetworkInfo.h>
#include <nn/lcs/detail/lcs_Resource.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationDownloadApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ContentDeliveryApi.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace lcs { namespace detail
{

    void GetRequestServerStopper(nn::ns::RequestServerStopper* request) NN_NOEXCEPT
    {
        nn::ns::GetRequestServerStopper(request);
    }

    Result CanGetApplicationInfo(Bit64 id) NN_NOEXCEPT
    {
        Result result;
        nn::ncm::ApplicationId appId = {};
        nn::ns::ApplicationDeliveryInfo appInfo[ApplicationDeliveryInfoCountMax] = {};
        nn::ns::ApplicationDeliveryAttribute attribute = {};
        int appInfoCount = 0;
        attribute.Reset();

        appId.value = id;
        attribute.Reset();
        attribute.Set<nn::ns::ApplicationDeliveryAttribute_RequestPatch>(true);

        result = nn::ns::GetApplicationDeliveryInfo(
            &appInfoCount, appInfo, ApplicationDeliveryInfoCountMax, appId, attribute);
        if (result.IsFailure())
        {
            if (nn::ns::ResultApplicationRecordNotFound::Includes(result))
            {
                return ResultApplicationNotFound();
            }
            else if (nn::ns::ResultApplyDeltaTaskExists::Includes(result))
            {
                return ResultApplyDeltaTaskExists();
            }
            else
            {
                NN_ABORT("GetApplicationDeliveryInfo Error : %08x\n", result.GetInnerValueForDebug());
            }
        }

        NN_LCS_LOG_DEBUG("--- CanGetApplicationInfo ---\n");
        for (int i = 0; i < appInfoCount; ++i)
        {
            NN_LCS_LOG_DEBUG("ID : %llx\n", appInfo[i].id.value);
            NN_LCS_LOG_DEBUG("Version : %x\n", appInfo[i].version);
            NN_LCS_LOG_DEBUG("RequiredVersion : %x\n", appInfo[i].requiredVersion);
            NN_LCS_LOG_DEBUG("RequiredSystemVersion : %x\n", appInfo[i].requiredSystemVersion);
        }
        NN_LCS_LOG_DEBUG("----------------------------\n");


        NN_RESULT_SUCCESS;
    }

    Result GetApplicationDeliveryInfo(int* pOutCount, ApplicationDeliveryInfo *pOutInfoList,
        int listCount, Bit64 id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutCount);
        NN_SDK_ASSERT_NOT_NULL(pOutInfoList);

        Result result;
        nn::ncm::ApplicationId appId = {};
        nn::ns::ApplicationDeliveryInfo appInfo[ApplicationDeliveryInfoCountMax] = {};
        nn::ns::ApplicationDeliveryAttribute attribute = {};

        appId.value = id;
        attribute.Reset();
        attribute.Set<nn::ns::ApplicationDeliveryAttribute_RequestPatch>(true);

        result = nn::ns::GetApplicationDeliveryInfo(
            pOutCount, appInfo, ApplicationDeliveryInfoCountMax, appId, attribute);
        if (result.IsFailure())
        {
            if (nn::ns::ResultApplicationRecordNotFound::Includes(result))
            {
                return ResultApplicationNotFound();
            }
            else
            {
                NN_ABORT("GetApplicationDeliveryInfo Error! : %08x\n", result.GetInnerValueForDebug());
            }
        }

        if (*pOutCount > listCount)
        {
            NN_ABORT("GetApplicationDeliveryInfo Size Error!\n");
        }
        ::std::memcpy(pOutInfoList->appDeliveryInfo,
            appInfo, (*pOutCount) * sizeof(ApplicationDeliveryInfo));

        NN_RESULT_SUCCESS;
    }

    Result HasAllContentsToDeliver(
        bool* outValue, const ApplicationDeliveryInfo* appInfoList, int listCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(appInfoList);

        nn::ns::ApplicationDeliveryInfo appInfo[ApplicationDeliveryInfoCountMax] = {};
        for (int i = 0; i < listCount; ++i)
        {
            ::std::memcpy(
                &appInfo[i], appInfoList[i].appDeliveryInfo, NsApplicationDeliveryInfoSize);
        }

        Result result = nn::ns::HasAllContentsToDeliver(outValue, appInfo, listCount);
        if(result.IsFailure())
        {
            return ResultNsError();
        }

        NN_RESULT_SUCCESS;
    }

    Result CompareApplicationDeliveryInfo(int* pOutValue,
        ApplicationDeliveryInfo* lhs, int lhsCount,
        ApplicationDeliveryInfo* rhs, int rhsCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        NN_SDK_ASSERT_NOT_NULL(lhs);
        NN_SDK_ASSERT_NOT_NULL(rhs);

        nn::ns::ApplicationDeliveryInfo lInfo[ApplicationDeliveryInfoCountMax] = {};
        nn::ns::ApplicationDeliveryInfo rInfo[ApplicationDeliveryInfoCountMax] = {};

        //NN_LCS_LOG_DEBUG("L Count : %d\n", lhsCount);
        for (int i = 0; i < lhsCount; ++i)
        {
            ::std::memcpy(&lInfo[i], lhs[i].appDeliveryInfo, NsApplicationDeliveryInfoSize);
            //NN_LCS_LOG_DEBUG("L Id[%d] : %llx\n", i, lInfo[i].id);
            //NN_LCS_LOG_DEBUG("L Version[%d] : %x\n", i, lInfo[i].version);
        }
        //NN_LCS_LOG_DEBUG("R Count : %d\n", rhsCount);
        for (int i = 0; i < rhsCount; ++i)
        {
            ::std::memcpy(&rInfo[i], rhs[i].appDeliveryInfo, NsApplicationDeliveryInfoSize);
            //NN_LCS_LOG_DEBUG("R Id[%d] : %llx\n", i, rInfo[i].id);
            //NN_LCS_LOG_DEBUG("R Version[%d] : %x\n", i, rInfo[i].version);
        }

        Result result =
            nn::ns::CompareApplicationDeliveryInfo(pOutValue, lInfo, lhsCount, rInfo, rhsCount);
        if (result.IsFailure())
        {
            return ResultNsError();
        }
        //NN_LCS_LOG_DEBUG("Value : %d\n", *pOutValue);
        NN_RESULT_SUCCESS;
    }

    Result CanDeliverApplication(
        bool* outValue, const ApplicationDeliveryInfo* receiverInfoList, int receiverInfoListCount,
        const ApplicationDeliveryInfo* senderInfoList, int senderInfoListCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(receiverInfoList);
        NN_SDK_ASSERT_NOT_NULL(senderInfoList);

        nn::ns::ApplicationDeliveryInfo receiverInfo[ApplicationDeliveryInfoCountMax] = {};
        nn::ns::ApplicationDeliveryInfo senderInfo[ApplicationDeliveryInfoCountMax] = {};

        for (int i = 0; i < receiverInfoListCount; ++i)
        {
            ::std::memcpy(
                &receiverInfo[i], receiverInfoList[i].appDeliveryInfo, NsApplicationDeliveryInfoSize);
        }
        for (int i = 0; i < senderInfoListCount; ++i)
        {
            ::std::memcpy(
                &senderInfo[i], senderInfoList[i].appDeliveryInfo, NsApplicationDeliveryInfoSize);
        }

        if (nn::ns::CanDeliverApplication(
            outValue, receiverInfo, receiverInfoListCount, senderInfo, senderInfoListCount).IsFailure())
        {
            // ToDo : 失敗を考える
        }
        NN_RESULT_SUCCESS;
    }

    Result CanDeliverApplication(
        const ApplicationDeliveryInfo* senderInfoList, int senderInfoListCount,
        void* appDeliveryInfoBuffer, size_t appDeliveryInfoBufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(appDeliveryInfoBuffer);
        NN_SDK_ASSERT_GREATER_EQUAL(appDeliveryInfoBufferSize, AppDeliveryInfoSize);
        NN_UNUSED(appDeliveryInfoBufferSize);

        nn::ns::ApplicationDeliveryInfo* sendAppDeliveryInfo =
            new(appDeliveryInfoBuffer)nn::ns::ApplicationDeliveryInfo[ApplicationDeliveryInfoCountMax];

        for (int i = 0; i < senderInfoListCount; ++i)
        {
            ::std::memcpy(
                &sendAppDeliveryInfo[i], senderInfoList[i].appDeliveryInfo, NsApplicationDeliveryInfoSize);
        }
        return nn::ns::CanDeliverApplication(sendAppDeliveryInfo, senderInfoListCount);
    }

    Result NeedsSystemUpdateToDeliverApplication(
        bool* outValue, const ApplicationDeliveryInfo* appInfoList, int appInfoListCount,
        const SystemDeliveryInfo& receiverSystemInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(appInfoList);

        nn::ns::ApplicationDeliveryInfo appInfo[ApplicationDeliveryInfoCountMax] = {};
        nn::ns::SystemDeliveryInfo sysInfo = {};

        for (int i = 0; i < appInfoListCount; ++i)
        {
            ::std::memcpy(
                &appInfo[i], appInfoList[i].appDeliveryInfo, NsApplicationDeliveryInfoSize);
        }
        ::std::memcpy(&sysInfo, &receiverSystemInfo, sizeof(nn::ns::SystemDeliveryInfo));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::ns::NeedsSystemUpdateToDeliverApplication(outValue, appInfo, appInfoListCount, sysInfo));

        NN_RESULT_SUCCESS;
    }

    Result ListContentMetaKeyToDeliverApplication(
        int* outCount, ContentMetaKey* outList, int listCount, Bit64 id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outList);

        nn::ncm::ApplicationId appId = {};
        nn::ns::ApplicationDeliveryInfo appDeliveryInfo[ApplicationDeliveryInfoCountMax] = {};
        int appDeliveryInfoCount = 0;
        nn::ns::ApplicationDeliveryAttribute attribute = {};
        nn::ncm::ContentMetaKey metaKey[ContentMetaKeyMax] = {};
        int metaKeyCount = 0;

        appId.value = id;
        attribute.Reset();
        attribute.Set<nn::ns::ApplicationDeliveryAttribute_RequestPatch>(true);

        nn::ns::GetApplicationDeliveryInfo(
            &appDeliveryInfoCount, appDeliveryInfo, ApplicationDeliveryInfoCountMax,
            appId, attribute);
        Result result = nn::ns::ListContentMetaKeyToDeliverApplication(
            &metaKeyCount, metaKey, ContentMetaKeyMax, 0, appDeliveryInfo, appDeliveryInfoCount);
        if(result.IsFailure())
        {
            if (nn::ns::ResultContentEntityNotFoundForContentDelivery().Includes(result))
            {
                return ResultNsError();
            }
            else
            {
                NN_ABORT("ListContentMetaKeyToDeliverApplication Error! : %08x\n", result.GetInnerValueForDebug());
            }
        }

        if (metaKeyCount > listCount)
        {
            NN_ABORT("ListContentMetaKeyToDeliverApplication Size Error!\n");
        }

        for (int i = 0; i < metaKeyCount; ++i)
        {
            ::std::memcpy(
                &outList[i].key, &metaKey[i], NcmContentMetaKeySize);
            NN_LCS_LOG_DEBUG("ID : %llx / version : %x / type : %x\n",
                metaKey[i].id, metaKey[i].version, metaKey[i].type);
        }
        *outCount = metaKeyCount;

        NN_RESULT_SUCCESS;
    }

    Result RequestReceiveApplication(nn::ns::AsyncResult* asyncResult,
        Ipv4Address fromAddress, uint16_t port,
        Bit64 id, ContentMetaKey* key, int keyCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(asyncResult);
        NN_SDK_ASSERT_NOT_NULL(key);

        NN_LCS_LOG_DEBUG("RequestReceiveApplication\n");
        NN_LCS_LOG_DEBUG("formAddress : %x / port : %d / id : %llx / keyCount : %d\n",
            fromAddress, port, id, keyCount);

        nn::ncm::ApplicationId appId = {};
        nn::ncm::ContentMetaKey metaKey[ContentMetaKeyMax] = {};

        appId.value = id;
        for (int i = 0; i < keyCount; ++i)
        {
            ::std::memcpy(&metaKey[i], key[i].key, NcmContentMetaKeySize);
            NN_LCS_LOG_DEBUG("ID : %llx / version : %x / type : %x\n",
                metaKey[i].id, metaKey[i].version, metaKey[i].type);
        }

        Result result = nn::ns::RequestReceiveApplication(asyncResult, fromAddress, port, appId, metaKey, keyCount);
        if (nn::ns::ResultInvalidContentDeliveryRequest().Includes(result))
        {
            return ResultInvalidContentDeliveryRequest();
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
        NN_RESULT_SUCCESS;
    }

    Result CommitReceiveApplication(Bit64 id) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("CommitReceiveApplication : %llx\n", id);
        nn::ncm::ApplicationId appId = {};
        appId.value = id;

        NN_RESULT_DO(nn::ns::CommitReceiveApplication(appId));

        NN_RESULT_SUCCESS;
    }

    Result GetReceiveApplicationProgress(
        uint64_t* downloadedSize, uint64_t* totalSize, Bit64 id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(downloadedSize);
        NN_SDK_ASSERT_NOT_NULL(totalSize);

        nn::ncm::ApplicationId appId = {};
        nn::ns::ReceiveApplicationProgress progress = {};

        appId.value = id;

        *downloadedSize = 0;
        *totalSize = 0;

        Result result = nn::ns::GetReceiveApplicationProgress(&progress, appId);
        if (result.IsSuccess())
        {
            *downloadedSize = static_cast<uint64_t>(progress.downloadedSize);
            *totalSize = static_cast<uint64_t>(progress.totalSize);
        }

        //NN_LCS_LOG_DEBUG("RecvAppProgress[ID : %llx] : %lld / %lld\n",
        //    id, progress.downloadedSize, progress.totalSize);

        NN_RESULT_SUCCESS;
    }

    Result ConfirmSendApplication(const ApplicationInfo& appInfo) NN_NOEXCEPT
    {
        nn::ns::ApplicationDeliveryInfo ownAppDeliveryInfo[ApplicationDeliveryInfoCountMax] = {};
        int ownAppDeliveryInfoCount = appInfo.deliveryInfoCount;
        for (int i = 0; i < ownAppDeliveryInfoCount; ++i)
        {
            ::std::memcpy(&ownAppDeliveryInfo[i], appInfo.deliveryInfo[i].appDeliveryInfo,
                NsApplicationDeliveryInfoSize);
        }

        bool hasOwnContents = false;
        NN_RESULT_DO(nn::ns::HasAllContentsToDeliver(
            &hasOwnContents, ownAppDeliveryInfo, ownAppDeliveryInfoCount));
        if (!hasOwnContents)
        {
            NN_LCS_LOG_DEBUG("Master did not have content at open\n");
            NN_RESULT_SUCCESS;
        }

        nn::ncm::ApplicationId appId = {};
        nn::ns::ApplicationDeliveryInfo appDeliveryInfo[ApplicationDeliveryInfoCountMax] = {};
        int appDeliveryInfoCount = 0;
        nn::ns::ApplicationDeliveryAttribute attribute = {};

        appId.value = appInfo.detailInfo.id;
        attribute.Reset();
        attribute.Set<nn::ns::ApplicationDeliveryAttribute_RequestPatch>(true);

        NN_RESULT_DO(nn::ns::GetApplicationDeliveryInfo(
            &appDeliveryInfoCount, appDeliveryInfo, ApplicationDeliveryInfoCountMax, appId, attribute));

        bool hasCurrentContents = false;
        NN_RESULT_DO(nn::ns::HasAllContentsToDeliver(
            &hasCurrentContents, appDeliveryInfo, appDeliveryInfoCount));
        if (!hasCurrentContents)
        {
            NN_LCS_LOG_DEBUG("CurrentContent HasAllContentsToDeliver Error\n");
            return ResultFailureStorageError();
        }

        NN_LCS_LOG_DEBUG("Own : %x / %x / %x\n", ownAppDeliveryInfo[0].version,
            ownAppDeliveryInfo[0].requiredSystemVersion, ownAppDeliveryInfo[0].requiredVersion);
        NN_LCS_LOG_DEBUG("Current : %x / %x / %x\n", appDeliveryInfo[0].version,
            appDeliveryInfo[0].requiredSystemVersion, appDeliveryInfo[0].requiredVersion);

        int value = 0;
        NN_RESULT_DO(nn::ns::CompareApplicationDeliveryInfo(
            &value, ownAppDeliveryInfo, ownAppDeliveryInfoCount, appDeliveryInfo, appDeliveryInfoCount));

        NN_LCS_LOG_DEBUG("Value : %d\n", value);

        if (value != 0)
        {
            NN_LCS_LOG_DEBUG("CompareApplicationDeliveryInfo : AppVersion is changed\n");
            return ResultFailureStorageError();
        }
        NN_RESULT_SUCCESS;
    }

    Result RequestSendApplication(nn::ns::AsyncResult* asyncResult,
        Ipv4Address toAddress, uint16_t port, Bit64 id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(asyncResult);

        Result result;
        nn::ncm::ApplicationId appId = {};
        nn::ns::ApplicationDeliveryInfo appDeliveryInfo[ApplicationDeliveryInfoCountMax] = {};
        int appDeliveryInfoCount = 0;
        nn::ns::ApplicationDeliveryAttribute attribute = {};
        nn::ncm::ContentMetaKey metaKey[ContentMetaKeyMax] = {};
        int metaKeyCount = 0;

        appId.value = id;
        attribute.Reset();
        attribute.Set<nn::ns::ApplicationDeliveryAttribute_RequestPatch>(true);

        nn::ns::GetApplicationDeliveryInfo(
            &appDeliveryInfoCount, appDeliveryInfo, ApplicationDeliveryInfoCountMax,
            appId, attribute);
        for (int i = 0; i < appDeliveryInfoCount; ++i)
        {
            NN_LCS_LOG_DEBUG("ID : %llx\n", appDeliveryInfo[i].id.value);
            NN_LCS_LOG_DEBUG("Version : %x\n", appDeliveryInfo[i].version);
            NN_LCS_LOG_DEBUG("RequiredVersion : %x\n", appDeliveryInfo[i].requiredVersion);
            NN_LCS_LOG_DEBUG("RequiredSystemVersion : %x\n", appDeliveryInfo[i].requiredSystemVersion);
        }
        result = nn::ns::ListContentMetaKeyToDeliverApplication(
            &metaKeyCount, metaKey, ContentMetaKeyMax, 0, appDeliveryInfo, appDeliveryInfoCount);
        if (result.IsFailure())
        {
            if (nn::ns::ResultContentEntityNotFoundForContentDelivery().Includes(result))
            {
                return ResultNsError();
            }
            else
            {
                NN_ABORT("ListContentMetaKeyToDeliverApplication Error! :%08x\n", result.GetInnerValueForDebug());
            }
        }

        for (int i = 0; i < metaKeyCount; ++i)
        {
            NN_LCS_LOG_DEBUG("ID : %llx / version : %x / type : %x\n",
                metaKey[i].id, metaKey[i].version, metaKey[i].type);
        }

        result = nn::ns::RequestSendApplication(asyncResult, toAddress, port, appId, metaKey, metaKeyCount);
        if (nn::ns::ResultInvalidContentDeliveryRequest().Includes(result))
        {
            return ResultInvalidContentDeliveryRequest();
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
        NN_RESULT_SUCCESS;
    }

    Result GetSendApplicationProgress(
        uint64_t* deliveredSize, uint64_t* totalSize, Bit64 id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(deliveredSize);
        NN_SDK_ASSERT_NOT_NULL(totalSize);

        nn::ncm::ApplicationId appId = {};
        nn::ns::SendApplicationProgress progress = {};

        appId.value = id;

        if (nn::ns::GetSendApplicationProgress(&progress, appId).IsFailure())
        {
            //NN_LCS_LOG_DEBUG("Fail GetSendApplicationProgress\n");
        }

        *deliveredSize = static_cast<uint64_t>(progress.sentSize);
        *totalSize = static_cast<uint64_t>(progress.totalSize);

        NN_RESULT_SUCCESS;
    }

    Result EvacuateDownloadTask(DownloadTaskInfo* pOutDownloadTaskInfo, Bit64 id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutDownloadTaskInfo);

        nn::ncm::ApplicationId applicationId = {};
        nn::ncm::ContentMetaKey contentMetaKey[ContentMetaKeyMax] = {};

        NN_LCS_LOG_DEBUG("EvacuateDownloadTask ID : %llx\n", id);

        applicationId.value = id;

        auto contentMetaKeyCount = nn::ns::ListNotCommittedContentMeta(
            contentMetaKey, ContentMetaKeyMax, applicationId, 0);

        NN_LCS_LOG_DEBUG("EvacuateDownloadTask ContentMetaKeyCount : %d\n", contentMetaKeyCount);
        NN_RESULT_THROW_UNLESS(contentMetaKeyCount > 0, ResultNotExsistDownloadTask());

        for (int i = 0; i < contentMetaKeyCount; ++i)
        {
            ::std::memcpy(&pOutDownloadTaskInfo->metaInfo.metaKey[i].key, &contentMetaKey[i], NcmContentMetaKeySize);
        }
        pOutDownloadTaskInfo->metaInfo.metaKeyCount = contentMetaKeyCount;
        pOutDownloadTaskInfo->id = id;

        nn::ns::CancelApplicationDownload(applicationId);
        nn::ns::CancelApplicationApplyDelta(applicationId);

        NN_RESULT_SUCCESS;
    }

    Result RecreateDownloadTask(const DownloadTaskInfo& DownloadTaskInfo) NN_NOEXCEPT
    {
        nn::ncm::ApplicationId applicationId = {};
        nn::ncm::ContentMetaKey contentMetaKey[ContentMetaKeyMax] = {};

        NN_LCS_LOG_DEBUG("RecreateDownloadTask ID : %llx\n", DownloadTaskInfo.id);

        applicationId.value = DownloadTaskInfo.id;

        NN_RESULT_THROW_UNLESS(DownloadTaskInfo.metaInfo.metaKeyCount > 0, ResultInvalidContentMetaKey());

        for (int i = 0; i < DownloadTaskInfo.metaInfo.metaKeyCount; ++i)
        {
            ::std::memcpy(&contentMetaKey[i], DownloadTaskInfo.metaInfo.metaKey[i].key, NcmContentMetaKeySize);
        }

        nn::ns::CreateDownloadTask(contentMetaKey, DownloadTaskInfo.metaInfo.metaKeyCount, applicationId);

        NN_RESULT_SUCCESS;
    }

    void ShowApplicationDeliveryInfo(const ApplicationDeliveryInfo& info) NN_NOEXCEPT
    {
        nn::ns::ApplicationDeliveryInfo appInfo = {};
        ::std::memcpy(&appInfo, &info, sizeof(nn::ns::ApplicationDeliveryInfo));
        NN_LCS_LOG_DEBUG("ID : %llx\n", appInfo.id.value);
        NN_LCS_LOG_DEBUG("ApplicationDeliveryProtocolVersion : %x\n", appInfo.applicationDeliveryProtocolVersion);
        NN_LCS_LOG_DEBUG("Version : %x\n", appInfo.version);
        NN_LCS_LOG_DEBUG("RequiredVersion : %x\n", appInfo.requiredVersion);
        NN_LCS_LOG_DEBUG("RequiredSystemVersion : %x\n", appInfo.requiredSystemVersion);
        NN_LCS_LOG_DEBUG("attributes : %x\n", appInfo.attributes);
    }

    void ShowContentMetaKey(const ContentMetaKey& key) NN_NOEXCEPT
    {
        nn::ncm::ContentMetaKey metaKey = {};
        ::std::memcpy(&metaKey, key.key, sizeof(nn::ncm::ContentMetaKey));
        NN_LCS_LOG_DEBUG("ID : %llx\n", metaKey.id);
        NN_LCS_LOG_DEBUG("Version : %x\n", metaKey.version);
        NN_LCS_LOG_DEBUG("Type : %x\n", metaKey.type);
    }


}}} // end of namespace nn::lcs
