﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/lcs_Types.h>
#include <nn/lcs/detail/lcs_NetworkApi.h>
#include <nn/lcs/detail/lcs_Util.h>
#include <nn/lcs/detail/lcs_Version.h>
#include <nn/lcs/detail/Advertise/lcs_AdvertiseManager.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Endian.h>

namespace nn { namespace lcs { namespace detail
{

    Result AdvertiseManager::MakeAdvertiseData(const SessionSettings& settings,
        const Version& version, const char* name, char* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(name);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, AppControlDataSize);

        Result result;
        ::std::memset(&m_AdvertiseData, 0, sizeof(AdvertiseData));

        m_AdvertiseData.lcsVersion = version._raw;
        m_AdvertiseData.sessionAttribute = SessionAttribute_Open;
        m_AdvertiseData.nodeCount = 1;
        m_AdvertiseData.nodeCountMax = settings.nodeCountMax;
        std::strncpy(m_AdvertiseData.hostUserName, name, UserNameBytesMax);

        for (int i = 0; i < settings.applicationCount; ++i)
        {
            // アプリケーションの情報取得
            ::std::memset(buffer, 0, AppControlDataSize);

            ApplicationDetailInfo detailInfo;
            NN_RESULT_DO(GetApplicationDetailInfo(&detailInfo,
                buffer, bufferSize, settings.applications[i].value));
            nn::util::StoreBigEndian(
                &m_AdvertiseData.contents[i].attributeFlag, detailInfo.attributeFlag);
            nn::util::StoreBigEndian(
                &m_AdvertiseData.contents[i].id, settings.applications[i].value);
            m_AdvertiseData.contents[i].contentsFlag = (0x01 << ContentsType::Patch::Index);
            ::std::memcpy(m_AdvertiseData.contents[i].displayVersion,
                detailInfo.displayVersion, DisplayVersionSizeMax);
        }

        m_AdvertiseData.contentCount = settings.applicationCount;

        NN_RESULT_SUCCESS;
    }

    void AdvertiseManager::GetAdvertiseData(AdvertiseData* pOutData) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutData);
        memcpy(pOutData, &m_AdvertiseData, sizeof(AdvertiseData));
    }

    void AdvertiseManager::SetAdvertiseData(const AdvertiseData& data) NN_NOEXCEPT
    {
        memcpy(&m_AdvertiseData, &data, sizeof(AdvertiseData));
    }

    void AdvertiseManager::SetNodeCount(int count) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_MINMAX(count, 0, nn::lcs::NodeCountMax);
        m_AdvertiseData.nodeCount = static_cast<uint8_t>(count);
    }

    void AdvertiseManager::SetContentInfo(
        size_t requiredSize, char* displayVersion, int appIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(displayVersion);
        NN_SDK_ASSERT_RANGE(appIndex, 0, nn::lcs::SharableContentsCountMax);
        nn::util::StoreBigEndian<uint64_t>(
            &m_AdvertiseData.contents[appIndex].requiredSize, requiredSize);
        ::std::memcpy(m_AdvertiseData.contents[appIndex].displayVersion,
            displayVersion, DisplayVersionSizeMax);
    }

    void AdvertiseManager::SetSessionSetting(SessionAttribute attribute) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(
            attribute == SessionAttribute_Close ||
            attribute == SessionAttribute_Open ||
            attribute == SessionAttribute_Limited);
        m_AdvertiseData.sessionAttribute = static_cast<uint8_t>(attribute);
    }

    void AdvertiseManager::SetSessionState(SessionState state) NN_NOEXCEPT
    {
        m_AdvertiseData.sessionState = static_cast<uint8_t>(state);
    }

    void AdvertiseManager::SetNodeProgress(
        AdvertiseNodeProgress *progress, int progressCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(progress);
        NN_SDK_ASSERT_MINMAX(progressCount, 0, NodeCountMax);

        for (int i = 0; i < progressCount; ++i)
        {
            m_AdvertiseData.progress[i].id = progress[i].id;
            m_AdvertiseData.progress[i].contentCount = progress[i].contentCount;
            m_AdvertiseData.progress[i].downloadedContentCount = progress[i].downloadedContentCount;
        }
    }

    void AdvertiseManager::SetConteShareSucceeded(bool isSuccess) NN_NOEXCEPT
    {
        m_AdvertiseData.isContentShareSucceeded = isSuccess;
    }

    Result AdvertiseManager::UpdateAdvertiseData() NN_NOEXCEPT
    {
        NN_RESULT_DO(SetAdvertiseDataForNetwork(&m_AdvertiseData, sizeof(AdvertiseData)));
        NN_RESULT_SUCCESS;
    }

}}} // end of namespace nn::lcs
