﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/kpr/kpr_KeyCodeComposer.h>

#include "kpr_KeyCodeComposerImpl.h"

namespace nn { namespace kpr { namespace detail {

namespace {

//!< デッドキーを処理します。
void ProcessDeadKey(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT;

//!< デッドキーを強制的に処理します。
void ProcessDeadKeyForcibly(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT;

//!< ローマ字変換を処理します。
void ProcessRomaji(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT;

//!< ローマ字変換を強制的に処理します。
void ProcessRomajiForcibly(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT;

} // namespace

void InitializeKeyCodeComposer(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);

    KeyCodeComposerStorage& composer = *pComposer;

    composer = KeyCodeComposerStorage();
}

KeyCodeComposerMode GetKeyCodeComposerMode(
    const KeyCodeComposerStorage& composer) NN_NOEXCEPT
{
    return static_cast<KeyCodeComposerMode>(composer.mode);
}

void SetKeyCodeComposerMode(
    KeyCodeComposerStorage* pComposer, KeyCodeComposerMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);

    KeyCodeComposerStorage& composer = *pComposer;

    if (mode != GetKeyCodeComposerMode(composer))
    {
        ClearKeyCodeComposer(&composer);
    }

    composer.mode = static_cast<int32_t>(mode);
}

int GetKeyCodeComposerCompositionCount(
    const KeyCodeComposerStorage& composer) NN_NOEXCEPT
{
    return composer.outputCount + composer.nativeCount;
}

int GetKeyCodeComposerComposition(
    const KeyCodeComposerStorage& composer, uint16_t outBuffer[], int count
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    int outCount = ::std::min(
        count, GetKeyCodeComposerCompositionCount(composer));

    ::std::copy(composer.queue, composer.queue + outCount, outBuffer);

    return outCount;
}

int EnqueueKeyCodeComposer(
    KeyCodeComposerStorage* pComposer, const uint16_t buffer[], int count
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    KeyCodeComposerStorage& composer = *pComposer;

    const KeyCodeComposerMode mode = GetKeyCodeComposerMode(composer);

    int outCount = 0;

    while (outCount < count)
    {
        if (0 < GetKeyCodeComposerDequeueableCount(composer))
        {
            break;
        }

        const int compositionCount =
            GetKeyCodeComposerCompositionCount(composer);

        if (KeyCodeCompositionCountMax <= compositionCount)
        {
            break;
        }

        composer.queue[compositionCount] = buffer[outCount];
        composer.nativeCount += 1;

        switch (mode)
        {
        case KeyCodeComposerMode_Default:
            composer.outputCount = GetKeyCodeComposerCompositionCount(composer);
            composer.nativeCount = 0;
            break;

        case KeyCodeComposerMode_DeadKey:
            ProcessDeadKey(&composer);
            break;

        case KeyCodeComposerMode_RomajiHiragana:
        case KeyCodeComposerMode_RomajiKatakana:
            ProcessRomaji(&composer);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        ++outCount;
    }

    return outCount;
}

int GetKeyCodeComposerDequeueableCount(
    const KeyCodeComposerStorage& composer) NN_NOEXCEPT
{
    return composer.outputCount;
}

int DequeueKeyCodeComposer(
    KeyCodeComposerStorage* pComposer, uint16_t outBuffer[], int count
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    KeyCodeComposerStorage& composer = *pComposer;

    const int outCount = ::std::min(
        count, GetKeyCodeComposerDequeueableCount(composer));

    if (0 < outCount)
    {
        ::std::copy(composer.queue, composer.queue + outCount, outBuffer);

        ::std::copy(
            composer.queue + outCount,
            composer.queue + GetKeyCodeComposerCompositionCount(composer),
            composer.queue);

        composer.outputCount -= static_cast<int32_t>(outCount);
    }

    return outCount;
}

int RemoveKeyCodeComposer(
    KeyCodeComposerStorage* pComposer, uint16_t outBuffer[], int count
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    KeyCodeComposerStorage& composer = *pComposer;

    const int outCount = ::std::min(count, composer.nativeCount);

    if (0 < outCount)
    {
        const int compositionCount =
            GetKeyCodeComposerCompositionCount(composer);

        ::std::copy(
            composer.queue + compositionCount - outCount,
            composer.queue + compositionCount, outBuffer);

        composer.nativeCount -= static_cast<int32_t>(outCount);
    }

    return outCount;
}

void FlushKeyCodeComposer(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);

    KeyCodeComposerStorage& composer = *pComposer;

    switch (GetKeyCodeComposerMode(composer))
    {
    case KeyCodeComposerMode_Default:
        composer.outputCount = GetKeyCodeComposerCompositionCount(composer);
        composer.nativeCount = 0;
        break;

    case KeyCodeComposerMode_DeadKey:
        ProcessDeadKeyForcibly(&composer);
        break;

    case KeyCodeComposerMode_RomajiHiragana:
    case KeyCodeComposerMode_RomajiKatakana:
        ProcessRomajiForcibly(&composer);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

void ClearKeyCodeComposer(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);

    KeyCodeComposerStorage& composer = *pComposer;

    const KeyCodeComposerMode mode = GetKeyCodeComposerMode(composer);

    InitializeKeyCodeComposer(&composer);

    SetKeyCodeComposerMode(&composer, mode);
}

namespace {

//!< デッドキー処理表
const uint16_t DeadKeyMap[][3] =
{
    { 0x0300, 0x0300, 0x0060 }, // ` + ` => `
    { 0x0300, 0x0041, 0x00C0 }, // ` + A => À
    { 0x0300, 0x0045, 0x00C8 }, // ` + E => È
    { 0x0300, 0x0049, 0x00CC }, // ` + I => Ì
    { 0x0300, 0x004F, 0x00D2 }, // ` + O => Ò
    { 0x0300, 0x0055, 0x00D9 }, // ` + U => Ù

    { 0x0300, 0x0061, 0x00E0 }, // ` + a => à
    { 0x0300, 0x0065, 0x00E8 }, // ` + e => è
    { 0x0300, 0x0069, 0x00EC }, // ` + i => ì
    { 0x0300, 0x006F, 0x00F2 }, // ` + o => ò
    { 0x0300, 0x0075, 0x00F9 }, // ` + u => ù

    { 0x0301, 0x0301, 0x00B4 }, // ´ + ´ => ´
    { 0x0301, 0x0041, 0x00C1 }, // ´ + A => Á
    { 0x0301, 0x0045, 0x00C9 }, // ´ + E => É
    { 0x0301, 0x0049, 0x00CD }, // ´ + I => Í
    { 0x0301, 0x004F, 0x00D3 }, // ´ + O => Ó
    { 0x0301, 0x0055, 0x00DA }, // ´ + U => Ú
    { 0x0301, 0x0059, 0x00DD }, // ´ + Y => Ý

    { 0x0301, 0x0061, 0x00E1 }, // ´ + a => á
    { 0x0301, 0x0065, 0x00E9 }, // ´ + e => é
    { 0x0301, 0x0069, 0x00ED }, // ´ + i => í
    { 0x0301, 0x006F, 0x00F3 }, // ´ + o => ó
    { 0x0301, 0x0075, 0x00FA }, // ´ + u => ú
    { 0x0301, 0x0079, 0x00FD }, // ´ + y => ý

    { 0x0302, 0x0302, 0x005E }, // ^ + ^ => ^
    { 0x0302, 0x0041, 0x00C2 }, // ^ + A => Â
    { 0x0302, 0x0045, 0x00CA }, // ^ + E => Ê
    { 0x0302, 0x0049, 0x00CE }, // ^ + I => Î
    { 0x0302, 0x004F, 0x00D4 }, // ^ + O => Ô
    { 0x0302, 0x0055, 0x00DB }, // ^ + U => Û

    { 0x0302, 0x0061, 0x00E2 }, // ^ + a => â
    { 0x0302, 0x0065, 0x00EA }, // ^ + e => ê
    { 0x0302, 0x0069, 0x00EE }, // ^ + i => î
    { 0x0302, 0x006F, 0x00F4 }, // ^ + o => ô
    { 0x0302, 0x0075, 0x00FB }, // ^ + u => û

    { 0x0303, 0x0303, 0x007E }, // ~ + ~ => ~
    { 0x0303, 0x0041, 0x00C3 }, // ~ + A => Ã
    { 0x0303, 0x004E, 0x00D1 }, // ~ + N => Ñ
    { 0x0303, 0x004F, 0x00D5 }, // ~ + O => Õ
    { 0x0303, 0x0061, 0x00E3 }, // ~ + a => ã
    { 0x0303, 0x006E, 0x00F1 }, // ~ + n => ñ
    { 0x0303, 0x006F, 0x00F5 }, // ~ + o => õ

    { 0x0308, 0x0308, 0x00A8 }, // ¨ + ¨ => ¨
    { 0x0308, 0x0041, 0x00C4 }, // ¨ + A => Ä
    { 0x0308, 0x0045, 0x00CB }, // ¨ + E => Ë
    { 0x0308, 0x0049, 0x00CF }, // ¨ + I => Ï
    { 0x0308, 0x004F, 0x00D6 }, // ¨ + O => Ö
    { 0x0308, 0x0055, 0x00DC }, // ¨ + U => Ü
    { 0x0308, 0x0059, 0x0178 }, // ¨ + Y => Ÿ

    { 0x0308, 0x0061, 0x00E4 }, // ¨ + a => ä
    { 0x0308, 0x0065, 0x00EB }, // ¨ + e => ë
    { 0x0308, 0x0069, 0x00EF }, // ¨ + i => ï
    { 0x0308, 0x006F, 0x00F6 }, // ¨ + o => ö
    { 0x0308, 0x0075, 0x00FC }, // ¨ + u => ü
    { 0x0308, 0x0079, 0x00FF }, // ¨ + y => ÿ

    { 0x0327, 0x0327, 0x00B8 }, // ¸ + ¸ => ¸
    { 0x0327, 0x0043, 0x00C7 }, // ¸ + C => Ç
    { 0x0327, 0x0063, 0x00E7 }, // ¸ + c => ç

    // US-international キーボード用
    { 0x030D, 0x030D, 0x0027 }, // ' + ' => '
    { 0x030D, 0x0041, 0x00C1 }, // ' + A => Á
    { 0x030D, 0x0045, 0x00C9 }, // ' + E => É
    { 0x030D, 0x0049, 0x00CD }, // ' + I => Í
    { 0x030D, 0x004F, 0x00D3 }, // ' + O => Ó
    { 0x030D, 0x0055, 0x00DA }, // ' + U => Ú
    { 0x030D, 0x0059, 0x00DD }, // ' + Y => Ý
    { 0x030D, 0x0043, 0x00C7 }, // ' + C => Ç

    { 0x030D, 0x0061, 0x00E1 }, // ' + a => á
    { 0x030D, 0x0065, 0x00E9 }, // ' + e => é
    { 0x030D, 0x0069, 0x00ED }, // ' + i => í
    { 0x030D, 0x006F, 0x00F3 }, // ' + o => ó
    { 0x030D, 0x0075, 0x00FA }, // ' + u => ú
    { 0x030D, 0x0079, 0x00FD }, // ' + y => ý
    { 0x030D, 0x0063, 0x00E7 }, // ' + c => ç

    { 0x030E, 0x030E, 0x0022 }, // " + " => "
    { 0x030E, 0x0041, 0x00C4 }, // " + A => Ä
    { 0x030E, 0x0045, 0x00CB }, // " + E => Ë
    { 0x030E, 0x0049, 0x00CF }, // " + I => Ï
    { 0x030E, 0x004F, 0x00D6 }, // " + O => Ö
    { 0x030E, 0x0055, 0x00DC }, // " + U => Ü
    { 0x030E, 0x0059, 0x0178 }, // " + Y => Ÿ

    { 0x030E, 0x0061, 0x00E4 }, // " + a => ä
    { 0x030E, 0x0065, 0x00EB }, // " + e => ë
    { 0x030E, 0x0069, 0x00EF }, // " + i => ï
    { 0x030E, 0x006F, 0x00F6 }, // " + o => ö
    { 0x030E, 0x0075, 0x00FC }, // " + u => ü
    { 0x030E, 0x0079, 0x00FF }, // " + y => ÿ

    // ギリシャ文字
    { 0x0344, 0x0344, 0x0385 }, // ΅ + ΅ => ΅
    { 0x0344, 0x03B9, 0x0390 }, // ΅ + ι => ΐ
    { 0x0344, 0x03C5, 0x03B0 }, // ΅ + υ => ΰ

    { 0x0308, 0x0399, 0x03AA }, // ¨ + Ι => Ϊ
    { 0x0308, 0x03A5, 0x03AB }, // ¨ + Υ => Ϋ

    { 0x0308, 0x03B9, 0x03CA }, // ¨ + ι => ϊ
    { 0x0308, 0x03C5, 0x03CB }, // ¨ + υ => ϋ

    { 0x0301, 0x0391, 0x0386 }, // ´ + Α => Ά
    { 0x0301, 0x0395, 0x0388 }, // ´ + Ε => Έ
    { 0x0301, 0x0397, 0x0389 }, // ´ + Η => Ή
    { 0x0301, 0x0399, 0x038A }, // ´ + Ι => Ί
    { 0x0301, 0x039F, 0x038C }, // ´ + Ο => Ό
    { 0x0301, 0x03A5, 0x038E }, // ´ + Υ => Ύ
    { 0x0301, 0x03A9, 0x038F }, // ´ + Ω => Ώ

    { 0x0301, 0x03B1, 0x03AC }, // ´ + α => ά
    { 0x0301, 0x03B5, 0x03AD }, // ´ + ε => έ
    { 0x0301, 0x03B7, 0x03AE }, // ´ + η => ή
    { 0x0301, 0x03B9, 0x03AF }, // ´ + ι => ί
    { 0x0301, 0x03BF, 0x03CC }, // ´ + ο => ό
    { 0x0301, 0x03C5, 0x03CD }, // ´ + υ => ύ
    { 0x0301, 0x03C9, 0x03CE }, // ´ + ω => ώ
};

//!< ローマ字 "っ" 変換表
const uint16_t RomajiSmallTsuMap[] =
{
    0x3063, // っ
    0x30C3, // ッ
};

//!< ローマ字 "ん" 変換表
const uint16_t RomajiNNMap[] =
{
    0x3093, // ん
    0x30F3, // ン
};

//!< ローマ字変換表のエントリを表す構造体です。
struct RomajiMapEntry
{
    int32_t  nativeCount;
    char     native[8];
    int32_t  outputCount;
    uint16_t output[2];
};

//!< ローマ字変換表
const RomajiMapEntry RomajiMap[] =
{
    { 4, "ltsa", 2, { 0x3063, 0x3041 } },
    { 4, "ltse", 2, { 0x3063, 0x3047 } },
    { 4, "ltsi", 2, { 0x3063, 0x3043 } },
    { 4, "ltso", 2, { 0x3063, 0x3049 } },
    { 4, "ltsu", 1, { 0x3063,        } },

    { 4, "lwha", 2, { 0x3045, 0x3041 } },
    { 4, "lwhe", 2, { 0x3045, 0x3047 } },
    { 4, "lwhi", 2, { 0x3045, 0x3043 } },
    { 4, "lwho", 2, { 0x3045, 0x3049 } },
    { 4, "lwhu", 1, { 0x3045,        } },

    { 4, "xtsa", 2, { 0x3063, 0x3041 } },
    { 4, "xtse", 2, { 0x3063, 0x3047 } },
    { 4, "xtsi", 2, { 0x3063, 0x3043 } },
    { 4, "xtso", 2, { 0x3063, 0x3049 } },
    { 4, "xtsu", 1, { 0x3063,        } },

    { 4, "xwha", 2, { 0x3045, 0x3041 } },
    { 4, "xwhe", 2, { 0x3045, 0x3047 } },
    { 4, "xwhi", 2, { 0x3045, 0x3043 } },
    { 4, "xwho", 2, { 0x3045, 0x3049 } },
    { 4, "xwhu", 1, { 0x3045,        } },

    { 3, "bya", 2, { 0x3073, 0x3083 } },
    { 3, "bye", 2, { 0x3073, 0x3047 } },
    { 3, "byi", 2, { 0x3073, 0x3043 } },
    { 3, "byo", 2, { 0x3073, 0x3087 } },
    { 3, "byu", 2, { 0x3073, 0x3085 } },

    { 3, "cha", 2, { 0x3061, 0x3083 } },
    { 3, "che", 2, { 0x3061, 0x3047 } },
    { 3, "chi", 1, { 0x3061,        } },
    { 3, "cho", 2, { 0x3061, 0x3087 } },
    { 3, "chu", 2, { 0x3061, 0x3085 } },

    { 3, "cya", 2, { 0x3061, 0x3083 } },
    { 3, "cye", 2, { 0x3061, 0x3047 } },
    { 3, "cyi", 2, { 0x3061, 0x3043 } },
    { 3, "cyo", 2, { 0x3061, 0x3087 } },
    { 3, "cyu", 2, { 0x3061, 0x3085 } },

    { 3, "dha", 2, { 0x3067, 0x3083 } },
    { 3, "dhe", 2, { 0x3067, 0x3047 } },
    { 3, "dhi", 2, { 0x3067, 0x3043 } },
    { 3, "dho", 2, { 0x3067, 0x3087 } },
    { 3, "dhu", 2, { 0x3067, 0x3085 } },

    { 3, "dya", 2, { 0x3062, 0x3083 } },
    { 3, "dyi", 2, { 0x3062, 0x3043 } },
    { 3, "dye", 2, { 0x3062, 0x3047 } },
    { 3, "dyo", 2, { 0x3062, 0x3087 } },
    { 3, "dyu", 2, { 0x3062, 0x3085 } },

    { 3, "fya", 2, { 0x3075, 0x3083 } },
    { 3, "fye", 2, { 0x3075, 0x3047 } },
    { 3, "fyi", 2, { 0x3075, 0x3043 } },
    { 3, "fyo", 2, { 0x3075, 0x3087 } },
    { 3, "fyu", 2, { 0x3075, 0x3085 } },

    { 3, "gya", 2, { 0x304E, 0x3083 } },
    { 3, "gye", 2, { 0x304E, 0x3047 } },
    { 3, "gyi", 2, { 0x304E, 0x3043 } },
    { 3, "gyo", 2, { 0x304E, 0x3087 } },
    { 3, "gyu", 2, { 0x304E, 0x3085 } },

    { 3, "hya", 2, { 0x3072, 0x3083 } },
    { 3, "hye", 2, { 0x3072, 0x3047 } },
    { 3, "hyi", 2, { 0x3072, 0x3043 } },
    { 3, "hyo", 2, { 0x3072, 0x3087 } },
    { 3, "hyu", 2, { 0x3072, 0x3085 } },

    { 3, "jya", 2, { 0x3058, 0x3083 } },
    { 3, "jye", 2, { 0x3058, 0x3047 } },
    { 3, "jyi", 2, { 0x3058, 0x3043 } },
    { 3, "jyo", 2, { 0x3058, 0x3087 } },
    { 3, "jyu", 2, { 0x3058, 0x3085 } },

    { 3, "kya", 2, { 0x304D, 0x3083 } },
    { 3, "kye", 2, { 0x304D, 0x3047 } },
    { 3, "kyi", 2, { 0x304D, 0x3043 } },
    { 3, "kyo", 2, { 0x304D, 0x3087 } },
    { 3, "kyu", 2, { 0x304D, 0x3085 } },

    { 3, "lya", 1, { 0x3083 } },
    { 3, "lye", 1, { 0x3047 } },
    { 3, "lyi", 1, { 0x3043 } },
    { 3, "lyo", 1, { 0x3087 } },
    { 3, "lyu", 1, { 0x3085 } },

    { 3, "ltu", 1, { 0x3063 } },

    { 3, "lwa", 1, { 0x308E } },

    { 3, "mya", 2, { 0x307F, 0x3083 } },
    { 3, "mye", 2, { 0x307F, 0x3047 } },
    { 3, "myi", 2, { 0x307F, 0x3043 } },
    { 3, "myo", 2, { 0x307F, 0x3087 } },
    { 3, "myu", 2, { 0x307F, 0x3085 } },

    { 3, "nya", 2, { 0x306B, 0x3083 } },
    { 3, "nye", 2, { 0x306B, 0x3047 } },
    { 3, "nyi", 2, { 0x306B, 0x3043 } },
    { 3, "nyo", 2, { 0x306B, 0x3087 } },
    { 3, "nyu", 2, { 0x306B, 0x3085 } },

    { 3, "pya", 2, { 0x3074, 0x3083 } },
    { 3, "pye", 2, { 0x3074, 0x3047 } },
    { 3, "pyi", 2, { 0x3074, 0x3043 } },
    { 3, "pyo", 2, { 0x3074, 0x3087 } },
    { 3, "pyu", 2, { 0x3074, 0x3085 } },

    { 3, "qya", 2, { 0x304F, 0x3083 } },
    { 3, "qye", 2, { 0x304F, 0x3047 } },
    { 3, "qyi", 2, { 0x304F, 0x3043 } },
    { 3, "qyo", 2, { 0x304F, 0x3087 } },
    { 3, "qyu", 2, { 0x304F, 0x3085 } },

    { 3, "rya", 2, { 0x308A, 0x3083 } },
    { 3, "rye", 2, { 0x308A, 0x3047 } },
    { 3, "ryi", 2, { 0x308A, 0x3043 } },
    { 3, "ryo", 2, { 0x308A, 0x3087 } },
    { 3, "ryu", 2, { 0x308A, 0x3085 } },

    { 3, "sha", 2, { 0x3057, 0x3083 } },
    { 3, "she", 2, { 0x3057, 0x3047 } },
    { 3, "shi", 1, { 0x3057,        } },
    { 3, "sho", 2, { 0x3057, 0x3087 } },
    { 3, "shu", 2, { 0x3057, 0x3085 } },

    { 3, "sya", 2, { 0x3057, 0x3083 } },
    { 3, "sye", 2, { 0x3057, 0x3047 } },
    { 3, "syi", 2, { 0x3057, 0x3043 } },
    { 3, "syo", 2, { 0x3057, 0x3087 } },
    { 3, "syu", 2, { 0x3057, 0x3085 } },

    { 3, "tha", 2, { 0x3066, 0x3083 } },
    { 3, "the", 2, { 0x3066, 0x3047 } },
    { 3, "thi", 2, { 0x3066, 0x3043 } },
    { 3, "tho", 2, { 0x3066, 0x3087 } },
    { 3, "thu", 2, { 0x3066, 0x3085 } },

    { 3, "tsa", 2, { 0x3064, 0x3041 } },
    { 3, "tse", 2, { 0x3064, 0x3047 } },
    { 3, "tsi", 2, { 0x3064, 0x3043 } },
    { 3, "tso", 2, { 0x3064, 0x3049 } },
    { 3, "tsu", 1, { 0x3064,        } },

    { 3, "tya", 2, { 0x3061, 0x3083 } },
    { 3, "tye", 2, { 0x3061, 0x3047 } },
    { 3, "tyi", 2, { 0x3061, 0x3043 } },
    { 3, "tyo", 2, { 0x3061, 0x3087 } },
    { 3, "tyu", 2, { 0x3061, 0x3085 } },

    { 3, "vya", 2, { 0x30F4, 0x3083 } },
    { 3, "vye", 2, { 0x30F4, 0x3047 } },
    { 3, "vyi", 2, { 0x30F4, 0x3043 } },
    { 3, "vyo", 2, { 0x30F4, 0x3087 } },
    { 3, "vyu", 2, { 0x30F4, 0x3085 } },

    { 3, "wha", 2, { 0x3046, 0x3041 } },
    { 3, "whe", 2, { 0x3046, 0x3047 } },
    { 3, "whi", 2, { 0x3046, 0x3043 } },
    { 3, "who", 2, { 0x3046, 0x3049 } },
    { 3, "whu", 1, { 0x3046,        } },

    { 3, "xya", 1, { 0x3083 } },
    { 3, "xye", 1, { 0x3047 } },
    { 3, "xyi", 1, { 0x3043 } },
    { 3, "xyo", 1, { 0x3087 } },
    { 3, "xyu", 1, { 0x3085 } },

    { 3, "xtu", 1, { 0x3063 } },

    { 3, "xwa", 1, { 0x308E } },

    { 3, "zya", 2, { 0x3058, 0x3083 } },
    { 3, "zye", 2, { 0x3058, 0x3047 } },
    { 3, "zyi", 2, { 0x3058, 0x3043 } },
    { 3, "zyo", 2, { 0x3058, 0x3087 } },
    { 3, "zyu", 2, { 0x3058, 0x3085 } },

    { 2, "ba", 1, { 0x3070 } },
    { 2, "be", 1, { 0x3079 } },
    { 2, "bi", 1, { 0x3073 } },
    { 2, "bo", 1, { 0x307C } },
    { 2, "bu", 1, { 0x3076 } },

    { 2, "ca", 1, { 0x304B } },
    { 2, "ce", 1, { 0x305B } },
    { 2, "ci", 1, { 0x3057 } },
    { 2, "co", 1, { 0x3053 } },
    { 2, "cu", 1, { 0x304F } },

    { 2, "da", 1, { 0x3060 } },
    { 2, "de", 1, { 0x3067 } },
    { 2, "di", 1, { 0x3062 } },
    { 2, "do", 1, { 0x3069 } },
    { 2, "du", 1, { 0x3065 } },

    { 2, "fa", 2, { 0x3075, 0x3041 } },
    { 2, "fe", 2, { 0x3075, 0x3047 } },
    { 2, "fi", 2, { 0x3075, 0x3043 } },
    { 2, "fo", 2, { 0x3075, 0x3049 } },
    { 2, "fu", 1, { 0x3075,        } },

    { 2, "ga", 1, { 0x304C } },
    { 2, "ge", 1, { 0x3052 } },
    { 2, "gi", 1, { 0x304E } },
    { 2, "go", 1, { 0x3054 } },
    { 2, "gu", 1, { 0x3050 } },

    { 2, "ha", 1, { 0x306F } },
    { 2, "he", 1, { 0x3078 } },
    { 2, "hi", 1, { 0x3072 } },
    { 2, "ho", 1, { 0x307B } },
    { 2, "hu", 1, { 0x3075 } },

    { 2, "ja", 2, { 0x3058, 0x3083 } },
    { 2, "je", 2, { 0x3058, 0x3047 } },
    { 2, "ji", 1, { 0x3058,        } },
    { 2, "jo", 2, { 0x3058, 0x3087 } },
    { 2, "ju", 2, { 0x3058, 0x3085 } },

    { 2, "ka", 1, { 0x304B } },
    { 2, "ke", 1, { 0x3051 } },
    { 2, "ki", 1, { 0x304D } },
    { 2, "ko", 1, { 0x3053 } },
    { 2, "ku", 1, { 0x304F } },

    { 2, "la", 1, { 0x3041 } },
    { 2, "le", 1, { 0x3047 } },
    { 2, "li", 1, { 0x3043 } },
    { 2, "lo", 1, { 0x3049 } },
    { 2, "lu", 1, { 0x3045 } },

    { 2, "ma", 1, { 0x307E } },
    { 2, "me", 1, { 0x3081 } },
    { 2, "mi", 1, { 0x307F } },
    { 2, "mo", 1, { 0x3082 } },
    { 2, "mu", 1, { 0x3080 } },

    { 2, "na", 1, { 0x306A } },
    { 2, "ne", 1, { 0x306D } },
    { 2, "ni", 1, { 0x306B } },
    { 2, "no", 1, { 0x306E } },
    { 2, "nu", 1, { 0x306C } },

    { 2, "pa", 1, { 0x3071 } },
    { 2, "pe", 1, { 0x307A } },
    { 2, "pi", 1, { 0x3074 } },
    { 2, "po", 1, { 0x307D } },
    { 2, "pu", 1, { 0x3077 } },

    { 2, "qa", 2, { 0x304F, 0x3041 } },
    { 2, "qe", 2, { 0x304F, 0x3047 } },
    { 2, "qi", 2, { 0x304F, 0x3043 } },
    { 2, "qo", 2, { 0x304F, 0x3049 } },
    { 2, "qu", 1, { 0x304F,        } },

    { 2, "ra", 1, { 0x3089 } },
    { 2, "re", 1, { 0x308C } },
    { 2, "ri", 1, { 0x308A } },
    { 2, "ro", 1, { 0x308D } },
    { 2, "ru", 1, { 0x308B } },

    { 2, "sa", 1, { 0x3055 } },
    { 2, "se", 1, { 0x305B } },
    { 2, "si", 1, { 0x3057 } },
    { 2, "so", 1, { 0x305D } },
    { 2, "su", 1, { 0x3059 } },

    { 2, "ta", 1, { 0x305F } },
    { 2, "te", 1, { 0x3066 } },
    { 2, "ti", 1, { 0x3061 } },
    { 2, "to", 1, { 0x3068 } },
    { 2, "tu", 1, { 0x3064 } },

    { 2, "va", 2, { 0x30F4, 0x3041 } },
    { 2, "ve", 2, { 0x30F4, 0x3047 } },
    { 2, "vi", 2, { 0x30F4, 0x3043 } },
    { 2, "vo", 2, { 0x30F4, 0x3049 } },
    { 2, "vu", 1, { 0x30F4,        } },

    { 2, "wa", 1, { 0x308F,        } },
    { 2, "we", 2, { 0x3046, 0x3047 } },
    { 2, "wi", 2, { 0x3046, 0x3043 } },
    { 2, "wo", 1, { 0x3092,        } },
    { 2, "wu", 1, { 0x3046,        } },

    { 2, "xa", 1, { 0x3041 } },
    { 2, "xe", 1, { 0x3047 } },
    { 2, "xi", 1, { 0x3043 } },
    { 2, "xo", 1, { 0x3049 } },
    { 2, "xu", 1, { 0x3045 } },

    { 2, "ya", 1, { 0x3084,        } },
    { 2, "ye", 2, { 0x3044, 0x3047 } },
    { 2, "yi", 1, { 0x3044,        } },
    { 2, "yo", 1, { 0x3088,        } },
    { 2, "yu", 1, { 0x3086,        } },

    { 2, "za", 1, { 0x3056 } },
    { 2, "ze", 1, { 0x305C } },
    { 2, "zi", 1, { 0x3058 } },
    { 2, "zo", 1, { 0x305E } },
    { 2, "zu", 1, { 0x305A } },

    { 2, "nn", 1, { 0x3093 } },

    { 1, "a", 1, { 0x3042 } },
    { 1, "e", 1, { 0x3048 } },
    { 1, "i", 1, { 0x3044 } },
    { 1, "o", 1, { 0x304A } },
    { 1, "u", 1, { 0x3046 } },
};

void ProcessDeadKey(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->nativeCount, 1, 2);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->outputCount, 0, KeyCodeCompositionCountMax - 1);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(*pComposer),
        1, KeyCodeCompositionCountMax);

    KeyCodeComposerStorage& composer = *pComposer;

    if (composer.nativeCount == 1)
    {
        for (const uint16_t (&entry)[3] : DeadKeyMap)
        {
            if (composer.queue[composer.outputCount] == entry[0])
            {
                return;
            }
        }

        composer.outputCount += 1;
        composer.nativeCount -= 1;
        return;
    }

    if (composer.nativeCount == 2)
    {
        uint16_t& head = composer.queue[composer.outputCount];
        uint16_t& tail = composer.queue[composer.outputCount + 1];

        uint16_t spacingDeadKeys[2] = {};

        for (const uint16_t (&entry)[3] : DeadKeyMap)
        {
            if (head == entry[0] && tail == entry[1])
            {
                head = entry[2];
                composer.outputCount += 1;
                composer.nativeCount -= 2;
                return;
            }

            if (head == entry[0] && head == entry[1])
            {
                spacingDeadKeys[0] = entry[2];
            }

            if (tail == entry[0] && tail == entry[1])
            {
                spacingDeadKeys[1] = entry[2];
            }
        }

        head = spacingDeadKeys[0];

        if (spacingDeadKeys[1] > 0)
        {
            tail = spacingDeadKeys[1];
        }

        composer.outputCount += 2;
        composer.nativeCount -= 2;
        return;
    }
}

void ProcessDeadKeyForcibly(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->nativeCount, 0, KeyCodeCompositionCountMax);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->outputCount, 0, KeyCodeCompositionCountMax);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(*pComposer),
        0, KeyCodeCompositionCountMax);

    KeyCodeComposerStorage& composer = *pComposer;

    for (int i = 0; i < composer.nativeCount; ++i)
    {
        for (const uint16_t (&entry)[3] : DeadKeyMap)
        {
            uint16_t& value = composer.queue[composer.outputCount + i];

            if (value == entry[0] && value == entry[1])
            {
                value = entry[2];

                break;
            }
        }
    }

    composer.outputCount += composer.nativeCount;
    composer.nativeCount = 0;
}

//!< 非英字でローマ字変換を解決します。
bool DispatchRomajiNonLetter(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->nativeCount, 1, KeyCodeCompositionCountMax);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->outputCount, 0, KeyCodeCompositionCountMax - 1);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(*pComposer),
        1, KeyCodeCompositionCountMax);

    KeyCodeComposerStorage& composer = *pComposer;

    const int compositionCount =
        GetKeyCodeComposerCompositionCount(composer);

    const uint16_t& tail = composer.queue[compositionCount - 1];

    if ('a' <= tail  && tail <= 'z')
    {
        return false;
    }
    else
    {
        if (composer.nativeCount > 1)
        {
            uint16_t& fore = composer.queue[compositionCount - 2];

            if (fore == 'n')
            {
                switch (GetKeyCodeComposerMode(composer))
                {
                case KeyCodeComposerMode_RomajiHiragana:
                    fore = RomajiNNMap[0];
                    break;

                case KeyCodeComposerMode_RomajiKatakana:
                    fore = RomajiNNMap[1];
                    break;

                default: NN_UNEXPECTED_DEFAULT;
                }
            }
        }

        composer.outputCount += composer.nativeCount;
        composer.nativeCount = 0;

        return true;
    }
}

//!< ローマ字変換表のエントリを返します。
const RomajiMapEntry* GetRomajiMapEntry(
    const KeyCodeComposerStorage& composer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(
        composer.nativeCount, 1, KeyCodeCompositionCountMax);
    NN_SDK_REQUIRES_MINMAX(
        composer.outputCount, 0, KeyCodeCompositionCountMax - 1);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(composer),
        1, KeyCodeCompositionCountMax);

    const uint16_t& queueTail =
        composer.queue[GetKeyCodeComposerCompositionCount(composer) - 1];

    for (const RomajiMapEntry& entry : RomajiMap)
    {
        if (composer.nativeCount >= entry.nativeCount)
        {
            const RomajiMapEntry* pEntry = &entry;

            const char& entryTail = entry.native[entry.nativeCount - 1];

            for (int i = 0; i < entry.nativeCount; ++i)
            {
                auto queueValue = *(&queueTail - i);
                auto entryValue = *(&entryTail - i);

                if (queueValue != static_cast<uint16_t>(entryValue))
                {
                    pEntry = nullptr;

                    break;
                }
            }

            if (pEntry != nullptr)
            {
                return pEntry;
            }
        }
    }

    return nullptr;
}

//!< キーコードをひらがなからカタカナへ変換します。
uint16_t ConvertHiraganaToKatakana(uint16_t value) NN_NOEXCEPT
{
    return (value < 0x30A1) ? static_cast<uint16_t>(value + 0x60u) : value;
}

//!< ローマ字変換表でローマ字変換を解決します。
bool DispatchRomajiMap(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->nativeCount, 1, KeyCodeCompositionCountMax);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->outputCount, 0, KeyCodeCompositionCountMax - 1);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(*pComposer),
        1, KeyCodeCompositionCountMax);

    KeyCodeComposerStorage& composer = *pComposer;

    const RomajiMapEntry* pEntry = GetRomajiMapEntry(composer);

    if (pEntry == nullptr)
    {
        return false;
    }

    const RomajiMapEntry& entry = *pEntry;

    const KeyCodeComposerMode mode = GetKeyCodeComposerMode(composer);

    if (composer.nativeCount > entry.nativeCount)
    {
        if (composer.nativeCount - entry.nativeCount == 2)
        {
            composer.outputCount += 2;
            composer.nativeCount -= 2;
        }
        else
        {
            uint16_t& value = composer.queue[composer.outputCount];

            if (value == composer.queue[composer.outputCount + 1])
            {
                switch (mode)
                {
                case KeyCodeComposerMode_RomajiHiragana:
                    value = RomajiSmallTsuMap[0];
                    break;

                case KeyCodeComposerMode_RomajiKatakana:
                    value = RomajiSmallTsuMap[1];
                    break;

                default: NN_UNEXPECTED_DEFAULT;
                }
            }
        }
    }

    uint16_t& head = composer.queue[
        GetKeyCodeComposerCompositionCount(composer) - entry.nativeCount];

    for (int i = 0; i < entry.outputCount; ++i)
    {
        uint16_t& value = *(&head + i);

        switch (mode)
        {
        case KeyCodeComposerMode_RomajiHiragana:
            value = entry.output[i];
            break;

        case KeyCodeComposerMode_RomajiKatakana:
            value = ConvertHiraganaToKatakana(entry.output[i]);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const int residueCount = composer.nativeCount - entry.nativeCount;

    composer.outputCount += residueCount + entry.outputCount;

    composer.nativeCount = 0;

    return true;
}

//!< "ん"でローマ字変換を解決します。
bool DispatchRomajiNn(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->nativeCount, 1, KeyCodeCompositionCountMax);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->outputCount, 0, KeyCodeCompositionCountMax - 1);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(*pComposer),
        1, KeyCodeCompositionCountMax);

    KeyCodeComposerStorage& composer = *pComposer;

    uint16_t& head = composer.queue[composer.outputCount];
    uint16_t& tail = composer.queue[composer.outputCount + 1];

    if (composer.nativeCount != 2 ||
        head != 'n' ||
        tail == 'a' || tail == 'e' || tail == 'i' || tail == 'o' ||
        tail == 'u' || tail == 'y')
    {
        return false;
    }
    else
    {
        switch (GetKeyCodeComposerMode(composer))
        {
        case KeyCodeComposerMode_RomajiHiragana:
            head = RomajiNNMap[0];
            break;

        case KeyCodeComposerMode_RomajiKatakana:
            head = RomajiNNMap[1];
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        composer.outputCount += 1;
        composer.nativeCount -= 1;

        return true;
    }
}

//!< 未確定部分 2 文字がローマ字変換の処理中か否かを表す値を返します。
bool AreTwoRomajiInProgress(const KeyCodeComposerStorage& composer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(composer.nativeCount, 2);
    NN_SDK_REQUIRES_MINMAX(
        composer.outputCount, 0, KeyCodeCompositionCountMax - 2);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(composer),
        2, KeyCodeCompositionCountMax);

    const uint16_t& head = composer.queue[composer.outputCount];
    const uint16_t& tail = *(&head + 1);

    if (head == tail)
    {
        return true;
    }

    if (head != 'w' && tail == 'y')
    {
        return true;
    }

    if (head == 't' && tail == 's')
    {
        return true;
    }

    if ((head == 'c' || head == 's' || head == 'd' || head == 't' ||
         head == 'w') && tail == 'h')
    {
        return true;
    }

    if ((head == 'l' || head == 'x') && (tail == 't' || tail == 'w'))
    {
        return true;
    }

    return false;
}

//!< 未確定部分 3 文字がローマ字変換の処理中か否かを表す値を返します。
bool AreThreeRomajiInProgress(
    const KeyCodeComposerStorage& composer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(composer.nativeCount, 3);
    NN_SDK_REQUIRES_MINMAX(
        composer.outputCount, 0, KeyCodeCompositionCountMax - 3);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(composer),
        3, KeyCodeCompositionCountMax);

    const uint16_t& head = composer.queue[composer.outputCount];
    const uint16_t& body = *(&head + 1);
    const uint16_t& tail = *(&head + 2);

    if (head == 'l' || head == 'x')
    {
        if (body == 't' && tail == 's')
        {
            return true;
        }

        if (body == 'w' && tail == 'h')
        {
            return true;
        }
    }

    if (head == body)
    {
        if (body != 'w' && body != 'y' && tail == 'y')
        {
            return true;
        }

        if (body == 't' && tail == 's')
        {
            return true;
        }

        if ((body == 'c' || body == 's' || body == 'd' || body == 't' ||
             body == 'w') && (tail == 'h'))
        {
            return true;
        }

        if ((body == 'l' || body == 'x') && (tail == 't' || tail == 'w'))
        {
            return true;
        }
    }

    return false;
}

void ProcessRomaji(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->nativeCount, 1, KeyCodeCompositionCountMax);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->outputCount, 0, KeyCodeCompositionCountMax - 1);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(*pComposer),
        1, KeyCodeCompositionCountMax);

    KeyCodeComposerStorage& composer = *pComposer;

    if (DispatchRomajiNonLetter(&composer))
    {
        return;
    }

    if (DispatchRomajiMap(&composer))
    {
        return;
    }

    if (DispatchRomajiNn(&composer))
    {
        return;
    }

    if (composer.nativeCount == 4)
    {
        composer.outputCount += 1;
        composer.nativeCount -= 1;
    }

    if (composer.nativeCount == 3 && !AreThreeRomajiInProgress(composer))
    {
        composer.outputCount += 1;
        composer.nativeCount -= 1;
    }

    if (composer.nativeCount == 2 && !AreTwoRomajiInProgress(composer))
    {
        composer.outputCount += 1;
        composer.nativeCount -= 1;
    }
}

void ProcessRomajiForcibly(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pComposer);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->nativeCount, 0, KeyCodeCompositionCountMax);
    NN_SDK_REQUIRES_MINMAX(
        pComposer->outputCount, 0, KeyCodeCompositionCountMax);
    NN_SDK_REQUIRES_MINMAX(
        GetKeyCodeComposerCompositionCount(*pComposer),
        0, KeyCodeCompositionCountMax);

    KeyCodeComposerStorage& composer = *pComposer;

    if (0 < composer.nativeCount)
    {
        uint16_t& value = composer.queue[
            GetKeyCodeComposerCompositionCount(composer) - 1];

        if (value == 'n')
        {
            switch (GetKeyCodeComposerMode(composer))
            {
            case KeyCodeComposerMode_RomajiHiragana:
                value = RomajiNNMap[0];
                break;

            case KeyCodeComposerMode_RomajiKatakana:
                value = RomajiNNMap[1];
                break;

            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    composer.outputCount += composer.nativeCount;
    composer.nativeCount = 0;
}

} // namespace

}}} // namespace nn::kpr::detail
