﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/irsensor/irsensor_IIrSensorServer.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorApiPrivate.h>
#include <nn/irsensor/irsensor_MarkerPositionReader.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include "detail/irsensor_IrCameraHandle.h"

#include <nn/xcd/xcd_IrsensorTypes.h>

#include <../source/OteteLib/src/OteteLib/OteteTypesSerialization.h>
#include <../source/OteteLib/src/OteteLib/BatmanPenUtils.h>
#include <../source/OteteLib/src/OteteLib/BatmanPenApi.h>
#include <../source/OteteLib/src/OteteLib/BatmanPenPrivateApi.h>

float const nerd_gillette_internal::math_constants<float>::PI = 3.1415f;

namespace nn { namespace irsensor {

batmanpen::BatmanTouchDetectionState g_BatmanTouchDetectionState[::nn::hid::system::IrSensorSupportedNpadIdsCount] = { };

::nn::Result RunMarkerPositionReader(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = detail::GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES(playerNumber >= 0 && playerNumber < ::nn::hid::system::IrSensorSupportedNpadIdsCount);

    const int MarkerPositionReaderMode = MarkerPositionReader - int(nn::xcd::IrProcessorType::TeraPlugin);
    nn::irsensor::TeraPluginProcessorConfig batmanConfig {};
    batmanConfig.modeOffset = MarkerPositionReaderMode;

    NN_RESULT_DO(nn::irsensor::RunTeraPluginProcessor(handle, batmanConfig));

    batmanpen::InitTouchDetectionState(g_BatmanTouchDetectionState[playerNumber]);

    NN_RESULT_SUCCESS;
}

::nn::Result GetMarkerPositionReaderStates(MarkerPositionReaderState* pOutValueArray, int *pReturnCount, int countMax, const IrCameraHandle& handle) NN_NOEXCEPT
{
    using namespace nerd;
    using namespace nerd::otete;
    using namespace nerd::otete::batman;

    NN_SDK_REQUIRES_NOT_NULL(pOutValueArray);

    auto playerNumber = detail::GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES(playerNumber >= 0 && playerNumber < ::nn::hid::system::IrSensorSupportedNpadIdsCount);

    // offset between camera and pentip
    nn::util::Float3 const PenTipOffset = {
        { { 0.0f, -15.0f, 10.0f } }
    };

    // derive position / rotation from packet data
    auto MarkerPacketToPenState = [&](batmanapi_private::BatmanPenState& out, batman::GothamAvenuePacket const& in)
    {
        out.timestamp = in.samplingNumber * in.samplingPeriod * 100; // microseconds
        out.rotation = float2(0);
        out.position = float2(0);

        bool isMarkerDetected = false;

        if (in.cameraMatrixFound)
        {
            size_t decodedBlockIdx = 0;
            out.patternFromCamera.Fill(0.f);
            if (gotham_avenue::DecodePositionAnd6dofFromPacket(out.patternFromCamera, out.position, decodedBlockIdx, in))
            {
                isMarkerDetected = true;
            }
        }

        return isMarkerDetected;
    };

    auto MatrixTranspose = [](nerd::float4x3 const& in)
    {
        batmanpen::FloatRowMajor4x3 out;
        for (size_t x = 0; x < 4; ++x)
            for (size_t y = 0; y < 3; ++y)
                out.m[x][y] = in(x, y);
        return out;
    };

    auto ScalePatternFromCameraMatrix = [](nerd::float4x3& M)
    {
        float const millimetersFromPixels = 5.f / 9.f;
        float const pixelsFromMillimeters = 1.f / millimetersFromPixels;
        nerd::float3x3 R;
        M.GetSubBlock(R, 0, 0);
        M.SetSubBlock(0, 0, R * pixelsFromMillimeters);
    };

    auto ComputePentipPositionFromCameraMatrix = [&](nn::util::Float2& pentipPosition, auto const& cameraMatrix)
    {
        pentipPosition.x = 0.f;
        pentipPosition.y = 0.f;

        using namespace nerd;
        float const pixelsFromMillimeters = 9.f / 5.f;
        float3 const tipInCam = pixelsFromMillimeters * float3(PenTipOffset.x, PenTipOffset.y, PenTipOffset.z);
        float3 const tipInPattern = cameraMatrix * tipInCam.xyz1();
        float3 const d = cameraMatrix.Column(2);
        float3 const n = float3(0.f, 0.f, 1.f);
        float const dotdn = dot(d, n);
        if (sqr(dotdn) < sqr(1e-6f))
            return;
        float const scale = -dot(tipInPattern, n) / dotdn;
        float3 const intersection = tipInPattern + d * scale;
        pentipPosition.x = intersection.y();
        pentipPosition.y = intersection.x();
    };

    unsigned int prevPacketId = 0;

    int count = 0;
    TeraPluginProcessorState states[TeraPluginProcessorStateCountMax];
    GetTeraPluginProcessorStates(states, &count, TeraPluginProcessorStateCountMax, 0, 0, 0, handle);
    if (count > countMax)
        count = countMax;

    *pReturnCount = 0;

    // parse packet
    for (int i = 0; i < count; ++i)
    {
        auto const& data = states[i].data;
        auto const size = TeraPluginProcessorStateDataSize;

        batman::GothamAvenuePacket packet{};
        InputBitStream inStream(data, size);
        BitStreamReader<InputBitStream> reader(inStream);
        bool ok = Serialize(packet, reader);

        if (!ok)
        {
            return ResultIrsensorNotReady();
        }

        if (packet.id >= prevPacketId)
        {
            prevPacketId = packet.id;
            batmanapi_private::BatmanPenState batmanPenState;
            ok = MarkerPacketToPenState(batmanPenState, packet);

            if (ok)
            {
              pOutValueArray[*pReturnCount].isMarkerDetected = true;
              pOutValueArray[*pReturnCount].samplingNumber = batmanPenState.timestamp;

              nerd::float4x3 patternFromCamera = batmanPenState.patternFromCamera;

              ComputePentipPositionFromCameraMatrix(pOutValueArray[*pReturnCount].pentipPosition, patternFromCamera);

              nn::util::MatrixIdentity(&pOutValueArray[*pReturnCount].cameraMatrix);
              memcpy(&pOutValueArray[*pReturnCount].cameraMatrix, &patternFromCamera, sizeof(nerd::float4x3) );

              ScalePatternFromCameraMatrix(patternFromCamera);

              batmanpen::BatmanTouchDetectionParams batmanTouchDetectionParams{};
              pOutValueArray[*pReturnCount].touchFactor = 1.0f - batmanpen::DetectTouch(g_BatmanTouchDetectionState[playerNumber], batmanTouchDetectionParams, MatrixTranspose(patternFromCamera));

              // increment packet count
              *pReturnCount = *pReturnCount + 1;

              if (countMax <= *pReturnCount)
                  break;
            }
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

} }


