﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <atomic>
#include <functional>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace irsensor { namespace detail {


// NOTE(mdelorme): The following code often compare a cursor position with another one to see which one
// is behind the other one.
//
// Be carefule for uint A, B: "A < B" is different from "int(A - B) < 0"
// Example with uint8: If A = 254 and B = 1, A < B == false, but int(A - B) == int(253) == -3 so A - B < 0 == true
//
// What you trying to do with int(A - B) is: if distance from A to B is bigger to UINT_MAX / 2 we consider that
// actually B is behind A.

template<typename ItemType, int Size>
class RingLifo
{
    NN_STATIC_ASSERT(Size >= 1);
    NN_DISALLOW_COPY(RingLifo);
    NN_DISALLOW_MOVE(RingLifo);

public:
    static const int Capacity = Size + 1;

protected:
    struct StartCount
    {
        uint64_t start;
        int      count;
    };

protected:
    ::std::atomic<uint64_t> m_NextWritePosition;
    ::std::atomic<int>      m_Count;

    ItemType m_Data[Capacity];
public:
    RingLifo() {
        Clear();
    }

    void Clear() NN_NOEXCEPT
    {
        m_Count.store(0);
        m_NextWritePosition.store(0);
    }

    /**
     *   @brief Append an object to the LIFO
     *
     *   @detail Append is not thread-safe, one thread only should ever call this method.
     */
    void Append(ItemType const& v) NN_NOEXCEPT
    {
        const uint64_t nextWritePos = m_NextWritePosition;
        m_Data[nextWritePos % Capacity] = v;
        m_NextWritePosition.store(nextWritePos + 1);
        if (m_Count < Capacity)
        {
            // NOTE: Has to be done after m_NextWritePosition has been updated.
            ++m_Count;
        }
    }


    /**
    *   @brief Read element of the ring buffer. Last in element is read first.
    *
    *   @detail Read at most "maxCount".
    */
    int Read(ItemType* pResult, const int maxCount, const uint32_t guardband = 0) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pResult);
        NN_SDK_ASSERT(maxCount + guardband <= Capacity);

        while(NN_STATIC_CONDITION(true))
        {
            StartCount sc = StartRead(m_NextWritePosition - maxCount, guardband);
            sc.count = std::min(maxCount, sc.count);
            CopyRangeLastInFirst(pResult, sc);
            if (EndRead(sc))
            {
                return sc.count;
            }
        }
    }

    /**
    *   @brief Read element of the ring buffer. Last in element is read first. But skip it if filter return false for the element
    *
    *   @detail Read at most "maxCount".
    */
    int ReadFiltered(ItemType* pResult, const int maxCount, std::function<bool(const ItemType&)> filter,  const uint32_t guardband = 0) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pResult);
        NN_SDK_ASSERT(maxCount + guardband <= Capacity);

        while (NN_STATIC_CONDITION(true))
        {
            // NOTE(mdelorme): We want to read all the ring buffer. Because the
            // position given to StartRead is for sure lower than lowest readable
            // position we are sure the StartCount return will be for the full
            // ring buffer.
            const uint64_t nextWritePos = m_NextWritePosition;
            const uint64_t maxReadPosition = nextWritePos - Capacity;
            const StartCount sc = StartRead(maxReadPosition, guardband);

            if (sc.count == 0)
            {
                return 0;
            }

            const uint64_t positionEnd = sc.start + static_cast<uint64_t>(sc.count) - 1;
            int count = 0;
            for (uint64_t position = positionEnd; static_cast<int64_t>(position - sc.start) >= 0; position--)
            {
                if (count >= maxCount)
                {
                    break;
                }

                const uint64_t inIndex = position % Capacity;
                NN_SDK_ASSERT(inIndex < Capacity);


                ItemType itemCopied = m_Data[inIndex];
                if (!EndRead(sc))
                {
                    break;
                }

                if (filter(itemCopied))
                {
                    const uint64_t outIndex = count;
                    NN_SDK_ASSERT(outIndex < maxCount);

                    pResult[outIndex] = m_Data[inIndex];
                    count += 1;
                }
            }

            if (EndRead(sc))
            {
                return count;
            }
        }
    }

protected:
    /**
     *   @brief Return a StartCount object according to the cursor the want to start reading.
     *
     *   @detail You want to read from a position, but maybe the position is being written
     *           or to close to be written. (according to guardBand value). StartRead return
     *           the close cursor you can read from the one you want to read, and return the
     *           number of element you can read.
     *
     *           The StartCount::cursor of returned startCount will be equal to wantedStartCursor
     *           or it will be higher.
     */
    StartCount StartRead(const uint64_t wantedStartCursor, const uint32_t guardBand) const
    {
        NN_SDK_ASSERT(guardBand + 1 < Capacity);

        const uint64_t nextWritePos = m_NextWritePosition;
        // NOTE(mdelorme): This this the lowest position you can read
        // without risking it to be overwritten by the writter during the read.
        const uint64_t lowestPositionReadableWithGuard = nextWritePos - Capacity + 1 + guardBand;
        // NOTE(mdelorme): This is the lowest position available according to the number of object in place.
        const uint64_t lowestPositionAccordingToCount = nextWritePos - m_Count;
        // NOTE(mdelorme): We want to choose the most recent position of the two previous.
        const uint64_t lowestPositionReadable =
            static_cast<int64_t>(lowestPositionAccordingToCount - lowestPositionReadableWithGuard) > 0
            ? lowestPositionAccordingToCount
            : lowestPositionReadableWithGuard;

        // NOTE(mdelorme): The start cursor is lowest between what is possible and what we want.
        const uint64_t startCursor = static_cast<int32_t>(wantedStartCursor - lowestPositionReadable) > 0 ? wantedStartCursor : lowestPositionReadable;
        const int64_t wantedCount  = nextWritePos - startCursor;
        const int64_t finalCount   = wantedCount > 0 ? wantedCount : 0;

        NN_SDK_ASSERT((finalCount & 0xffffffff00000000) == 0, "finalCount has to fit inside an int");

        StartCount startCount;
        startCount.start = startCursor;
        startCount.count = static_cast<int>(finalCount);

        return startCount;
    }

    /**
     *   @brief Tell if a StartCount have been read safely.
     *
     *   @detail StartCount describe a range of element you can read. EndRead check
     *   if this range has been overwritten. If it return true, what you read was
     *   valid.
     */
    bool EndRead(const StartCount& startCount) const NN_NOEXCEPT
    {
        // NOTE(mdelorme): If the startCount.Count is zero, we did not try to read
        // anything. The the reading cannot have failed.

        if (startCount.count == 0)
        {
            return true;
        }

        const uint64_t nextWritePos = m_NextWritePosition;
        const uint64_t count        = m_Count;

        // NOTE(mdelorme): If the current cursor is before the starting, maybe the
        // data structure has been reset. Anyway, the read has failed.
        if (int64_t(nextWritePos - startCount.start) <= 0)
        {
            return false;
        }

        // NOTE(mdelorme): If the current count is lower the starting count, maybe the
        // data structure has been reset. Anyway, the read has failed.
        if (count < startCount.count)
        {
            return false;
        }

        // NOTE(mdelorme): Highest position which may have been overwritten. We are
        // checking every which has been read is after this position.
        uint64_t const highestPositionPossiblyWrittenValue = nextWritePos - Capacity;
        return static_cast<int64_t>(startCount.start - highestPositionPossiblyWrittenValue) > 0;
    }

protected:
    void CopyRangeFirstInFirst(ItemType* pOutValueArray, const StartCount& startCount) const NN_NOEXCEPT
    {
        if (startCount.Count == 0)
        {
            return;
        }

        NN_SDK_ASSERT(startCount.Count > 0);

        const uint64_t moduloStart = startCount.Start % Capacity;
        const uint64_t moduloEnd   = ((startCount.Start + startCount.Count - 1) % Capacity) + 1;

        if (moduloEnd > moduloStart)
        {
            memcpy(pOutValueArray, m_Data + moduloStart, (moduloEnd - moduloStart) * sizeof(ItemType));
        }
        else if (moduloEnd < moduloStart)
        {
            NN_SDK_ASSERT(Capacity > moduloStart);
            NN_SDK_ASSERT(moduloEnd > 0);
            memcpy(pOutValueArray, m_Data + moduloStart, (Capacity - moduloStart) * sizeof(ItemType));
            memcpy(pOutValueArray + Capacity - moduloStart, m_Data, moduloEnd * sizeof(ItemType));
        }
    }

    void CopyRangeLastInFirst(ItemType* pOutValueArray, const StartCount& startCount) const NN_NOEXCEPT
    {
        if (startCount.count == 0)
        {
            return;
        }

        NN_SDK_ASSERT(startCount.count > 0);

        const int64_t moduloStart = static_cast<int64_t>(startCount.start % Capacity);
        const int64_t moduloEnd   = static_cast<int64_t>(((startCount.start + startCount.count - 1) % Capacity) + 1);

        if (moduloEnd > moduloStart)
        {
            for (int64_t inIndex = moduloEnd - 1; inIndex >= moduloStart; inIndex--)
            {
                const int64_t outIndex = moduloEnd - 1 - inIndex;
                NN_SDK_ASSERT(outIndex >= 0);
                NN_SDK_ASSERT(inIndex < Capacity);

                pOutValueArray[outIndex] = m_Data[inIndex];
            }
        }
        else if (moduloEnd < moduloStart)
        {
            NN_SDK_ASSERT(Capacity > moduloStart);
            NN_SDK_ASSERT(moduloEnd > 0);

            for (int64_t inIndex = moduloEnd - 1; inIndex >= 0; inIndex--)
            {
                const int64_t outIndex = moduloEnd - 1 - inIndex;
                NN_SDK_ASSERT(outIndex >= 0);
                NN_SDK_ASSERT(inIndex < Capacity);

                pOutValueArray[outIndex] = m_Data[inIndex];
            }
            for (int64_t inIndex = Capacity - 1; inIndex >= moduloStart; inIndex--)
            {
                NN_SDK_ASSERT(inIndex < Capacity);

                const int64_t outIndex = moduloEnd + Capacity - 1 - inIndex;
                pOutValueArray[outIndex] = m_Data[inIndex];
            }
        }
    }
};


}}}
