﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/util/util_ScopeExit.h>

#include "irsensor_ResourceManager.h"
#include "irsensor_IrCameraHandle.h"
#include "irsensor_RequestHandlerTask.h"

namespace nn { namespace irsensor { namespace detail {

namespace {

const auto RetryInterval = ::nn::TimeSpan::FromMilliSeconds(100);
const auto McuErrorCountMax = 3;
const auto McuVersionPollingCountMax = 300;

} // namespace

RequestHandlerTask::RequestHandlerTask() NN_NOEXCEPT
    : m_ManagerMutex(false)
    , m_TimerEvent()
    , m_pManager(nullptr)
    , m_IsManagerInitialized(false)
    , m_RequestMutex(false)
    , m_RequestMode(IrSensorMode::None)
    , m_CurrentMode(IrSensorMode::None)
    , m_RunRequestEvent()
    , m_ResumeRequestEvent()
    , m_StopRequestEvent()
    , m_SuspendRequestEvent()
    , m_McuErrorCounter(0)
    , m_IsMcuVersionRequested(false)
    , m_RequiredVersion()
    , m_LatestRequestType(RequestType::None)
{
    // 何もしない
}

RequestHandlerTask::~RequestHandlerTask() NN_NOEXCEPT
{
    // 何もしない
}

void RequestHandlerTask::SetDriverManager(DriverManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimerEvent(m_TimerEvent.GetBase());
    m_pManager = pManager;
}

::nn::Result RequestHandlerTask::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    {
        ::std::lock_guard<decltype(m_ManagerMutex)> locker(m_ManagerMutex);
        NN_RESULT_DO(m_pManager->Activate());
        m_IsManagerInitialized = true;
    }
    // 初期状態を停止状態としておく。
    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_SuccessfulTerminated);

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::Activate(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedFunctionLevel& functionLevel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    {
        ::std::lock_guard<decltype(m_ManagerMutex)> locker(m_ManagerMutex);
        NN_RESULT_DO(m_pManager->Activate());
        m_IsManagerInitialized = true;
        m_IsMcuVersionRequested = false;
    }
    // FunctionLevel を設定
    SetFunctionLevel(aruid, functionLevel);

    // 初期状態を停止状態としておく。
    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_SuccessfulTerminated);

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::Deactivate(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);

    // 各種リクエストが残っている場合は、処理が行われるのを待つ
    // この間、Finalize の IPC をブロックする。
    // 333フレーム x 15ms の 5 秒間待っても停止しない場合は上位にエラーを返してアボートする。
    // 3.x 以前のシステムで呼び出された Finalize に対しては、StopImageProcessor が完了同期するため、
    // 必ずこの時点でデバイスは終了されている。 そのためブロックしない。
    NN_RESULT_DO(WaitForProcessorStopped(333));

    // 初期状態の停止状態に戻しておく。
    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_SuccessfulTerminated);
    m_IsMcuVersionRequested = false;
    // FunctionLevel を初期値に戻す
    PackedFunctionLevel functionLevel = {};
    functionLevel.level = ::nn::irsensor::IrSensorFunctionLevel_0;
    SetFunctionLevel(aruid, functionLevel);
    {
        ::std::lock_guard<decltype(m_ManagerMutex)> locker(m_ManagerMutex);
        NN_RESULT_DO(m_pManager->Deactivate());
        m_IsManagerInitialized = false;
        m_LatestRequestType = RequestType::None;
    }

    NN_RESULT_DO(this->DeactivateThread());
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestFirmwareVersion(
    const PackedMcuVersion& requiredVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);

    ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);

    if (!m_IsMcuVersionRequested)
    {
        // リクエストフラグを立てる
        m_IsMcuVersionRequested = true;

        // 要求バージョンを登録
        m_RequiredVersion = requiredVersion;

        // Request を出してすぐ返す
        NN_RESULT_DO(m_pManager->RequestFirmwareVersion());

        // 内部状態をバージョンチェック中に変更
        m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_CheckVersionIncompleted);
    }
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::CheckFirmwareVersion() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);

    // Mcu のバージョンチェックを行い、結果を保存します。 リトライが必要な場合はエラーを返します。
    nn::Result result = m_pManager->CheckFirmwareVersion(m_RequiredVersion);
    if (::nn::irsensor::ResultIrsensorNeedUpdate::Includes(result))
    {
        // このイベントは、CheckFirmwareUpdateNecessity 発行時に、Teraを起こしてバージョンチェックする場合のみ
        m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_NeedUpdateOnInitial);
    }
    else if (::nn::irsensor::ResultIrsensorBusy::Includes(result)
        || ::nn::irsensor::ResultIrsensorMcuNotAvailable::Includes(result))
    {
        if (m_McuErrorCounter > McuVersionPollingCountMax)
        {
            // CheckFirmwareUpdateNecessity 関数呼び出し直後にコントローラ切断した際、または何等かの要因で 4.5秒間 バージョンが取得できなかった場合は、
            // FailSafe のため、更新必要なしとして成功を返すようにしておく。
            m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_SuccessfulTerminated);
        }
        else
        {
            m_McuErrorCounter++;
            NN_RESULT_THROW(result);
        }
    }
    else if (::nn::irsensor::ResultIrsensorAppletResourceNotAvailable::Includes(result))
    {
        // BG状態の場合は、 IrCameraInternalStatus_CheckVersionIncompleted の状態をキープしてリトライします。
        m_McuErrorCounter = 0;
        NN_RESULT_THROW(result);
    }
    else if (result.IsSuccess())
    {
        // 成功時は、元の停止状態に戻す
        m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_SuccessfulTerminated);
    }
    else
    {
        NN_ABORT("[irsensor] FirmwareVersionCheck FatalError");
    }

    // RequiredVersion のクリア
    m_RequiredVersion.major = 0;
    m_RequiredVersion.minor = 0;
    m_McuErrorCounter = 0;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::SetFunctionLevel(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedFunctionLevel& functionLevel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);

    NN_RESULT_DO(m_pManager->SetFunctionLevel(aruid, functionLevel));
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::StopImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    // 互換性維持のため、IAAA-4020 の不具合を維持した版。
    // NXAddon 3.x 系以前のSDKからはこちらが呼び出される。
    // プロセス内で WAIT するため、その間他 hid の IPC をブロックする。
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    {
        // ロックは Request->Manager の順
        ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);

        if (m_RequestMode == IrSensorMode::ImageTransfer)
        {
            ReleaseTransferMemory(aruid);
        }

        m_RequestMode = IrSensorMode::None;
        m_LatestRequestType = RequestType::Stop;
        m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_Available);
        {
            ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
            NN_RESULT_DO(m_pManager->StopImageProcessorWithCompatibility());
            m_CurrentMode = IrSensorMode::None;
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::StopImageProcessorAsync(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);
    // ImageTransfer 起動中でトランスファーメモリを Attach済みの場合は開放しておく
    void* address = nullptr;
    m_pManager->GetTransferMemoryWorkBufferAddressWithAruid(&address, aruid);
    if (address != nullptr)
    {
        // サンプリングなどを停止する
        m_pManager->StopSampling(aruid);
        m_pManager->SetImageValidity(aruid, false);
        ReleaseTransferMemory(aruid);
    }
    m_StopRequestEvent.Signal();
    m_RequestMode = IrSensorMode::None;
    m_LatestRequestType = RequestType::Stop;
    NN_RESULT_SUCCESS;
}

bool RequestHandlerTask::IsProcessorStopped() NN_NOEXCEPT
{
    // イベントが全て呼び出し済みかつ、 CurrentMode が None の状態
    bool isStopped = !m_StopRequestEvent.GetBase()->_signalState
        && !m_SuspendRequestEvent.GetBase()->_signalState
        && !m_RunRequestEvent.GetBase()->_signalState
        && !m_ResumeRequestEvent.GetBase()->_signalState
        && m_CurrentMode == IrSensorMode::None;
    return isStopped;
}

::nn::Result RequestHandlerTask::WaitForProcessorStopped(int waitPacketCount) NN_NOEXCEPT
{
    auto counter = 0;
    const auto RetryIntervalFrame = ::nn::TimeSpan::FromMilliSeconds(15);
    while (!IsProcessorStopped())
    {
        // 負数だった場合は、無限ループする
        if (waitPacketCount >= 0)
        {
            counter++;
            // 指定した時間待っても停止しない場合はタイムアウトする。
            if (counter > waitPacketCount)
            {
                break;
            }
        }
        ::nn::os::SleepThread(RetryIntervalFrame);
    }
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::SuspendImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);

    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    // ImageTransfer 起動中でトランスファーメモリを Attach済みの場合は開放しておく
    void* address = nullptr;
    m_pManager->GetTransferMemoryWorkBufferAddressWithAruid(&address, aruid);
    if (address != nullptr)
    {
        // サンプリングなどを停止する
        m_pManager->StopSampling(aruid);
        m_pManager->SetImageValidity(aruid, false);
        ReleaseTransferMemory(aruid);
    }
    m_SuspendRequestEvent.Signal();
    m_RequestMode = IrSensorMode::None;
    m_LatestRequestType = RequestType::Suspend;

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::StopSampling(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    NN_RESULT_DO(m_pManager->StopSampling(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RunMomentProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedMomentProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);

    ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);
    // Run 直後には、Setting 状態にする
    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_Setting);

    // 直前が Run 状態だったため、Suspend された場合、または同じモードへの遷移の場合は Resume を発行する
    if (m_LatestRequestType == RequestType::Suspend || m_RequestMode == IrSensorMode::Moment)
    {
        if (m_RequestMode == IrSensorMode::Moment)
        {
            // 同じモードへの遷移の場合はサンプリングを一旦停止する
            NN_RESULT_DO(StopSampling(aruid));
        }
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestResumeImageProcessor(aruid, config));
    }
    // 直前が Stop 状態だった場合は、従来どおり Run を発行する
    else
    {
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestRunImageProcessor(aruid, config));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RunClusteringProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedClusteringProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);
    // Run 直後には、Setting 状態にする
    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_Setting);

    // 直前が Run 状態だったため、Suspend された場合、または同じモードへの遷移の場合は Resume を発行する
    if (m_LatestRequestType == RequestType::Suspend || m_RequestMode == IrSensorMode::Clustering)
    {
        if (m_RequestMode == IrSensorMode::Clustering)
        {
            // 同じモードへの遷移の場合はサンプリングを一旦停止する
            NN_RESULT_DO(StopSampling(aruid));
        }
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestResumeImageProcessor(aruid, config));
    }
    // 直前が Stop 状態だった場合は、従来どおり Run を発行する
    else
    {
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestRunImageProcessor(aruid, config));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RunImageTransferProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedImageTransferProcessorExConfig& config,
    ::nn::os::NativeHandle&& transferMemoryHandle,
    size_t size,
    bool isManaged
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);
    // Run 直後には、Setting 状態にする
    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_Setting);

    // 直前が Run 状態だったため、Suspend された場合、または同じモードへの遷移の場合は Resume を発行する
    if (m_LatestRequestType == RequestType::Suspend || m_RequestMode == IrSensorMode::ImageTransfer)
    {
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestResumeImageProcessor(
            aruid, config, ::std::move(transferMemoryHandle), size, isManaged, true));
    }
    // 直前が Stop 状態だった場合は、従来どおり Run を発行する
    else
    {
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestRunImageProcessor(
            aruid, config, ::std::move(transferMemoryHandle), size, isManaged));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RunPointingProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedPointingProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);
    // Run 直後には、Setting 状態にする
    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_Setting);

    // 直前が Run 状態だったため、Suspend された場合、または同じモードへの遷移の場合は Resume を発行する
    if (m_LatestRequestType == RequestType::Suspend || m_RequestMode == IrSensorMode::Pointing)
    {
        if (m_RequestMode == IrSensorMode::Pointing)
        {
            // 同じモードへの遷移の場合はサンプリングを一旦停止する
            NN_RESULT_DO(StopSampling(aruid));
        }
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestResumeImageProcessor(aruid, config));
    }
    // 直前が Stop 状態だった場合は、従来どおり Run を発行する
    else
    {
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestRunImageProcessor(aruid, config));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RunTeraPluginProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedTeraPluginProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    // Run 直後には、Setting 状態にする
    ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);
    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_Setting);

    // 直前が Run 状態だったため、Suspend された場合は Resume を発行する
    if (m_LatestRequestType == RequestType::Suspend)
    {
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestResumeImageProcessor(aruid, config));
    }
    // 直前が Stop 状態だった場合は、従来どおり Run を発行する
    else
    {
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestRunImageProcessor(aruid, config));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RunIrLedProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedIrLedProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);

    ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);
    if (m_LatestRequestType == RequestType::Suspend || m_RequestMode == IrSensorMode::IrLed)
    {
        if (m_RequestMode == IrSensorMode::IrLed)
        {
            // 同じモードへの遷移の場合はサンプリングを一旦停止する
            NN_RESULT_DO(StopSampling(aruid));
        }
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestResumeImageProcessor(aruid, config));
    }
    // 直前が Stop 状態だった場合は、従来どおり Run を発行する
    else
    {
        // リクエストだけを発行しておきユーザにはすぐ制御を返す
        NN_RESULT_DO(RequestRunImageProcessor(aruid, config));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestRunImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedMomentProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        MomentProcessorExposureTimeMin <= config.irCameraConfig.exposureTime
        && config.irCameraConfig.exposureTime <= MomentProcessorExposureTimeMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        IrCameraGainMin <= config.irCameraConfig.gain
        && config.irCameraConfig.gain <= IrCameraGainMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.preprocess == MomentProcessorPreprocess_Binarize
        || config.preprocess == MomentProcessorPreprocess_Cutoff,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.preprocessIntensityThreshold
        && config.preprocessIntensityThreshold <= ::nn::irsensor::IrCameraIntensityMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.x,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        MomentProcessorBlockColumnCount <= config.windowOfInterest.width,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.x + config.windowOfInterest.width <= IrCameraImageWidth,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.y,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        MomentProcessorBlockRowCount <= config.windowOfInterest.height,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.y + config.windowOfInterest.height <= IrCameraImageHeight,
        ResultIrsensorInvalidConfig());

    IrSensorConfig irSensorConfig;
    irSensorConfig.moment = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::Moment;
    m_RunRequestEvent.Signal();
    m_LatestRequestType = RequestType::Run;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestResumeImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedMomentProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        MomentProcessorExposureTimeMin <= config.irCameraConfig.exposureTime
        && config.irCameraConfig.exposureTime <= MomentProcessorExposureTimeMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        IrCameraGainMin <= config.irCameraConfig.gain
        && config.irCameraConfig.gain <= IrCameraGainMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.preprocess == MomentProcessorPreprocess_Binarize
        || config.preprocess == MomentProcessorPreprocess_Cutoff,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.preprocessIntensityThreshold
        && config.preprocessIntensityThreshold <= ::nn::irsensor::IrCameraIntensityMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.x,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        MomentProcessorBlockColumnCount <= config.windowOfInterest.width,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.x + config.windowOfInterest.width <= IrCameraImageWidth,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.y,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        MomentProcessorBlockRowCount <= config.windowOfInterest.height,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.y + config.windowOfInterest.height <= IrCameraImageHeight,
        ResultIrsensorInvalidConfig());

    IrSensorConfig irSensorConfig;
    irSensorConfig.moment = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::Moment;
    m_ResumeRequestEvent.Signal();
    m_LatestRequestType = RequestType::Resume;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestRunImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedClusteringProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        ClusteringProcessorExposureTimeMin <= config.irCameraConfig.exposureTime
        && config.irCameraConfig.exposureTime <= ClusteringProcessorExposureTimeMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        IrCameraGainMin <= config.irCameraConfig.gain
        && config.irCameraConfig.gain <= IrCameraGainMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.objectIntensityMin && config.objectIntensityMin <= ::nn::irsensor::IrCameraIntensityMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.objectPixelCountMin
        && config.objectPixelCountMin <= ClusteringProcessorObjectPixelCountMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.objectPixelCountMax
        && config.objectPixelCountMax <= ClusteringProcessorObjectPixelCountMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.x,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        1 <= config.windowOfInterest.width,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.x + config.windowOfInterest.width <= IrCameraImageWidth,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.y,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        1 <= config.windowOfInterest.height,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.y + config.windowOfInterest.height <= IrCameraImageHeight,
        ResultIrsensorInvalidConfig());

    IrSensorConfig irSensorConfig;
    irSensorConfig.clustering = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::Clustering;
    m_RunRequestEvent.Signal();
    m_LatestRequestType = RequestType::Run;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestResumeImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedClusteringProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        ClusteringProcessorExposureTimeMin <= config.irCameraConfig.exposureTime
        && config.irCameraConfig.exposureTime <= ClusteringProcessorExposureTimeMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        IrCameraGainMin <= config.irCameraConfig.gain
        && config.irCameraConfig.gain <= IrCameraGainMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.objectIntensityMin && config.objectIntensityMin <= ::nn::irsensor::IrCameraIntensityMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.objectPixelCountMin
        && config.objectPixelCountMin <= ClusteringProcessorObjectPixelCountMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.objectPixelCountMax
        && config.objectPixelCountMax <= ClusteringProcessorObjectPixelCountMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.x,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        1 <= config.windowOfInterest.width,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.x + config.windowOfInterest.width <= IrCameraImageWidth,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.y,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        1 <= config.windowOfInterest.height,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.y + config.windowOfInterest.height <= IrCameraImageHeight,
        ResultIrsensorInvalidConfig());

    IrSensorConfig irSensorConfig;
    irSensorConfig.clustering = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::Clustering;
    m_ResumeRequestEvent.Signal();
    m_LatestRequestType = RequestType::Resume;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestRunImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedImageTransferProcessorExConfig& config,
    ::nn::os::NativeHandle&& transferMemoryHandle,
    size_t size,
    bool isManaged
    ) NN_NOEXCEPT
{
    auto needsRollback = true;

    // Transfer メモリの情報を取得
    ::nn::os::TransferMemoryType* pTransferMemory = nullptr;
    m_pManager->GetTransferMemoryType(&pTransferMemory, aruid);

    ::nn::os::AttachTransferMemory(
        pTransferMemory, size, transferMemoryHandle, isManaged);

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            ::nn::os::DestroyTransferMemory(pTransferMemory);
        }
    };

    NN_RESULT_THROW_UNLESS(
        ImageTransferProcessorExposureTimeMin <= config.irCameraConfig.exposureTime
        && config.irCameraConfig.exposureTime <= ImageTransferProcessorExposureTimeMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        IrCameraGainMin <= config.irCameraConfig.gain
        && config.irCameraConfig.gain <= IrCameraGainMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.trimmingFormat == ImageTransferProcessorFormat_320x240
        || config.trimmingFormat == ImageTransferProcessorFormat_160x120
        || config.trimmingFormat == ImageTransferProcessorFormat_80x60
        || config.trimmingFormat == ImageTransferProcessorFormat_40x30
        || config.trimmingFormat == ImageTransferProcessorFormat_20x15,
        ResultIrsensorInvalidConfig());

    NN_RESULT_THROW_UNLESS(config.trimmingFormat >= config.origFormat,
        ResultIrsensorInvalidConfig());

    auto origWidth = IrCameraImageWidth >> config.origFormat;
    auto origHeight = IrCameraImageHeight >> config.origFormat;
    auto destWidth = IrCameraImageWidth >> config.trimmingFormat;
    auto destHeight = IrCameraImageHeight >> config.trimmingFormat;
    NN_RESULT_THROW_UNLESS(
        0 <= config.trimmingStartX
        && config.trimmingStartX <= origWidth - destWidth,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.trimmingStartY
        && config.trimmingStartY <= origHeight - destHeight,
        ResultIrsensorInvalidConfig());

    IrSensorConfig irSensorConfig;
    irSensorConfig.imageTransfer = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);

    void* address = nullptr;
    NN_RESULT_TRY(::nn::os::MapTransferMemory(
        &address,
        pTransferMemory,
        ::nn::os::MemoryPermission_None))
        NN_RESULT_CATCH_ALL
        {
            return ResultIrsensorCreateTransferMemoryFailed();
        }
    NN_RESULT_END_TRY

    // 取得したワークバッファの先頭アドレスを設定
    m_pManager->SetTransferMemoryWorkBufferAddress(aruid, address);
    m_RequestMode = IrSensorMode::ImageTransfer;
    m_RunRequestEvent.Signal();
    m_LatestRequestType = RequestType::Run;
    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestResumeImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedImageTransferProcessorExConfig& config,
    ::nn::os::NativeHandle&& transferMemoryHandle,
    size_t size,
    bool isManaged,
    bool isTransferMemoryHandled
    ) NN_NOEXCEPT
{
    auto needsRollback = true;

    // Transfer メモリの情報を取得
    ::nn::os::TransferMemoryType* pTransferMemory = nullptr;
    m_pManager->GetTransferMemoryType(&pTransferMemory, aruid);

    if (isTransferMemoryHandled)
    {
        ::nn::os::AttachTransferMemory(
            pTransferMemory, size, transferMemoryHandle, isManaged);
    }

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback && isTransferMemoryHandled)
        {
            ::nn::os::DestroyTransferMemory(pTransferMemory);
        }
    };

    NN_RESULT_THROW_UNLESS(
        ImageTransferProcessorExposureTimeMin <= config.irCameraConfig.exposureTime
        && config.irCameraConfig.exposureTime <= ImageTransferProcessorExposureTimeMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        IrCameraGainMin <= config.irCameraConfig.gain
        && config.irCameraConfig.gain <= IrCameraGainMax,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.trimmingFormat == ImageTransferProcessorFormat_320x240
        || config.trimmingFormat == ImageTransferProcessorFormat_160x120
        || config.trimmingFormat == ImageTransferProcessorFormat_80x60
        || config.trimmingFormat == ImageTransferProcessorFormat_40x30
        || config.trimmingFormat == ImageTransferProcessorFormat_20x15,
        ResultIrsensorInvalidConfig());

    NN_RESULT_THROW_UNLESS(config.trimmingFormat >= config.origFormat,
        ResultIrsensorInvalidConfig());

    auto origWidth = IrCameraImageWidth >> config.origFormat;
    auto origHeight = IrCameraImageHeight >> config.origFormat;
    auto destWidth = IrCameraImageWidth >> config.trimmingFormat;
    auto destHeight = IrCameraImageHeight >> config.trimmingFormat;
    NN_RESULT_THROW_UNLESS(
        0 <= config.trimmingStartX
        && config.trimmingStartX <= origWidth - destWidth,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.trimmingStartY
        && config.trimmingStartY <= origHeight - destHeight,
        ResultIrsensorInvalidConfig());

    IrSensorConfig irSensorConfig;
    irSensorConfig.imageTransfer = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);

    if (isTransferMemoryHandled)
    {
        void* address = nullptr;
        NN_RESULT_TRY(::nn::os::MapTransferMemory(
            &address,
            pTransferMemory,
            ::nn::os::MemoryPermission_None))
            NN_RESULT_CATCH_ALL
            {
                return ResultIrsensorCreateTransferMemoryFailed();
            }
        NN_RESULT_END_TRY

       // 取得したワークバッファの先頭アドレスを設定
       m_pManager->SetTransferMemoryWorkBufferAddress(aruid, address);
    }
    m_RequestMode = IrSensorMode::ImageTransfer;
    m_ResumeRequestEvent.Signal();
    m_LatestRequestType = RequestType::Resume;
    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestRunImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedPointingProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.x,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        1 <= config.windowOfInterest.width,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.x + config.windowOfInterest.width <= IrCameraImageWidth,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.y,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        1 <= config.windowOfInterest.height,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.y + config.windowOfInterest.height <= IrCameraImageHeight,
        ResultIrsensorInvalidConfig());

    IrSensorConfig irSensorConfig;
    irSensorConfig.pointing = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::Pointing;
    m_RunRequestEvent.Signal();
    m_LatestRequestType = RequestType::Run;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestResumeImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedPointingProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.x,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        1 <= config.windowOfInterest.width,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.x + config.windowOfInterest.width <= IrCameraImageWidth,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        0 <= config.windowOfInterest.y,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        1 <= config.windowOfInterest.height,
        ResultIrsensorInvalidConfig());
    NN_RESULT_THROW_UNLESS(
        config.windowOfInterest.y + config.windowOfInterest.height <= IrCameraImageHeight,
        ResultIrsensorInvalidConfig());

    IrSensorConfig irSensorConfig;
    irSensorConfig.pointing = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::Pointing;
    m_ResumeRequestEvent.Signal();
    m_LatestRequestType = RequestType::Resume;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestRunImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedTeraPluginProcessorConfig& config
    ) NN_NOEXCEPT
{
    IrSensorConfig irSensorConfig;
    irSensorConfig.teraPlugin = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::TeraPlugin;
    m_RunRequestEvent.Signal();
    m_LatestRequestType = RequestType::Run;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestResumeImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedTeraPluginProcessorConfig& config
    ) NN_NOEXCEPT
{
    IrSensorConfig irSensorConfig;
    irSensorConfig.teraPlugin = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::TeraPlugin;
    m_ResumeRequestEvent.Signal();
    m_LatestRequestType = RequestType::Resume;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestRunImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedIrLedProcessorConfig& config
    ) NN_NOEXCEPT
{
    IrSensorConfig irSensorConfig;
    irSensorConfig.irLed = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::IrLed;
    m_RunRequestEvent.Signal();
    m_LatestRequestType = RequestType::Run;
    NN_RESULT_SUCCESS;
}

::nn::Result RequestHandlerTask::RequestResumeImageProcessor(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedIrLedProcessorConfig& config
    ) NN_NOEXCEPT
{
    IrSensorConfig irSensorConfig;
    irSensorConfig.irLed = config;
    m_pManager->SetIrSensorConfig(aruid, irSensorConfig);
    m_RequestMode = IrSensorMode::IrLed;
    m_ResumeRequestEvent.Signal();
    m_LatestRequestType = RequestType::Resume;
    NN_RESULT_SUCCESS;
}

void RequestHandlerTask::ReleaseTransferMemory(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);

    // ARUID テーブルの情報をクリアしておく
    m_pManager->SetTransferMemoryWorkBufferAddress(aruid, nullptr);

    // Transfer メモリの情報を取得
    ::nn::os::TransferMemoryType* pTransferMemory = nullptr;
    m_pManager->GetTransferMemoryType(&pTransferMemory, aruid);

    // Transfer メモリを破棄
    ::nn::os::UnmapTransferMemory(pTransferMemory);
    ::nn::os::DestroyTransferMemory(pTransferMemory);
}

void RequestHandlerTask::SetThreadName(
    ::nn::os::ThreadType* pThreadType) NN_NOEXCEPT
{
    ::nn::os::SetThreadName(pThreadType, NN_SYSTEM_THREAD_NAME(irsensor, RequestHandler));
}

void RequestHandlerTask::LinkEvent(
    ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    // Link順が多重待ち時の優先度となるので注意が必要
    m_TimerEvent.Link(pMultiWait);
    m_StopRequestEvent.Link(pMultiWait);
    m_SuspendRequestEvent.Link(pMultiWait);
    m_RunRequestEvent.Link(pMultiWait);
    m_ResumeRequestEvent.Link(pMultiWait);
}

void RequestHandlerTask::UnlinkEvent() NN_NOEXCEPT
{
    m_TimerEvent.Unlink();
    m_RunRequestEvent.Unlink();
    m_ResumeRequestEvent.Unlink();
    m_StopRequestEvent.Unlink();
    m_SuspendRequestEvent.Unlink();
}

::nn::Result RequestHandlerTask::GetImageTransferProcessorState(
    ImageTransferProcessorState* pOutState,
    void* pOutImage,
    const ::nn::applet::AppletResourceUserId& aruid,
    size_t size
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(pOutImage);

    // トランスファーメモリが開放されている場合はデータ取得を行わない
    void* address = nullptr;
    m_pManager->GetTransferMemoryWorkBufferAddressWithAruid(&address, aruid);
    NN_RESULT_THROW_UNLESS(address != nullptr, nn::irsensor::ResultIrsensorNotReady());

    NN_RESULT_TRY(
        m_pManager->GetImageProcessorState(pOutState, pOutImage, size))
        NN_RESULT_CATCH(nn::irsensor::ResultIrsensorAppletResourceNotAvailable)
        {
            NN_RESULT_THROW(nn::irsensor::ResultIrsensorNotReady());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

void RequestHandlerTask::HandleEvent(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);

    if (waitId == m_StopRequestEvent.GetWaitId()
        || waitId == m_SuspendRequestEvent.GetWaitId())
    {
        bool isStopRequest = false;
        if (waitId == m_StopRequestEvent.GetWaitId())
        {
            isStopRequest = true;
        }
        // クライアントからのリクエストを処理する。
        // 次の要求の取りこぼしを防ぐために最初にクリアする
        if (isStopRequest)
        {
            m_StopRequestEvent.Clear();
            m_SuspendRequestEvent.Clear();
        }
        else
        {
            m_SuspendRequestEvent.Clear();
        }

        // stop
        if (isStopRequest)
        {
            ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
            nn::Result result = m_pManager->StopImageProcessor();
            if (result.IsSuccess())
            {
                m_CurrentMode = IrSensorMode::None;
                m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_SuccessfulTerminated);
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
        else
        // suspend
        {
            ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
            nn::Result result = m_pManager->SuspendImageProcessor();
            if (result.IsSuccess())
            {
                m_CurrentMode = IrSensorMode::None;
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
    }
    else if (waitId == m_RunRequestEvent.GetWaitId()
        || waitId == m_ResumeRequestEvent.GetWaitId())
    {
        bool isRunRequest = false;
        if (waitId == m_RunRequestEvent.GetWaitId())
        {
            isRunRequest = true;
        }
        // クライアントからのリクエストを処理する。
        // 次の要求の取りこぼしを防ぐために最初にクリアする
        if (isRunRequest)
        {
            m_RunRequestEvent.Clear();
            m_ResumeRequestEvent.Clear();
        }
        else
        {
            m_ResumeRequestEvent.Clear();
        }

        IrSensorMode mode;
        IrSensorConfig config;
        ::nn::Result result = nn::ResultSuccess();

        bool isRetryRunRequest = false;
        bool isSettingEnabled = false;
        {
            ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);
            if (m_LatestRequestType != RequestType::Stop
                && m_LatestRequestType != RequestType::Suspend
                && m_LatestRequestType != RequestType::None)
            {
                isSettingEnabled = true;
            }

            if (isRunRequest && m_LatestRequestType == RequestType::Suspend)
            {
                // Run 実行中にクライアント側から Suspend 要求が来た場合は、Run をリトライする
                isRetryRunRequest = true;
            }
            mode = m_RequestMode;
            m_pManager->GetIrSensorConfig(&config);
        }
        if (isSettingEnabled)
        {
            {
                ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                // 既に m_pManager が無効であれば処理をやめる
                if (!m_IsManagerInitialized)
                {
                    return;
                }
                m_pManager->UpdateIrCameraStatus();
            }

            switch (mode)
            {
            case IrSensorMode::Moment:
                {
                    ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                    m_pManager->PrepareMomentProcessor();
                    if (isRunRequest)
                    {
                        result = m_pManager->RunImageProcessor(config.moment);
                    }
                    else
                    {
                        result = m_pManager->ResumeImageProcessor(config.moment);
                    }
                }
                break;
            case IrSensorMode::Clustering:
                {
                    ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                    m_pManager->PrepareClusteringProcessor();
                    if (isRunRequest)
                    {
                        result = m_pManager->RunImageProcessor(config.clustering);
                    }
                    else
                    {
                        result = m_pManager->ResumeImageProcessor(config.clustering);
                    }
                }
                break;
            case IrSensorMode::ImageTransfer:
                {
                    ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                    void* address = nullptr;
                    m_pManager->GetTransferMemoryWorkBufferAddress(&address);
                    if (address == nullptr)
                    {
                        // TransferMemory 解放直後に処理が回ってきた場合は、リトライする
                        result = nn::irsensor::ResultIrsensorBusy();
                    }
                    else
                    {
                        m_pManager->PrepareImageTransferProcessor();
                        if(isRunRequest)
                        {
                            result = m_pManager->RunImageProcessor(
                                config.imageTransfer, address);
                        }
                        else
                        {
                            result = m_pManager->ResumeImageProcessor(
                                config.imageTransfer, address);
                        }
                    }
                }
                break;
            case IrSensorMode::Pointing:
                {
                    ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                    m_pManager->PreparePointingProcessor();
                    if (isRunRequest)
                    {
                        result = m_pManager->RunImageProcessor(config.pointing);
                    }
                    else
                    {
                        result = m_pManager->ResumeImageProcessor(config.pointing);
                    }
                }
                break;
            case IrSensorMode::TeraPlugin:
                {
                    ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                    m_pManager->PrepareTeraPluginProcessor();
                    if (isRunRequest)
                    {
                        result = m_pManager->RunImageProcessor(config.teraPlugin);
                    }
                    else
                    {
                        result = m_pManager->ResumeImageProcessor(config.teraPlugin);
                    }
                }
                break;
            case IrSensorMode::IrLed:
                {
                    ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                    m_pManager->PrepareIrLedProcessor();
                    if (isRunRequest)
                    {
                        result = m_pManager->RunImageProcessor(config.irLed);
                    }
                    else
                    {
                        result = m_pManager->ResumeImageProcessor(config.irLed);
                    }
                }
                break;
            case IrSensorMode::None:
                // RunImageProcessor の呼び出し直後に StopImageProcessor が呼ばれた
                return;
            default:
                NN_ABORT();
            }

            // 現在のモードに反映
            m_CurrentMode = mode;

            // NeedUpdate エラーの場合は InternalStatus を設定して、Get 時にエラービューア表示する。
            if (::nn::irsensor::ResultIrsensorNeedUpdate::Includes(result))
            {
                m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_NeedUpdate);
            }
            else if (::nn::irsensor::ResultIrsensorHardwareError::Includes(result))
            {
                m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_HardwareError);
            }
            else if (::nn::irsensor::ResultIrsensorNeedCharge::Includes(result))
            {
                m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_NeedCharge);
            }
            else if (::nn::irsensor::ResultHidResourceError::Includes(result))
            {
                m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_HidResourceError);
            }
            else if (::nn::irsensor::ResultIrsensorAppletResourceNotAvailable::Includes(result))
            {
                // 別アプレット起動中の場合は何もしない
            }
            else if (::nn::irsensor::ResultIrsensorBusy::Includes(result))
            {
                // Busy の場合は何もせずリトライする
            }
            else if (::nn::irsensor::ResultIrsensorMcuNotAvailable::Includes(result))
            {
                // Tera MCU と通信できない場合は上限を決めてリトライする
                // しばらく成功しない場合は、BT-Tera 間の通信異常として HardwareError を通知する。
                if (m_McuErrorCounter > McuErrorCountMax)
                {
                    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_HardwareError);
                }
                m_McuErrorCounter++;
            }
            else if (::nn::irsensor::ResultIrsensorRegisterAccessTimeout::Includes(result))
            {
                // レジスタアクセスに時間がかかった場合もリトライする
            }
            else if (result.IsSuccess()
                || ::nn::irsensor::ResultIrsensorUnavailable::Includes(result))
            {
                // 成功時、または未接続、非搭載の場合は異常状態になりえないので、Availableにしておく
                if (!m_ResumeRequestEvent.GetBase()->_signalState)
                {
                    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_Available);
                }
                else
                {
                    // すでに次のリクエストが送られている場合は、Setting 状態を維持する
                    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_Setting);
                }
                m_McuErrorCounter = 0;
            }
            else
            {
                // その他のエラーの場合もリトライする
                // NN_SDK_LOG("[RequestHandler] result:%08X\n", result);
            }

            // 失敗した場合には、動的 Run であっても一度停止して、リトライする
            if (result.IsFailure())
            {
                ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                m_pManager->ClearLifo();
                m_pManager->SleepImageProcessor();
                m_RunRequestEvent.Signal();
            }
        }
        else if (isRetryRunRequest)
        {
            ::nn::os::SleepThread(RetryInterval);
            m_RunRequestEvent.Signal();
        }
        if (result.IsFailure())
        {
            ::nn::os::SleepThread(RetryInterval);
        }
    }
    else if (waitId == m_TimerEvent.GetWaitId())
    {
        // タイマによるサンプリングリクエストを処理する。
        m_TimerEvent.Clear();

        {
            bool isSamplingEnabled = false;
            {
                ::std::lock_guard<decltype(m_RequestMutex)> requestLocker(m_RequestMutex);
                if (m_LatestRequestType != RequestType::Stop
                    && m_LatestRequestType != RequestType::Suspend
                    && m_LatestRequestType != RequestType::None)
                {
                    isSamplingEnabled = true;
                }
            }

            {
                // クライアントアプレットの状態などを更新
                ::std::lock_guard<decltype(m_ManagerMutex)> locker(m_ManagerMutex);
                // 既に m_pManager が無効であれば処理をやめる
                if (!m_IsManagerInitialized)
                {
                    return;
                }
                m_pManager->UpdateIrSensorAppletStatus();
                m_pManager->UpdateIrCameraStatus();
            }

            if (isSamplingEnabled)
            {
                nn::Result result = nn::ResultSuccess();
                if (m_CurrentMode != IrSensorMode::None)
                {
                    ::std::lock_guard<decltype(m_ManagerMutex)> locker(m_ManagerMutex);
                    result = m_pManager->Sample();
                }

                if (ResultIrsensorBusy::Includes(result)
                    || ResultIrsensorAppletResourceNotAvailable::Includes(result)
                    || ResultIrsensorSamplingTimeout::Includes(result))
                {
                    if (m_CurrentMode != IrSensorMode::None)
                    {
                        ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                        // コントローラが切断されるなど不正な場合, アプレットが BG に遷移した場合は、
                        // IR センサーのサンプリングを止める。
                        // Run のリトライに入り、次回 Run が成功した時に起きる。
                        m_pManager->ClearLifo();
                        m_pManager->SleepImageProcessor();
                        m_RunRequestEvent.Signal();
                    }
                }
                else if (ResultIrsensorNeedCharge::Includes(result))
                {
                    // 電池残量が足りなくなった場合は、NeedCharge 状態を設定してリトライする
                    ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                    m_pManager->SetIrCameraInternalStatus(::nn::irsensor::IrCameraInternalStatus_NeedCharge);
                    m_pManager->ClearLifo();
                    m_pManager->SleepImageProcessor();
                    m_RunRequestEvent.Signal();
                }
            }
            else
            {
                // サンプリングが始まっていない間、バージョンチェックのポーリングを行う
                ::std::lock_guard<decltype(m_ManagerMutex)> managerLocker(m_ManagerMutex);
                if (m_IsMcuVersionRequested)
                {
                    auto result = CheckFirmwareVersion();
                    if (result.IsSuccess())
                    {
                        m_IsMcuVersionRequested = false;
                    }
                }
            }
        }
    }
} // NOLINT(impl/function_size)

}}} // namespace nn::irsensor::detail
