﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/irsensor/irsensor_PackedTypes.h>
#include <nn/irsensor/irsensor_IIrSensorDriver.h>

#include "irsensor_ActivationCount.h"
#include "irsensor_StatusManagerHolder.h"
#include "irsensor_AppletResourceManager.h"

namespace nn { namespace irsensor { namespace detail {

class DriverManager final
{
    NN_DISALLOW_COPY(DriverManager);
    NN_DISALLOW_MOVE(DriverManager);

public:
    static const ::nn::TimeSpan SamplingInterval;
    static const int PointingSamplingStatesCountMax;

private:
    IrCameraHandle m_Handle;

    IIrSensorDriver* m_pDriver;

    StatusManagerHolder* m_pStatusManagerHolder;

    ::nn::os::Mutex* m_pAppletResourceMutex;

    AppletResourceManager* m_pAppletResourceManager;

    ActivationCount m_ActivationCount;

    ::nn::os::TimerEventType* m_pTimerEvent;

    //!< コントローラの状態通知用イベント
    ::nn::os::SystemEventType m_DeviceStatusEvent;

    //!< デバイスが Suspend 状態かどうかを判別するフラグ
    bool m_IsDeviceSuspended;

public:
    DriverManager() NN_NOEXCEPT;

    ~DriverManager() NN_NOEXCEPT;

    void SetTimerEvent(::nn::os::TimerEventType* pTimerEvent) NN_NOEXCEPT;

    void SetDriver(IIrSensorDriver* pDriver) NN_NOEXCEPT;

    void SetStatusManagerHolder(
        StatusManagerHolder* pStatusManagerHolder) NN_NOEXCEPT;

    void SetIrCameraHandle(IrCameraHandle handle) NN_NOEXCEPT;

    void BindIrSensorEvent() NN_NOEXCEPT;

    void SetAppletResourceManager(
        ::nn::os::Mutex* pAppletResourceMutex,
        AppletResourceManager* pAppletResourceManager) NN_NOEXCEPT;

    bool IsDriverAvailable() NN_NOEXCEPT;

    ::nn::Result Activate() NN_NOEXCEPT;

    ::nn::Result Deactivate() NN_NOEXCEPT;

    ::nn::Result GetLifoHandle(::nn::os::NativeHandle* outValue) NN_NOEXCEPT;

    ::nn::Result CheckFirmwareVersion(
        const PackedMcuVersion& requiredVersion) NN_NOEXCEPT;

    ::nn::Result RequestFirmwareVersion() NN_NOEXCEPT;

    ::nn::Result SetFunctionLevel(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedFunctionLevel& functionLevel) NN_NOEXCEPT;

    ::nn::Result SleepImageProcessor() NN_NOEXCEPT;

    ::nn::Result StopImageProcessor() NN_NOEXCEPT;

    ::nn::Result StopImageProcessorWithCompatibility() NN_NOEXCEPT;

    ::nn::Result SuspendImageProcessor() NN_NOEXCEPT;

    ::nn::Result StopSampling(
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    ::nn::Result RunImageProcessor(
        const PackedMomentProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunImageProcessor(
        const PackedClusteringProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunImageProcessor(
        const PackedImageTransferProcessorExConfig& config,
        void* workBuffer
        ) NN_NOEXCEPT;

    ::nn::Result RunImageProcessor(
        const PackedPointingProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunImageProcessor(
        const PackedTeraPluginProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunImageProcessor(
        const PackedIrLedProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result ResumeImageProcessor(
        const PackedMomentProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result ResumeImageProcessor(
        const PackedClusteringProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result ResumeImageProcessor(
        const PackedImageTransferProcessorExConfig& config,
        void* workBuffer
        ) NN_NOEXCEPT;

    ::nn::Result ResumeImageProcessor(
        const PackedPointingProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result ResumeImageProcessor(
        const PackedTeraPluginProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result ResumeImageProcessor(
        const PackedIrLedProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result GetImageProcessorState(
        ImageTransferProcessorState* pOutState,
        void* pOutImage,
        size_t size) NN_NOEXCEPT;

    ::nn::Result PrepareMomentProcessor() NN_NOEXCEPT;

    ::nn::Result PrepareClusteringProcessor() NN_NOEXCEPT;

    ::nn::Result PrepareImageTransferProcessor() NN_NOEXCEPT;

    ::nn::Result PreparePointingProcessor() NN_NOEXCEPT;

    ::nn::Result PrepareTeraPluginProcessor() NN_NOEXCEPT;

    ::nn::Result PrepareIrLedProcessor() NN_NOEXCEPT;

    void UpdateIrCameraStatus() NN_NOEXCEPT;

    void SetIrCameraInternalStatus(IrCameraInternalStatus status) NN_NOEXCEPT;

    void UpdateIrSensorAppletStatus() NN_NOEXCEPT;

    ::nn::Result Sample() NN_NOEXCEPT;

    ::nn::Result ClearLifo() NN_NOEXCEPT;

    ::nn::Result SetIrSensorConfig(
        const ::nn::applet::AppletResourceUserId& aruid,
        IrSensorConfig config) NN_NOEXCEPT;

    ::nn::Result GetIrSensorConfig(
        IrSensorConfig* pConfig) NN_NOEXCEPT;

    ::nn::Result GetTransferMemoryType(
        ::nn::os::TransferMemoryType** pTransferMemory,
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    ::nn::Result SetTransferMemoryWorkBufferAddress(
        const ::nn::applet::AppletResourceUserId& aruid,
        void* address) NN_NOEXCEPT;

    ::nn::Result GetTransferMemoryWorkBufferAddressWithAruid(
        void** pAddress,
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    ::nn::Result GetTransferMemoryWorkBufferAddress(
        void** pAddress) NN_NOEXCEPT;

    ::nn::Result SetImageValidity(
        const ::nn::applet::AppletResourceUserId& aruid,
        bool isValid) NN_NOEXCEPT;

private:
    void ActivateTimerEvent() NN_NOEXCEPT;

    void DeactivateTimerEvent() NN_NOEXCEPT;

    ::nn::Result ActivateIrsensorForHid() NN_NOEXCEPT;

    ::nn::Result DeactivateIrsensorForHid() NN_NOEXCEPT;
};

}}} // namespace nn::irsensor::detail
