﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>


// TO REMOVE---->
#include <gillette_Api.h>
#include <nn/irsensor/irsensor_HandAnalysis.h>
#include <nn/irsensor/irsensor_HandAnalysisConstants.h>
#include <nn/irsensor/irsensor_HandAnalysisTypesPrivate.h>
// <----

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/err/err_ShowErrorApi.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/hid/hid_ResultControllerSupport.h>
#include <nn/irsensor/irsensor_IIrSensorServer.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>
#include <nn/irsensor/irsensor_IrCameraTypesPrivate.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/irsensor/detail/irsensor_Log.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_Thread.h>
#include <nn/sf/sf_ISharedObject.h>

#include "irsensor_IrSensorServer.h"
#include "irsensor_IrCameraHandle.h"
#include "irsensor_ApiImpl.h"
#include "irsensor_LockableMutexType.h"
#include "irsensor_SessionManager.h"
#include "irsensor_AppletResourceUserId.h"
#include "irsensor_ActivationCount.h"

namespace nn { namespace irsensor { namespace detail {

namespace {

//!<  IR センサーライブラリが要求する TeraMcu のバージョン
static const PackedMcuVersion RequiredMcuVersion = { 5, 24 };

ActivationCount s_ActivationCount;

void PackConfig(
    PackedIrCameraConfig* pOutValue,
    const IrCameraConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->exposureTime = config.exposureTime;
    pOutValue->gain = static_cast<int8_t>(config.gain);
    pOutValue->isNegativeImageUsed = config.isNegativeImageUsed;
    pOutValue->lightTarget = static_cast<int8_t>(config.lightTarget);
}

void PackConfig(
    PackedMomentProcessorConfig* pOutValue,
    const MomentProcessorConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    PackConfig(&pOutValue->irCameraConfig, config.irCameraConfig);
    pOutValue->windowOfInterest = config.windowOfInterest;
    pOutValue->preprocess = static_cast<int8_t>(config.preprocess);
    pOutValue->preprocessIntensityThreshold =
        static_cast<uint8_t>(config.preprocessIntensityThreshold);
    pOutValue->requiredVersion = RequiredMcuVersion;
}

void PackConfig(
    PackedClusteringProcessorConfig* pOutValue,
    const ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    PackConfig(&pOutValue->irCameraConfig, config.irCameraConfig);
    pOutValue->windowOfInterest = config.windowOfInterest;
    pOutValue->objectIntensityMin =
        static_cast<uint8_t>(config.objectIntensityMin);
    pOutValue->objectPixelCountMax =
        static_cast<int32_t>(config.objectPixelCountMax);
    pOutValue->objectPixelCountMin =
        static_cast<int32_t>(config.objectPixelCountMin);
    pOutValue->isExternalLightFilterEnebled = config.isExternalLightFilterEnabled;
    pOutValue->requiredVersion = RequiredMcuVersion;
}

void PackConfig(
    PackedImageTransferProcessorConfig* pOutValue,
    const ImageTransferProcessorConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    PackConfig(&pOutValue->irCameraConfig, config.irCameraConfig);
    pOutValue->format = static_cast<int8_t>(config.format);
    pOutValue->requiredVersion = RequiredMcuVersion;
}

void PackConfig(
    PackedImageTransferProcessorExConfig* pOutValue,
    const ImageTransferProcessorExConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    PackConfig(&pOutValue->irCameraConfig, config.irCameraConfig);
    pOutValue->requiredVersion = RequiredMcuVersion;
    pOutValue->origFormat = static_cast<int8_t>(config.origFormat);
    pOutValue->trimmingFormat = static_cast<int8_t>(config.trimmingFormat);
    pOutValue->trimmingStartX = static_cast<uint16_t>(config.trimmingStartX);
    pOutValue->trimmingStartY = static_cast<uint16_t>(config.trimmingStartY);
    pOutValue->isExternalLightFilterEnebled = config.isExternalLightFilterEnabled;
}

void PackConfig(
    PackedPointingProcessorConfig* pOutValue,
    const PointingProcessorConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->windowOfInterest = config.windowOfInterest;
    pOutValue->requiredVersion = RequiredMcuVersion;
}

void PackConfig(
    PackedTeraPluginProcessorConfig* pOutValue,
    const TeraPluginProcessorConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->mode = config.modeOffset;
    pOutValue->requiredVersion = RequiredMcuVersion;
    pOutValue->param.setting = ((config.isParameterEnabled << 7) & 0x80) | static_cast<int8_t>(TeraPluginParameterSize);
    for (auto i = 0; i < static_cast<int8_t>(TeraPluginParameterSize); i++)
    {
        pOutValue->param.data[i] = config.param.data[i];
    }
}

void PackConfig(
    PackedIrLedProcessorConfig* pOutValue,
    const IrLedProcessorConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->lightTarget = static_cast<int8_t>(config.lightTarget);
    pOutValue->requiredVersion = RequiredMcuVersion;
}

} // namespace

::nn::Result GetIrCameraHandle(
    IrCameraHandle* pOutValue,
    const ::nn::hid::NpadIdType& npadIdType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto proxy = ::nn::sf::SharedPointer<IIrSensorServer>();
    CreateIrSensorServerProxy(&proxy);
    NN_RESULT_DO(proxy->GetNpadIrCameraHandle(pOutValue, npadIdType));

    NN_RESULT_SUCCESS;
}

::nn::Result InitializeIrCamera(
    const IrSensorFunctionLevel& functionLevel,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    if (s_ActivationCount.IsZero())
    {
        NN_RESULT_DO(pSession->Activate(functionLevel));
        // FunctionLevel を設定
        pSession->SetFunctionLevel(handle, functionLevel);
    }
    ++s_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeIrCamera(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_UNUSED(handle);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    if (!pSession->IsProcessorStopped(handle))
    {
        // StopImageProcessor が行われていなかった場合は、ここで同期的に行う
        NN_RESULT_DO(StopImageProcessor(handle));
    }

    --s_ActivationCount;
    if (s_ActivationCount.IsZero())
    {
        NN_RESULT_DO(pSession->Deactivate());

        GetSessionManager().FinalizeSession(GetAppletResourceUserId());
        // FunctionLevel をクリア
        pSession->SetFunctionLevel(handle, nn::irsensor::IrSensorFunctionLevel_0);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetIrCameraStatus(
    IrCameraStatus* pOutValue,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();
    *pOutValue = pSession->GetIrCameraStatus(handle);

    NN_RESULT_SUCCESS;
}

::nn::Result GetIrCameraInternalStatus(
    IrCameraInternalStatus* pOutValue,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();
    *pOutValue = pSession->GetIrCameraInternalStatus(handle);

    NN_RESULT_SUCCESS;
}

::nn::Result GetImageProcessorStatus(
    ImageProcessorStatus* pOutValue,
    const bool& isInternalCall,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    // 内部呼び出しの場合は、FunctionLevel によらず許可する
    if (!isInternalCall)
    {
        // FunctionLevel 1以上で利用可能
        IrSensorFunctionLevel functionLevel = IrSensorFunctionLevel_Latest;
        pSession->GetFunctionLevel(&functionLevel, handle);
        NN_SDK_ASSERT_GREATER_EQUAL(functionLevel, IrSensorFunctionLevel_1);
    }

    ImageProcessorStatus imageProcessorStatus = ImageProcessorStatus_Stopped;
    IrCameraInternalStatus internalStatus = IrCameraInternalStatus_Available;
    GetIrCameraInternalStatus(&internalStatus, handle);
    if (internalStatus == IrCameraInternalStatus_SuccessfulTerminated
        || internalStatus == IrCameraInternalStatus_NeedUpdateOnInitial
        || internalStatus == IrCameraInternalStatus_CheckVersionIncompleted)
    {
        // 上記の内部状態の場合は、停止が保証される
        imageProcessorStatus = ImageProcessorStatus_Stopped;
    }
    else
    {
        // それ以外の内部状態は、動作時にのみ発生する
        imageProcessorStatus = ImageProcessorStatus_Running;
    }
    *pOutValue = imageProcessorStatus;
    NN_RESULT_SUCCESS;
}

::nn::Result CheckLibraryAppletAvailability(
    bool* pOutIsLibraryAppletCallEnabled,
    const IrCameraInternalStatus& status,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsLibraryAppletCallEnabled);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();
    *pOutIsLibraryAppletCallEnabled = pSession->IsLibraryAppletCallEnabled(status, handle);

    NN_RESULT_SUCCESS;
}

::nn::Result CheckFirmwareUpdateNecessity(
    bool* pIsUpdateNeeded,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pIsUpdateNeeded);

    bool isUpdateNeeded = false;

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();

    // BG の時は FirmwareCheckVersionIncompleted を返す
    NN_RESULT_THROW_UNLESS(pSession->IsAppletForeground(), ResultIrsensorFirmwareCheckIncompleted());

    // コントローラの状態チェック
    NN_RESULT_DO(CheckStatus(nn::ResultSuccess(), handle));

    // バージョンチェックのリクエスト発行
    bool isVersionCheckRequestEnabled = pSession->IsVersionCheckRequestEnabled(handle);
    if (isVersionCheckRequestEnabled)
    {
        NN_RESULT_DO(pSession->CheckFirmwareVersion(handle, RequiredMcuVersion));
        pSession->SetVersionCheckRequestEnableFlag(handle, false);
    }

    // 共有メモリの状態チェック
    IrCameraInternalStatus internalStatus = pSession->GetIrCameraInternalStatus(handle);
    switch (internalStatus)
    {
    case IrCameraInternalStatus_NeedUpdateOnInitial:
        {
            isUpdateNeeded = true;
        }
        break;
    case IrCameraInternalStatus_CheckVersionIncompleted:
        {
            NN_RESULT_THROW(ResultIrsensorFirmwareCheckIncompleted());
        }
        break;
    case IrCameraInternalStatus_Available:
    case IrCameraInternalStatus_Setting:
    case IrCameraInternalStatus_SuccessfulTerminated:
        {
            isUpdateNeeded = false;
        }
        break;
    default:
        {
            NN_DETAIL_IRSENSOR_INFO("Invalid InternalStatus:%d\n", internalStatus);
            NN_RESULT_THROW(ResultIrsensorFirmwareCheckIncompleted());
        }
    }
    // 成功の場合はリクエスト発行可能状態に遷移
    pSession->SetVersionCheckRequestEnableFlag(handle, true);
    *pIsUpdateNeeded = isUpdateNeeded;
    NN_RESULT_SUCCESS;
}

::nn::Result CheckStatus(
    const nn::Result defaultResult,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    IrCameraStatus status;
    GetIrCameraStatus(&status, handle);
    switch (status)
    {
    case IrCameraStatus_Available:
        {
            NN_RESULT_THROW(defaultResult);
        }
        break;
    case IrCameraStatus_Unconnected:
        {
            NN_RESULT_THROW(ResultIrsensorUnconnected());
        }
        break;
    case IrCameraStatus_Unsupported:
        {
            NN_RESULT_THROW(ResultIrsensorUnsupported());
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

::nn::Result CheckInternalStatus(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    IrCameraInternalStatus internalStatus;
    GetIrCameraInternalStatus(&internalStatus, handle);

    bool isLibraryAppletCallEnabled = false;
    CheckLibraryAppletAvailability(&isLibraryAppletCallEnabled, internalStatus, handle);

    switch (internalStatus)
    {
    case IrCameraInternalStatus_NeedUpdate:
        {
            if (isLibraryAppletCallEnabled)
            {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
                // 選択 UI なしの強制アップデート版コンサポ呼び出し
                nn::hid::ControllerFirmwareUpdateArg arg;
                arg.enableForceUpdate = true;
                NN_RESULT_TRY(nn::hid::ShowControllerFirmwareUpdate(arg))
                    NN_RESULT_CATCH(nn::hid::ResultControllerFirmwareUpdateFailed)
                    {
                        // 失敗した場合は UpdateFailed を返して、DeviceError としてハンドリングしてもらう
                        NN_RESULT_THROW(ResultIrsensorUpdateFailed());
                    }
                NN_RESULT_END_TRY
#elif defined( NN_BUILD_CONFIG_OS_WIN )
                NN_DETAIL_IRSENSOR_INFO("Invoke ControllerFirmwareUpdate is not suppported on Windows environment.\n");
                NN_RESULT_THROW(ResultIrsensorNeedUpdate());
#else
            #error "unsupported os"
#endif
            }
            // 成功した場合、スキップされた場合は NotReady を返す
            return ResultIrsensorNotReady();
        }
        break;
    case IrCameraInternalStatus_HardwareError:
        {
            if (isLibraryAppletCallEnabled)
            {
                nn::err::ShowError(ResultIrsensorHardwareError());
            }
            return ResultIrsensorHardwareError();
        }
        break;
    case IrCameraInternalStatus_NeedCharge:
        {
            if (isLibraryAppletCallEnabled)
            {
                nn::err::ShowError(ResultIrsensorNeedCharge());
            }
            return ResultIrsensorNeedCharge();
        }
        break;
    case IrCameraInternalStatus_HidResourceError:
        {
            if (isLibraryAppletCallEnabled)
            {
                NN_DETAIL_IRSENSOR_INFO("Hid Resource Error\n");
            }
            return ResultIrsensorNotReady();
        }
        break;
    case IrCameraInternalStatus_CheckVersionIncompleted:
        {
            return ResultIrsensorNotReady();
        }
        break;
    case IrCameraInternalStatus_NeedUpdateOnInitial:
    case IrCameraInternalStatus_Setting:
        {
            return ResultIrsensorNotReady();
        }
        break;
    case IrCameraInternalStatus_Available:
    case IrCameraInternalStatus_SuccessfulTerminated:
        {
            NN_RESULT_SUCCESS;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

::nn::Result StopImageProcessor(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    // 対象ハンドルの内部エラー管理状態をリセット
    pSession->ResetInternalStatusForLibraryAppletCall(handle);

    // Stop 時にバージョンチェックリクエストの再呼び出しを許可
    pSession->SetVersionCheckRequestEnableFlag(handle, true);

    // IAAA-4020 の不具合修正のため、非同期 Stop を発行し完了をshim 側で待つ
    NN_RESULT_DO(pSession->StopImageProcessorAsync(handle));

    // プロセッサの停止を待つ
    auto counter = 0;
    const auto RetryCountMax = 333;
    const auto RetryIntervalFrame = ::nn::TimeSpan::FromMilliSeconds(15);
    ImageProcessorStatus processorStatus = ImageProcessorStatus_Stopped;
    GetImageProcessorStatus(&processorStatus, true, handle);
    while (processorStatus != ImageProcessorStatus_Stopped)
    {
        ::nn::os::SleepThread(RetryIntervalFrame);
        counter++;
        // 遅くとも 5s 程度待てば処理は完了するはずなので、タイムアウトする。
        if (counter > RetryCountMax)
        {
            NN_DETAIL_IRSENSOR_INFO("StopImageProcessor timed out. Device can be incompleted state.\n");
            break;
        }
        GetImageProcessorStatus(&processorStatus, true, handle);
    }
    NN_RESULT_SUCCESS;
}

::nn::Result StopImageProcessorAsync(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    // FunctionLevel 1以上で利用可能
    IrSensorFunctionLevel functionLevel = IrSensorFunctionLevel_Latest;
    pSession->GetFunctionLevel(&functionLevel, handle);
    NN_SDK_ASSERT_GREATER_EQUAL(functionLevel, IrSensorFunctionLevel_1);

    // 対象ハンドルの内部エラー管理状態をリセット
    pSession->ResetInternalStatusForLibraryAppletCall(handle);

    // Stop 時にバージョンチェックリクエストの再呼び出しを許可
    pSession->SetVersionCheckRequestEnableFlag(handle, true);

    NN_RESULT_DO(pSession->StopImageProcessorAsync(handle));

    NN_RESULT_SUCCESS;
}

::nn::Result RunMomentProcessor(
    const IrCameraHandle& handle,
    const MomentProcessorConfig& config) NN_NOEXCEPT
{
    PackedMomentProcessorConfig packedConfig = {};
    PackConfig(&packedConfig, config);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    NN_RESULT_DO(pSession->RunMomentProcessor(handle, packedConfig));

    NN_RESULT_SUCCESS;
}

::nn::Result GetMomentProcessorState(
    MomentProcessorState* pOutValue,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    int count;
    NN_UNUSED(count);
    NN_RESULT_TRY(detail::GetMomentProcessorStates(pOutValue, &count, 1, handle))
        NN_RESULT_CATCH_ALL
        {
            auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
            NN_RESULT_THROW_UNLESS(
                pSession != nullptr,
                ResultIrsensorInvalidHandle());
            pSession->GetMomentProcessorStateDefault(pOutValue, handle);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result GetMomentProcessorStates(
    MomentProcessorState* pOutStates,
    int* pOutCount, int countMax,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER(countMax, 0);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();

    // 初期値を設定しておく
    *pOutCount = 0;
    pSession->GetMomentProcessorStateDefault(pOutStates, handle);

    // BG の時は NotReady を返す
    NN_RESULT_THROW_UNLESS(pSession->IsAppletForeground(), ResultIrsensorNotReady());

    // 内部エラーをチェックします。
    NN_RESULT_DO(CheckInternalStatus(handle));

    auto pLifo = pSession->GetMomentLifo(handle);
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultIrsensorNotReady());

    int count = ::std::min(countMax, ::nn::irsensor::MomentProcessorStateCountMax);

    *pOutCount = pLifo->Read(pOutStates, count);

    if (*pOutCount == 0)
    {
        // 通常のエラーをチェックします。
        NN_RESULT_DO(CheckStatus(nn::irsensor::ResultIrsensorNotReady(), handle));
    }

    // (IAAA-4350: WorkAround)
    // WOI のサイズに合わせて輝度平均値を変換する
    Rect woi = pSession->GetWindowOfInterestConfig(handle);
    float factor = IrCameraImageWidth * IrCameraImageHeight / static_cast<float>(woi.width * woi.height);
    for (auto i = 0 ; i < *pOutCount; i++)
    {
        for (auto j = 0; j < ::nn::irsensor::MomentProcessorBlockCount; j++)
        {
            pOutStates[i].blocks[j].averageIntensity *= factor;
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RunClusteringProcessor(
    const IrCameraHandle& handle,
    const ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    PackedClusteringProcessorConfig packedConfig = {};
    PackConfig(&packedConfig, config);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    NN_RESULT_DO(pSession->RunClusteringProcessor(handle, packedConfig));

    NN_RESULT_SUCCESS;
}

::nn::Result GetClusteringProcessorState(
    ClusteringProcessorState* pOutValue,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    int count;
    NN_UNUSED(count);
    NN_RESULT_TRY(detail::GetClusteringProcessorStates(pOutValue, &count, 1, handle))
        NN_RESULT_CATCH_ALL
        {
            auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
            NN_RESULT_THROW_UNLESS(
                pSession != nullptr,
                ResultIrsensorInvalidHandle());
            pSession->GetClusteringProcessorStateDefault(pOutValue, handle);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result GetClusteringProcessorStates(
    ClusteringProcessorState* pOutStates,
    int* pOutCount, int countMax,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER(countMax, 0);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();

    // 初期値を設定しておく
    *pOutCount = 0;
    pSession->GetClusteringProcessorStateDefault(pOutStates, handle);

    // BG の時は NotReady を返す
    NN_RESULT_THROW_UNLESS(pSession->IsAppletForeground(), ResultIrsensorNotReady());

    // 内部エラーをチェックします。
    NN_RESULT_DO(CheckInternalStatus(handle));

    auto pLifo = pSession->GetClusteringLifo(handle);
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultIrsensorNotReady());

    int count = ::std::min(countMax, ::nn::irsensor::ClusteringProcessorStateCountMax);

    *pOutCount = pLifo->Read(pOutStates, count);

    if (*pOutCount == 0)
    {
        // 通常のエラーをチェックします。
        NN_RESULT_DO(CheckStatus(nn::irsensor::ResultIrsensorNotReady(), handle));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RunImageTransferProcessor(
    const IrCameraHandle& handle,
    const ImageTransferProcessorConfig& config,
    void* workBuffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_ALIGNED(workBuffer, ::nn::os::MemoryPageSize);

    PackedImageTransferProcessorConfig packedConfig = {};
    PackConfig(&packedConfig, config);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    NN_RESULT_DO(pSession->RunImageTransferProcessor(
        handle, packedConfig, workBuffer, size));

    NN_RESULT_SUCCESS;
}

::nn::Result RunImageTransferProcessor(
    const IrCameraHandle& handle,
    const ImageTransferProcessorExConfig& config,
    void* workBuffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_ALIGNED(workBuffer, ::nn::os::MemoryPageSize);

    PackedImageTransferProcessorExConfig packedConfig = {};
    PackConfig(&packedConfig, config);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    NN_RESULT_DO(pSession->RunImageTransferProcessor(
        handle, packedConfig, workBuffer, size));

    NN_RESULT_SUCCESS;
}

::nn::Result GetImageTransferProcessorState(
    ImageTransferProcessorState* pOutState,
    void* pOutImage,
    size_t size,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(pOutImage);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();

    // BG の時は NotReady を返す
    NN_RESULT_THROW_UNLESS(pSession->IsAppletForeground(), ResultIrsensorNotReady());

    // 内部エラーをチェックします。
    NN_RESULT_DO(CheckInternalStatus(handle));

    auto result = pSession->GetImageTransferState(handle, pOutState, pOutImage, size);
    if (result.IsFailure())
    {
        // 通常のエラーをチェックします。
        NN_RESULT_DO(CheckStatus(nn::irsensor::ResultIrsensorNotReady(), handle));

        // 想定内のエラーの場合にはデフォルト値を設定する
        if (ResultIrsensorUnavailable::Includes(result)
            || ResultIrsensorNotReady::Includes(result))
        {
            pSession->GetImageTransferProcessorStateDefault(
                pOutState, pOutImage, size, handle);
            NN_RESULT_THROW(result);
        }

        // 想定外のエラーの場合にはアボートする
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RunPointingProcessor(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_DETAIL_IRSENSOR_INFO("This function needs a specific hardware. If you want to use it as a product, please contact Nintendo\n");

    PackedPointingProcessorConfig packedConfig = {};
    PointingProcessorConfig config;
    config.windowOfInterest = { 0, 0, IrCameraImageWidth, IrCameraImageHeight };
    PackConfig(&packedConfig, config);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    NN_RESULT_DO(pSession->RunPointingProcessor(handle, packedConfig));

    NN_RESULT_SUCCESS;
}

::nn::Result GetPointingProcessorStates(
    PointingProcessorMarkerState* pOutStates,
    int* pOutCount, int countMax,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER(countMax, 0);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();

    // BG の時は NotReady を返す
    NN_RESULT_THROW_UNLESS(pSession->IsAppletForeground(), ResultIrsensorNotReady());

    // 内部エラーをチェックします。
    NN_RESULT_DO(CheckInternalStatus(handle));

    auto pLifo = pSession->GetPointingLifo(handle);
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultIrsensorNotReady());

    int count = ::std::min(countMax, ::nn::irsensor::PointingProcessorStateCountMax);

    *pOutCount = pLifo->Read(pOutStates, count);

    if (*pOutCount == 0)
    {
        // 通常のエラーをチェックします。
        NN_RESULT_DO(CheckStatus(nn::irsensor::ResultIrsensorNotReady(), handle));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RunTeraPluginProcessor(
    const IrCameraHandle& handle,
    const TeraPluginProcessorConfig& config) NN_NOEXCEPT
{
    PackedTeraPluginProcessorConfig packedConfig = {};
    PackConfig(&packedConfig, config);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    NN_RESULT_DO(pSession->RunTeraPluginProcessor(handle, packedConfig));

    NN_RESULT_SUCCESS;
}

::nn::Result GetTeraPluginProcessorStates(
    TeraPluginProcessorState* pOutStates,
    int* pOutCount, int countMax,
    int64_t infSamplingNumber,
    uint32_t prefix,
    int prefixSize,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER(countMax, 0);
    NN_SDK_REQUIRES_LESS(prefixSize, static_cast<int>(8 * sizeof(prefix)));

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();

    // BG の時は NotReady を返す
    NN_RESULT_THROW_UNLESS(pSession->IsAppletForeground(), ResultIrsensorNotReady());

    // 内部エラーをチェックします。
    NN_RESULT_DO(CheckInternalStatus(handle));

    {
        auto pLifo = pSession->GetTeraPluginLifo(handle);
        NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultIrsensorNotReady());
        int count = ::std::min(countMax, ::nn::irsensor::TeraPluginProcessorStateCountMax);

        *pOutCount = pLifo->ReadFiltered(pOutStates, count, [&infSamplingNumber, &prefix, &prefixSize](const TeraPluginProcessorState& state) {
            bool prefixTest = true;

            size_t cursor = 0;
            ::nerd::otete::Mode mode;
            ::nerd::otete::jaimev2::DecodePacketModeROK(mode, reinterpret_cast<const unsigned char*>(&state.data), sizeof(state.data), &cursor);

            int payload = (state.data[0] & 0x3);
            if (payload == HandAnalysisPrefixImage)
            {
                NN_SDK_ASSERT(mode == nerd::otete::Mode_Image);
            }
            if (payload == HandAnalysisPrefixSilouette)
            {
                NN_SDK_ASSERT(mode == nerd::otete::Mode_Silhouette);
            }

            for (int i = 0; i < prefixSize; i++)
            {
                const int byteIndex = i / 8;
                const int bitIndex = i - 8 * byteIndex;
                NN_SDK_ASSERT_RANGE(bitIndex, 0, prefixSize);
                NN_SDK_ASSERT_LESS(byteIndex, static_cast<int>(sizeof(state.data)));

                uint8_t value       = (state.data[byteIndex] >> bitIndex) & 1;
                uint8_t prefixBit   = (prefix >> i) & 1;

                if (value != prefixBit)
                {
                    prefixTest = false;
                    break;
                }
            }

            return prefixTest && (state.samplingNumber >= infSamplingNumber);
        });
    }

    if (*pOutCount == 0)
    {
        // 通常のエラーをチェックします。
        NN_RESULT_DO(CheckStatus(nn::irsensor::ResultIrsensorNotReady(), handle));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result RunIrLedProcessor(
    const IrCameraHandle& handle,
    const IrLedProcessorConfig& config) NN_NOEXCEPT
{
    PackedIrLedProcessorConfig packedConfig = {};
    PackConfig(&packedConfig, config);

    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    NN_RESULT_DO(pSession->RunIrLedProcessor(handle, packedConfig));

    NN_RESULT_SUCCESS;
}

::nn::Result GetIrLedProcessorState(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto pSession = GetSessionManager().GetSession(GetAppletResourceUserId());
    NN_RESULT_THROW_UNLESS(
        pSession != nullptr,
        ResultIrsensorInvalidHandle());

    pSession->AttachSharedMemoryIfNotMapped();

    // BG の時は NotReady を返す
    NN_RESULT_THROW_UNLESS(pSession->IsAppletForeground(), ResultIrsensorNotReady());

    // 内部エラーをチェックします。
    NN_RESULT_DO(CheckInternalStatus(handle));

    auto pLifo = pSession->GetMomentLifo(handle);
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultIrsensorNotReady());

    // 上位には渡さないが、エラー処理用に Moment の Lifo からデータを取得する
    MomentProcessorState state = {};
    auto count = pLifo->Read(&state, 1);
    if (count == 0)
    {
        // 通常のエラーをチェックします。
        NN_RESULT_DO(CheckStatus(nn::irsensor::ResultIrsensorNotReady(), handle));
    }

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::irsensor::detail
