﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <new>

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MemoryFence.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/result/result_HandlingUtility.h>

#include "hidbus_ResourceHolder.h"
#include "../detail/hidbus_InternalUtility.h"
#include "../detail/hidbus_InternalDefinition.h"

namespace nn { namespace hidbus { namespace server {

namespace {

//!< 他プロセスに許可されるメモリアクセス権
const ::nn::os::MemoryPermission OtherPermission =
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Visual Studio の std::atomic 型が load 時にロックしてしまうため
    ::nn::os::MemoryPermission_ReadWrite;
#else
    ::nn::os::MemoryPermission_ReadOnly;
#endif

} // namespace

ResourceHolder::ResourceHolder() NN_NOEXCEPT
    : m_IsCreated(false)
{
    for (int i = 0; i < nn::hidbus::detail::MaxHidbusNumber; i++)
    {
        m_TransferMemoryHolder->pMappedAddress = nullptr;
        m_TransferMemoryHolder->size = 0;
        m_EnabledAruid[i] = nn::applet::AppletResourceUserId::GetInvalidId();
        m_InitializedAruid[i] = nn::applet::AppletResourceUserId::GetInvalidId();
        m_TransferMemoryMutex[i] = NN_OS_SDK_MUTEX_INITIALIZER();
        m_IsResourceInitialized[i] = false;
        m_InternalStateMutex[i] = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();
    }
}

ResourceHolder::~ResourceHolder() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result ResourceHolder::Initialize() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::os::CreateSharedMemory(
        &m_SharedMemory,
        ::nn::os::MemoryPageSize,
        ::nn::os::MemoryPermission_ReadWrite,
        OtherPermission));

    ::nn::os::MapSharedMemory(&m_SharedMemory,
        ::nn::os::MemoryPermission_ReadWrite);

    m_IsCreated = true;

    auto pAccessor = reinterpret_cast<nn::hidbus::detail::StatusManager*>(
        ::nn::os::GetSharedMemoryAddress(&m_SharedMemory));
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    new (pAccessor) nn::hidbus::detail::StatusManager();
    ::nn::os::FenceMemoryStoreLoad();

    NN_RESULT_SUCCESS;
}

void ResourceHolder::Finalize() NN_NOEXCEPT
{
    if (m_IsCreated)
    {
        auto pAccessor = GetStatusManager();
        pAccessor->~StatusManager();
        pAccessor = nullptr;

        // 共有メモリに対するアクセスの完了を保証
        ::nn::os::FenceMemoryAnyStore();
    }

    m_IsCreated = false;

    if (::nn::os::GetSharedMemoryAddress(&m_SharedMemory) != nullptr)
    {
        ::nn::os::UnmapSharedMemory(&m_SharedMemory);
    }

    ::nn::os::DestroySharedMemory(&m_SharedMemory);
}

::nn::Result ResourceHolder::ClearSharedMemory() NN_NOEXCEPT
{
    if (m_IsCreated)
    {
        auto pAccessor = GetStatusManager();
        // コンストラクタ呼び出しで、データをクリア
        new (pAccessor) nn::hidbus::detail::StatusManager();
        // 共有メモリに対するアクセスの完了を保証
        ::nn::os::FenceMemoryStoreLoad();
    }
    NN_RESULT_SUCCESS;
}

void ResourceHolder::AttachSharedMemory(
    ::nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT
{
    ::nn::os::AttachSharedMemory(&m_SharedMemory,
                                 ::nn::os::MemoryPageSize,
                                 handle,
                                 managed);

    ::nn::os::MapSharedMemory(&m_SharedMemory, OtherPermission);
}

size_t ResourceHolder::GetSharedMemorySize() const NN_NOEXCEPT
{
    return ::nn::os::GetSharedMemorySize(&m_SharedMemory);
}

::nn::os::NativeHandle ResourceHolder::GetSharedMemoryHandle(
    ) const NN_NOEXCEPT
{
    return ::nn::os::GetSharedMemoryHandle(&m_SharedMemory);
}

nn::hidbus::detail::StatusManager* ResourceHolder::GetStatusManager() const NN_NOEXCEPT
{
    auto pAccessor = reinterpret_cast<nn::hidbus::detail::StatusManager*>(
        ::nn::os::GetSharedMemoryAddress(&m_SharedMemory));
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    return pAccessor;
}

bool ResourceHolder::IsSharedMemoryMapped() const NN_NOEXCEPT
{
    return ::nn::os::GetSharedMemoryAddress(&m_SharedMemory) != nullptr;
}

template<typename PollingDataLifoFormatT, int PollingDataSize>
nn::hidbus::detail::PollingDataAccessor<PollingDataLifoFormatT, PollingDataSize>* ResourceHolder::GetPollingDataAccessor(int index) NN_NOEXCEPT
{
    auto pAccessor = reinterpret_cast<nn::hidbus::detail::PollingDataAccessor<PollingDataLifoFormatT, PollingDataSize>*>(
        m_TransferMemoryHolder[index].pMappedAddress);

    return pAccessor;
}

ResourceHolder& GetResourceHolder() NN_NOEXCEPT
{
    return nn::hidbus::detail::StaticObject<ResourceHolder>::Get();
}

nn::hidbus::detail::JoyEnableSixAxisPollingDataAccessor* GetJoyEnableSixAxisPollingDataAccessor(int index) NN_NOEXCEPT
{
    return GetResourceHolder().GetPollingDataAccessor<
        detail::JoyConEnableSixAxisSensorLifoFormat,
        detail::JoyConEnableSixAxisSensorLifoCount>(index);
}

nn::hidbus::detail::JoyDisableSixAxisPollingDataAccessor* GetJoyDisableSixAxisPollingDataAccessor(int index) NN_NOEXCEPT
{
    return GetResourceHolder().GetPollingDataAccessor<
        detail::JoyConDisableSixAxisSensorLifoFormat,
        detail::JoyConDisableSixAxisSensorLifoCount>(index);
}

}}} // namespace nn::hidbus::server
