﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_SystemEvent.h>

#include "hidbus_StatusManager.h"
#include "hidbus_PollingDataAccessor.h"
#include "hidbus_InternalDefinition.h"

namespace nn { namespace hidbus { namespace detail {

class ClientResourceHolder final
{
    NN_DISALLOW_COPY(ClientResourceHolder);
    NN_DISALLOW_MOVE(ClientResourceHolder);

private:
    //!< 新規に構築されたか否か
    bool m_IsCreated;

    //!< 共有メモリがマップ済みかどうか
    bool m_IsSharedMemoryMapped;

    //!< 共有メモリがマップ済みかどうかを守るための mutex
    nn::os::SdkMutexType m_IsSharedMemoryMappedMutex;

    //!< 共有メモリオブジェクト
    ::nn::os::SharedMemoryType m_SharedMemory;

    //!< TransferMemory でサーバー側へ渡したバッファへのポインタ
    void* m_pTransferdBuffer[MaxHidbusNumber];

    // SendAndReceive で使用するための SystemEvenType
    nn::os::SystemEventType m_SendAndReceiveEvent[nn::hid::NpadStateCountMax];

    // SendAndReceive で使用するための SystemEvenType が初期化済みか管理する bool
    bool m_IsSystemEventInitialized[nn::hid::NpadStateCountMax] = { 0 };

    // index ごとの取得した handle
    BusHandle m_HandleHolder[MaxHidbusNumber];

    // index ごとの排他処理を行う StaticMutex
    nn::os::SdkRecursiveMutexType m_ClientMutexForEachIndex[MaxHidbusNumber];

    // FwUpdate 実行中かどうかを管理する bool
    bool m_IsFwUpdateRunning;

    // FwUpdate 実行中を管理するフラグに対する mutex
    nn::os::SdkMutexType m_FwUpdateCheckMutex;

public:
    ClientResourceHolder() NN_NOEXCEPT;

    ~ClientResourceHolder() NN_NOEXCEPT;

    ::nn::Result Clear() NN_NOEXCEPT;

    void Attach(::nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT;
    void Detach() NN_NOEXCEPT;

    StatusManager* GetStatusManager() const NN_NOEXCEPT;

    size_t GetSharedMemorySize() const NN_NOEXCEPT;

    ::nn::os::NativeHandle GetSharedMemoryHandle() const NN_NOEXCEPT;

    bool IsSharedMemoryMapped() NN_NOEXCEPT;

    void SetTransferdBuffer(void* pBuffer, int index) NN_NOEXCEPT;

    template<typename PollingDataLifoFormatT, int PollingDataSize>
    nn::hidbus::detail::PollingDataAccessor<PollingDataLifoFormatT, PollingDataSize>* GetPollingDataAccessor(int index) NN_NOEXCEPT;

    nn::os::SystemEventType* GetSystemEventTypeForSendAndReceive(int index) NN_NOEXCEPT
    {
        return &m_SendAndReceiveEvent[index];
    }

    void SetInitializedFlagForSendAndReceiveEvent(bool isInitialized, int index) NN_NOEXCEPT
    {
        m_IsSystemEventInitialized[index] = isInitialized;
    }

    bool IsSendAndReceiveEventInitialized(int index) NN_NOEXCEPT
    {
        return m_IsSystemEventInitialized[index];
    }

    void SetHandle(BusHandle handle, int index)
    {
        m_HandleHolder[index]._storage = handle._storage;
    }

    bool IsValidHandle(BusHandle handle, int index)
    {
        if (m_HandleHolder[index]._storage == handle._storage)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    void LockMutexPerEachIndex(int index) NN_NOEXCEPT
    {
        m_ClientMutexForEachIndex[index].Lock();
    }

    void UnlockMutexPerEachIndex(int index) NN_NOEXCEPT
    {
        m_ClientMutexForEachIndex[index].Unlock();
    }

    bool IsFwUpdateAvailable() NN_NOEXCEPT
    {
        ::std::lock_guard<decltype(m_FwUpdateCheckMutex)> locker(m_FwUpdateCheckMutex);
        if (!m_IsFwUpdateRunning)
        {
            // Update が走ってない時は true を返しつつ、 FwUpdate が Running 状態にする
            m_IsFwUpdateRunning = true;
            return true;
        }

        return false;
    }

    void SetFwUpdateDone() NN_NOEXCEPT
    {
        ::std::lock_guard<decltype(m_FwUpdateCheckMutex)> locker(m_FwUpdateCheckMutex);
        m_IsFwUpdateRunning = false;
    }
};

ClientResourceHolder& GetClientResourceHolder() NN_NOEXCEPT;

nn::hidbus::detail::JoyEnableSixAxisPollingDataAccessor* GetJoyEnableSixAxisPollingDataAccessor(int index) NN_NOEXCEPT;
nn::hidbus::detail::JoyDisableSixAxisPollingDataAccessor* GetJoyDisableSixAxisPollingDataAccessor(int index) NN_NOEXCEPT;

}}} // namespace nn::hidbus::detail
