﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/err/err_ShowErrorApi.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/hid.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/hid/hid_ResultControllerSupport.h>
#include <nn/hidbus/hidbus.h>
#include <nn/hidbus/hidbus_ResultPrivate.h>
#include <nn/hidbus/hidbus_IHidbusServer.sfdl.h>
#include <nn/hidbus/detail/hidbus_Log.h>

#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_ISharedObject.h>

#include <nn/applet/applet.h>
#include <nn/oe.h>

#include "hidbus_ApiImpl.h"
#include "hidbus_HidbusServerByHipc.h"
#include "hidbus_ClientResourceHolder.h"
#include "hidbus_InternalUtility.h"
#include "hidbus_AppletResourceUserId.h"

namespace
{

// 現時点では version 情報は Joy-Con + アルファ前提で uint64_t の幅で与える。(必要になったら shim で分岐ロジックを組む)

//JoyConFW の割り当て
// | reserved(16bit) | bt mejor(8bit) | bt minor(8bit) | mcu mejor(16bit) | mcu minor(16bit |
const uint64_t RequiredVersion = 0x03ULL << 40 | 0x89ULL << 32 | 0x05ULL << 16 | 0x18ULL;

// JoyCon 向けに１回で Send できるデータの最大値
const size_t MaxSendCommandSizeForJoy = 37;

}

namespace nn { namespace hidbus { namespace detail {

bool GetBusHandle(nn::hidbus::BusHandle* pOutHandle, const nn::hid::NpadIdType &id, BusType busType) NN_NOEXCEPT
{
    bool hasHandle = false;
    nn::hidbus::BusHandle handle;
    auto proxy = nn::sf::SharedPointer<IHidbusServer>();
    CreateHidbusServerProxy(&proxy);
    auto aruid = detail::GetAppletResourceUserId();
    auto result = proxy->GetBusHandle(&handle, &hasHandle, id, static_cast<uint64_t>(busType), aruid);
    if (result.IsFailure() || !hasHandle)
    {
        return false;
    }

    pOutHandle->_storage = handle._storage;

    auto index = GetInternalIndexFromHandle(handle);
    GetClientResourceHolder().SetHandle(handle, index);

    return true;
}

nn::Result Initialize(const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    auto index = GetInternalIndexFromHandle(handle);
    GetClientResourceHolder().LockMutexPerEachIndex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetClientResourceHolder().UnlockMutexPerEachIndex(index);
    };

    if (!GetClientResourceHolder().IsValidHandle(handle, index))
    {
        return nn::hidbus::ResultInvalidHandle();
    }

    nn::sf::NativeHandle sfEventHandle;
    nn::sf::NativeHandle sfSharedMemoryHandle;
    auto proxy = nn::sf::SharedPointer<IHidbusServer>();
    CreateHidbusServerProxy(&proxy);
    // SharedMemory の Mapping
    if (!GetClientResourceHolder().IsSharedMemoryMapped())
    {
        NN_RESULT_DO(proxy->GetSharedMemoryHandle(&sfSharedMemoryHandle));
        GetClientResourceHolder().Attach(sfSharedMemoryHandle.GetOsHandle(), sfSharedMemoryHandle.IsManaged());
        // ハンドルの管理権を放棄
        sfSharedMemoryHandle.Detach();
    }

    auto aruid = detail::GetAppletResourceUserId();

    auto initializeResult = proxy->Initialize(handle, aruid);
    if (nn::hidbus::ResultAlreadyInitialized::Includes(initializeResult))
    {
        // すでに初期化済みなら、何もせずに返る
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_DO(initializeResult);
    NN_RESULT_DO(proxy->SetEventForSendCommandAsycResult(&sfEventHandle, handle));

    auto pSystemEventType = GetClientResourceHolder().GetSystemEventTypeForSendAndReceive(index);

    // Joy-Con の切断などで Shim 側の SystemEvent が Destroy されずに初期化のままになっている場合、
    // Destroy する
    if (GetClientResourceHolder().IsSendAndReceiveEventInitialized(index))
    {
        nn::os::DestroySystemEvent(pSystemEventType);
    }

    // イベントの登録
    nn::os::AttachReadableHandleToSystemEvent(pSystemEventType,
        sfEventHandle.GetOsHandle(),
        sfEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear);

    sfEventHandle.Detach();

    GetClientResourceHolder().SetInitializedFlagForSendAndReceiveEvent(true, index);

    NN_RESULT_SUCCESS;
}

void Finalize(const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    auto index = GetInternalIndexFromHandle(handle);
    GetClientResourceHolder().LockMutexPerEachIndex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetClientResourceHolder().UnlockMutexPerEachIndex(index);
    };

    if (!GetClientResourceHolder().IsValidHandle(handle, index))
    {
        return;
    }

    auto proxy = nn::sf::SharedPointer<IHidbusServer>();
    CreateHidbusServerProxy(&proxy);
    auto aruid = detail::GetAppletResourceUserId();
    auto result = proxy->Finalize(handle, aruid);
    if (result.IsFailure())
    {
        // ここで失敗する場合は handle が Invalid もしくはすでに終了済みであり、すでに内部的に終了相当の処理が行われているので、
        // 呼び出し元へ返す
        return;
    }

    // index が取れて、かつ、 SystemEvent が Initialize 済みなら Destroy するが、
    // そうではない時はスキップして Initialize 時に任せる。(本当にリークしないかは要チェック)
    auto pSystemEventType = GetClientResourceHolder().GetSystemEventTypeForSendAndReceive(index);

    nn::os::DestroySystemEvent(pSystemEventType);

    // SharedMemory はアプリが死んだ時に開放されるので、 Finalize では特にマップしなおしたりしない。
    //GetClientResourceHolder().Detach();

    GetClientResourceHolder().SetInitializedFlagForSendAndReceiveEvent(false, index);
}

nn::Result EnableExternalDevice(bool isEnabled, uint32_t deviceId, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    auto index = GetInternalIndexFromHandle(handle);
    GetClientResourceHolder().LockMutexPerEachIndex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetClientResourceHolder().UnlockMutexPerEachIndex(index);
    };

    if (!GetClientResourceHolder().IsValidHandle(handle, index))
    {
        return nn::hidbus::ResultInvalidHandle();
    }

    if (!GetClientResourceHolder().GetStatusManager()->IsInFocus(index))
    {
        return nn::hidbus::ResultNotInForcus();
    }

    bool isConnected;
    NN_RESULT_DO(GetClientResourceHolder().GetStatusManager()->IsConnected(&isConnected, index));
    if (!isConnected && isEnabled)
    {
        return nn::hidbus::ResultExternalDeviceNotAttached();
    }

    auto aruid = detail::GetAppletResourceUserId();
    auto proxy = nn::sf::SharedPointer<IHidbusServer>();
    CreateHidbusServerProxy(&proxy);
    auto result = proxy->EnableExternalDevice(handle, isEnabled, RequiredVersion, aruid);
    if (nn::xcd::ResultAttachmentDeviceNeedUpdate::Includes(result) ||
        nn::hid::ResultExternalBusDeviceNeedUpdate::Includes(result))
    {
        if (GetClientResourceHolder().IsFwUpdateAvailable())
        {
            NN_UTIL_SCOPE_EXIT
            {
                GetClientResourceHolder().SetFwUpdateDone();
            };

            // アップデーターを立ち上げる
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
            // 選択 UI なしの強制アップデート版コンサポ呼び出し
            nn::hid::ControllerFirmwareUpdateArg arg;
            arg.enableForceUpdate = true;
            NN_RESULT_TRY(nn::hid::ShowControllerFirmwareUpdate(arg))
                NN_RESULT_CATCH(nn::hid::ResultControllerFirmwareUpdateFailed)
            {
                // 失敗した場合は UpdateFailed を返す
                NN_RESULT_THROW(nn::hidbus::ResultFWUpdateFailed());
            }
            NN_RESULT_END_TRY

            // Update の完了後、成功していたら、そのまま Enable を試みる
            NN_RESULT_DO(proxy->EnableExternalDevice(handle, isEnabled, RequiredVersion, aruid));
#elif defined( NN_BUILD_CONFIG_OS_WIN )
            ::nn::applet::AppletResourceUserId::GetInvalidId();
            NN_DETAIL_HIDBUS_INFO("Invoke ControllerFirmwareUpdate is not suppported on Windows environment.\n");
            NN_RESULT_THROW(nn::hid::ResultControllerFirmwareUpdateFailed());
#else
#error "unsupported os"
#endif
        }
        else
        {
            // すでに別スレッドで FwUpdater を呼び出している状態なので、 NotInFocus を返す
            return nn::hidbus::ResultNotInForcus();
        }
    }
    else if (nn::hidbus::ResultNotInitialized::Includes(result))
    {
        // Initialize を呼ばずに Enable が呼ばれており、復旧できなくなるので、 ABORT する。
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    if (result.IsFailure())
    {
        return result;
    }

    if (isEnabled)
    {
        // ID の取得
        uint32_t deviceIdInfo;
        NN_RESULT_DO(proxy->GetExternalDeviceId(&deviceIdInfo, handle));
        if (deviceIdInfo == deviceId)
        {
            return nn::ResultSuccess();
        }
        else
        {
            // DeviceId が違った場合、Disable にする。
            NN_RESULT_DO(nn::hidbus::EnableExternalDevice(false, deviceId, handle));
            return nn::hidbus::ResultExternalDeviceInvalidId();

        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result SendAndReceive(size_t* pOutSize,
    void* pOutBuffer, size_t outBufferSize,
    const nn::hidbus::BusHandle& handle,
    const void* pInBuffer, size_t inBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(inBufferSize, MaxSendCommandSizeForJoy);
    NN_ABORT_UNLESS_LESS_EQUAL(inBufferSize, MaxSendCommandSizeForJoy);

    auto index = GetInternalIndexFromHandle(handle);
    GetClientResourceHolder().LockMutexPerEachIndex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetClientResourceHolder().UnlockMutexPerEachIndex(index);
    };

    if (!GetClientResourceHolder().IsValidHandle(handle, index))
    {
        return nn::hidbus::ResultInvalidHandle();
    }

    if (!GetClientResourceHolder().GetStatusManager()->IsEnabled(index))
    {
        return nn::hidbus::ResultExternalDeviceNotEnabled();
    }

    uint32_t outSize;
    nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), outBufferSize);
    nn::sf::InBuffer inBuffer(reinterpret_cast<const char*>(pInBuffer), inBufferSize);
    auto proxy = nn::sf::SharedPointer<IHidbusServer>();
    CreateHidbusServerProxy(&proxy);

    auto pSystemEventType = GetClientResourceHolder().GetSystemEventTypeForSendAndReceive(index);

    NN_RESULT_DO(proxy->SendCommandAsync(inBuffer, handle));
    nn::os::WaitSystemEvent(pSystemEventType);
    nn::os::ClearSystemEvent(pSystemEventType);

    auto result = proxy->GetSendCommandAsynceResult(&outSize, outBuffer, handle);
    *pOutSize = outSize;
    return result;
}

nn::Result EnableJoyPollingReceiveMode(const nn::hidbus::BusHandle& handle,
    const void* pInCommand, size_t inCommandSize,
    void* workBuffer, size_t workBufferSize,
    JoyPollingMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_ALIGNED(workBuffer, ::nn::os::MemoryPageSize);
    NN_SDK_REQUIRES_LESS_EQUAL(inCommandSize, MaxSendCommandSizeForJoy);
    NN_ABORT_UNLESS_LESS_EQUAL(inCommandSize, MaxSendCommandSizeForJoy);

    auto index = GetInternalIndexFromHandle(handle);
    GetClientResourceHolder().LockMutexPerEachIndex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetClientResourceHolder().UnlockMutexPerEachIndex(index);
    };

    if (!GetClientResourceHolder().IsValidHandle(handle, index))
    {
        return nn::hidbus::ResultInvalidHandle();
    }

    if (!GetClientResourceHolder().GetStatusManager()->IsEnabled(index))
    {
        return nn::hidbus::ResultExternalDeviceNotEnabled();
    }
    else if (GetClientResourceHolder().GetStatusManager()->IsPollingMode(index))
    {
        NN_DETAIL_HIDBUS_INFO("Already PollingMode\n");
        // すでに PolingMode なら何もせず返す
        NN_RESULT_SUCCESS;
    }

    auto proxy = nn::sf::SharedPointer<IHidbusServer>();
    CreateHidbusServerProxy(&proxy);
    nn::sf::InBuffer inBuffer(reinterpret_cast<const char*>(pInCommand), inCommandSize);

    ::nn::os::TransferMemory pollingReceiveBuf(workBuffer, workBufferSize, nn::os::MemoryPermission_ReadOnly);
    auto sfHandle = nn::sf::NativeHandle(pollingReceiveBuf.Detach(), true);

    // 内部管理用に Transferd した領域をセットしておく
    GetClientResourceHolder().SetTransferdBuffer(workBuffer, index);

    return proxy->EnableJoyPollingReceiveMode(inBuffer, std::move(sfHandle), static_cast<uint32_t>(workBufferSize), static_cast<uint32_t>(mode), handle);
}

nn::Result DisableJoyPollingReceiveMode(const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    auto index = GetInternalIndexFromHandle(handle);
    GetClientResourceHolder().LockMutexPerEachIndex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetClientResourceHolder().UnlockMutexPerEachIndex(index);
    };

    if (!GetClientResourceHolder().IsValidHandle(handle, index))
    {
        return nn::hidbus::ResultInvalidHandle();
    }

    if (!GetClientResourceHolder().GetStatusManager()->IsEnabled(index))
    {
        return nn::hidbus::ResultExternalDeviceNotEnabled();
    }
    auto proxy = nn::sf::SharedPointer<IHidbusServer>();
    CreateHidbusServerProxy(&proxy);
    return proxy->DisableJoyPollingReceiveMode(handle);
}

nn::Result GetJoyPollingReceivedData(JoyPollingReceivedData* pOutData, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    auto index = GetInternalIndexFromHandle(handle);
    if (!GetClientResourceHolder().IsValidHandle(handle, index))
    {
        return nn::hidbus::ResultInvalidHandle();
    }

    if (!GetClientResourceHolder().GetStatusManager()->IsEnabled(index))
    {
        return nn::hidbus::ResultExternalDeviceNotEnabled();
    }

    // 接続状態を StatusManager から取得 (SharedMemory)
    bool isConnected;
    NN_RESULT_DO(GetClientResourceHolder().GetStatusManager()->IsConnected(&isConnected, index));
    if (!isConnected)
    {
        return nn::hidbus::ResultExternalDeviceNotEnabled();
    }

    // PollingMode ではない時はデータサイズを 0 にして返す。
    if (!GetClientResourceHolder().GetStatusManager()->IsPollingMode(index))
    {
        pOutData->outSize = 0;
        NN_RESULT_SUCCESS;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    //Windows では TransferMemory した領域にクライアントは触れないため、 DFC でデータをコピーしてくる。
    JoyPollingReceivedData data;
    nn::sf::OutArray<JoyPollingReceivedData> outArray(&data, 1);

    auto proxy = nn::sf::SharedPointer<IHidbusServer>();
    CreateHidbusServerProxy(&proxy);

    NN_RESULT_DO(proxy->GetPollingData(outArray, handle));

    memcpy(pOutData->data, outArray[0].data, outArray[0].outSize);
    pOutData->outSize = outArray[0].outSize;
    pOutData->samplingNumber = outArray[0].samplingNumber;
    NN_RESULT_SUCCESS;

#else

    int pollingMode = GetClientResourceHolder().GetStatusManager()->GetPollingMode(index);
    // NOTE : JoyCon 向け処理なので、他のコントローラが対応したら変更する
    switch (static_cast<nn::hidbus::JoyPollingMode>(pollingMode))
    {
    case nn::hidbus::JoyPollingMode_SixAxisSensorEnable:
    {
        JoyConEnableSixAxisSensorLifoFormat format;
        NN_RESULT_DO(GetJoyEnableSixAxisPollingDataAccessor(index)->GetPollingData(&format, 1));
        memcpy(pOutData->data, format.data, format.dataSize);
        pOutData->outSize = format.dataSize;
        pOutData->samplingNumber = format.samplingNumber;
    }
    break;

    case nn::hidbus::JoyPollingMode_SixAxisSensorDisable:
    {
        JoyConDisableSixAxisSensorLifoFormat format;
        NN_RESULT_DO(GetJoyDisableSixAxisPollingDataAccessor(index)->GetPollingData(&format, 1));
        memcpy(pOutData->data, format.data, format.dataSize);
        pOutData->outSize = format.dataSize;
        pOutData->samplingNumber = format.samplingNumber;
    }
    break;

    default:NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
#endif
}

nn::Result GetJoyPollingReceivedData(JoyPollingReceivedData* pOutData, int count, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    auto index = GetInternalIndexFromHandle(handle);
    if (!GetClientResourceHolder().IsValidHandle(handle, index))
    {
        return nn::hidbus::ResultInvalidHandle();
    }

    if (!GetClientResourceHolder().GetStatusManager()->IsEnabled(index))
    {
        return nn::hidbus::ResultExternalDeviceNotEnabled();
    }

    // 接続状態を StatusManager から取得 (SharedMemory)
    bool isConnected;
    NN_RESULT_DO(GetClientResourceHolder().GetStatusManager()->IsConnected(&isConnected, index));
    if (!isConnected)
    {
        return nn::hidbus::ResultExternalDeviceNotEnabled();
    }

    // PollingMode ではない時はデータサイズを 0 にして返す。
    if (!GetClientResourceHolder().GetStatusManager()->IsPollingMode(index))
    {
        for (int i = 0; i < count; i++)
        {
            pOutData[i].outSize = 0;
        }

        NN_RESULT_SUCCESS;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    //Windows では TransferMemory した領域にクライアントは触れないため、 DFC でデータをコピーしてくる。
    JoyPollingReceivedData data[JoyConEnableSixAxisSensorLifoCount];

    int cpySize;
    if (count <= JoyConEnableSixAxisSensorLifoCount)
    {
        cpySize = count;
    }
    else
    {
        cpySize = JoyConEnableSixAxisSensorLifoCount;
    }

    nn::sf::OutArray<JoyPollingReceivedData> outArray(data, cpySize);

    auto proxy = nn::sf::SharedPointer<IHidbusServer>();
    CreateHidbusServerProxy(&proxy);

    NN_RESULT_DO(proxy->GetPollingData(outArray, handle));

    for (int i = 0; i < cpySize; i++)
    {
        memcpy(&pOutData[i].data, outArray[i].data, outArray[i].outSize);
        pOutData[i].outSize = outArray[i].outSize;
        pOutData[i].samplingNumber = outArray[i].samplingNumber;
    }
    NN_RESULT_SUCCESS;
#else

    int pollingMode = GetClientResourceHolder().GetStatusManager()->GetPollingMode(index);
    // NOTE : JoyCon 向け処理なので、他のコントローラが対応したら変更する
    switch (static_cast<nn::hidbus::JoyPollingMode>(pollingMode))
    {
    case nn::hidbus::JoyPollingMode_SixAxisSensorEnable:
    {
        int cpySize = count <= JoyConEnableSixAxisSensorLifoCount ? count : JoyConEnableSixAxisSensorLifoCount;

        JoyConEnableSixAxisSensorLifoFormat format[JoyConEnableSixAxisSensorLifoCount];
        NN_RESULT_DO(GetJoyEnableSixAxisPollingDataAccessor(index)->GetPollingData(format, cpySize));
        for (int i = 0; i < cpySize; i++)
        {
            memcpy(&pOutData[i].data, format[i].data, format[i].dataSize);
            pOutData[i].outSize = format[i].dataSize;
            pOutData[i].samplingNumber = format[i].samplingNumber;
        }
    }
    break;

    case nn::hidbus::JoyPollingMode_SixAxisSensorDisable:
    {
        int cpySize = count <= JoyConDisableSixAxisSensorLifoCount ? count : JoyConDisableSixAxisSensorLifoCount;

        JoyConDisableSixAxisSensorLifoFormat format[JoyConDisableSixAxisSensorLifoCount];
        NN_RESULT_DO(GetJoyDisableSixAxisPollingDataAccessor(index)->GetPollingData(format, cpySize));
        for (int i = 0; i < cpySize; i++)
        {
            memcpy(&pOutData[i].data, format[i].data, format[i].dataSize);
            pOutData[i].outSize = format[i].dataSize;
            pOutData[i].samplingNumber = format[i].samplingNumber;
        }
    }
    break;

    default:NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
#endif
}

}}} // namespace nn::hidbus::detail
