﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_SixAxisSensorAccurateCalibration.h>
#include <nn/hid/debug/hid_UniquePad.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include "detail/hid_UniquePadImpl.h"

namespace nn { namespace hid { namespace system {

void BindUniquePadConnectionEvent(::nn::os::SystemEventType* pEvent,
                                  ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::BindUniquePadConnectionEvent(pEvent, clearMode));
}

int ListUniquePads(UniquePadId* outValues,
                   int count) NN_NOEXCEPT
{
    int gotCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::ListUniquePads(&gotCount, outValues, count));
    return gotCount;
}

UniquePadType GetUniquePadType(UniquePadId id) NN_NOEXCEPT
{
    UniquePadType type;

    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::hid::detail::GetUniquePadType(&type, id));

    return type;
}

Result GetUniquePadInterface(UniquePadInterface* pOutValue,
                             UniquePadId id) NN_NOEXCEPT
{
    return ::nn::hid::detail::GetUniquePadInterface(pOutValue, id);
}

Result GetUniquePadControllerNumber(int* pOutValue,
                                    UniquePadId id) NN_NOEXCEPT
{
    return ::nn::hid::detail::GetUniquePadControllerNumber(pOutValue, id);
}

Result GetUniquePadSerialNumber(UniquePadSerialNumber*pOutValue, UniquePadId id) NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::hid::detail::GetUniquePadSerialNumber(pOutValue, id));

    NN_RESULT_SUCCESS;
}

::nn::Result GetUniquePadBluetoothAddress(nn::bluetooth::Address *pOutValue, UniquePadId id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutValue);
    auto result = detail::GetUniquePadBluetoothAddress(pOutValue, id);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultUniquePadError)
    {
        NN_RESULT_THROW(result);
    }
    NN_RESULT_END_TRY;
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

::nn::Result DisconnectUniquePad(system::UniquePadId id) NN_NOEXCEPT
{
    auto result = detail::DisconnectUniquePad(id);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultUniquePadError)
    {
        NN_RESULT_THROW(result);
    }
    NN_RESULT_END_TRY;
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

int ListSixAxisSensorHandles(UniqueSixAxisSensorHandle* pOutValues,
                             const UniquePadId& id,
                             int count) NN_NOEXCEPT
{
    int gotCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::ListSixAxisSensorHandles(&gotCount, pOutValues, id, count));
    return gotCount;
}

bool IsSixAxisSensorUserCalibrationSupported(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    bool isSupported;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::IsSixAxisSensorUserCalibrationSupported(&isSupported,
                                                                                   handle));
    return isSupported;
}

Result ResetSixAxisSensorCalibrationValues(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    return detail::ResetSixAxisSensorCalibrationValues(handle);
}

Result StartSixAxisSensorUserCalibration(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    return detail::StartSixAxisSensorUserCalibration(handle);
}

nn::Result GetSixAxisSensorUserCalibrationStage(SixAxisSensorUserCalibrationStage* pOutValue,
                                                const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    return detail::GetSixAxisSensorUserCalibrationStage(pOutValue, handle);
}

void CancelSixAxisSensorUserCalibration(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::CancelSixAxisSensorUserCalibration(handle));
}

bool IsSixAxisSensorAccurateUserCalibrationSupported(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    bool isSupported = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::IsSixAxisSensorAccurateUserCalibrationSupported(&isSupported, handle));

    return isSupported;
}

Result StartSixAxisSensorAccurateUserCalibration(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    return detail::StartSixAxisSensorAccurateUserCalibration(handle);
}

void CancelSixAxisSensorAccurateUserCalibration(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::CancelSixAxisSensorAccurateUserCalibration(handle));
}

Result GetSixAxisSensorAccurateUserCalibrationState(SixAxisSensorAccurateUserCalibrationState* pOutValue,
                                                    const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    return detail::GetSixAxisSensorAccurateUserCalibrationState(pOutValue, handle);
}

}}} // namespace nn::hid::system

namespace nn { namespace hid { namespace debug {

Result UpdateControllerColor(const nn::util::Color4u8Type& mainColor,
                             const nn::util::Color4u8Type& subColor,
                             const system::UniquePadId& id) NN_NOEXCEPT
{
    auto result = detail::UpdateControllerColor(mainColor, subColor, id);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(system::ResultUniquePadError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY;

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result UpdateDesignInfo(const ::nn::util::Color4u8Type& mainColor,
                        const ::nn::util::Color4u8Type& subColor,
                        const ::nn::util::Color4u8Type& thirdColor,
                        const ::nn::util::Color4u8Type& forthColor,
                        uint8_t variation,
                        const system::UniquePadId& id) NN_NOEXCEPT
{
    auto result = detail::UpdateDesignInfo(mainColor, subColor, thirdColor, forthColor, variation, id);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(system::ResultUniquePadError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY;

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

void ConnectUsbPadsAsync() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ConnectUsbPadsAsync());
}

void DisconnectUsbPadsAsync() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::DisconnectUsbPadsAsync());
}

void GetUniquePadDriverState(detail::PadDriverState* pOutValue, const system::UniquePadId& id) NN_NOEXCEPT
{
    auto result = detail::GetUniquePadDriverState(pOutValue, id);
    if(nn::hid::system::ResultUniquePadError().Includes(result))
    {
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

int GetSixAxisSensorDriverStates(detail::SixAxisSensorDriverState* pOutStates,
                                 int count,
                                 const system::UniquePadId& id) NN_NOEXCEPT
{
    int outCount = 0;
    auto result = detail::GetSixAxisSensorDriverStates(&outCount, pOutStates, count, id);
    if(nn::hid::system::ResultUniquePadError().Includes(result))
    {
        return 0;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return outCount;

}

void GetRxPacketHistory(detail::RxPacketHistory* pOutValue, const system::UniquePadId& id) NN_NOEXCEPT
{
    auto result = detail::GetRxPacketHistory(pOutValue, id);
    if(nn::hid::system::ResultUniquePadError().Includes(result))
    {
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

::nn::Result BindSerialFlashEventHandle(
    ::nn::os::SystemEventType* pEvent,
    ::nn::os::EventClearMode clearMode,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    NN_RESULT_TRY(detail::BindSerialFlashEventHandle(pEvent, clearMode, id));
    NN_RESULT_CATCH(nn::hid::system::ResultUniquePadError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH(nn::hid::ResultUniquePadPrivateError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
    }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

::nn::Result ReadSerialFlash(
    const uint32_t address,
    uint8_t* pOutBuffer,
    int size,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES_GREATER(size, 0);
    NN_RESULT_TRY(detail::ReadSerialFlash(address, pOutBuffer, size, id));
    NN_RESULT_CATCH(nn::hid::system::ResultUniquePadError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH(nn::hid::ResultUniquePadPrivateError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
    }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

::nn::Result WriteSerialFlash(
    const uint32_t address,
    const uint8_t* pBuffer,
    int bufferSize,
    int writeSize,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0);
    NN_SDK_REQUIRES_GREATER(writeSize, 0);
    NN_RESULT_TRY(detail::WriteSerialFlash(address, pBuffer, bufferSize, writeSize, id));
    NN_RESULT_CATCH(nn::hid::system::ResultUniquePadError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH(nn::hid::ResultUniquePadPrivateError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
    }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

::nn::Result GetSerialFlashResult(const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_RESULT_TRY(detail::GetSerialFlashResult(id));
    NN_RESULT_CATCH(nn::hid::system::ResultUniquePadError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH(nn::hid::ResultUniquePadPrivateError)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
    }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}


}}} // namespace nn::hid::debug
