﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_JoyXpad.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Quaternion.h>

#include "detail/hid_NpadImpl.h"
#include "detail/hid_SixAxisSensorHandle.h"
#include "detail/hid_SixAxisSensorImpl.h"

namespace nn { namespace hid {

void SixAxisSensorState::GetQuaternion(nn::util::Vector4fType* pOutQuaternion) const NN_NOEXCEPT
{
    const nn::hid::DirectionState* const pDirection = &this->direction;

    nn::util::MatrixRowMajor4x3f rotationMatrix4x3f(
        pDirection->x.x, pDirection->x.y, pDirection->x.z,
        pDirection->y.x, pDirection->y.y, pDirection->y.z,
        pDirection->z.x, pDirection->z.y, pDirection->z.z,
        0.0f, 0.0f, 0.0f);

    *pOutQuaternion = nn::util::Quaternion::FromRotationMatrix(rotationMatrix4x3f);
}

int GetSixAxisSensorStates(SixAxisSensorState* outStates,
                  int count,
                  const BasicXpadId& xpadId) NN_NOEXCEPT
{
    int gotCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetSixAxisSensorStates(&gotCount, outStates, count, xpadId));
    return gotCount;
}

void GetSixAxisSensorHandles(SixAxisSensorHandle* pOutLeftHandle,
                             SixAxisSensorHandle* pOutRightHandle,
                             JoyXpadId xpadId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetSixAxisSensorHandles(pOutLeftHandle,
                                        pOutRightHandle,
                                        xpadId));
}

void GetSixAxisSensorHandle(SixAxisSensorHandle* pOutHandle,
                            BasicXpadId xpadId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetSixAxisSensorHandle(pOutHandle, xpadId));
}

int GetSixAxisSensorHandles(SixAxisSensorHandle* pOutValues, int count, const NpadIdType& id, NpadStyleSet style) NN_NOEXCEPT
{
    int outCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetSixAxisSensorHandle(pOutValues, &outCount, count, id, style));

    return outCount;
}

void StartSixAxisSensor(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::StartSixAxisSensor(handle));
}

void StopSixAxisSensor(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::StopSixAxisSensor(handle));
}


bool IsSixAxisSensorFusionEnabled(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    bool isEnabled;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::IsSixAxisSensorFusionEnabled(&isEnabled,
                                                                        handle));
    return isEnabled;
}

void EnableSixAxisSensorFusion(const SixAxisSensorHandle& handle,
                               bool enable) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::EnableSixAxisSensorFusion(handle, enable));
}

void SetSixAxisSensorFusionParameters(const SixAxisSensorHandle& handle,
                                      float revisePower,
                                      float reviseRange) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::SetSixAxisSensorFusionParameters(handle,
                                                                            revisePower,
                                                                            reviseRange));
}

void GetSixAxisSensorFusionParameters(float* pOutRevisePower,
                                      float* pOutReviseRange,
                                      const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetSixAxisSensorFusionParameters(pOutRevisePower,
                                                                            pOutReviseRange,
                                                                            handle));
}


void ResetSixAxisSensorFusionParameters(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ResetSixAxisSensorFusionParameters(handle));
}

void SetAccelerometerParameters(const SixAxisSensorHandle& handle,
                                float playRadius,
                                float sensitivity) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::SetAccelerometerParameters(handle,
                                                                      playRadius,
                                                                      sensitivity));
}

void GetAccelerometerParameters(float* pOutPlayRadius,
                                float* pOutSensitivity,
                                const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetAccelerometerParameters(pOutPlayRadius,
                                                                      pOutSensitivity,
                                                                      handle));
}

void ResetAccelerometerParameters(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ResetAccelerometerParameters(handle));
}

void SetAccelerometerPlayMode(const SixAxisSensorHandle& handle,
                              const AccelerometerPlayMode& mode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::SetAccelerometerPlayMode(handle,
                                                                    mode));
}

void GetAccelerometerPlayMode(AccelerometerPlayMode* pOutMode,
                              const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetAccelerometerPlayMode(pOutMode,
                                                                    handle));
}

void ResetAccelerometerPlayMode(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ResetAccelerometerPlayMode(handle));
}

void SetGyroscopeZeroDriftMode(const SixAxisSensorHandle& handle,
                               const GyroscopeZeroDriftMode& mode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::SetGyroscopeZeroDriftMode(handle,
                                                                     mode));
}

GyroscopeZeroDriftMode GetGyroscopeZeroDriftMode(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    GyroscopeZeroDriftMode mode;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetGyroscopeZeroDriftMode(&mode,
                                                                     handle));
    return mode;
}

void ResetGyroscopeZeroDriftMode(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ResetGyroscopeZeroDriftMode(handle));
}

void GetSixAxisSensorState(SixAxisSensorState* outValue,
                           const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetSixAxisSensorState(outValue, handle));
}

int GetSixAxisSensorStates(SixAxisSensorState* outStates,
                           int count,
                           const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    int gotCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetSixAxisSensorStates(&gotCount, outStates, count, handle));
    return gotCount;
}

bool IsSixAxisSensorAtRest(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    bool isAtRest;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::IsSixAxisSensorAtRest(&isAtRest,
                                                                 handle));
    return isAtRest;
}

}} // namespace nn::hid

namespace nn { namespace hid { namespace debug {

void SetShiftAccelerometerCalibrationValue(const SixAxisSensorHandle& handle,
                                           float shiftOrigin,
                                           float shiftSensitivity) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::SetShiftAccelerometerCalibrationValue(handle, shiftOrigin, shiftSensitivity));
}

void GetShiftAccelerometerCalibrationValue(float* pOutShiftOrigin,
                                           float* pOutShiftSensitivity,
                                           const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetShiftAccelerometerCalibrationValue(pOutShiftOrigin,
                                                                                 pOutShiftSensitivity,
                                                                                 handle));
}

void SetShiftGyroscopeCalibrationValue(const SixAxisSensorHandle& handle,
                                       float shiftOrigin,
                                       float shiftSensitivity) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::SetShiftGyroscopeCalibrationValue(handle, shiftOrigin, shiftSensitivity));
}

void GetShiftGyroscopeCalibrationValue(float* pOutShiftOrigin,
                                       float* pOutShiftSensitivity,
                                       const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetShiftGyroscopeCalibrationValue(pOutShiftOrigin,
                                                                             pOutShiftSensitivity,
                                                                             handle));
}

}}} // namespace nn::hid::debug
