﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_Palma.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include "detail/hid_PalmaImpl.h"

namespace nn { namespace hid {

Result GetPalmaConnectionHandle(PalmaConnectionHandle* pOutHandle, const NpadIdType& id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHandle);

    auto result = detail::GetPalmaConnectionHandle(pOutHandle, id);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result InitializePalma(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto result = detail::InitializePalma(handle);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result BindPalmaOperationCompleteEvent(const PalmaConnectionHandle& handle,
                                      nn::os::SystemEventType* pEvent,
                                      ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pEvent);

    auto result = detail::BindPalmaOperationCompleteEvent(handle, pEvent, clearMode);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result GetPalmaOperationInfo(PalmaOperationInfo* pOutValue,
                            const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutValue);

    auto result = detail::GetPalmaOperationInfo(pOutValue, handle);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result PlayPalmaActivity(const PalmaConnectionHandle& handle, const PalmaActivityIndex& index) NN_NOEXCEPT
{
    auto result = detail::PlayPalmaActivity(handle, index);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result SetPalmaFrModeType(const PalmaConnectionHandle& handle, const PalmaFrModeType& type) NN_NOEXCEPT
{
    auto result = detail::SetPalmaFrModeType(handle, type);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result ReadPalmaStep(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto result = detail::ReadPalmaStep(handle);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result EnablePalmaStep(const PalmaConnectionHandle& handle, bool isEnabled) NN_NOEXCEPT
{
    auto result = detail::EnablePalmaStep(handle, isEnabled);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result ResetPalmaStep(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto result = detail::ResetPalmaStep(handle);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result ReadPalmaApplicationSection(const PalmaConnectionHandle& handle, int address, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_LESS_EQUAL(size, PalmaApplicationSectionAccessUnitSizeMax);
    auto result = detail::ReadPalmaApplicationSection(handle, address, size);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result WritePalmaApplicationSection(const PalmaConnectionHandle& handle, int address, size_t size, const PalmaApplicationSectionAccessBuffer& buffer) NN_NOEXCEPT
{
    auto result = detail::WritePalmaApplicationSection(handle, address, size, buffer);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result ReadPalmaUniqueCode(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto result = detail::ReadPalmaUniqueCode(handle);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result SetPalmaUniqueCodeInvalid(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto result = detail::SetPalmaUniqueCodeInvalid(handle);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result WritePalmaActivityEntry(const PalmaConnectionHandle& handle, const PalmaActivityIndex& index, const PalmaActivityEntry* pEntry) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pEntry);

    auto result = detail::WritePalmaActivityEntry(handle, index, pEntry);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result WritePalmaRgbLedPatternEntry(const PalmaConnectionHandle& handle, const PalmaRgbLedPatternIndex& index, const void* pRgbLedPatternBuffer, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pRgbLedPatternBuffer);

    auto result = detail::WritePalmaRgbLedPatternEntry(handle, index, pRgbLedPatternBuffer, size);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result WritePalmaWaveEntry(const PalmaConnectionHandle& handle, const PalmaWaveSet& waveSet, const PalmaWaveIndex& index, const void* pWaveBuffer, size_t bufferSize, size_t waveDataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pWaveBuffer);

    auto result = detail::WritePalmaWaveEntry(handle, waveSet, index, pWaveBuffer, bufferSize, waveDataSize);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result SetPalmaDataBaseIdentificationVersion(const PalmaConnectionHandle& handle, int version) NN_NOEXCEPT
{
    auto result = detail::SetPalmaDataBaseIdentificationVersion(handle, version);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result GetPalmaDataBaseIdentificationVersion(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto result = detail::GetPalmaDataBaseIdentificationVersion(handle);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result SuspendPalmaFeature(const PalmaConnectionHandle& handle, const PalmaFeatureSet& suspendFeatureSet) NN_NOEXCEPT
{
    auto result = detail::SuspendPalmaFeature(handle, suspendFeatureSet);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result ReadPalmaPlayLog(const PalmaConnectionHandle& handle, const PalmaPlayLogFieldIndex& index) NN_NOEXCEPT
{
    auto result = detail::ReadPalmaPlayLog(handle, index);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

Result ResetPalmaPlayLog(const PalmaConnectionHandle& handle, const PalmaPlayLogFieldIndex& index) NN_NOEXCEPT
{
    auto result = detail::ResetPalmaPlayLog(handle, index);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultPalma)
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

void EnablePairedPalmaConnection() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::EnablePairedPalmaConnection());
}

void DisablePairedPalmaConnection() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::DisablePairedPalmaConnection());
}

void EnableAnyPalmaConnection() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::EnableAnyPalmaConnection());
}
void DisableAnyPalmaConnection() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::DisableAnyPalmaConnection());
}

void PairPalma(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::PairPalma(handle));
}

void SetPalmaBoostMode(bool enabled) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::SetPalmaBoostMode(enabled));
}

}} // namespace nn::hid
