﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_KeyCode.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_TypedStorage.h>

namespace nn { namespace hid {

namespace {

//!< キーコードの組み立て処理キューのサイズです。
const int QueueSize = 5;

//!< キーコードの組み立てモード定義です。
struct Mode final
{
    typedef ::nn::util::BitFlagSet<32, Mode>::Flag<0>
            DeadKey;        //!< デッドキー処理モード
    typedef ::nn::util::BitFlagSet<32, Mode>::Flag<1>
            RomajiHiragana; //!< ローマ字変換（ひらがな）モード
    typedef ::nn::util::BitFlagSet<32, Mode>::Flag<2>
            RomajiKatakana; //!< ローマ字変換（カタカナ）モード
};

//!< キーコードの組み立てモードの制御フラグ集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, Mode> ModeSet;

//!< Alt コードのフィールド定義です。
struct AtlCodeField final
{
    typedef ::nn::util::BitPack<uint32_t, AtlCodeField>::Field< 0, 31, uint32_t>
            Value;              //!< Alt コードの値
    typedef ::nn::util::BitPack<uint32_t, AtlCodeField>::Field<31,  1, bool>
            AnsiCodePageFlag;   //!< ANSI コードページフラグ
};

//!< Alt コードを表す型です。
typedef ::nn::util::BitPack<uint32_t, AtlCodeField> AtlCode;

//!< Alt コードを処理します。
void ProcessAltCode(detail::KeyCodeComposerStorage* pStorage) NN_NOEXCEPT;

//!< デッドキーを処理します。
void ProcessDeadKey(detail::KeyCodeComposerStorage* pStorage) NN_NOEXCEPT;

//!< デッドキーを強制的に処理します。
void ProcessDeadKeyForcibly(detail::KeyCodeComposerStorage* pStorage
                            ) NN_NOEXCEPT;

//!< ローマ字変換を処理します。
void ProcessRomaji(detail::KeyCodeComposerStorage* pStorage) NN_NOEXCEPT;

//!< ローマ字変換を強制的に処理します。
void ProcessRomajiForcibly(detail::KeyCodeComposerStorage* pStorage
                           ) NN_NOEXCEPT;

} // namespace

namespace detail {

//!< キーコードの組み立ての内部状態を表す構造体です。
struct NN_ALIGNAS(8) KeyCodeComposerStorage
{
    uint16_t queue[QueueSize];  //!< 組み立て処理キュー
    NN_PADDING2;
    ModeSet mode;               //!< 組み立てモード
    int32_t outputCount;        //!< 組み立て中キーコード列の確定部分の長さ
    int32_t nativeCount;        //!< 組み立て中キーコード列の未確定部分の長さ
    AtlCode altCode;            //!< Alt コード
    NN_PADDING4;
};

} // namespace detail

KeyCodeComposer::KeyCodeComposer() NN_NOEXCEPT
{
    ::nn::util::Get(m_Storage) = detail::KeyCodeComposerStorage();
}

void KeyCodeComposer::SetDeadKeyMode(bool enables) NN_NOEXCEPT
{
    detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);

    if (enables != this->GetDeadKeyMode())
    {
        storage.mode.Reset();

        this->Clear();
    }

    storage.mode.Set<Mode::DeadKey>(enables);
}

bool KeyCodeComposer::GetDeadKeyMode() const NN_NOEXCEPT
{
    return ::nn::util::Get(m_Storage).mode.Test<Mode::DeadKey>();
}

void KeyCodeComposer::SetRomajiHiraganaMode(bool enables) NN_NOEXCEPT
{
    detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);

    if (enables != this->GetRomajiHiraganaMode())
    {
        storage.mode.Reset();

        this->Clear();
    }

    storage.mode.Set<Mode::RomajiHiragana>(enables);
}

bool KeyCodeComposer::GetRomajiHiraganaMode() const NN_NOEXCEPT
{
    return ::nn::util::Get(m_Storage).mode.Test<Mode::RomajiHiragana>();
}

void KeyCodeComposer::SetRomajiKatakanaMode(bool enables) NN_NOEXCEPT
{
    detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);

    if (enables != this->GetRomajiKatakanaMode())
    {
        storage.mode.Reset();

        this->Clear();
    }

    storage.mode.Set<Mode::RomajiKatakana>(enables);
}

bool KeyCodeComposer::GetRomajiKatakanaMode() const NN_NOEXCEPT
{
    return ::nn::util::Get(m_Storage).mode.Test<Mode::RomajiKatakana>();
}

int KeyCodeComposer::GetCompositionCount() const NN_NOEXCEPT
{
    const detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           0, QueueSize);
    return (storage.outputCount + storage.nativeCount);
}

int KeyCodeComposer::GetComposition(uint16_t outBuffer[], int count
                                    ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    const auto valueCount = ::std::min<int>(count, this->GetCompositionCount());

    if (valueCount <= 0)
    {
        return 0;
    }

    const detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);

    ::std::copy(storage.queue, storage.queue + valueCount, outBuffer);

    return valueCount;
}

int KeyCodeComposer::GetOutputCount() const NN_NOEXCEPT
{
    const detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           0, QueueSize);
    return storage.outputCount;
}

int KeyCodeComposer::DequeueOutputs(uint16_t outBuffer[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    const auto valueCount = ::std::min<int>(count, this->GetOutputCount());

    if (valueCount <= 0)
    {
        return 0;
    }

    detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);

    ::std::copy(storage.queue, storage.queue + valueCount, outBuffer);

    ::std::copy(storage.queue + valueCount,
                storage.queue + this->GetCompositionCount(),
                storage.queue);

    storage.outputCount -= static_cast<int32_t>(valueCount);

    return valueCount;
}

int KeyCodeComposer::EnqueueKeyCodes(const uint16_t buffer[], int count
                                     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    if (0 < this->GetOutputCount())
    {
        return 0;
    }

    ProcessAltCode(&::nn::util::Get(m_Storage));

    detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);

    int valueCount = 0;

    while (valueCount < count && this->GetCompositionCount() < QueueSize)
    {
        storage.queue[this->GetCompositionCount()] = buffer[valueCount];

        ++storage.nativeCount;

        if (storage.mode.Test<Mode::DeadKey>())
        {
            ProcessDeadKey(&storage);
        }
        else if (storage.mode.Test<Mode::RomajiHiragana>() ||
                 storage.mode.Test<Mode::RomajiKatakana>())
        {
            ProcessRomaji(&storage);
        }
        else
        {
            storage.outputCount = this->GetCompositionCount();
            storage.nativeCount = 0;
        }

        ++valueCount;

        if (0 < this->GetOutputCount())
        {
            break;
        }
    }

    return valueCount;
}

int KeyCodeComposer::RemoveKeyCodes(uint16_t outBuffer[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);

    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           0, QueueSize);

    const auto valueCount = ::std::min<int>(count, storage.nativeCount);

    if (valueCount <= 0)
    {
        return 0;
    }

    ::std::copy(storage.queue + this->GetCompositionCount() - valueCount,
                storage.queue + this->GetCompositionCount(),
                outBuffer);

    storage.nativeCount -= static_cast<int32_t>(valueCount);

    return valueCount;
}

void KeyCodeComposer::FlushKeyCodes() NN_NOEXCEPT
{
    detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);

    ProcessAltCode(&storage);

    if (storage.mode.Test<Mode::DeadKey>())
    {
        ProcessDeadKeyForcibly(&storage);
    }
    else if (storage.mode.Test<Mode::RomajiHiragana>() ||
             storage.mode.Test<Mode::RomajiKatakana>())
    {
        ProcessRomajiForcibly(&storage);
    }
    else
    {
        storage.outputCount = this->GetCompositionCount();
        storage.nativeCount = 0;
    }
}

int KeyCodeComposer::EnqueueAltCodes(const int buffer[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    if (0 < this->GetOutputCount())
    {
        return 0;
    }

    if (QueueSize <= this->GetCompositionCount())
    {
        return 0;
    }

    AtlCode& altCode = ::nn::util::Get(m_Storage).altCode;

    for (int i = 0; i < count; ++i)
    {
        NN_SDK_REQUIRES_MINMAX(buffer[i], 0, 9);

        const auto value = static_cast<uint32_t>(buffer[i]);

        uint32_t code = altCode.Get<AtlCodeField::Value>();

        if (code == 0u)
        {
            if (value == 0u)
            {
                altCode.Set<AtlCodeField::AnsiCodePageFlag>(true);
            }
            else
            {
                altCode.Set<AtlCodeField::Value>(value);
            }
        }
        else
        {
            if (code > ((1 << 30) / 10))
            {
                break;
            }
            else
            {
                altCode.Set<AtlCodeField::Value>(code * 10 + value);
            }
        }
    }

    return count;
}

void KeyCodeComposer::FlushAltCodes() NN_NOEXCEPT
{
    ProcessAltCode(&::nn::util::Get(m_Storage));
}

void KeyCodeComposer::Clear() NN_NOEXCEPT
{
    detail::KeyCodeComposerStorage& storage = ::nn::util::Get(m_Storage);
    const ModeSet mode = storage.mode;
    storage = detail::KeyCodeComposerStorage();
    storage.mode = mode;
}

namespace {

//!< ANSI コードページ
const uint16_t AnsiCodePage[] =
{
    0x20AC, 0x0000, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
    0x02C6, 0x2030, 0x0160, 0x2039, 0x0152, 0x0000, 0x017D, 0x0000,
    0x0000, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
    0x02DC, 0x2122, 0x0161, 0x203A, 0x0153, 0x0000, 0x017E, 0x0178,
};

//!< OEM コードページ
const uint16_t OemCodePage[] =
{
    0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7,
    0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x00EC, 0x00C4, 0x00C5,
    0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00F2, 0x00FB, 0x00F9,
    0x00FF, 0x00D6, 0x00DC, 0x00A2, 0x00A3, 0x00A5, 0x20A7, 0x0192,
    0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA,
    0x00BF, 0x2310, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB,
    0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
    0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510,
    0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F,
    0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567,
    0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B,
    0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580,
    0x03B1, 0x00DF, 0x0393, 0x03C0, 0x03A3, 0x03C3, 0x00B5, 0x03C4,
    0x03A6, 0x0398, 0x03A9, 0x03B4, 0x221E, 0x03C6, 0x03B5, 0x2229,
    0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248,
    0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0x00A0
};

//!< デッドキー処理表
const uint16_t DeadKeyMap[][3] =
{
    { 0x0300, 0x0300, 0x0060 }, // ` + ` => `
    { 0x0300, 0x0041, 0x00C0 }, // ` + A => À
    { 0x0300, 0x0045, 0x00C8 }, // ` + E => È
    { 0x0300, 0x0049, 0x00CC }, // ` + I => Ì
    { 0x0300, 0x004F, 0x00D2 }, // ` + O => Ò
    { 0x0300, 0x0055, 0x00D9 }, // ` + U => Ù

    { 0x0300, 0x0061, 0x00E0 }, // ` + a => à
    { 0x0300, 0x0065, 0x00E8 }, // ` + e => è
    { 0x0300, 0x0069, 0x00EC }, // ` + i => ì
    { 0x0300, 0x006F, 0x00F2 }, // ` + o => ò
    { 0x0300, 0x0075, 0x00F9 }, // ` + u => ù

    { 0x0301, 0x0301, 0x00B4 }, // ´ + ´ => ´
    { 0x0301, 0x0041, 0x00C1 }, // ´ + A => Á
    { 0x0301, 0x0045, 0x00C9 }, // ´ + E => É
    { 0x0301, 0x0049, 0x00CD }, // ´ + I => Í
    { 0x0301, 0x004F, 0x00D3 }, // ´ + O => Ó
    { 0x0301, 0x0055, 0x00DA }, // ´ + U => Ú
    { 0x0301, 0x0059, 0x00DD }, // ´ + Y => Ý

    { 0x0301, 0x0061, 0x00E1 }, // ´ + a => á
    { 0x0301, 0x0065, 0x00E9 }, // ´ + e => é
    { 0x0301, 0x0069, 0x00ED }, // ´ + i => í
    { 0x0301, 0x006F, 0x00F3 }, // ´ + o => ó
    { 0x0301, 0x0075, 0x00FA }, // ´ + u => ú
    { 0x0301, 0x0079, 0x00FD }, // ´ + y => ý

    { 0x0302, 0x0302, 0x005E }, // ^ + ^ => ^
    { 0x0302, 0x0041, 0x00C2 }, // ^ + A => Â
    { 0x0302, 0x0045, 0x00CA }, // ^ + E => Ê
    { 0x0302, 0x0049, 0x00CE }, // ^ + I => Î
    { 0x0302, 0x004F, 0x00D4 }, // ^ + O => Ô
    { 0x0302, 0x0055, 0x00DB }, // ^ + U => Û

    { 0x0302, 0x0061, 0x00E2 }, // ^ + a => â
    { 0x0302, 0x0065, 0x00EA }, // ^ + e => ê
    { 0x0302, 0x0069, 0x00EE }, // ^ + i => î
    { 0x0302, 0x006F, 0x00F4 }, // ^ + o => ô
    { 0x0302, 0x0075, 0x00FB }, // ^ + u => û

    { 0x0303, 0x0303, 0x007E }, // ~ + ~ => ~
    { 0x0303, 0x0041, 0x00C3 }, // ~ + A => Ã
    { 0x0303, 0x004E, 0x00D1 }, // ~ + N => Ñ
    { 0x0303, 0x004F, 0x00D5 }, // ~ + O => Õ
    { 0x0303, 0x0061, 0x00E3 }, // ~ + a => ã
    { 0x0303, 0x006E, 0x00F1 }, // ~ + n => ñ
    { 0x0303, 0x006F, 0x00F5 }, // ~ + o => õ

    { 0x0308, 0x0308, 0x00A8 }, // ¨ + ¨ => ¨
    { 0x0308, 0x0041, 0x00C4 }, // ¨ + A => Ä
    { 0x0308, 0x0045, 0x00CB }, // ¨ + E => Ë
    { 0x0308, 0x0049, 0x00CF }, // ¨ + I => Ï
    { 0x0308, 0x004F, 0x00D6 }, // ¨ + O => Ö
    { 0x0308, 0x0055, 0x00DC }, // ¨ + U => Ü
    { 0x0308, 0x0059, 0x0178 }, // ¨ + Y => Ÿ

    { 0x0308, 0x0061, 0x00E4 }, // ¨ + a => ä
    { 0x0308, 0x0065, 0x00EB }, // ¨ + e => ë
    { 0x0308, 0x0069, 0x00EF }, // ¨ + i => ï
    { 0x0308, 0x006F, 0x00F6 }, // ¨ + o => ö
    { 0x0308, 0x0075, 0x00FC }, // ¨ + u => ü
    { 0x0308, 0x0079, 0x00FF }, // ¨ + y => ÿ

    { 0x0327, 0x0327, 0x00B8 }, // ¸ + ¸ => ¸
    { 0x0327, 0x0043, 0x00C7 }, // ¸ + C => Ç
    { 0x0327, 0x0063, 0x00E7 }, // ¸ + c => ç

    // US-international キーボード用
    { 0x030D, 0x030D, 0x0027 }, // ' + ' => '
    { 0x030D, 0x0041, 0x00C1 }, // ' + A => Á
    { 0x030D, 0x0045, 0x00C9 }, // ' + E => É
    { 0x030D, 0x0049, 0x00CD }, // ' + I => Í
    { 0x030D, 0x004F, 0x00D3 }, // ' + O => Ó
    { 0x030D, 0x0055, 0x00DA }, // ' + U => Ú
    { 0x030D, 0x0059, 0x00DD }, // ' + Y => Ý
    { 0x030D, 0x0043, 0x00C7 }, // ' + C => Ç

    { 0x030D, 0x0061, 0x00E1 }, // ' + a => á
    { 0x030D, 0x0065, 0x00E9 }, // ' + e => é
    { 0x030D, 0x0069, 0x00ED }, // ' + i => í
    { 0x030D, 0x006F, 0x00F3 }, // ' + o => ó
    { 0x030D, 0x0075, 0x00FA }, // ' + u => ú
    { 0x030D, 0x0079, 0x00FD }, // ' + y => ý
    { 0x030D, 0x0063, 0x00E7 }, // ' + c => ç

    { 0x030E, 0x030E, 0x0022 }, // " + " => "
    { 0x030E, 0x0041, 0x00C4 }, // " + A => Ä
    { 0x030E, 0x0045, 0x00CB }, // " + E => Ë
    { 0x030E, 0x0049, 0x00CF }, // " + I => Ï
    { 0x030E, 0x004F, 0x00D6 }, // " + O => Ö
    { 0x030E, 0x0055, 0x00DC }, // " + U => Ü
    { 0x030E, 0x0059, 0x0178 }, // " + Y => Ÿ

    { 0x030E, 0x0061, 0x00E4 }, // " + a => ä
    { 0x030E, 0x0065, 0x00EB }, // " + e => ë
    { 0x030E, 0x0069, 0x00EF }, // " + i => ï
    { 0x030E, 0x006F, 0x00F6 }, // " + o => ö
    { 0x030E, 0x0075, 0x00FC }, // " + u => ü
    { 0x030E, 0x0079, 0x00FF }, // " + y => ÿ

    // ギリシャ文字
    { 0x0344, 0x0344, 0x0385 }, // ΅ + ΅ => ΅
    { 0x0344, 0x03B9, 0x0390 }, // ΅ + ι => ΐ
    { 0x0344, 0x03C5, 0x03B0 }, // ΅ + υ => ΰ

    { 0x0308, 0x0399, 0x03AA }, // ¨ + Ι => Ϊ
    { 0x0308, 0x03A5, 0x03AB }, // ¨ + Υ => Ϋ

    { 0x0308, 0x03B9, 0x03CA }, // ¨ + ι => ϊ
    { 0x0308, 0x03C5, 0x03CB }, // ¨ + υ => ϋ

    { 0x0301, 0x0391, 0x0386 }, // ´ + Α => Ά
    { 0x0301, 0x0395, 0x0388 }, // ´ + Ε => Έ
    { 0x0301, 0x0397, 0x0389 }, // ´ + Η => Ή
    { 0x0301, 0x0399, 0x038A }, // ´ + Ι => Ί
    { 0x0301, 0x039F, 0x038C }, // ´ + Ο => Ό
    { 0x0301, 0x03A5, 0x038E }, // ´ + Υ => Ύ
    { 0x0301, 0x03A9, 0x038F }, // ´ + Ω => Ώ

    { 0x0301, 0x03B1, 0x03AC }, // ´ + α => ά
    { 0x0301, 0x03B5, 0x03AD }, // ´ + ε => έ
    { 0x0301, 0x03B7, 0x03AE }, // ´ + η => ή
    { 0x0301, 0x03B9, 0x03AF }, // ´ + ι => ί
    { 0x0301, 0x03BF, 0x03CC }, // ´ + ο => ό
    { 0x0301, 0x03C5, 0x03CD }, // ´ + υ => ύ
    { 0x0301, 0x03C9, 0x03CE }, // ´ + ω => ώ
};

//!< ローマ字 "っ" 変換表
const uint16_t RomajiSmallTsuMap[] =
{
    0x3063, // っ
    0x30C3, // ッ
};

//!< ローマ字 "ん" 変換表
const uint16_t RomajiNNMap[] =
{
    0x3093, // ん
    0x30F3, // ン
};

//!< ローマ字変換表のエントリを表す構造体です。
struct RomajiMapEntry
{
    int32_t  nativeCount;
    char     native[8];
    int32_t  outputCount;
    uint16_t output[2];
};

//!< ローマ字変換表
const RomajiMapEntry RomajiMap[] =
{
    { 4, "ltsa", 2, { 0x3063, 0x3041 } },
    { 4, "ltse", 2, { 0x3063, 0x3047 } },
    { 4, "ltsi", 2, { 0x3063, 0x3043 } },
    { 4, "ltso", 2, { 0x3063, 0x3049 } },
    { 4, "ltsu", 1, { 0x3063,        } },

    { 4, "lwha", 2, { 0x3045, 0x3041 } },
    { 4, "lwhe", 2, { 0x3045, 0x3047 } },
    { 4, "lwhi", 2, { 0x3045, 0x3043 } },
    { 4, "lwho", 2, { 0x3045, 0x3049 } },
    { 4, "lwhu", 1, { 0x3045,        } },

    { 4, "xtsa", 2, { 0x3063, 0x3041 } },
    { 4, "xtse", 2, { 0x3063, 0x3047 } },
    { 4, "xtsi", 2, { 0x3063, 0x3043 } },
    { 4, "xtso", 2, { 0x3063, 0x3049 } },
    { 4, "xtsu", 1, { 0x3063,        } },

    { 4, "xwha", 2, { 0x3045, 0x3041 } },
    { 4, "xwhe", 2, { 0x3045, 0x3047 } },
    { 4, "xwhi", 2, { 0x3045, 0x3043 } },
    { 4, "xwho", 2, { 0x3045, 0x3049 } },
    { 4, "xwhu", 1, { 0x3045,        } },

    { 3, "bya", 2, { 0x3073, 0x3083 } },
    { 3, "bye", 2, { 0x3073, 0x3047 } },
    { 3, "byi", 2, { 0x3073, 0x3043 } },
    { 3, "byo", 2, { 0x3073, 0x3087 } },
    { 3, "byu", 2, { 0x3073, 0x3085 } },

    { 3, "cha", 2, { 0x3061, 0x3083 } },
    { 3, "che", 2, { 0x3061, 0x3047 } },
    { 3, "chi", 1, { 0x3061,        } },
    { 3, "cho", 2, { 0x3061, 0x3087 } },
    { 3, "chu", 2, { 0x3061, 0x3085 } },

    { 3, "cya", 2, { 0x3061, 0x3083 } },
    { 3, "cye", 2, { 0x3061, 0x3047 } },
    { 3, "cyi", 2, { 0x3061, 0x3043 } },
    { 3, "cyo", 2, { 0x3061, 0x3087 } },
    { 3, "cyu", 2, { 0x3061, 0x3085 } },

    { 3, "dha", 2, { 0x3067, 0x3083 } },
    { 3, "dhe", 2, { 0x3067, 0x3047 } },
    { 3, "dhi", 2, { 0x3067, 0x3043 } },
    { 3, "dho", 2, { 0x3067, 0x3087 } },
    { 3, "dhu", 2, { 0x3067, 0x3085 } },

    { 3, "dya", 2, { 0x3062, 0x3083 } },
    { 3, "dyi", 2, { 0x3062, 0x3043 } },
    { 3, "dye", 2, { 0x3062, 0x3047 } },
    { 3, "dyo", 2, { 0x3062, 0x3087 } },
    { 3, "dyu", 2, { 0x3062, 0x3085 } },

    { 3, "fya", 2, { 0x3075, 0x3083 } },
    { 3, "fye", 2, { 0x3075, 0x3047 } },
    { 3, "fyi", 2, { 0x3075, 0x3043 } },
    { 3, "fyo", 2, { 0x3075, 0x3087 } },
    { 3, "fyu", 2, { 0x3075, 0x3085 } },

    { 3, "gya", 2, { 0x304E, 0x3083 } },
    { 3, "gye", 2, { 0x304E, 0x3047 } },
    { 3, "gyi", 2, { 0x304E, 0x3043 } },
    { 3, "gyo", 2, { 0x304E, 0x3087 } },
    { 3, "gyu", 2, { 0x304E, 0x3085 } },

    { 3, "hya", 2, { 0x3072, 0x3083 } },
    { 3, "hye", 2, { 0x3072, 0x3047 } },
    { 3, "hyi", 2, { 0x3072, 0x3043 } },
    { 3, "hyo", 2, { 0x3072, 0x3087 } },
    { 3, "hyu", 2, { 0x3072, 0x3085 } },

    { 3, "jya", 2, { 0x3058, 0x3083 } },
    { 3, "jye", 2, { 0x3058, 0x3047 } },
    { 3, "jyi", 2, { 0x3058, 0x3043 } },
    { 3, "jyo", 2, { 0x3058, 0x3087 } },
    { 3, "jyu", 2, { 0x3058, 0x3085 } },

    { 3, "kya", 2, { 0x304D, 0x3083 } },
    { 3, "kye", 2, { 0x304D, 0x3047 } },
    { 3, "kyi", 2, { 0x304D, 0x3043 } },
    { 3, "kyo", 2, { 0x304D, 0x3087 } },
    { 3, "kyu", 2, { 0x304D, 0x3085 } },

    { 3, "lya", 1, { 0x3083 } },
    { 3, "lye", 1, { 0x3047 } },
    { 3, "lyi", 1, { 0x3043 } },
    { 3, "lyo", 1, { 0x3087 } },
    { 3, "lyu", 1, { 0x3085 } },

    { 3, "ltu", 1, { 0x3063 } },

    { 3, "lwa", 1, { 0x308E } },

    { 3, "mya", 2, { 0x307F, 0x3083 } },
    { 3, "mye", 2, { 0x307F, 0x3047 } },
    { 3, "myi", 2, { 0x307F, 0x3043 } },
    { 3, "myo", 2, { 0x307F, 0x3087 } },
    { 3, "myu", 2, { 0x307F, 0x3085 } },

    { 3, "nya", 2, { 0x306B, 0x3083 } },
    { 3, "nye", 2, { 0x306B, 0x3047 } },
    { 3, "nyi", 2, { 0x306B, 0x3043 } },
    { 3, "nyo", 2, { 0x306B, 0x3087 } },
    { 3, "nyu", 2, { 0x306B, 0x3085 } },

    { 3, "pya", 2, { 0x3074, 0x3083 } },
    { 3, "pye", 2, { 0x3074, 0x3047 } },
    { 3, "pyi", 2, { 0x3074, 0x3043 } },
    { 3, "pyo", 2, { 0x3074, 0x3087 } },
    { 3, "pyu", 2, { 0x3074, 0x3085 } },

    { 3, "qya", 2, { 0x304F, 0x3083 } },
    { 3, "qye", 2, { 0x304F, 0x3047 } },
    { 3, "qyi", 2, { 0x304F, 0x3043 } },
    { 3, "qyo", 2, { 0x304F, 0x3087 } },
    { 3, "qyu", 2, { 0x304F, 0x3085 } },

    { 3, "rya", 2, { 0x308A, 0x3083 } },
    { 3, "rye", 2, { 0x308A, 0x3047 } },
    { 3, "ryi", 2, { 0x308A, 0x3043 } },
    { 3, "ryo", 2, { 0x308A, 0x3087 } },
    { 3, "ryu", 2, { 0x308A, 0x3085 } },

    { 3, "sha", 2, { 0x3057, 0x3083 } },
    { 3, "she", 2, { 0x3057, 0x3047 } },
    { 3, "shi", 1, { 0x3057,        } },
    { 3, "sho", 2, { 0x3057, 0x3087 } },
    { 3, "shu", 2, { 0x3057, 0x3085 } },

    { 3, "sya", 2, { 0x3057, 0x3083 } },
    { 3, "sye", 2, { 0x3057, 0x3047 } },
    { 3, "syi", 2, { 0x3057, 0x3043 } },
    { 3, "syo", 2, { 0x3057, 0x3087 } },
    { 3, "syu", 2, { 0x3057, 0x3085 } },

    { 3, "tha", 2, { 0x3066, 0x3083 } },
    { 3, "the", 2, { 0x3066, 0x3047 } },
    { 3, "thi", 2, { 0x3066, 0x3043 } },
    { 3, "tho", 2, { 0x3066, 0x3087 } },
    { 3, "thu", 2, { 0x3066, 0x3085 } },

    { 3, "tsa", 2, { 0x3064, 0x3041 } },
    { 3, "tse", 2, { 0x3064, 0x3047 } },
    { 3, "tsi", 2, { 0x3064, 0x3043 } },
    { 3, "tso", 2, { 0x3064, 0x3049 } },
    { 3, "tsu", 1, { 0x3064,        } },

    { 3, "tya", 2, { 0x3061, 0x3083 } },
    { 3, "tye", 2, { 0x3061, 0x3047 } },
    { 3, "tyi", 2, { 0x3061, 0x3043 } },
    { 3, "tyo", 2, { 0x3061, 0x3087 } },
    { 3, "tyu", 2, { 0x3061, 0x3085 } },

    { 3, "vya", 2, { 0x30F4, 0x3083 } },
    { 3, "vye", 2, { 0x30F4, 0x3047 } },
    { 3, "vyi", 2, { 0x30F4, 0x3043 } },
    { 3, "vyo", 2, { 0x30F4, 0x3087 } },
    { 3, "vyu", 2, { 0x30F4, 0x3085 } },

    { 3, "wha", 2, { 0x3046, 0x3041 } },
    { 3, "whe", 2, { 0x3046, 0x3047 } },
    { 3, "whi", 2, { 0x3046, 0x3043 } },
    { 3, "who", 2, { 0x3046, 0x3049 } },
    { 3, "whu", 1, { 0x3046,        } },

    { 3, "xya", 1, { 0x3083 } },
    { 3, "xye", 1, { 0x3047 } },
    { 3, "xyi", 1, { 0x3043 } },
    { 3, "xyo", 1, { 0x3087 } },
    { 3, "xyu", 1, { 0x3085 } },

    { 3, "xtu", 1, { 0x3063 } },

    { 3, "xwa", 1, { 0x308E } },

    { 3, "zya", 2, { 0x3058, 0x3083 } },
    { 3, "zye", 2, { 0x3058, 0x3047 } },
    { 3, "zyi", 2, { 0x3058, 0x3043 } },
    { 3, "zyo", 2, { 0x3058, 0x3087 } },
    { 3, "zyu", 2, { 0x3058, 0x3085 } },

    { 2, "ba", 1, { 0x3070 } },
    { 2, "be", 1, { 0x3079 } },
    { 2, "bi", 1, { 0x3073 } },
    { 2, "bo", 1, { 0x307C } },
    { 2, "bu", 1, { 0x3076 } },

    { 2, "ca", 1, { 0x304B } },
    { 2, "ce", 1, { 0x305B } },
    { 2, "ci", 1, { 0x3057 } },
    { 2, "co", 1, { 0x3053 } },
    { 2, "cu", 1, { 0x304F } },

    { 2, "da", 1, { 0x3060 } },
    { 2, "de", 1, { 0x3067 } },
    { 2, "di", 1, { 0x3062 } },
    { 2, "do", 1, { 0x3069 } },
    { 2, "du", 1, { 0x3065 } },

    { 2, "fa", 2, { 0x3075, 0x3041 } },
    { 2, "fe", 2, { 0x3075, 0x3047 } },
    { 2, "fi", 2, { 0x3075, 0x3043 } },
    { 2, "fo", 2, { 0x3075, 0x3049 } },
    { 2, "fu", 1, { 0x3075,        } },

    { 2, "ga", 1, { 0x304C } },
    { 2, "ge", 1, { 0x3052 } },
    { 2, "gi", 1, { 0x304E } },
    { 2, "go", 1, { 0x3054 } },
    { 2, "gu", 1, { 0x3050 } },

    { 2, "ha", 1, { 0x306F } },
    { 2, "he", 1, { 0x3078 } },
    { 2, "hi", 1, { 0x3072 } },
    { 2, "ho", 1, { 0x307B } },
    { 2, "hu", 1, { 0x3075 } },

    { 2, "ja", 2, { 0x3058, 0x3083 } },
    { 2, "je", 2, { 0x3058, 0x3047 } },
    { 2, "ji", 1, { 0x3058,        } },
    { 2, "jo", 2, { 0x3058, 0x3087 } },
    { 2, "ju", 2, { 0x3058, 0x3085 } },

    { 2, "ka", 1, { 0x304B } },
    { 2, "ke", 1, { 0x3051 } },
    { 2, "ki", 1, { 0x304D } },
    { 2, "ko", 1, { 0x3053 } },
    { 2, "ku", 1, { 0x304F } },

    { 2, "la", 1, { 0x3041 } },
    { 2, "le", 1, { 0x3047 } },
    { 2, "li", 1, { 0x3043 } },
    { 2, "lo", 1, { 0x3049 } },
    { 2, "lu", 1, { 0x3045 } },

    { 2, "ma", 1, { 0x307E } },
    { 2, "me", 1, { 0x3081 } },
    { 2, "mi", 1, { 0x307F } },
    { 2, "mo", 1, { 0x3082 } },
    { 2, "mu", 1, { 0x3080 } },

    { 2, "na", 1, { 0x306A } },
    { 2, "ne", 1, { 0x306D } },
    { 2, "ni", 1, { 0x306B } },
    { 2, "no", 1, { 0x306E } },
    { 2, "nu", 1, { 0x306C } },

    { 2, "pa", 1, { 0x3071 } },
    { 2, "pe", 1, { 0x307A } },
    { 2, "pi", 1, { 0x3074 } },
    { 2, "po", 1, { 0x307D } },
    { 2, "pu", 1, { 0x3077 } },

    { 2, "qa", 2, { 0x304F, 0x3041 } },
    { 2, "qe", 2, { 0x304F, 0x3047 } },
    { 2, "qi", 2, { 0x304F, 0x3043 } },
    { 2, "qo", 2, { 0x304F, 0x3049 } },
    { 2, "qu", 1, { 0x304F,        } },

    { 2, "ra", 1, { 0x3089 } },
    { 2, "re", 1, { 0x308C } },
    { 2, "ri", 1, { 0x308A } },
    { 2, "ro", 1, { 0x308D } },
    { 2, "ru", 1, { 0x308B } },

    { 2, "sa", 1, { 0x3055 } },
    { 2, "se", 1, { 0x305B } },
    { 2, "si", 1, { 0x3057 } },
    { 2, "so", 1, { 0x305D } },
    { 2, "su", 1, { 0x3059 } },

    { 2, "ta", 1, { 0x305F } },
    { 2, "te", 1, { 0x3066 } },
    { 2, "ti", 1, { 0x3061 } },
    { 2, "to", 1, { 0x3068 } },
    { 2, "tu", 1, { 0x3064 } },

    { 2, "va", 2, { 0x30F4, 0x3041 } },
    { 2, "ve", 2, { 0x30F4, 0x3047 } },
    { 2, "vi", 2, { 0x30F4, 0x3043 } },
    { 2, "vo", 2, { 0x30F4, 0x3049 } },
    { 2, "vu", 1, { 0x30F4,        } },

    { 2, "wa", 1, { 0x308F,        } },
    { 2, "we", 2, { 0x3046, 0x3047 } },
    { 2, "wi", 2, { 0x3046, 0x3043 } },
    { 2, "wo", 1, { 0x3092,        } },
    { 2, "wu", 1, { 0x3046,        } },

    { 2, "xa", 1, { 0x3041 } },
    { 2, "xe", 1, { 0x3047 } },
    { 2, "xi", 1, { 0x3043 } },
    { 2, "xo", 1, { 0x3049 } },
    { 2, "xu", 1, { 0x3045 } },

    { 2, "ya", 1, { 0x3084,        } },
    { 2, "ye", 2, { 0x3044, 0x3047 } },
    { 2, "yi", 1, { 0x3044,        } },
    { 2, "yo", 1, { 0x3088,        } },
    { 2, "yu", 1, { 0x3086,        } },

    { 2, "za", 1, { 0x3056 } },
    { 2, "ze", 1, { 0x305C } },
    { 2, "zi", 1, { 0x3058 } },
    { 2, "zo", 1, { 0x305E } },
    { 2, "zu", 1, { 0x305A } },

    { 2, "nn", 1, { 0x3093 } },

    { 1, "a", 1, { 0x3042 } },
    { 1, "e", 1, { 0x3048 } },
    { 1, "i", 1, { 0x3044 } },
    { 1, "o", 1, { 0x304A } },
    { 1, "u", 1, { 0x3046 } },
};

void ProcessAltCode(detail::KeyCodeComposerStorage* pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    detail::KeyCodeComposerStorage& storage = *pStorage;

    AtlCode& altCode = storage.altCode;

    uint32_t code = altCode.Get<AtlCodeField::Value>();

    if (code != 0)
    {
        NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize - 1);
        NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 0, QueueSize);
        NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                               0, QueueSize - 1);

        if (128 <= code && code <= 255)
        {
            if (altCode.Get<AtlCodeField::AnsiCodePageFlag>())
            {
                if (code < 160)
                {
                    code = AnsiCodePage[code - 128];
                }
            }
            else
            {
                code = OemCodePage[code - 128];
            }
        }
        else if (255 < code)
        {
            if (((1 << 30) / 10) < code)
            {
                code = static_cast<uint32_t>(' ');
            }
            else
            {
                code &= 0xff;
            }
        }

        ::std::copy_backward(
            storage.queue + storage.outputCount,
            storage.queue + storage.outputCount + storage.nativeCount,
            storage.queue + storage.outputCount + storage.nativeCount + 1);

        storage.queue[storage.outputCount] = static_cast<uint16_t>(code);

        ++storage.outputCount;
    }

    altCode.Clear();
}

void ProcessDeadKey(detail::KeyCodeComposerStorage* pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    detail::KeyCodeComposerStorage& storage = *pStorage;

    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize - 1);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 1, 2);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           1, QueueSize);

    if (storage.nativeCount == 1)
    {
        for (const uint16_t (&entry)[3] : DeadKeyMap)
        {
            if (storage.queue[storage.outputCount] == entry[0])
            {
                return;
            }
        }

        storage.outputCount += 1;
        storage.nativeCount -= 1;
        return;
    }

    if (storage.nativeCount == 2)
    {
        uint16_t spacingDeadKeys[2] = {};

        for (const uint16_t (&entry)[3] : DeadKeyMap)
        {
            if (storage.queue[storage.outputCount    ] == entry[0] &&
                storage.queue[storage.outputCount + 1] == entry[1])
            {
                storage.queue[storage.outputCount] = entry[2];
                storage.outputCount += 1;
                storage.nativeCount -= 2;
                return;
            }

            if (storage.queue[storage.outputCount] == entry[0] &&
                storage.queue[storage.outputCount] == entry[1])
            {
                spacingDeadKeys[0] = entry[2];
            }

            if (storage.queue[storage.outputCount + 1] == entry[0] &&
                storage.queue[storage.outputCount + 1] == entry[1])
            {
                spacingDeadKeys[1] = entry[2];
            }
        }

        storage.queue[storage.outputCount] = spacingDeadKeys[0];

        if (spacingDeadKeys[1] > 0)
        {
            storage.queue[storage.outputCount + 1] = spacingDeadKeys[1];
        }

        storage.outputCount += 2;
        storage.nativeCount -= 2;
        return;
    }
}

void ProcessDeadKeyForcibly(detail::KeyCodeComposerStorage* pStorage
                            ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    detail::KeyCodeComposerStorage& storage = *pStorage;

    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           0, QueueSize);

    for (int i = 0; i < storage.nativeCount; ++i)
    {
        for (const uint16_t (&entry)[3] : DeadKeyMap)
        {
            uint16_t& value = storage.queue[storage.outputCount + i];

            if (value == entry[0] && value == entry[1])
            {
                value = entry[2];

                break;
            }
        }
    }

    storage.outputCount += storage.nativeCount;
    storage.nativeCount = 0;
}

//!< 非英字についてローマ字変換を処理します。
bool ProcessRomajiNonLetter(detail::KeyCodeComposerStorage* pStorage
                            ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    detail::KeyCodeComposerStorage& storage = *pStorage;

    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize - 1);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 1, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           1, QueueSize);

    const int tail = storage.outputCount + storage.nativeCount - 1;

    if (storage.queue[tail] >= static_cast<uint16_t>('a') &&
        storage.queue[tail] <= static_cast<uint16_t>('z'))
    {
        return false;
    }
    else
    {
        if (storage.nativeCount > 1)
        {
            if (storage.queue[tail - 1] == static_cast<uint16_t>('n'))
            {
                if (storage.mode.Test<Mode::RomajiHiragana>())
                {
                    storage.queue[tail - 1] = RomajiNNMap[0];
                }
                else if (storage.mode.Test<Mode::RomajiKatakana>())
                {
                    storage.queue[tail - 1] = RomajiNNMap[1];
                }
            }
        }

        storage.outputCount += storage.nativeCount;
        storage.nativeCount = 0;

        return true;
    }
}

//!< ローマ字変換表のエントリを返します。
const RomajiMapEntry* GetRomajiMapEntry(
    const detail::KeyCodeComposerStorage* pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    const detail::KeyCodeComposerStorage& storage = *pStorage;

    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize - 1);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 1, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           1, QueueSize);

    const uint16_t* queueTail =
        &storage.queue[storage.outputCount + storage.nativeCount - 1];

    for (const RomajiMapEntry& entry : RomajiMap)
    {
        if (storage.nativeCount >= entry.nativeCount)
        {
            const RomajiMapEntry* pEntry = &entry;

            const char* entryTail = &entry.native[entry.nativeCount - 1];

            for (int i = 0; i < entry.nativeCount; ++i)
            {
                if (*(queueTail - i) != static_cast<uint16_t>(*(entryTail - i)))
                {
                    pEntry = nullptr;

                    break;
                }
            }

            if (pEntry != nullptr)
            {
                return pEntry;
            }
        }
    }

    return nullptr;
}

//!< キーコードをひらがなからカタカナへ変換します。
uint16_t ConvertHiraganaToKatakana(uint16_t value) NN_NOEXCEPT
{
    return (value < 0x30A1) ? static_cast<uint16_t>(value + 0x60u) : value;
}

//!< ローマ字変換表に合致したローマ字変換を処理します。
bool ProcessRomajiMap(detail::KeyCodeComposerStorage* pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    detail::KeyCodeComposerStorage& storage = *pStorage;

    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize - 1);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 1, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           1, QueueSize);

    const RomajiMapEntry* pEntry = GetRomajiMapEntry(&storage);

    if (pEntry == nullptr)
    {
        return false;
    }

    const RomajiMapEntry& entry = *pEntry;

    if (storage.nativeCount > entry.nativeCount)
    {
        if (storage.nativeCount - entry.nativeCount == 2)
        {
            storage.outputCount += 2;
            storage.nativeCount -= 2;
        }
        else if (storage.queue[storage.outputCount] ==
                 storage.queue[storage.outputCount + 1])
        {
            if (storage.mode.Test<Mode::RomajiHiragana>())
            {
                storage.queue[storage.outputCount] = RomajiSmallTsuMap[0];
            }
            else if (storage.mode.Test<Mode::RomajiKatakana>())
            {
                storage.queue[storage.outputCount] = RomajiSmallTsuMap[1];
            }
        }
    }

    for (int i = 0; i < entry.outputCount; ++i)
    {
        uint16_t& value = storage.queue[
            storage.outputCount + storage.nativeCount - entry.nativeCount + i];

        if (storage.mode.Test<Mode::RomajiHiragana>())
        {
            value = entry.output[i];
        }
        else if (storage.mode.Test<Mode::RomajiKatakana>())
        {
            value = ConvertHiraganaToKatakana(entry.output[i]);
        }
    }

    storage.outputCount += storage.nativeCount - entry.nativeCount
                                               + entry.outputCount;
    storage.nativeCount = 0;

    return true;
}

//!< かな "ん" のローマ字変換を処理します。
bool ProcessRomajiNn(detail::KeyCodeComposerStorage* pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    detail::KeyCodeComposerStorage& storage = *pStorage;

    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize - 1);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 1, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           1, QueueSize);

    if (storage.nativeCount != 2 ||
        storage.queue[storage.outputCount    ] != static_cast<uint16_t>('n') ||
        storage.queue[storage.outputCount + 1] == static_cast<uint16_t>('a') ||
        storage.queue[storage.outputCount + 1] == static_cast<uint16_t>('e') ||
        storage.queue[storage.outputCount + 1] == static_cast<uint16_t>('i') ||
        storage.queue[storage.outputCount + 1] == static_cast<uint16_t>('o') ||
        storage.queue[storage.outputCount + 1] == static_cast<uint16_t>('u') ||
        storage.queue[storage.outputCount + 1] == static_cast<uint16_t>('y'))
    {
        return false;
    }
    else
    {
        if (storage.mode.Test<Mode::RomajiHiragana>())
        {
            storage.queue[storage.outputCount] = RomajiNNMap[0];
        }
        else if (storage.mode.Test<Mode::RomajiKatakana>())
        {
            storage.queue[storage.outputCount] = RomajiNNMap[1];
        }

        storage.outputCount += 1;
        storage.nativeCount -= 1;

        return true;
    }
}

void ProcessRomaji(detail::KeyCodeComposerStorage* pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    detail::KeyCodeComposerStorage& storage = *pStorage;

    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize - 1);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 1, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           1, QueueSize);

    if (ProcessRomajiNonLetter(&storage))
    {
        return;
    }

    if (ProcessRomajiMap(&storage))
    {
        return;
    }

    if (ProcessRomajiNn(&storage))
    {
        return;
    }

    if (storage.nativeCount == 4)
    {
        storage.outputCount += 1;
        storage.nativeCount -= 1;
    }

    if (storage.nativeCount == 3)
    {
        const uint16_t* head = &storage.queue[storage.outputCount];

        if ((head[0] == static_cast<uint16_t>('l') ||
             head[0] == static_cast<uint16_t>('x')) &&
            ((head[1] == static_cast<uint16_t>('t') &&
              head[2] == static_cast<uint16_t>('s')) ||
             (head[1] == static_cast<uint16_t>('w') &&
              head[2] == static_cast<uint16_t>('h'))))
        {
            return;
        }

        if ((head[0] == head[1]) &&
            ((head[1] != static_cast<uint16_t>('w') &&
              head[1] != static_cast<uint16_t>('y') &&
              head[2] == static_cast<uint16_t>('y')) ||
             (head[1] == static_cast<uint16_t>('t') &&
              head[2] == static_cast<uint16_t>('s')) ||
             ((head[1] == static_cast<uint16_t>('c') ||
               head[1] == static_cast<uint16_t>('s') ||
               head[1] == static_cast<uint16_t>('d') ||
               head[1] == static_cast<uint16_t>('t') ||
               head[1] == static_cast<uint16_t>('w')) &&
              head[2] == static_cast<uint16_t>('h')) ||
             ((head[1] == static_cast<uint16_t>('l') ||
               head[1] == static_cast<uint16_t>('x')) &&
              (head[2] == static_cast<uint16_t>('t') ||
               head[2] == static_cast<uint16_t>('w')))))
        {
            return;
        }

        storage.outputCount += 1;
        storage.nativeCount -= 1;
    }

    if (storage.nativeCount == 2)
    {
        const uint16_t* head = &storage.queue[storage.outputCount];

        if ((head[0] == head[1]) ||
            (head[0] != static_cast<uint16_t>('w') &&
             head[1] == static_cast<uint16_t>('y')) ||
            (head[0] == static_cast<uint16_t>('t') &&
             head[1] == static_cast<uint16_t>('s')) ||
            ((head[0] == static_cast<uint16_t>('c') ||
              head[0] == static_cast<uint16_t>('s') ||
              head[0] == static_cast<uint16_t>('d') ||
              head[0] == static_cast<uint16_t>('t') ||
              head[0] == static_cast<uint16_t>('w')) &&
             head[1] == static_cast<uint16_t>('h')) ||
            ((head[0] == static_cast<uint16_t>('l') ||
              head[0] == static_cast<uint16_t>('x')) &&
             (head[1] == static_cast<uint16_t>('t') ||
              head[1] == static_cast<uint16_t>('w'))))
        {
            return;
        }

        storage.outputCount += 1;
        storage.nativeCount -= 1;
    }
}

void ProcessRomajiForcibly(detail::KeyCodeComposerStorage* pStorage
                           ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    detail::KeyCodeComposerStorage& storage = *pStorage;

    NN_SDK_REQUIRES_MINMAX(storage.outputCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.nativeCount, 0, QueueSize);
    NN_SDK_REQUIRES_MINMAX(storage.outputCount + storage.nativeCount,
                           0, QueueSize);

    if (storage.nativeCount > 0)
    {
        uint16_t& value = storage.queue[
            storage.outputCount + storage.nativeCount - 1];

        if (value == static_cast<uint16_t>('n'))
        {
            if (storage.mode.Test<Mode::RomajiHiragana>())
            {
                value = RomajiNNMap[0];
            }
            else if (storage.mode.Test<Mode::RomajiKatakana>())
            {
                value = RomajiNNMap[1];
            }
        }
    }

    storage.outputCount += storage.nativeCount;
    storage.nativeCount = 0;
}

} // namespace

}} // namespace nn::hid
