﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   コントローラーサポートの UI 呼び出しに関する API の定義
 */

#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/hid/hid_NpadCommon.h>
#include <nn/util/util_StringUtil.h>

#include "detail/hid_ControllerSupportImpl.h"
#include "detail/hid_NpadId.h"

//---------------------------------------------------------------------------
// C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace hid {

//---------------------------------------------------------------------------
//
nn::Result ShowControllerSupport( const ControllerSupportArg& showControllerSupportArg ) NN_NOEXCEPT
{
    ControllerSupportResultInfo dummy; // 呼び出し元には返さない
    return ShowControllerSupport( &dummy, showControllerSupportArg );
}

//---------------------------------------------------------------------------
//
nn::Result ShowControllerSupport( ControllerSupportResultInfo* pOutValue, const ControllerSupportArg& showControllerSupportArg ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutValue );

    system::ControllerSupportArgPrivate privateArg;
    privateArg.SetDefault();

    return detail::ShowControllerSupportCore( pOutValue, showControllerSupportArg, privateArg );
}

//---------------------------------------------------------------------------
//
void SetExplainText( ControllerSupportArg* pOutControllerSupportArg, const char* pStr, const NpadIdType& id ) NN_NOEXCEPT
{
    // コントローラーサポート用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutControllerSupportArg != nullptr ), "[controller support] ControllerSupportArg must not be null.\n" );

    // NpadId::No1～NpadId::No4以外が指定されたらエラー
    auto index = detail::GetIndexFromNpadIdType( id );
    NN_SDK_ASSERT( ( 0 <= index && index < 4 ), "[controller support] Invalid NpadIdType : %d ", index );

    nn::util::Strlcpy< char >( &( pOutControllerSupportArg->explainText[ index ][ 0 ] ), pStr, nn::hid::ExplainTextMaxBufferSize );
}

//---------------------------------------------------------------------------
//
void ShowControllerStrapGuide() NN_NOEXCEPT
{
    system::ControllerSupportArgPrivate privateArg;
    privateArg.SetDefault();
    privateArg.mode = nn::hid::system::ControllerSupportMode_StrapGuide;
    privateArg.caller = nn::hid::system::ControllerSupportCaller_Application;

    // ダミー引数
    ControllerSupportResultInfo dummyInfo; // 呼び出し元には返さない
    ControllerSupportArg dummyArg; // 呼び出し元には使わせない
    dummyArg.SetDefault();

    detail::ShowControllerSupportCore( &dummyInfo, dummyArg, privateArg );
}

//---------------------------------------------------------------------------
//
nn::Result ShowControllerFirmwareUpdate( const ControllerFirmwareUpdateArg& showControllerFirmwareUpdateArg ) NN_NOEXCEPT
{
    system::ControllerSupportArgPrivate privateArg;
    privateArg.SetDefault();
    privateArg.mode = nn::hid::system::ControllerSupportMode_FirmwareUpdate;
    privateArg.caller = nn::hid::system::ControllerSupportCaller_Application;

    return detail::ShowControllerFirmwareUpdateCore( showControllerFirmwareUpdateArg, privateArg );
}

}} // namespace nn::hid

