﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_GenericPadAxis.h>
#include <nn/settings/settings_GenericPadButton.h>
#include <nn/settings/settings_Xpad.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_WindowsGenericPadAccessor-os.win.h"
#include "hid_XpadMappingManager.h"

namespace nn { namespace hid { namespace detail {

XpadMappingManager::XpadMappingManager() NN_NOEXCEPT
    : m_Mutex(false)
{
    m_BasicXpadMap.isGenericPadMapInitialized = false;
    m_BasicXpadMap.genericPadMap = ::nn::settings::BasicXpadGenericPadMap();
    m_BasicXpadMap.keyboardMap = ::nn::settings::BasicXpadKeyboardMap();
}

bool XpadMappingManager::IsValidBasicXpadGenericPadMap(
    const ::nn::settings::BasicXpadGenericPadMap& value) NN_NOEXCEPT
{
    // 汎用コントローラ ID が有効かつ有効な値が一つでも設定されていれば有効
    return IsValidWindowsGenericPadId(value.genericPadId) &&
           (value.buttonA > 0 || value.buttonB > 0 || value.buttonX > 0 ||
            value.buttonY > 0 || value.buttonStickL > 0 ||
            value.buttonStickR > 0 || value.buttonL > 0 || value.buttonR > 0 ||
            value.buttonZL > 0 || value.buttonZR > 0 ||
            value.buttonStart > 0 || value.buttonSelect > 0 ||
            value.buttonLeft > 0 || value.buttonUp > 0 ||
            value.buttonRight > 0 || value.buttonDown > 0);
}

bool XpadMappingManager::IsValidBasicXpadKeyboardMap(
    const ::nn::settings::BasicXpadKeyboardMap& value) NN_NOEXCEPT
{
    // 有効な値が一つでも設定されていれば有効
    return (value.buttonA > 0 || value.buttonB > 0 || value.buttonX > 0 ||
            value.buttonY > 0 || value.buttonStickL > 0 ||
            value.buttonStickR > 0 || value.buttonL > 0 || value.buttonR > 0 ||
            value.buttonZL > 0 || value.buttonZR > 0 ||
            value.buttonStart > 0 || value.buttonSelect > 0 ||
            value.buttonLeft > 0 || value.buttonUp > 0 ||
            value.buttonRight > 0 || value.buttonDown > 0);
}

::nn::Result XpadMappingManager::GetBasicXpadGenericPadMap(
    ::nn::settings::BasicXpadGenericPadMap* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultBasicXpadNullGenericPadMapBuffer());

    if (!m_BasicXpadMap.isGenericPadMapInitialized)
    {
        // 未初期化の場合のみ割り当ての初期化を実施

        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        if (!m_BasicXpadMap.isGenericPadMapInitialized)
        {
            // 排他した状態での再チェックにも失敗した場合のみ初期化処理を実施

            ::nn::settings::BasicXpadGenericPadMap map;

            GetDefaultBasicXpadGenericPadMap(&map);

            NN_RESULT_DO(GetUnbusyWindowsGenericPad(&map.genericPadId));

            m_BasicXpadMap.genericPadMap = map;

            m_BasicXpadMap.isGenericPadMapInitialized = true;
        }
    }

    *pOutValue = m_BasicXpadMap.genericPadMap;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadMappingManager::SetBasicXpadGenericPadMap(
    const ::nn::settings::BasicXpadGenericPadMap& value) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    const int32_t newId = value.genericPadId;

    if (!m_BasicXpadMap.isGenericPadMapInitialized)
    {
        // 未初期化の場合は汎用コントローラの割り当てを実施

        NN_RESULT_DO(AddWindowsGenericPadUser(newId));

        m_BasicXpadMap.isGenericPadMapInitialized = true;
    }
    else
    {
        const int32_t oldId = m_BasicXpadMap.genericPadMap.load().genericPadId;

        if (newId != oldId)
        {
            // 汎用コントローラの割り当てに変更がある場合は状態を更新

            auto needsRollback = true;

            NN_RESULT_DO(AddWindowsGenericPadUser(newId));

            NN_UTIL_SCOPE_EXIT
            {
                if (needsRollback)
                {
                    RemoveWindowsGenericPadUser(newId);
                }
            };

            NN_RESULT_DO(RemoveWindowsGenericPadUser(oldId));

            needsRollback = false;
        }
    }

    m_BasicXpadMap.genericPadMap = value;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadMappingManager::ResetBasicXpadGenericPadMap() NN_NOEXCEPT
{
    BasicXpadMap& xpadMap = m_BasicXpadMap;

    if (xpadMap.isGenericPadMapInitialized)
    {
        // 初期化済みだった場合のみ規定値へリセット

        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        ::nn::settings::BasicXpadGenericPadMap map = xpadMap.genericPadMap;

        int genericPadId = map.genericPadId;

        GetDefaultBasicXpadGenericPadMap(&map);

        map.genericPadId = genericPadId;

        xpadMap.genericPadMap = map;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result XpadMappingManager::GetBasicXpadKeyboardMap(
    ::nn::settings::BasicXpadKeyboardMap* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultBasicXpadNullKeyboardMapBuffer());

    *pOutValue = m_BasicXpadMap.keyboardMap;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadMappingManager::SetBasicXpadKeyboardMap(
    const ::nn::settings::BasicXpadKeyboardMap& value) NN_NOEXCEPT
{
    m_BasicXpadMap.keyboardMap = value;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadMappingManager::ResetBasicXpadKeyboardMap() NN_NOEXCEPT
{
    m_BasicXpadMap.keyboardMap = ::nn::settings::BasicXpadKeyboardMap();

    NN_RESULT_SUCCESS;
}

void XpadMappingManager::GetDefaultBasicXpadGenericPadMap(
    ::nn::settings::BasicXpadGenericPadMap* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->genericPadId = 0;
    pOutValue->buttonA = ::nn::settings::GenericPadButton_4;
    pOutValue->buttonB = ::nn::settings::GenericPadButton_3;
    pOutValue->buttonX = ::nn::settings::GenericPadButton_2;
    pOutValue->buttonY = ::nn::settings::GenericPadButton_1;
    pOutValue->buttonStickL = ::nn::settings::GenericPadButton_11;
    pOutValue->buttonStickR = ::nn::settings::GenericPadButton_12;
    pOutValue->buttonL = ::nn::settings::GenericPadButton_5;
    pOutValue->buttonR = ::nn::settings::GenericPadButton_6;
    pOutValue->buttonZL = ::nn::settings::GenericPadButton_7;
    pOutValue->buttonZR = ::nn::settings::GenericPadButton_8;
    pOutValue->buttonStart = ::nn::settings::GenericPadButton_10;
    pOutValue->buttonSelect = ::nn::settings::GenericPadButton_9;
    pOutValue->buttonLeft = ::nn::settings::GenericPadButton_Left;
    pOutValue->buttonUp = ::nn::settings::GenericPadButton_Up;
    pOutValue->buttonRight = ::nn::settings::GenericPadButton_Right;
    pOutValue->buttonDown = ::nn::settings::GenericPadButton_Down;
    pOutValue->axisAnalogStickRX = ::nn::settings::GenericPadAxis_Z;
    pOutValue->axisAnalogStickRY = ::nn::settings::GenericPadAxis_R;
    pOutValue->axisAnalogStickLX = ::nn::settings::GenericPadAxis_X;
    pOutValue->axisAnalogStickLY = ::nn::settings::GenericPadAxis_Y;
    pOutValue->attributes.Reset();
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>(false);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>(true);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>(false);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>(true);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>(false);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>(false);
}

}}} // namespace nn::hid::detail
