﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>

#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_IXpadDriver.h"
#include "hid_JoyXpadManager.h"
#include "hid_RingLifoHolder.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_XpadLifo.h"
#include "hid_SynchronizedTimer.h"
#include "hid_SixAxisSensorLifo.h"

namespace nn { namespace hid { namespace detail {

//!< アプレットリソースの Xpad 用拡張エントリを表す構造体です。
struct XpadAppletResourceEntry final
{
    //!< ARUID
    ::nn::applet::AppletResourceUserId aruid;

    //!< 基本的な構成を持つ Xpad のデジタルボタンのマスク
    BasicXpadButtonSet buttonMask;

    NN_PADDING4;
};

//!< Xpad インターフェイスの解決を担うマネージャを扱うクラスです。
class XpadManager final
{
    NN_DISALLOW_COPY(XpadManager);
    NN_DISALLOW_MOVE(XpadManager);

public:
    //!< サンプリング間隔
    static const ::nn::TimeSpan SamplingInterval;

private:
    //!< タイマーイベント
    SynchronizedTimer* m_pTimer;

    //!< ドライバ
    IXpadDriver* m_pDriver;

    //!< アプレットリソースマネージャ
    AppletResourceManager* m_pAppletResourceManager;

    //!< アプレットリソースマネージャのミューテックス
    ::nn::os::SdkRecursiveMutex* m_pAppletResourceManagerMutex;

    //!< サンプリングを行うか否かを表す値です。
    bool m_IsSamplingEnabled;

    //!< 基本的な構成を持つ Xpad がアクティブ化された回数
    ActivationCount m_BasicXpadActivationCount;

    //!< 基本的な構成を持つ Xpad の ID
    BasicXpadId m_BasicXpadId;

    //!< 基本的な構成を持つ Xpad のサンプリング番号
    int64_t m_BasicXpadSamplingNumber;

    //!< 基本的な構成を持つ Xpad の入力状態
    BasicXpadState m_BasicXpadState;

    //!< 基本的な構成を持つ Xpad の入力状態計算用の一時的メモリ領域
    BasicXpadState m_BasicXpadTempState;

    //!< 基本的な構成を持つ Xpad のドライバの入力状態
    BasicXpadState m_BasicXpadDriverState;

    //!< 基本的な構成を持つ Xpad の自動操作が有効か否かを表す値
    bool m_IsBasicXpadAutoPilotEnabled;

    //!< 基本的な構成を持つ Xpad の自動操作状態
    ::nn::hid::debug::BasicXpadAutoPilotState m_BasicXpadAutoPilotState;

    //!< JoyXpad がアクティブ化された回数
    ActivationCount m_JoyXpadActivationCount;

    //!< JoyXpad の実装クラス
    JoyXpadManager m_JoyXpadManager;

    //!< 6軸センサーがアクティブ化された回数
    ActivationCount m_SixAxisSensorActivationCount;

    //!< 6軸センサーのサンプリング番号
    int64_t m_SixAxisSensorSamplingNumber;

    //!< 6軸センサーの LIFO
    RingLifoHolder<SixAxisSensorLifo> m_SixAxisSensorLifoHolder;

    //!< 6軸センサーの入力状態
    SixAxisSensorState m_SixAxisSensorState;

    //!< JoyXpad の6軸センサーがアクティブ化された回数
    ActivationCount m_JoySixAxisSensorActivationCount;

    //!< ARUID 処理用の一時的メモリ領域
    ::nn::applet::AppletResourceUserId m_TempAruid;

    //!< アプレットリソースの Xpad 用拡張エントリ
    XpadAppletResourceEntry
        m_AppletResourceEntries[AppletResourceEntryCountMax];

public:
    XpadManager() NN_NOEXCEPT;

    //!< タイマーイベントを設定します。
    void SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT;

    //!< ドライバを設定します。
    void SetDriver(IXpadDriver* pDriver) NN_NOEXCEPT;

    //!< アプレットリソースマネージャを設定します。
    void SetAppletResourceManager(
        AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
        ) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad を設定します。
    void SetXpadId(const BasicXpadId& xpadId) NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< サンプリングを有効化します。
    void EnableSampling() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad についてのみマネージャをアクティブ化します。
    ::nn::Result ActivateBasicXpad() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad についてのみマネージャを非アクティブ化します。
    ::nn::Result DeactivateBasicXpad() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureBasicXpadAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 6軸センサーについてのみマネージャをアクティブ化します。
    ::nn::Result ActivateSixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーについてのみマネージャを非アクティブ化します。
    ::nn::Result DeactivateSixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーの LIFO のハンドルを取得します。
    ::nn::Result GetSixAxisSensorLifoHandle(
        ::nn::os::NativeHandle* outValue) NN_NOEXCEPT;

    //!< 6軸センサーについてのみマネージャをアクティブ化します。
    ::nn::Result ActivateJoySixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーについてのみマネージャを非アクティブ化します。
    ::nn::Result DeactivateJoySixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーの LIFO のハンドルを取得します。
    ::nn::Result GetJoySixAxisSensorLifoHandle(
        ::nn::os::NativeHandle* outValue) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad に自動操作状態を設定します。
    ::nn::Result SetBasicXpadAutoPilotState(
        const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の自動操作状態を解除します。
    ::nn::Result UnsetBasicXpadAutoPilotState() NN_NOEXCEPT;

    //!< JoyXpad についてのみマネージャをアクティブ化します。
    ::nn::Result ActivateJoyXpad() NN_NOEXCEPT;

    //!< JoyXpad についてのみマネージャを非アクティブ化します。
    ::nn::Result DeactivateJoyXpad() NN_NOEXCEPT;

    //!< JoyXpad の LIFO のハンドルを取得します。
    ::nn::Result GetJoyXpadLifoHandle(
        ::nn::os::NativeHandle* outValue) NN_NOEXCEPT;

    //!< 入力状態を解決します。
    void Sample() NN_NOEXCEPT;

private:
    //!< 基本的な構成を持つ Xpad の入力状態を更新します。
    void UpdateBasicXpad() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の共有メモリを処理します。
    void ProcessSharedMemory(
        void (*processor)(
            XpadManager* that,
            BasicXpadSharedMemoryEntry* address,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            BasicXpadButtonSet& buttonMask) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の自動操作を無効化します。
    void DisableBasicXpadAutoPilot() NN_NOEXCEPT;

    //!< 入力状態を 6軸センサーの LIFO に追加します。
    void AppendStateToSixAxisSensorLifo() NN_NOEXCEPT;

    //!< タイマーイベントをアクティブ化します。
    void ActivateTimerEvent() NN_NOEXCEPT;

    //!< タイマーイベントを非アクティブ化します。
    void DeactivateTimerEvent() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
