﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <mutex>
#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceManager.h"
#include "hid_CommonStateUtility.h"
#include "hid_IXpadDriver.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_XpadId.h"
#include "hid_XpadLifo.h"
#include "hid_XpadManager.h"
#include "hid_XpadStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 基本的な構成を持つ Xpad の入力状態を計算します。
void CalculateXpadState(BasicXpadState* pOutValue,
                        const BasicXpadState& state,
                        bool enablesInput,
                        BasicXpadButtonSet& buttonMask) NN_NOEXCEPT;

} // namespace

const ::nn::TimeSpan XpadManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(5);

XpadManager::XpadManager() NN_NOEXCEPT
    : m_pTimer(nullptr)
    , m_pDriver(nullptr)
    , m_pAppletResourceManager(nullptr)
    , m_pAppletResourceManagerMutex(nullptr)
    , m_IsSamplingEnabled(false)
    , m_BasicXpadId()
    , m_BasicXpadSamplingNumber(0)
    , m_BasicXpadState()
    , m_BasicXpadTempState()
    , m_BasicXpadDriverState()
    , m_IsBasicXpadAutoPilotEnabled(false)
    , m_BasicXpadAutoPilotState()
    , m_SixAxisSensorSamplingNumber(0)
    , m_SixAxisSensorLifoHolder()
    , m_TempAruid()
{
    for (XpadAppletResourceEntry& extraEntry : m_AppletResourceEntries)
    {
        extraEntry = XpadAppletResourceEntry();
    }
}

void XpadManager::SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(timer);
    NN_SDK_REQUIRES(m_BasicXpadActivationCount.IsZero());
    m_pTimer = timer;
}

void XpadManager::SetDriver(IXpadDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_BasicXpadActivationCount.IsZero());
    m_pDriver = pDriver;
}

void XpadManager::SetAppletResourceManager(
    AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_BasicXpadActivationCount.IsZero());
    m_pAppletResourceManager = pManager;
    m_pAppletResourceManagerMutex = pMutex;
}

void XpadManager::SetXpadId(const BasicXpadId& xpadId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_BasicXpadActivationCount.IsZero());
    m_BasicXpadId = xpadId;
}

::nn::Result XpadManager::Activate() NN_NOEXCEPT
{
    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateBasicXpad());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateBasicXpad();
        }
    };

    NN_RESULT_DO(this->ActivateJoyXpad());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateJoyXpad();
        }
    };

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_DO(this->DeactivateBasicXpad());
    NN_RESULT_DO(this->DeactivateJoyXpad());
    NN_RESULT_SUCCESS;
}

void XpadManager::EnableSampling() NN_NOEXCEPT
{
    m_IsSamplingEnabled = true;
}

::nn::Result XpadManager::ActivateBasicXpad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_BasicXpadActivationCount.IsMax(),
                           ResultBasicXpadActivationUpperLimitOver());

    if (m_BasicXpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // ドライバをアクティブ化
        NN_RESULT_DO(m_pDriver->Activate());

        this->ProcessSharedMemory([] (
            XpadManager* that,
            BasicXpadSharedMemoryEntry* address,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            BasicXpadButtonSet& buttonMask) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(that);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);
            NN_UNUSED(buttonMask);
            ::nn::util::Get(address->lifo).Clear();
        });

        // タイマーイベントをアクティブ化
        this->ActivateTimerEvent();

        m_BasicXpadState = BasicXpadState();

        // 入力状態を更新
        this->UpdateBasicXpad();
    }

    ++m_BasicXpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::DeactivateBasicXpad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_BasicXpadActivationCount.IsZero(),
                           ResultBasicXpadDeactivationLowerLimitOver());

    --m_BasicXpadActivationCount;

    if (m_BasicXpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // 自動操作を無効化
        this->DisableBasicXpadAutoPilot();

        // タイマーイベントを非アクティブ化
        this->DeactivateTimerEvent();

        // ドライバを非アクティブ化
        NN_RESULT_DO(m_pDriver->Deactivate());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::EnsureBasicXpadAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_TempAruid = aruid;

    this->ProcessSharedMemory([] (XpadManager* that,
                                  BasicXpadSharedMemoryEntry* address,
                                  ::nn::applet::AppletResourceUserId haystack,
                                  bool enablesInput,
                                  BasicXpadButtonSet& buttonMask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        BasicXpadState* pOutValue = &that->m_BasicXpadTempState;

        const BasicXpadState& state = that->m_BasicXpadState;

        const ::nn::applet::AppletResourceUserId& needle = that->m_TempAruid;

        BasicXpadLifo& lifo = ::nn::util::Get(address->lifo);

        if (lifo.IsEmpty() && needle == haystack)
        {
            CalculateXpadState(pOutValue, state, enablesInput, buttonMask);

            lifo.Append(*pOutValue);
        }
    });

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::GetJoyXpadLifoHandle(
    ::nn::os::NativeHandle* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_RESULT_THROW_UNLESS(!m_JoyXpadActivationCount.IsZero(),
                           ResultJoyXpadNotInitialized());
    m_JoyXpadManager.GetJoyXpadLifoHandle(outValue);
    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::SetBasicXpadAutoPilotState(
    const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_BasicXpadActivationCount.IsZero(),
                           ResultBasicXpadNotInitialized());

    m_IsBasicXpadAutoPilotEnabled = true;

    switch (value.powerState)
    {
    case PowerState_Disconnected:
    case PowerState_OnBattery:
    case PowerState_NoBattery:
    case PowerState_Charging:
    case PowerState_Charged:
        m_BasicXpadAutoPilotState.powerState = value.powerState;
        break;
    default:
        // 未知の状態が指定された場合はバッテリー駆動状態に設定
        m_BasicXpadAutoPilotState.powerState = PowerState_OnBattery;
        break;
    }

    m_BasicXpadAutoPilotState.buttons = RestrictXpadButtons(value.buttons);

    m_BasicXpadAutoPilotState.analogStickL = LimitAnalogStick(
        value.analogStickL, 0, AnalogStickMax, AnalogStickMax);

    m_BasicXpadAutoPilotState.analogStickR = LimitAnalogStick(
        value.analogStickR, 0, AnalogStickMax, AnalogStickMax);

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::UnsetBasicXpadAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_BasicXpadActivationCount.IsZero(),
                           ResultBasicXpadNotInitialized());
    this->DisableBasicXpadAutoPilot();
    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::ActivateJoyXpad() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_JoyXpadActivationCount.IsMax(),
                           ResultJoyXpadActivationUpperLimitOver());

    if (m_JoyXpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        // JoyXpadManager をアクティブ化
        NN_RESULT_DO(m_JoyXpadManager.ActivateJoyXpad());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_JoyXpadManager.ActivateJoyXpad();
            }
        };

        // タイマーイベントをアクティブ化
        this->ActivateTimerEvent();

        needsRollback = false;
    }

    ++m_JoyXpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::DeactivateJoyXpad() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_JoyXpadActivationCount.IsZero(),
                           ResultJoyXpadDeactivationLowerLimitOver());

    --m_JoyXpadActivationCount;

    if (m_JoyXpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // タイマーイベントを非アクティブ化
        this->DeactivateTimerEvent();

        // JoyXpadManager を非アクティブ化
        NN_RESULT_DO(m_JoyXpadManager.DeactivateJoyXpad());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::ActivateSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_SixAxisSensorActivationCount.IsMax(),
                           ResultSixAxisSensorActivationUpperLimitOver());

    if (m_SixAxisSensorActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        // LIFO を初期化
        NN_RESULT_DO(m_SixAxisSensorLifoHolder.Initialize());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_SixAxisSensorLifoHolder.Finalize();
            }
        };

        // LIFO の確保に失敗した場合は失敗
        NN_RESULT_THROW_UNLESS(m_SixAxisSensorLifoHolder.GetLifo() != nullptr,
                               ResultSixAxisSensorLifoAllocationFailed());

        // LIFO 上に最低 1 つの入力状態があることを保証
        this->AppendStateToSixAxisSensorLifo();

        needsRollback = false;
    }

    ++m_SixAxisSensorActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::DeactivateSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_SixAxisSensorActivationCount.IsZero(),
                           ResultSixAxisSensorDeactivationLowerLimitOver());
    --m_SixAxisSensorActivationCount;

    if (m_SixAxisSensorActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // LIFO を開放
        m_SixAxisSensorLifoHolder.Finalize();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::GetSixAxisSensorLifoHandle(
    ::nn::os::NativeHandle* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_RESULT_THROW_UNLESS(!m_BasicXpadActivationCount.IsZero(),
                           ResultSixAxisSensorNotInitialized());
    *outValue = m_SixAxisSensorLifoHolder.GetSharedMemoryHandle();
    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::ActivateJoySixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_JoySixAxisSensorActivationCount.IsMax(),
                           ResultSixAxisSensorActivationUpperLimitOver());

    if (m_JoySixAxisSensorActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        // 6軸センサーをアクティブ化
        NN_RESULT_DO(m_JoyXpadManager.ActivateSixAxisSensor());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_JoyXpadManager.DeactivateSixAxisSensor();
            }
        };
        needsRollback = false;
    }

    ++m_JoySixAxisSensorActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadManager::DeactivateJoySixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_JoySixAxisSensorActivationCount.IsZero(),
                           ResultSixAxisSensorDeactivationLowerLimitOver());

    --m_JoySixAxisSensorActivationCount;

    if (m_JoySixAxisSensorActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // ドライバを非アクティブ化
        NN_RESULT_DO(m_JoyXpadManager.DeactivateSixAxisSensor());
    }

    NN_RESULT_SUCCESS;

}

::nn::Result XpadManager::GetJoySixAxisSensorLifoHandle(
    ::nn::os::NativeHandle* outValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_JoyXpadActivationCount.IsZero(),
                           ResultJoyXpadNotInitialized());
    m_JoyXpadManager.GetSixAxisSensorLifoHandle(outValue);
    NN_RESULT_SUCCESS;
}

void XpadManager::Sample() NN_NOEXCEPT
{
    if (!m_IsSamplingEnabled)
    {
        return;
    }

    if (!m_BasicXpadActivationCount.IsZero())
    {
        this->UpdateBasicXpad();

        this->ProcessSharedMemory([] (
            XpadManager* that,
            BasicXpadSharedMemoryEntry* address,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            BasicXpadButtonSet& buttonMask) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(aruid);

            BasicXpadState* pOutValue = &that->m_BasicXpadTempState;

            const BasicXpadState& state = that->m_BasicXpadState;

            CalculateXpadState(pOutValue, state, enablesInput, buttonMask);

            ::nn::util::Get(address->lifo).Append(*pOutValue);
        });
    }

    if (!m_JoyXpadActivationCount.IsZero())
    {
        m_JoyXpadManager.Sample();
    }
}

void XpadManager::UpdateBasicXpad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    // 入力状態を取得
    m_pDriver->GetState(&m_BasicXpadState);

    // サンプリング番号を設定
    m_BasicXpadState.samplingNumber = m_BasicXpadSamplingNumber++;

    if (m_IsBasicXpadAutoPilotEnabled)
    {
        // 自動操作が有効な場合は自動操作状態をマージ

        if (m_BasicXpadAutoPilotState.powerState != PowerState_Disconnected)
        {
            m_BasicXpadState.attributes.Set<BasicXpadAttribute::IsConnected>();
        }

        m_BasicXpadState.buttons = MergeXpadButtons(
            m_BasicXpadState.buttons,
            m_BasicXpadAutoPilotState.buttons);

        m_BasicXpadState.analogStickL = MergeAnalogStick(
            m_BasicXpadState.analogStickL,
            m_BasicXpadAutoPilotState.analogStickL);

        m_BasicXpadState.analogStickR = MergeAnalogStick(
            m_BasicXpadState.analogStickR,
            m_BasicXpadAutoPilotState.analogStickR);
    }
}

void XpadManager::ProcessSharedMemory(
    void (*processor)(XpadManager* that,
                      BasicXpadSharedMemoryEntry* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput,
                      BasicXpadButtonSet& buttonMask) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);

    const auto index =
        static_cast<size_t>(GetXpadIdPlayerNumber(m_BasicXpadId));

    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)
                      > locker(*m_pAppletResourceManagerMutex);

    const AppletResourceEntry (&entries)[AppletResourceEntryCountMax] =
        m_pAppletResourceManager->GetAppletResourceEntries();

    for (size_t i = 0; i < AppletResourceEntryCountMax; ++i)
    {
        const AppletResourceEntry& entry = entries[i];

        XpadAppletResourceEntry& extraEntry = m_AppletResourceEntries[i];

        if (!entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            extraEntry = XpadAppletResourceEntry();
        }
        else
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);
            NN_SDK_ASSERT_LESS(
                index,
                ::std::extent<
                    decltype(entry.address->basicXpad.entries)>::value);

            if (entry.aruid != extraEntry.aruid)
            {
                extraEntry = XpadAppletResourceEntry();

                extraEntry.aruid = entry.aruid;
            }

            processor(this,
                      &entry.address->basicXpad.entries[index],
                      entry.aruid,
                      entry.flags.Test<AppletResourceFlag::EnablesInput>(),
                      extraEntry.buttonMask);
        }
    }
}

void XpadManager::DisableBasicXpadAutoPilot() NN_NOEXCEPT
{
    m_IsBasicXpadAutoPilotEnabled = false;
    m_BasicXpadAutoPilotState = ::nn::hid::debug::BasicXpadAutoPilotState();
}

void XpadManager::AppendStateToSixAxisSensorLifo() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_SixAxisSensorLifoHolder.GetLifo());

    // 互換性担保用に空の 6軸センサーデータを書き込む
    m_SixAxisSensorState.samplingNumber = m_SixAxisSensorSamplingNumber++;
    m_SixAxisSensorLifoHolder.GetLifo()->Append(m_SixAxisSensorState);
}

void XpadManager::ActivateTimerEvent() NN_NOEXCEPT
{
    if (m_pTimer != nullptr &&
        m_BasicXpadActivationCount.IsZero() &&
        m_JoyXpadActivationCount.IsZero())
    {
        m_pTimer->Enable(SamplingInterval);
    }
}

void XpadManager::DeactivateTimerEvent() NN_NOEXCEPT
{
    if (m_pTimer != nullptr &&
        m_BasicXpadActivationCount.IsZero() &&
        m_JoyXpadActivationCount.IsZero())
    {
        m_pTimer->Disable();
    }
}

namespace {

void CalculateXpadState(BasicXpadState* pOutValue,
                        const BasicXpadState& state,
                        bool enablesInput,
                        BasicXpadButtonSet& buttonMask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = state;

    if (enablesInput)
    {
        buttonMask |= ~state.buttons;

        pOutValue->buttons &= buttonMask;
    }
    else
    {
        buttonMask.Reset();

        pOutValue->buttons.Reset();
        pOutValue->analogStickL = AnalogStickState();
        pOutValue->analogStickR = AnalogStickState();
    }
}

} // namespace

}}} // namespace nn::hid::detail
