﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_Xpad.h>
#include <nn/settings/settings_GenericPadAxis.h>
#include <nn/settings/settings_GenericPadButton.h>

#include "hid_CommonStateUtility.h"
#include "hid_HidShellPortFile-os.win.h"
#include "hid_XpadDriver.win32.h"
#include "hid_XpadMappingManager.h"
#include "hid_XpadStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 基本的な構成を持つ Xpad のデジタルボタンの押下状態を返します。
BasicXpadButtonSet GetButtons(const WindowsGenericPad& genericPad) NN_NOEXCEPT
{
    WindowsGenericPadButtonSet buttons = genericPad.GetButtons();
    BasicXpadButtonSet value = {};
    value.Set<BasicXpadButton::A>(
        buttons.Test<WindowsGenericPadButton::B>());
    value.Set<BasicXpadButton::B>(
        buttons.Test<WindowsGenericPadButton::A>());
    value.Set<BasicXpadButton::X>(
        buttons.Test<WindowsGenericPadButton::Y>());
    value.Set<BasicXpadButton::Y>(
        buttons.Test<WindowsGenericPadButton::X>());
    value.Set<BasicXpadButton::StickL>(
        buttons.Test<WindowsGenericPadButton::ThumbL>());
    value.Set<BasicXpadButton::StickR>(
        buttons.Test<WindowsGenericPadButton::ThumbR>());
    value.Set<BasicXpadButton::L>(
        buttons.Test<WindowsGenericPadButton::L>());
    value.Set<BasicXpadButton::R>(
        buttons.Test<WindowsGenericPadButton::R>());
    value.Set<BasicXpadButton::ZL>(
        buttons.Test<WindowsGenericPadButton::TriggerL>());
    value.Set<BasicXpadButton::ZR>(
        buttons.Test<WindowsGenericPadButton::TriggerR>());
    value.Set<BasicXpadButton::Start>(
        buttons.Test<WindowsGenericPadButton::Start>());
    value.Set<BasicXpadButton::Select>(
        buttons.Test<WindowsGenericPadButton::Back>());
    value.Set<BasicXpadButton::Left>(
        buttons.Test<WindowsGenericPadButton::Left>());
    value.Set<BasicXpadButton::Up>(
        buttons.Test<WindowsGenericPadButton::Up>());
    value.Set<BasicXpadButton::Right>(
        buttons.Test<WindowsGenericPadButton::Right>());
    value.Set<BasicXpadButton::Down>(
        buttons.Test<WindowsGenericPadButton::Down>());
    return value;
}

//!< 指定の割り当てに基づいた Xpad のデジタルボタンの押下状態を返します。
BasicXpadButtonSet GetButtons(
    const WindowsGenericPad& genericPad,
    const ::nn::settings::BasicXpadGenericPadMap& map) NN_NOEXCEPT
{
    BasicXpadButtonSet value = {};
    value.Set<BasicXpadButton::A>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonA)));
    value.Set<BasicXpadButton::B>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonB)));
    value.Set<BasicXpadButton::X>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonX)));
    value.Set<BasicXpadButton::Y>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonY)));
    value.Set<BasicXpadButton::StickL>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonStickL)));
    value.Set<BasicXpadButton::StickR>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonStickR)));
    value.Set<BasicXpadButton::L>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonL)));
    value.Set<BasicXpadButton::R>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonR)));
    value.Set<BasicXpadButton::ZL>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonZL)));
    value.Set<BasicXpadButton::ZR>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonZR)));
    value.Set<BasicXpadButton::Start>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonStart)));
    value.Set<BasicXpadButton::Select>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonSelect)));
    value.Set<BasicXpadButton::Left>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonLeft)));
    value.Set<BasicXpadButton::Up>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonUp)));
    value.Set<BasicXpadButton::Right>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonRight)));
    value.Set<BasicXpadButton::Down>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonDown)));
    return value;
}

//!< 基本的な構成を持つ Xpad の自動操作状態を取得します。
void GetXpadAutoPilotState(
    ::nn::hid::debug::BasicXpadAutoPilotState* pOutValue,
    const ::nn::hid::BasicXpadState& state) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    pOutValue->powerState =
        state.attributes.Test<BasicXpadAttribute::IsConnected>()
            ? PowerState_OnBattery
            : PowerState_Disconnected;
    pOutValue->buttons = state.buttons;
    pOutValue->analogStickR = state.analogStickR;
    pOutValue->analogStickL = state.analogStickL;
}

//!< 基本的な構成を持つ Xpad の状態を取得します。
void GetXpadState(
    ::nn::hid::BasicXpadState* pOutValue,
    const ::nn::hid::debug::BasicXpadAutoPilotState& state) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    pOutValue->attributes.Set<BasicXpadAttribute::IsConnected>(
        state.powerState != PowerState_Disconnected);
    pOutValue->buttons = RestrictXpadButtons(state.buttons);
    pOutValue->analogStickR = LimitAnalogStick(
        state.analogStickR, 0, AnalogStickMax, AnalogStickMax);
    pOutValue->analogStickL = LimitAnalogStick(
        state.analogStickL, 0, AnalogStickMax, AnalogStickMax);
}

} // namespace

XpadDriver::XpadDriver() NN_NOEXCEPT
    : m_SamplingNumber(0)
    , m_HidShellPort(0)
    , m_pMappingManager(nullptr)
{
    // 何もしない
}

XpadDriver::~XpadDriver() NN_NOEXCEPT
{
    // 何もしない
}

void XpadDriver::SetHidShellPort(int port) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidHidShellXpadPort(port));
    m_HidShellPort = port;
}

void XpadDriver::SetMappingManager(XpadMappingManager* pMappingManager
                                   ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMappingManager);
    m_pMappingManager = pMappingManager;
}

bool XpadDriver::IsActivated() NN_NOEXCEPT
{
    return !m_ActivationCount.IsZero();
}

::nn::Result XpadDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        m_WindowsGenericPad.Reset();

        m_WindowsKeyboard.Reset();
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    NN_RESULT_SUCCESS;
}

void XpadDriver::GetState(BasicXpadState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    NN_SDK_REQUIRES_NOT_NULL(m_pMappingManager);

    // 無入力状態に初期化
    *pOutValue = BasicXpadState();
    pOutValue->samplingNumber = m_SamplingNumber++;

    // 汎用ゲームパッドとのマッピングを取得
    m_pMappingManager->GetBasicXpadGenericPadMap(&m_GenericPadMap);

    if (!m_pMappingManager->IsValidBasicXpadGenericPadMap(m_GenericPadMap))
    {
        // マッピングされた識別子が無効ならば内部状態をリセット
        m_WindowsGenericPad.Reset();
    }
    else
    {
        // マッピングされた識別子が有効ならば対象とする汎用コントローラに設定
        m_WindowsGenericPad.SetGenericPadId(m_GenericPadMap.genericPadId);

        // 汎用コントローラの内部状態を更新
        m_WindowsGenericPad.Update();

        // Xpad の入力状態を決定
        pOutValue->attributes.Set<BasicXpadAttribute::IsConnected>(
            m_WindowsGenericPad.IsDetected());
        if (m_WindowsGenericPad.IsXInput())
        {
            pOutValue->buttons = RestrictXpadButtons(
                GetButtons(m_WindowsGenericPad));
            pOutValue->analogStickR = LimitAnalogStick(
                m_WindowsGenericPad.GetStickR(),
                0, XInputAnalogStickMaxCount,
                XInputAnalogStickTheoreticalCount);
            pOutValue->analogStickL = LimitAnalogStick(
                m_WindowsGenericPad.GetStickL(),
                0, XInputAnalogStickMaxCount,
                XInputAnalogStickTheoreticalCount);
        }
        else
        {
            pOutValue->buttons = RestrictXpadButtons(
                GetButtons(m_WindowsGenericPad, m_GenericPadMap));
            pOutValue->analogStickR = LimitAnalogStick(
                m_WindowsGenericPad.GetStickR(m_GenericPadMap),
                XInputAnalogStickRDeadZoneCount, XInputAnalogStickMaxCount,
                XInputAnalogStickTheoreticalCount);
            pOutValue->analogStickL = LimitAnalogStick(
                m_WindowsGenericPad.GetStickL(m_GenericPadMap),
                XInputAnalogStickLDeadZoneCount, XInputAnalogStickMaxCount,
                XInputAnalogStickTheoreticalCount);
        }
    }

    // キーボードとのマッピングを取得
    m_pMappingManager->GetBasicXpadKeyboardMap(&m_KeyboardMap);

    if (!m_pMappingManager->IsValidBasicXpadKeyboardMap(m_KeyboardMap))
    {
        // マッピングが無効ならば内部状態をリセット
        m_WindowsKeyboard.Reset();
    }
    else
    {
        // Keyboard のキーの状態について内部状態を更新
        m_WindowsKeyboard.UpdateKeys();

        BasicXpadButtonSet buttons = RestrictXpadButtons(
            m_WindowsKeyboard.GetButtons(m_KeyboardMap));

        // マッピングが有効ならば接続状態
        pOutValue->attributes.Set<BasicXpadAttribute::IsConnected>(true);

        // キーボードによるエミュレーション結果をマージ
        pOutValue->buttons = MergeXpadButtons(pOutValue->buttons, buttons);
    }

    //!< 基本的な構成を持つ Xpad の HidShell 向けに共有される入力状態を更新
    ::nn::hid::debug::BasicXpadAutoPilotState basicXpadAutoPilotState = {};
    GetXpadAutoPilotState(&basicXpadAutoPilotState, *pOutValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        SetHidShellXpadState(m_HidShellPort, basicXpadAutoPilotState));

    //!< 基本的な構成を持つ Xpad の HidShell 向けに共有される自動操作状態を取得
    bool isBasicXpadAutoPilotEnabled = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        GetHidShellXpadState(&isBasicXpadAutoPilotEnabled,
                             &basicXpadAutoPilotState, m_HidShellPort));
    if (isBasicXpadAutoPilotEnabled)
    {
        GetXpadState(pOutValue, basicXpadAutoPilotState);
    }
}

}}} // namespace nn::hid::detail
