﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEventTypes.h>

#include "hid_ActivationCount.h"
#include "hid_MultiWaitEvent.h"
#include "hid_SynchronizedTimer.h"

namespace nn { namespace hid { namespace detail {

//!< Xcdを初期化/終了及び駆動するために使われるクラスです。XCDを使うためには本クラスを通じてActivateしておく必要があります
class XcdDriver final
{
    NN_DISALLOW_COPY(XcdDriver);
    NN_DISALLOW_MOVE(XcdDriver);

private:
    //!< スレッド操作排他用のミューテックス
    ::nn::os::Mutex m_ThreadMutex;

    //!< アクティブ化回数
    ActivationCount m_ActivationCount;

    //!< 多重待ちオブジェクト
    ::nn::os::MultiWaitType* m_pMultiWait;

public:
    XcdDriver() NN_NOEXCEPT;

    ~XcdDriver() NN_NOEXCEPT;

    //!< Xcd ドライバをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< Xcd ドライバを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< 多重待ちオブジェクトを Xcd ドライバにリンクします。
    void LinkMultiWait(::nn::os::MultiWaitType* pMultiWaitType) NN_NOEXCEPT;

    //!< 多重待ちオブジェクトを Xcd ドライバにアンリンクします。
    void UnlinkMultiWait() NN_NOEXCEPT;

    //!< イベントシグナル時の Xcd の処理を実行します。
    bool Proceed(const ::nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

    //!< 定常的な Xcd の処理を実行します。
    void ProceedPeriodic() NN_NOEXCEPT;

    //!< 定常的な Xcd の処理を待機します
    nn::os::MultiWaitHolderType* WaitEvent() NN_NOEXCEPT;

    //!< ボタンペアリングを開始します。
    void StartButtonPairing() NN_NOEXCEPT;

    //!< Xcd をサスペンドします
    Result Suspend() NN_NOEXCEPT;

    //!< Xcd をレジュームします
    Result Resume() NN_NOEXCEPT;

    //!< 有線デバイス登録を有効にします。
    Result EnableWiredDeviceRegistration() NN_NOEXCEPT;

    //!< 有線デバイス登録を無効にします。
    Result DisableWiredDeviceRegistration() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
