﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Windows.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid { namespace detail {

//!< 汎用コントローラの能力の制御フラグ定義です。
struct WindowsGenericPadAbilityFlag final
{
    typedef ::nn::util::BitFlagSet<32, WindowsGenericPadAbilityFlag>::Flag<0>
            IsXInput;   //!< XInput 対応コントローラとして利用可能か
    typedef ::nn::util::BitFlagSet<32, WindowsGenericPadAbilityFlag>::Flag<1>
            IsJoystick; //!< ジョイスティックとして利用可能か
    typedef ::nn::util::BitFlagSet<32, WindowsGenericPadAbilityFlag>::Flag<4>
            HasPov;     //!< POV を持つか否か
    typedef ::nn::util::BitFlagSet<32, WindowsGenericPadAbilityFlag>::Flag<8>
            HasZ;       //!< Z 軸を持つか否か
    typedef ::nn::util::BitFlagSet<32, WindowsGenericPadAbilityFlag>::Flag<9>
            HasR;       //!< R 軸を持つか否か
    typedef ::nn::util::BitFlagSet<32, WindowsGenericPadAbilityFlag>::Flag<10>
            HasU;       //!< U 軸を持つか否か
    typedef ::nn::util::BitFlagSet<32, WindowsGenericPadAbilityFlag>::Flag<11>
            HasV;       //!< V 軸を持つか否か
};

//!< 汎用コントローラの能力の制御フラグ集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, WindowsGenericPadAbilityFlag>
        WindowsGenericPadAbilityFlagSet;

//!< 汎用コントローラの能力を表す構造体です。
struct WindowsGenericPadAbility final
{
    WindowsGenericPadAbilityFlagSet flags;  //!< 制御フラグ
    uint32_t index;                         //!< インデックス
    char deviceName[256];                   //!< デバイス名
    uint16_t vid;                           //!< ベンダ ID
    uint16_t pid;                           //!< プロダクト ID
    uint32_t buttonCount;                   //!< ボタンの数
    int32_t xMax;                           //!< X 軸の最大値
    int32_t xMin;                           //!< X 軸の最小値
    int32_t yMax;                           //!< Y 軸の最大値
    int32_t yMin;                           //!< Y 軸の最小値
    int32_t zMax;                           //!< Z 軸の最大値
    int32_t zMin;                           //!< Z 軸の最小値
    int32_t rMax;                           //!< R 軸の最大値
    int32_t rMin;                           //!< R 軸の最小値
    int32_t uMax;                           //!< U 軸の最大値
    int32_t uMin;                           //!< U 軸の最小値
    int32_t vMax;                           //!< V 軸の最大値
    int32_t vMin;                           //!< V 軸の最小値
};

//!< 汎用コントローラの識別子が有効か否かを表す値を返します。
bool IsValidWindowsGenericPadId(int id) NN_NOEXCEPT;

//!< 汎用コントローラの使用者を追加します。
::nn::Result AddWindowsGenericPadUser(int id) NN_NOEXCEPT;

//!< 汎用コントローラの使用者を削除します。
::nn::Result RemoveWindowsGenericPadUser(int id) NN_NOEXCEPT;

//!< 使用者数の少ない汎用コントローラを取得します。
::nn::Result GetUnbusyWindowsGenericPad(int* pOutValue) NN_NOEXCEPT;

//!< 汎用コントローラの能力を取得します。
::nn::Result GetWindowsGenericPadAbility(
    WindowsGenericPadAbility* pOutValue, int id) NN_NOEXCEPT;

//!< 汎用コントローラの能力を更新します。
::nn::Result UpdateWindowsGenericPadAbilities(
    const WindowsGenericPadAbility abilities[], size_t count) NN_NOEXCEPT;

}}} // namespace nn::hid::detail
