﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <iterator>
#include <limits>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_FileMappingObject-os.win.h"
#include "hid_StaticObject.h"
#include "hid_WindowsGenericPadAccessor-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 汎用コントローラの管理情報のファイルマッピングオブジェクトの名前です。
const char* const FileMappingObjectName = "NINTENDO_GENERIC_PAD_BROKER";

//!< 汎用コントローラの最大数
const int WindowsGenericPadCountMax = 16;

//!< 汎用コントローラのクライアントの最大数
const int WindowsGenericPadClientCountMax = 64;

//!< 不正な汎用コントローラのクライアント ID
const int64_t InvalidWindowsGenericPadClientId = 0;

//!< 汎用コントローラのクライアントのタイムアウト時間（ミリ秒）
const int64_t WindowsGenericPadClientTimeOut = 500;

//!< 汎用コントローラのクライアントを表す構造体です。
struct WindowsGenericPadClient;

//!< 汎用コントローラの管理情報のアドレスマップを表す構造体です。
struct WindowsGenericPadMap;

//!< 汎用コントローラの管理情報の保持を扱うクラスです。
class WindowsGenericPadMapHolder final
{
    NN_DISALLOW_COPY(WindowsGenericPadMapHolder);
    NN_DISALLOW_MOVE(WindowsGenericPadMapHolder);

private:
    //!< ファイルマッピングオブジェクト
    FileMappingObject m_FileMappingObject;

    //!< キャッシュされているか否か
    bool m_IsCached;

    //!< キャッシュされた汎用コントローラのエントリ
    WindowsGenericPadMap* m_CachedMapAddress;

    //!< 当該インスタンスのクライアント ID
    int64_t m_ClientId;

    //!< 当該インスタンスから汎用コントローラを利用しているユーザの数
    int32_t m_UserCounts[WindowsGenericPadCountMax];

public:
    WindowsGenericPadMapHolder() NN_NOEXCEPT;

    //!< 汎用コントローラの使用者をエントリに追加します。
    ::nn::Result AddUser(int id) NN_NOEXCEPT;

    //!< 汎用コントローラの使用者をエントリから削除します。
    ::nn::Result RemoveUser(int id) NN_NOEXCEPT;

    //!< 使用者数の少ない汎用コントローラの ID を取得します。
    ::nn::Result GetUnbusyId(int* pOutValue) NN_NOEXCEPT;

    //!< 汎用コントローラの能力を取得します。
    ::nn::Result GetAbility(
        WindowsGenericPadAbility* pOutValue, int id) NN_NOEXCEPT;

    //!< 汎用コントローラの能力を更新します。
    ::nn::Result UpdateAbilities(
        const WindowsGenericPadAbility abilities[], size_t count) NN_NOEXCEPT;

private:
    template<typename T>
    ::nn::Result ProcessMap(
        const T& args,
        ::nn::Result (*processor)(
            WindowsGenericPadMapHolder& that,
            WindowsGenericPadMap& map, const T& args) NN_NOEXCEPT
        ) NN_NOEXCEPT;

    template<typename T>
    ::nn::Result ProcessClient(
        const T& args,
        ::nn::Result (*processor)(
            WindowsGenericPadMapHolder& that,
            WindowsGenericPadMap& map, WindowsGenericPadClient& client,
            const T& args) NN_NOEXCEPT
        ) NN_NOEXCEPT;
};

//!< WindowsGenericPadMapHolder を返します。
WindowsGenericPadMapHolder& GetWindowsGenericPadMapHolder() NN_NOEXCEPT;

} // namespace

bool IsValidWindowsGenericPadId(int id) NN_NOEXCEPT
{
    return (0 <= id && id < WindowsGenericPadCountMax);
}

::nn::Result AddWindowsGenericPadUser(int id) NN_NOEXCEPT
{
    NN_RESULT_DO(GetWindowsGenericPadMapHolder().AddUser(id));
    NN_RESULT_SUCCESS;
}

::nn::Result RemoveWindowsGenericPadUser(int id) NN_NOEXCEPT
{
    NN_RESULT_DO(GetWindowsGenericPadMapHolder().RemoveUser(id));
    NN_RESULT_SUCCESS;
}

::nn::Result GetUnbusyWindowsGenericPad(int* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_DO(GetWindowsGenericPadMapHolder().GetUnbusyId(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result GetWindowsGenericPadAbility(
    WindowsGenericPadAbility* pOutValue, int id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_DO(
        GetWindowsGenericPadMapHolder().GetAbility(pOutValue, id));
    NN_RESULT_SUCCESS;
}

::nn::Result UpdateWindowsGenericPadAbilities(
    const WindowsGenericPadAbility abilities[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(abilities);
    NN_RESULT_DO(
        GetWindowsGenericPadMapHolder().UpdateAbilities(abilities, count));
    NN_RESULT_SUCCESS;
}

namespace {

struct WindowsGenericPadClient
{
    //!< 汎用コントローラのクライアント ID
    int64_t clientId;

    //!< 汎用コントローラのクライアントのタイムスタンプ
    ::nn::TimeSpanType timeStamp;

    //!< 汎用コントローラを利用しているユーザの数
    int32_t userCounts[WindowsGenericPadCountMax];
};

struct WindowsGenericPadMap final
{
    //!< 汎用コントローラの能力
    WindowsGenericPadAbility abilities[WindowsGenericPadCountMax];

    //!< 汎用コントローラの最新のクライアント ID
    int64_t latestClientId;

    //!< 汎用コントローラのクライアント
    WindowsGenericPadClient clients[WindowsGenericPadClientCountMax];
};

template<typename T>
class MutexWrapper final
{
    NN_DISALLOW_COPY(MutexWrapper);
    NN_DISALLOW_MOVE(MutexWrapper);

private:
    T& m_Impl;

public:
    explicit MutexWrapper(T& impl) NN_NOEXCEPT
        : m_Impl(impl)
    {
        // 何もしない
    }

    void lock() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Impl.Lock());
    }

    void unlock() NN_NOEXCEPT
    {
        m_Impl.Unlock();
    }
};

//!< 取り付けられた汎用コントローラを追加します。
void AppendAttachedWindowsGenericPads(
    WindowsGenericPadAbility (&oldies)[WindowsGenericPadCountMax],
    const WindowsGenericPadAbility newies[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(newies);

    // 汎用コントローラが登録済みか否かを判定するためのマスク
    const WindowsGenericPadAbilityFlagSet mask =
        WindowsGenericPadAbilityFlag::IsXInput::Mask |
        WindowsGenericPadAbilityFlag::IsJoystick::Mask;

    for (size_t i = 0; i < count; ++i)
    {
        const WindowsGenericPadAbility& newie = newies[i];

        const WindowsGenericPadAbilityFlagSet newieFlags = newie.flags & mask;

        if (newieFlags.IsAnyOn())
        {
            auto position = -1;

            auto exists = false;

            for (int j = 0; j < NN_ARRAY_SIZE(oldies); ++j)
            {
                WindowsGenericPadAbility& oldie = oldies[j];

                const WindowsGenericPadAbilityFlagSet oldieFlags =
                    oldie.flags &  mask;

                if (position == -1 && oldieFlags.IsAllOff())
                {
                    position = j;
                }
                else if (
                    newieFlags == oldieFlags && newie.index == oldie.index &&
                    ::std::strncmp(
                        newie.deviceName, oldie.deviceName,
                        NN_ARRAY_SIZE(newie.deviceName)) == 0)
                {
                    // 登録済みであれば更新
                    position = j;

                    exists = true;

                    break;
                }
            }

            // XInput 対応コントローラは極力 as-is で割り当て
            if (!exists &&
                newie.flags.Test<WindowsGenericPadAbilityFlag::IsXInput>() &&
                newie.index < NN_ARRAY_SIZE(oldies) &&
                (oldies[newie.index].flags & mask).IsAllOff())
            {
                position = static_cast<int>(newie.index);
            }

            if (0 <= position)
            {
                oldies[position] = newie;
            }
        }
    }
}

//!< 取り外された汎用コントローラを削除します。
void RemoveDetachedWindowsGenericPads(
    WindowsGenericPadAbility (&oldies)[WindowsGenericPadCountMax],
    const WindowsGenericPadAbility newies[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(newies);

    // 汎用コントローラが登録済みか否かを判定するためのマスク
    const WindowsGenericPadAbilityFlagSet mask =
        WindowsGenericPadAbilityFlag::IsXInput::Mask |
        WindowsGenericPadAbilityFlag::IsJoystick::Mask;

    for (WindowsGenericPadAbility& oldie : oldies)
    {
        const WindowsGenericPadAbilityFlagSet oldieFlags = oldie.flags & mask;

        if (oldieFlags.IsAnyOn())
        {
            auto finds = false;

            for (size_t i = 0; i < count; ++i)
            {
                const WindowsGenericPadAbility& newie = newies[i];

                const WindowsGenericPadAbilityFlagSet newieFlags =
                    newie.flags & mask;

                if (oldie.index == newie.index && oldieFlags == newieFlags &&
                    ::std::strncmp(
                        oldie.deviceName, newie.deviceName,
                        NN_ARRAY_SIZE(oldie.deviceName)) == 0)
                {
                    finds = true;

                    break;
                }
            }

            if (!finds)
            {
                oldie = WindowsGenericPadAbility();
            }
        }
    }
}

//!< 汎用コントローラの使用者の数を返します。
int32_t GetWindowsGenericPadUserCount(
    const WindowsGenericPadClient (&clients)[WindowsGenericPadClientCountMax],
    int id) NN_NOEXCEPT
{
    int32_t count = 0;

    if (IsValidWindowsGenericPadId(id))
    {
        for (const WindowsGenericPadClient& client : clients)
        {
            count += client.userCounts[id];
        }
    }

    return count;
}

//!< 汎用コントローラのクライアントを検索します。
WindowsGenericPadClient* FindWindowsGenericPadClient(
    WindowsGenericPadClient (&clients)[WindowsGenericPadClientCountMax],
    int64_t clientId) NN_NOEXCEPT
{
    for (WindowsGenericPadClient& client : clients)
    {
        if (client.clientId == clientId)
        {
            return &client;
        }
    }

    return nullptr;
}

//!< 汎用コントローラのクライアントを登録します。
WindowsGenericPadClient* RegisterWindowsGenericPadClient(
    WindowsGenericPadClient (&clients)[WindowsGenericPadClientCountMax],
    int64_t clientId,
    ::nn::TimeSpanType timeStamp,
    const int32_t (&userCounts)[WindowsGenericPadCountMax]) NN_NOEXCEPT
{
    for (WindowsGenericPadClient& client : clients)
    {
        if (client.clientId == InvalidWindowsGenericPadClientId)
        {
            client.clientId = clientId;

            client.timeStamp = timeStamp;

            ::std::copy(
                ::std::begin(userCounts), ::std::end(userCounts),
                client.userCounts);

            return &client;
        }
    }

    return nullptr;
}

//!< 無効な汎用コントローラのクライアントを削除します。
void RemoveExpiredWindowsGenericPadClient(
    WindowsGenericPadClient (&clients)[WindowsGenericPadClientCountMax],
    ::nn::TimeSpanType timeStamp) NN_NOEXCEPT
{
    const auto timeOut =
        ::nn::TimeSpanType::FromMilliSeconds(WindowsGenericPadClientTimeOut);

    for (WindowsGenericPadClient& client : clients)
    {
        if (client.clientId != InvalidWindowsGenericPadClientId &&
            timeOut < (timeStamp - client.timeStamp))
        {
            client = WindowsGenericPadClient();
        }
    }
}

WindowsGenericPadMapHolder::WindowsGenericPadMapHolder() NN_NOEXCEPT
    : m_FileMappingObject(sizeof(WindowsGenericPadMap), FileMappingObjectName)
    , m_IsCached(false)
    , m_CachedMapAddress(nullptr)
    , m_ClientId(InvalidWindowsGenericPadClientId)
{
    for (int32_t& userCount : m_UserCounts)
    {
        userCount = 0;
    }
}

::nn::Result WindowsGenericPadMapHolder::AddUser(int id) NN_NOEXCEPT
{
    NN_RESULT_DO(this->ProcessClient<int>(id, [](
        WindowsGenericPadMapHolder& that,
        WindowsGenericPadMap& map, WindowsGenericPadClient& client,
        const int& id) NN_NOEXCEPT -> ::nn::Result
    {
        if (IsValidWindowsGenericPadId(id))
        {
            // 使用者数が上限に達した場合は失敗
            NN_RESULT_THROW_UNLESS(
                GetWindowsGenericPadUserCount(
                    map.clients, id) < ::std::numeric_limits<int32_t>::max(),
                ResultJoystickUserUpperLimitOver());

            client.userCounts[id] += 1;

            that.m_UserCounts[id] += 1;
        }

        NN_RESULT_SUCCESS;
    }));

    NN_RESULT_SUCCESS;
}

::nn::Result WindowsGenericPadMapHolder::RemoveUser(int id) NN_NOEXCEPT
{
    NN_RESULT_DO(this->ProcessClient<int>(id, [](
        WindowsGenericPadMapHolder& that,
        WindowsGenericPadMap& map, WindowsGenericPadClient& client,
        const int& id) NN_NOEXCEPT -> ::nn::Result
    {
        NN_UNUSED(map);

        if (IsValidWindowsGenericPadId(id))
        {
            int32_t& userCount = client.userCounts[id];

            // 使用者数が下限に達した場合は失敗
            NN_RESULT_THROW_UNLESS(
                0 < userCount, ResultJoystickUserUpperLimitOver());

            userCount -= 1;

            that.m_UserCounts[id] -= 1;
        }

        NN_RESULT_SUCCESS;
    }));

    NN_RESULT_SUCCESS;
}

::nn::Result WindowsGenericPadMapHolder::GetUnbusyId(int* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    typedef decltype(pOutValue) T;

    NN_RESULT_DO(this->ProcessClient<T>(pOutValue, [](
        WindowsGenericPadMapHolder& that,
        WindowsGenericPadMap& map, WindowsGenericPadClient& client,
        const T& pOutValue) NN_NOEXCEPT -> ::nn::Result
    {
        int id = WindowsGenericPadCountMax;

        int32_t minimumCount = ::std::numeric_limits<int32_t>::max();

        for (int i = 0; i < WindowsGenericPadCountMax; ++i)
        {
            const int32_t count = GetWindowsGenericPadUserCount(map.clients, i);

            if (count < minimumCount)
            {
                id = i;

                minimumCount = count;
            }

            if (minimumCount == 0)
            {
                break;
            }
        }

        // エントリに空きが無ければ失敗
        NN_RESULT_THROW_UNLESS(
            id < WindowsGenericPadCountMax, ResultJoystickNoRemains());

        client.userCounts[id] += 1;

        that.m_UserCounts[id] += 1;

        *pOutValue = id;

        NN_RESULT_SUCCESS;
    }));

    NN_RESULT_SUCCESS;
}

::nn::Result WindowsGenericPadMapHolder::GetAbility(
    WindowsGenericPadAbility* pOutValue, int id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    struct LocalArgs
    {
        WindowsGenericPadAbility* pOutValue;
        int id;
        ::nn::TimeSpanType timeStamp;
    };

    LocalArgs localArgs = {
        pOutValue, id, ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick())
    };

    NN_RESULT_DO(this->ProcessClient<LocalArgs>(localArgs, [](
        WindowsGenericPadMapHolder& that,
        WindowsGenericPadMap& map, WindowsGenericPadClient& client,
        const LocalArgs& localArgs) NN_NOEXCEPT -> ::nn::Result
    {
        NN_UNUSED(that);

        client.timeStamp = localArgs.timeStamp;

        RemoveExpiredWindowsGenericPadClient(map.clients, localArgs.timeStamp);

        *localArgs.pOutValue =
            !IsValidWindowsGenericPadId(localArgs.id)
                ? WindowsGenericPadAbility()
                : map.abilities[localArgs.id];

        NN_RESULT_SUCCESS;
    }));

    NN_RESULT_SUCCESS;
}

::nn::Result WindowsGenericPadMapHolder::UpdateAbilities(
    const WindowsGenericPadAbility abilities[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(abilities);

    struct LocalArgs
    {
        const WindowsGenericPadAbility* abilities;
        size_t count;
    };

    LocalArgs localArgs = { abilities, count };

    NN_RESULT_DO(this->ProcessMap<LocalArgs>(localArgs, [](
        WindowsGenericPadMapHolder& that,
        WindowsGenericPadMap& map, const LocalArgs& localArgs
        ) NN_NOEXCEPT -> ::nn::Result
    {
        NN_UNUSED(that);

        RemoveDetachedWindowsGenericPads(
            map.abilities, localArgs.abilities, localArgs.count);

        AppendAttachedWindowsGenericPads(
            map.abilities, localArgs.abilities, localArgs.count);

        NN_RESULT_SUCCESS;
    }));

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result WindowsGenericPadMapHolder::ProcessMap(
    const T& args,
    ::nn::Result (*processor)(
        WindowsGenericPadMapHolder& that,
        WindowsGenericPadMap& map, const T& args) NN_NOEXCEPT
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);

    MutexWrapper<FileMappingObject> mutex(m_FileMappingObject);

    ::std::lock_guard<decltype(mutex)> locker(mutex);

    if (!m_IsCached)
    {
        auto exists = false;

        WindowsGenericPadMap* address = nullptr;

        NN_RESULT_DO(
            m_FileMappingObject.GetAddress(
                reinterpret_cast<LPVOID*>(&address), &exists));

        NN_SDK_ASSERT_NOT_NULL(address);

        NN_SDK_ASSERT_ALIGNED(address, NN_ALIGNOF(WindowsGenericPadMap));

        NN_SDK_ASSERT_EQUAL(
            static_cast<size_t>(m_FileMappingObject.GetCount()),
            sizeof(WindowsGenericPadMap));

        if (!exists)
        {
            // 最初に汎用コントローラのエントリを取得した場合は全て未使用状態に
            *address = WindowsGenericPadMap();
        }

        m_IsCached = true;

        m_CachedMapAddress = address;

        m_ClientId = address->latestClientId + 1;

        address->latestClientId = m_ClientId;
    }

    NN_SDK_ASSERT_NOT_NULL(m_CachedMapAddress);

    NN_RESULT_DO(processor(*this, *m_CachedMapAddress, args));

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result WindowsGenericPadMapHolder::ProcessClient(
    const T& args,
    ::nn::Result (*processor)(
        WindowsGenericPadMapHolder& that,
        WindowsGenericPadMap& map, WindowsGenericPadClient& client,
        const T& args) NN_NOEXCEPT
    ) NN_NOEXCEPT
{
    class LocalArgs
    {
    private:
        decltype(processor) m_Processor;
        decltype(args) m_Args;
        ::nn::TimeSpan m_TimeStamp;

    public:
        LocalArgs(
            decltype(processor) processor, decltype(args) args,
            ::nn::TimeSpan timeStamp) NN_NOEXCEPT
            : m_Processor(processor)
            , m_Args(args)
            , m_TimeStamp(timeStamp) {}

        decltype(processor) GetProcessor() const NN_NOEXCEPT
        {
            return m_Processor;
        }

        decltype(args) GetArgs() const NN_NOEXCEPT { return m_Args; }

        ::nn::TimeSpan GetTimeStamp() const NN_NOEXCEPT { return m_TimeStamp; }
    };

    LocalArgs localArgs(
        processor, args,
        ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick()));

    NN_RESULT_DO(this->ProcessMap<LocalArgs>(localArgs, [](
        WindowsGenericPadMapHolder& that,
        WindowsGenericPadMap& map, const LocalArgs& localArgs
        ) NN_NOEXCEPT -> ::nn::Result
    {
        const int64_t clientId = that.m_ClientId;

        WindowsGenericPadClient* pClient =
            FindWindowsGenericPadClient(map.clients, clientId);

        if (pClient == nullptr)
        {
            const ::nn::TimeSpanType timeStamp = localArgs.GetTimeStamp();

            RemoveExpiredWindowsGenericPadClient(map.clients, timeStamp);

            pClient = RegisterWindowsGenericPadClient(
                map.clients, clientId, timeStamp, that.m_UserCounts);

            NN_RESULT_THROW_UNLESS(
                pClient != nullptr, ResultJoystickClientUpperLimitOver());
        }

        NN_RESULT_DO(
            localArgs.GetProcessor()(that, map, *pClient, localArgs.GetArgs()));

        NN_RESULT_SUCCESS;
    }));

    NN_RESULT_SUCCESS;
}

WindowsGenericPadMapHolder& GetWindowsGenericPadMapHolder() NN_NOEXCEPT
{
    return StaticObject<WindowsGenericPadMapHolder>::Get();
}

} // namespace

}}} // namespace nn::hid::detail
