﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/hid_ResultPrivate.h>

#include "hid_VibrationFileApiImpl.h"

namespace {

const int MetaDataSizeOffset = 0;
const int FormatIdOffset = 4;
const int SamplingRateOffset = 6;
const int LoopStartOffset = 8;
const int LoopEndOffset = 12;
const int LoopIntervalOffset = 16;

const uint32_t MetaDataSizeMin = 4;
const uint32_t MetaDataSizeForLoopInfo = 12;
const uint32_t MetaDataSizeForLoopIntervalInfo = 16;
const uint16_t ExpectedFormatId = 3;
const uint16_t ExpectedSamplingRate = 200;
const uint32_t LoopIntervalMax = 0x7FFFFFFF;

struct VibrationFileFormat
{
    //!< バイト配列から数値を取得します。
    template<typename T>
    static inline T Get(const uint8_t* p)
    {
        T value = 0;
        for(size_t i = 0; i < sizeof(T); i++)
        {
            value += static_cast<T>(p[i]) << (i * 8);
        }
        return value;
    }
};

//!< バイトを振幅値に変換した値を取得します。
inline float GetAmplitudeFromByte(uint8_t b)
{
    return static_cast<float>(b) / 255.0f;
}

//!< バイトを周波数値に変換した値を取得します。
inline float GetFrequencyFromByte(uint8_t b)
{
    const int FreqMin = 10;
    const int FreqResolution = 32;
    const float powTable[FreqResolution] = {
        1.000000f, 1.021897f, 1.044274f, 1.067140f, 1.090508f, 1.114387f, 1.138789f, 1.163725f,
        1.189207f, 1.215247f, 1.241858f, 1.269051f, 1.296840f, 1.325237f, 1.354256f, 1.383910f,
        1.414214f, 1.445181f, 1.476826f, 1.509164f, 1.542211f, 1.575981f, 1.610490f, 1.645755f,
        1.681793f, 1.718619f, 1.756252f, 1.794709f, 1.834008f, 1.874168f, 1.915207f, 1.957144f,
    };

    return static_cast<float>(FreqMin << (b / FreqResolution)) * powTable[b % FreqResolution];
}

} // anonymous-namespace

namespace nn { namespace hid { namespace detail {

Result ParseVibrationFile(
    VibrationFileInfo* pOutInfo,
    VibrationFileParserContextImpl* pOutContext,
    const void* address,
    size_t fileSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(address);

    const uint8_t* p = reinterpret_cast<const uint8_t*>(address);

    // メタデータ領域をパース
    pOutInfo->metaDataSize =
        VibrationFileFormat::Get<uint32_t>(p + MetaDataSizeOffset);
    pOutInfo->formatId =
        VibrationFileFormat::Get<uint16_t>(p + FormatIdOffset);
    pOutInfo->samplingRate =
        VibrationFileFormat::Get<uint16_t>(p + SamplingRateOffset);
    NN_RESULT_THROW_UNLESS(
        pOutInfo->metaDataSize >= MetaDataSizeMin,
        ResultVibrationFileInvalidMetaDataSize());
    NN_RESULT_THROW_UNLESS(
        pOutInfo->formatId == ExpectedFormatId,
        ResultVibrationFileInvalidFormatId());
    NN_RESULT_THROW_UNLESS(
        pOutInfo->samplingRate == ExpectedSamplingRate,
        ResultVibrationFileInvalidSamplingRate());

    if (pOutInfo->metaDataSize >= MetaDataSizeForLoopInfo)
    {
        pOutInfo->isLoop = 1;
        pOutInfo->loopInterval = 0;

        // ループ情報をパース
        pOutInfo->loopStartPosition =
            VibrationFileFormat::Get<uint32_t>(p + LoopStartOffset);
        pOutInfo->loopEndPosition =
            VibrationFileFormat::Get<uint32_t>(p + LoopEndOffset);
        NN_RESULT_THROW_UNLESS(
            pOutInfo->loopStartPosition <= pOutInfo->loopEndPosition,
            ResultVibrationFileInvalidLoopInfo());

        if(pOutInfo->metaDataSize >= MetaDataSizeForLoopIntervalInfo)
        {
            // ループインターバル情報をパース
            pOutInfo->loopInterval =
                VibrationFileFormat::Get<uint32_t>(p + LoopIntervalOffset);
            NN_RESULT_THROW_UNLESS(
                pOutInfo->loopInterval <= LoopIntervalMax,
                ResultVibrationFileInvalidLoopInfo());
        }
    }
    else
    {
        pOutInfo->isLoop = 0;
    }

    // データ領域のオフセットを計算
    int dataSizeOffset = sizeof(uint32_t) + pOutInfo->metaDataSize;
    int dataOffset = dataSizeOffset + sizeof(uint32_t);
    NN_RESULT_THROW_UNLESS(
        dataOffset <= static_cast<int>(fileSize),
        ResultVibrationFileInvalidMetaDataSize());

    // データ領域をパース
    pOutInfo->dataSize =
        VibrationFileFormat::Get<uint32_t>(p + dataSizeOffset);
    pOutInfo->sampleLength = pOutInfo->dataSize / sizeof(uint32_t);

    // ループ情報を持たない場合はデフォルト値をセット
    if (pOutInfo->isLoop == 0)
    {
        pOutInfo->loopStartPosition = 0;
        pOutInfo->loopEndPosition = static_cast<uint32_t>(pOutInfo->sampleLength);
        pOutInfo->loopInterval = 0;
    }

    NN_RESULT_THROW_UNLESS(
        pOutInfo->loopEndPosition <= static_cast<uint32_t>(pOutInfo->sampleLength),
        ResultVibrationFileInvalidLoopInfo());

    // パーサーコンテクストに必要な情報をセット
    pOutContext->dataAddress = p + dataOffset;
    pOutContext->sampleLength = pOutInfo->sampleLength;

    // データ終端の位置をチェック
    int dataEnd = dataOffset + pOutInfo->dataSize;
    NN_RESULT_THROW_UNLESS(
        dataEnd <= static_cast<int>(fileSize),
        ResultVibrationFileInvalidDataSize());

    // [Note] FormatId == 3 ではデコーダの初期化は不要
    //        将来的に FormatId == 1, 2 に対応する際にはここでデコーダコンテクストを初期化する

    NN_RESULT_SUCCESS;
}

Result RetrieveVibrationValue(
    VibrationValue* pOutValue,
    int position,
    VibrationFileParserContextImpl* pContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_RANGE(position, 0, pContext->sampleLength);

    const uint8_t* p = pContext->dataAddress;
    NN_SDK_REQUIRES_NOT_NULL(p);

    p += position * sizeof(uint32_t);
    pOutValue->amplitudeLow = GetAmplitudeFromByte(p[0]);
    pOutValue->frequencyLow = GetFrequencyFromByte(p[1]);
    pOutValue->amplitudeHigh = GetAmplitudeFromByte(p[2]);
    pOutValue->frequencyHigh = GetFrequencyFromByte(p[3]);

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
