﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BitPack.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/hid_NpadPalma.h>
#include <nn/hid/system/hid_Npad.h>
#include "hid_VibrationDeviceHandle.h"
#include "hid_NpadId.h"

namespace nn { namespace hid { namespace detail {

NpadVibrationDeviceIdx GetVibrationDeviceIdxFromPosition(VibrationDevicePosition position) NN_NOEXCEPT
{
    switch(position)
    {
    case VibrationDevicePosition_Left:
        return NpadVibrationDeviceIdx_Left;
    case VibrationDevicePosition_Right:
        return NpadVibrationDeviceIdx_Right;
    case VibrationDevicePosition_None:
        return NpadVibrationDeviceIdx_None;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

//!< NpadVibrationDeviceIdx から VibrationDevicePosition を取得します。
VibrationDevicePosition GetVibrationDevicePositionFromIdx(VibrationDeviceType vibrationDeviceType, NpadVibrationDeviceIdx vibrationDeviceIdx) NN_NOEXCEPT
{
    if (vibrationDeviceType == VibrationDeviceType_LinearResonantActuator)
    {
        switch (vibrationDeviceIdx)
        {
        case NpadVibrationDeviceIdx_Left:
            return VibrationDevicePosition_Left;
        case NpadVibrationDeviceIdx_Right:
            return VibrationDevicePosition_Right;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    else if(vibrationDeviceType == VibrationDeviceType_GcErm)
    {
        return VibrationDevicePosition_None;
    }

    return VibrationDevicePosition_None;
}

//<! VibrationDeviceHandle のビットフィールド定義です。
struct VibrationDeviceHandleField final
{
    //!< Npad の種類
    typedef ::nn::util::BitPack32::Field<0, 8, uint8_t> NpadTypeValue;

    //!< プレイヤー番号
    typedef ::nn::util::BitPack32::Field<8, 8, uint8_t> PlayerNumber;

    //!< 振動子番号
    typedef ::nn::util::BitPack32::Field<16, 8, uint8_t> VibrationDeviceIdx;
};

VibrationDeviceHandle MakeVibrationDeviceHandle(uint8_t npadStyleId, int playerNumber, int vibrationDeviceIdx) NN_NOEXCEPT
{
    auto bitPack = ::nn::util::BitPack32();
    bitPack.Set<VibrationDeviceHandleField::NpadTypeValue>(npadStyleId);
    bitPack.Set<VibrationDeviceHandleField::PlayerNumber>(static_cast<int8_t>(playerNumber));
    bitPack.Set<VibrationDeviceHandleField::VibrationDeviceIdx>(static_cast<int8_t>(vibrationDeviceIdx));

    VibrationDeviceHandle handle = { bitPack.storage };
    return handle;
}

::nn::Result GetVibrationDeviceHandles(VibrationDeviceHandle* pOutHandle,
                                      int* pOutCount,
                                      int size,
                                      NpadIdType npadId,
                                      NpadStyleSet style) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutCount);
    NN_ABORT_UNLESS_GREATER_EQUAL(size, 1);
    NN_RESULT_DO(detail::VerifyValidNpadId(npadId));

    if (style.Test<NpadStyleFullKey>() == true)
    {
        pOutHandle[0] = detail::MakeVibrationDeviceHandle(NpadStyleFullKey::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Left));
        *pOutCount = 1;
        if (size > 1)
        {
            pOutHandle[1] = detail::MakeVibrationDeviceHandle(NpadStyleFullKey::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Right));
            (*pOutCount)++;
        }
    }
    else if (style.Test<NpadStyleHandheld> () == true)
    {
        pOutHandle[0] = detail::MakeVibrationDeviceHandle(NpadStyleHandheld::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Left));
        *pOutCount = 1;
        if (size > 1)
        {
            pOutHandle[1] = detail::MakeVibrationDeviceHandle(NpadStyleHandheld::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Right));
            (*pOutCount)++;
        }
    }
    else if (style.Test<NpadStyleJoyDual>() == true)
    {
        pOutHandle[0] = detail::MakeVibrationDeviceHandle(NpadStyleJoyDual::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Left));
        *pOutCount = 1;
        if (size > 1)
        {
            pOutHandle[1] = detail::MakeVibrationDeviceHandle(NpadStyleJoyDual::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Right));
            (*pOutCount)++;
        }
    }
    else if (style.Test<NpadStyleJoyLeft> () == true)
    {
        *pOutHandle = detail::MakeVibrationDeviceHandle(NpadStyleJoyLeft::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Left));
        *pOutCount = 1;
    }
    else if (style.Test<NpadStyleJoyRight>() == true)
    {
        *pOutHandle = detail::MakeVibrationDeviceHandle(NpadStyleJoyRight::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Right));
        *pOutCount = 1;
    }
    else if (style.Test<NpadStyleGc>() == true)
    {
        // 暫定的に DeviceIndex は Left を用いる
        *pOutHandle = detail::MakeVibrationDeviceHandle(NpadStyleGc::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_None));
        *pOutCount = 1;
    }
    else if (style.Test<system::NpadStyleSystem> () == true)
    {
        pOutHandle[0] = detail::MakeVibrationDeviceHandle(system::NpadStyleSystem::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Left));
        *pOutCount = 1;
        if (size > 1)
        {
            pOutHandle[1] = detail::MakeVibrationDeviceHandle(system::NpadStyleSystem::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Right));
            (*pOutCount)++;
        }
    }
    else if (style.Test<system::NpadStyleSystemExt>() == true)
    {
        pOutHandle[0] = detail::MakeVibrationDeviceHandle(system::NpadStyleSystemExt::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Left));
        *pOutCount = 1;
        if (size > 1)
        {
            pOutHandle[1] = detail::MakeVibrationDeviceHandle(system::NpadStyleSystemExt::Index + NpadVibrationTypeValueOffset, static_cast<int>(npadId), static_cast<int>(NpadVibrationDeviceIdx_Right));
            (*pOutCount)++;
        }
    }
    else if (style.Test<NpadStylePalma>() == true)
    {
        // Palma は振動対応なし
        *pOutCount = 0;
    }

    NN_RESULT_SUCCESS;
}

uint8_t GetVibrationDeviceHandleXpadTypeValue(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    ::nn::util::BitPack32 bitPack = { handle._storage };
    return bitPack.Get<VibrationDeviceHandleField::NpadTypeValue>();
}

int GetVibrationDeviceHandlePlayerNumber(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    ::nn::util::BitPack32 bitPack = { handle._storage };
    return bitPack.Get<VibrationDeviceHandleField::PlayerNumber>();
}

int GetVibrationDeviceHandleDeviceIdx(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    ::nn::util::BitPack32 bitPack = { handle._storage };
    return bitPack.Get<VibrationDeviceHandleField::VibrationDeviceIdx>();
}

}}} // namespace nn::hid::detail
