﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/hid_IHidServer.sfdl.h>

#include "hid_AppletResourceUserId.h"
#include "hid_HidServer.h"
#include "hid_LockableMutexType.h"
#include "hid_VibrationDeviceApiImpl.h"

namespace nn { namespace hid { namespace detail {

namespace {

nn::sf::SharedPointer<IHidServer>& GetProxy() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));
    NN_FUNCTION_LOCAL_STATIC(nn::sf::SharedPointer<IHidServer>, s_pProxy);

    if(!s_IsInitialized)
    {
        static LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if(!s_IsInitialized)
        {
            CreateHidServerProxy(&s_pProxy);
            s_IsInitialized = true;
        }
    }

    return s_pProxy;
}

}

Result PermitVibration(bool isPermitted) NN_NOEXCEPT
{
    auto pProxy = GetProxy();

    NN_RESULT_DO(pProxy->PermitVibration(isPermitted));
    NN_RESULT_SUCCESS;
}

Result IsVibrationPermitted(bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto pProxy = GetProxy();

    NN_RESULT_DO(pProxy->IsVibrationPermitted(pOutValue));
    NN_RESULT_SUCCESS;
}

Result BeginPermitVibrationSession() NN_NOEXCEPT
{
    auto pProxy = GetProxy();

    NN_RESULT_DO(pProxy->BeginPermitVibrationSession(GetAppletResourceUserId()));
    NN_RESULT_SUCCESS;
}

Result EndPermitVibrationSession() NN_NOEXCEPT
{
    auto pProxy = GetProxy();

    NN_RESULT_DO(pProxy->EndPermitVibrationSession());
    NN_RESULT_SUCCESS;
}

Result InitializeVibrationDevice(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsListCreated, (false));
    NN_FUNCTION_LOCAL_STATIC(nn::sf::SharedPointer<IActiveVibrationDeviceList>, s_pList);

    auto pProxy = GetProxy();

    if (!s_IsListCreated)
    {
        static LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsListCreated)
        {
            NN_RESULT_DO(pProxy->CreateActiveVibrationDeviceList(&s_pList));
            s_IsListCreated = true;
        }
    }

    s_pList->ActivateVibrationDevice(handle);

    NN_RESULT_SUCCESS;
}

Result GetVibrationDeviceInfo(VibrationDeviceInfo* pOutValue, const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto pProxy = GetProxy();

    VibrationDeviceInfoForIpc info;
    NN_RESULT_DO(pProxy->GetVibrationDeviceInfo(&info, handle));
    *pOutValue = info.ToVibrationDeviceInfo();
    NN_RESULT_SUCCESS;
}

Result SendVibrationValue(const VibrationDeviceHandle& handle, const VibrationValue& value) NN_NOEXCEPT
{
    auto pProxy = GetProxy();

    NN_RESULT_DO(pProxy->SendVibrationValue(GetAppletResourceUserId(), handle, value));
    NN_RESULT_SUCCESS;
}

Result SendVibrationValues(const VibrationDeviceHandle* handles, const VibrationValue* values, int valueCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handles);
    NN_SDK_REQUIRES_NOT_NULL(values);
    NN_SDK_REQUIRES_LESS_EQUAL(valueCount, SendableVibrationValueCountMax);

    auto pProxy = GetProxy();

    NN_RESULT_DO(pProxy->SendVibrationValues(
        GetAppletResourceUserId(),
        ::nn::sf::InArray<VibrationDeviceHandle>(handles, static_cast<size_t>(valueCount)),
        ::nn::sf::InArray<VibrationValue>(values, static_cast<size_t>(valueCount))));
    NN_RESULT_SUCCESS;
}

Result GetActualVibrationValue(VibrationValue* pOutValue, const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    auto pProxy = GetProxy();

    NN_RESULT_DO(pProxy->GetActualVibrationValue(pOutValue, GetAppletResourceUserId(), handle));
    NN_RESULT_SUCCESS;
}

Result SendVibrationGcErmCommand(const VibrationDeviceHandle& handle, const VibrationGcErmCommand& command) NN_NOEXCEPT
{
    auto pProxy = GetProxy();

    NN_RESULT_DO(pProxy->SendVibrationGcErmCommand(GetAppletResourceUserId(), handle, command));
    NN_RESULT_SUCCESS;
}

Result GetActualVibrationGcErmCommand(VibrationGcErmCommand* pOutValue, const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    auto pProxy = GetProxy();

    NN_RESULT_DO(pProxy->GetActualVibrationGcErmCommand(pOutValue, GetAppletResourceUserId(), handle));
    NN_RESULT_SUCCESS;
}


}}} // namespace nn::hid::detail
