﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/detail/hid_PadTypes.h>
#include <nn/hid/detail/hid_SensorTypes.h>

#include "hid_ActivationCount.h"
#include "hid_UniquePadAssignmentManager.h"
#include "hid_UniquePadManager.h"
#include "hid_XcdFirmwareUpdater.h"

namespace nn { namespace hid { namespace detail {

//!< UniquePad に関するリソースを管理するクラスです。
class UniquePadResourceManager final
{
private:
    //!< Npad の Activation された回数
    ActivationCount m_ActivationCount;

    //!< UniquePad 操作排他用のミューテックス
    ::nn::os::Mutex m_UniquePadMutex;

    //!< UniquePad マネージャ
    UniquePadManager m_UniquePadManagers[system::UniquePadIdCountMax];

    //!< UniquePad Control マネージャ
    UniquePadAssignmentManager m_AssignmentManager;

    //!< Xcd を利用したファームウェア更新
    XcdFirmwareUpdater m_XcdFirmwareUpdater;

public:
    UniquePadResourceManager() NN_NOEXCEPT;

    ~UniquePadResourceManager() NN_NOEXCEPT;

    //!< UniquePad を初期化します
    void Initialize(HandheldManager* pHandheldManager,
                    InterruptSceneNotifier* pInterruptSceneNotifier) NN_NOEXCEPT;

    //!< SystemEvent を登録します
    void SetSystemEvents(nn::os::SystemEventType* pBluetoothFirmwareUpdateEvent,
                         nn::os::SystemEventType* pMcuFirmwareUpdateEvent,
                         nn::os::SystemEventType* pSampleUpdateEvent,
                         nn::os::TimerEventType* pResetTimeoutEvent) NN_NOEXCEPT;

    //!< IAbstractedPad をセットします
    void AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT;

    //!< UniquePadResourceManager をアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< UniquePadResourceManager を非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< UniquePad の接続状態変化通知イベントをバインドします。
    ::nn::Result AcquireUniquePadConnectionEventHandle(::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT;

    //!< 最前面 AppletResourceUserId の変更を通知します。
    void NotifyAppletResourceUserId(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 定期処理をハンドリングします
    void HandlePeriodicalEvent() NN_NOEXCEPT;

    //!< デバイスリストの更新をハンドリングします
    void HandleDeviceUpdate() NN_NOEXCEPT;

    //!< Bluetooth のアップデートのイベントをハンドリングします
    void HandleBluetoothFirmwareUpdate() NN_NOEXCEPT;

    //!< Mcu のアップデートのイベントをハンドリングします
    void HandleMcuFirmwareUpdate() NN_NOEXCEPT;

    //!< 新しいサンプルの受信をハンドリングします
    void HandleNewSample() NN_NOEXCEPT;

    //!< ファームウェアアップデート中のリセットのタイムアウトをハンドリングします
    void HandleResetTimedOut() NN_NOEXCEPT;

    //!< コントローラーへの接続処理のタイムアウトをハンドリングします
    void HandleConnectionTriggerTimeout() NN_NOEXCEPT;

    //!< UniquePad ID を取得します。
    ::nn::Result GetIds(int* pOutCount,
                        system::UniquePadId* outUniquePadIds,
                        int count) NN_NOEXCEPT;

    //!< UniquePad の接続状態を取得します。
    bool IsConnected(system::UniquePadId& id) NN_NOEXCEPT;

    //!< UniquePad のコントローラー番号を取得します。
    ::nn::Result GetControllerNumber(int* pOutValue, system::UniquePadId& id) NN_NOEXCEPT;

    //!< UniquePad のデバイス種別を取得します。
    ::nn::Result GetType(system::UniquePadType* pOutValue, system::UniquePadId& id) NN_NOEXCEPT;

    //!< UniquePad の Bluetooth Address を取得します。
    ::nn::Result GetBluetoothAddress(::nn::bluetooth::Address* pOutAddress,
                                     system::UniquePadId& id) NN_NOEXCEPT;

    //!< UniquePad の接続インターフェースを取得します。
    ::nn::Result GetInterface(system::UniquePadInterface* pOutValue,
                              system::UniquePadId& id) NN_NOEXCEPT;

    //!< UniquePad のシリアル番号を取得します。
    ::nn::Result GetSerialNumber(system::UniquePadSerialNumber* pOutValue,
                                 system::UniquePadId& id) NN_NOEXCEPT;

    //!< UniquePad を切断します。
    ::nn::Result DisconnectUniquePad(system::UniquePadId& id) NN_NOEXCEPT;

    //!< ファームウェア更新処理を有効にします。
    Result ActivateFirmwareUpdate() NN_NOEXCEPT;

    //!< ファームウェア更新処理を有効にします。
    Result DeactivateFirmwareUpdate() NN_NOEXCEPT;

    //!< ファームウェア更新処理を開始します。
    Result StartFirmwareUpdate(
        system::FirmwareUpdateDeviceHandle* pOutDeviceHandle,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< デバッグ用のファームウェア更新処理を開始します。
    Result StartFirmwareUpdateForDebug(
        system::UniquePadId id) NN_NOEXCEPT;

    //!< 巻き戻し用のファームウェア更新処理を開始します。
    Result StartFirmwareUpdateForRevert(
        system::UniquePadId id) NN_NOEXCEPT;

    //!< ファームウェア更新を開始します。 (TransferMemory 版)
    Result StartFirmwareUpdateByTransferMemory(
        system::FirmwareUpdateDeviceHandle* pOutDeviceHandle,
        system::UniquePadId id,
        XcdFirmwareUpdater::FirmwareUpdateTarget target,
        const XcdFirmwareImageInfo& imageInfo) NN_NOEXCEPT;

    //!< ファームウェア更新処理を中断します。
    Result AbortFirmwareUpdate() NN_NOEXCEPT;

    //!< ファームウェアバージョンを取得します
    Result GetFirmwareVersion(
        system::FirmwareVersion* pOutValue,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< 更新先のファームウェアバージョンを取得します
    Result GetDestinationFirmwareVersion(
        system::FirmwareVersion *pOutValue,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< 巻き戻し用の更新先ファームウェアバージョンを取得します
    Result GetDestinationFirmwareVersionForRevert(
        debug::FirmwareVersion *pOutValue,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< 更新可能ファームウェアの有無を確認します。
    Result IsFirmwareUpdateAvailable(
        bool* pIsAvailable,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< ファームウェア更新が要求されているか確認します。
    Result CheckFirmwareUpdateRequired(
        system::FirmwareUpdateRequiredReason* pOutReason,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< ファームウェアの更新状況を取得します。
    Result GetFirmwareUpdateStage(
        debug::FirmwareUpdateStage* pOutStage,
        uint8_t* pOutProgress) NN_NOEXCEPT;

    //!< ファームウェアの更新状況を取得します。
    Result GetFirmwareUpdateState(
        system::FirmwareUpdateState* pOutState,
        system::FirmwareUpdateDeviceHandle handle) NN_NOEXCEPT;

    //!< ファームウェア更新中のデバイスかどうか確認します。
    Result IsFirmwareUpdatingDevice(
        bool* pOutIsUpdating,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< デバッグ/巻き戻し用ファームウェア情報のキャッシュを破棄します。
    Result DiscardFirmwareInfoCacheForRevert() NN_NOEXCEPT;

    Result SetFirmwareHotfixUpdateSkipEnabled(bool isEnabled) NN_NOEXCEPT;

    //!< アナログスティックの手動キャリブレーション処理を開始します。
    Result StartAnalogStickManualCalibration(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< 現在のアナログスティックのキャリブレーション処理をやり直します。
    Result RetryCurrentAnalogStickManualCalibrationStage(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< 進行中アナログスティックのキャリブレーション処理をキャンセルします。
    Result CancelAnalogStickManualCalibration(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< アナログスティックのキャリブレーション値を初期値に戻します。
    Result ResetAnalogStickManualCalibration(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< アナログスティックの状態を取得します。
    void GetAnalogStickState(AnalogStickState* pOutValue, system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< アナログスティックのキャリブレーションの状態を取得します。
    Result GetAnalogStickManualCalibrationStage(system::AnalogStickManualCalibrationStage* pOutValue,
                                                system::UniquePadId& id,
                                                system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< アナログスティックの状態を取得します。
    bool IsAnalogStickButtonPressed(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< キャリブレーション中のアナログスティックがリリースポジションにある場合は true を返します。
    bool IsAnalogStickInReleasePosition(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< キャリブレーション中のアナログスティックが外周にある場合は true を返します。
    bool IsAnalogStickInCircumference(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< UniqueSixAxisSensorHandle を取得します。
    ::nn::Result GetUniqueSixAxisSensorHandles(int* pOutCount,
                                               system::UniqueSixAxisSensorHandle* outUniqueSixAxisSensorHandles,
                                               int count,
                                               system::UniquePadId id) NN_NOEXCEPT;

    //!< 6 軸センサーのユーザーキャリブレーション機能のサポート有無を取得します。
    ::nn::Result IsSixAxisSensorUserCalibrationSupported(bool* pOutIsSupported,
                                                         system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< 工程出荷時の 6 軸センサーのキャリブレーション値を書き込みます。
    ::nn::Result ResetSixAxisSensorCalibrationValues(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< 6軸センサーのユーザーキャリブレーション処理を開始します。
    Result StartSixAxisSensorUserCalibration(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< 6軸センサーのユーザーキャリブレーション処理をキャンセルします。
    Result CancelSixAxisSensorUserCalibration(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< 6軸センサーのユーザーキャリブレーションの状況を取得します。
    ::nn::Result GetSixAxisSensorUserCalibrationStage(system::SixAxisSensorUserCalibrationStage* pOutValue,
                                                      system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< コントローラーが USB 接続されているかどうかを取得します
    ::nn::Result IsUsbConnected(bool* pOutConnected, system::UniquePadId id) NN_NOEXCEPT;

    //!< コントローラーが USB 接続されているかどうかを取得します
    int GetAllowedBluetoothLinksCount() NN_NOEXCEPT;

    //!< コントローラーの色情報を更新する
    Result UpdateControllerColor(const nn::util::Color4u8Type& mainColor,
                                 const nn::util::Color4u8Type& subColor,
                                 const system::UniquePadId& id) NN_NOEXCEPT;

    //!< コントローラーのデザイン情報を更新する
    Result UpdateDesignInfo(const nn::util::Color4u8Type& mainColor,
                            const nn::util::Color4u8Type& subColor,
                            const nn::util::Color4u8Type& thirdColor,
                            const nn::util::Color4u8Type& forthColor,
                            uint8_t variation,
                            const system::UniquePadId& id) NN_NOEXCEPT;

    //!< USB コントローラを接続します
    Result ConnectUsbPadsAsync() NN_NOEXCEPT;

    //!< USB コントローラを接続します
    Result DisconnectUsbPadsAsync() NN_NOEXCEPT;

    //!< UniquePad の入力状態を取得します。
    Result GetPadDriverState(PadDriverState* pOutStates,
                             const system::UniquePadId& id) NN_NOEXCEPT;

    //!< 6軸センサーの状態を取得します。
    Result GetSixAxisSensorDriverStates(int* pOutCount,
                                        SixAxisSensorDriverState* outStates,
                                        int count,
                                        const system::UniquePadId& id) NN_NOEXCEPT;

    //!< パケットの受信履歴を取得します。
    Result GetRxPacketHistory(RxPacketHistory* pOutValue, const system::UniquePadId& id) NN_NOEXCEPT;

    //!<  シリアルフラッシュへの操作が完了した際に通知するイベントをバインドします。
    Result AcquireSerialFlashEventHandle(
        ::nn::os::NativeHandle* pOutHandle,
        const system::UniquePadId& id) NN_NOEXCEPT;

    //!<  シリアルフラッシュの任意のアドレスから読み込みを行います
    Result ReadSerialFlash(
        const uint32_t address,
        const nn::os::NativeHandle& osHandle,
        bool isManaged,
        int size,
        const system::UniquePadId& id) NN_NOEXCEPT;

    //!<  シリアルフラッシュの任意のアドレスへの書き込みを行います
    Result WriteSerialFlash(
        const uint32_t address,
        const nn::os::NativeHandle& osHandle,
        bool isManaged,
        int bufferSize,
        int writeSize,
        const system::UniquePadId& id) NN_NOEXCEPT;

    //!<  シリアルフラッシュへの操作結果を取得します
    Result GetSerialFlashResult(const system::UniquePadId& id) NN_NOEXCEPT;

private:
    //!< UniquePadManager を取得する
    UniquePadManager* GetUniquePadManager(const system::UniquePadId& id) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
