﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/detail/hid_PadTypes.h>
#include <nn/hid/detail/hid_SensorTypes.h>
#include <nn/hid/detail/hid_RxPacketHistoryTypes.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemoryTypes.h>

#include "hid_AnalogStickManualCalibrationTypes.h"
#include "hid_HandheldManager.h"
#include "hid_UniquePadAbstractedPadHolder.h"
#include "hid_UniquePadAnalogStickDriver.h"
#include "hid_UniquePadSixAxisSensorDriver.h"

namespace nn { namespace hid { namespace detail {

//!< UniquePad インターフェイスの解決を担うマネージャを扱うクラスです。
class UniquePadManager final
{
    NN_DISALLOW_COPY(UniquePadManager);
    NN_DISALLOW_MOVE(UniquePadManager);

private:
    //!< UniquePad に割り当てられた AbstractedPad を管理するクラス
    UniquePadAbstractedPadHolder m_AbstractedPadHolder;

    //!< AnalogStick ドライバ
    UniquePadAnalogStickDriver m_AnalogStickDrivers[AnalogStickPositionCountMax];

    //!< SixAxisSensor ドライバ
    UniquePadSixAxisSensorDriver m_SixAxisSensorDriver;

    //!< UniquePad Id
    system::UniquePadId m_UniquePadId;

    //!< SerialFlash への操作が完了したときに通知するシステムイベント
    ::nn::os::SystemEventType m_SerialFlashEvent;

    //!< バッファの読み書きに用いる TransferMemory
    ::nn::os::TransferMemoryType m_TransferMemory;

    //!< SystemEventType が初期化されているかどうか
    bool m_IsSerialFlashEventInitialized;

public:
    UniquePadManager() NN_NOEXCEPT;

    ~UniquePadManager() NN_NOEXCEPT;

    //!< 初期化を行います。
    void Initialize() NN_NOEXCEPT;

    //!< Handheld マネージャを設定します。
    void SetHandheldManager(HandheldManager* pManager) NN_NOEXCEPT;

    //!< AnalogStick のドライバーを取得します。
    UniquePadAnalogStickDriver& GetAnalogStickDriver(system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< SixAxisSensor のドライバーを取得します。
    UniquePadSixAxisSensorDriver& GetSixAxisSensorDriver(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< UniqueSixAxisSensorHandle を取得します。
    ::nn::Result GetUniqueSixAxisSensorHandles(int* pOutCount,
                                               system::UniqueSixAxisSensorHandle* outUniqueSixAxisSensorHandles,
                                               int count) const NN_NOEXCEPT;

    //!< 接続されたデバイスをもとに、ドライバを有効にします。
    void AttachAbstractedPad(IAbstractedPad* pPad, system::UniquePadId id) NN_NOEXCEPT;

    //!< 切断済みであればデバイスを切断します。
    bool CheckForDeviceDetach() NN_NOEXCEPT;

    //!< 接続されているか否かを返します。
    bool IsConnected() const NN_NOEXCEPT;

    //!< 接続済みの AbstractedPad と一致するかどうか
    bool IsConnectedAbstractedPad(const AbstractedPadId& id) const NN_NOEXCEPT;

    //!< 接続済みの AbstractedPad を取得します。
    bool GetIAbstractedPad(IAbstractedPad** ppOutValue) const NN_NOEXCEPT;

    //!< UniquePadId を取得します
    system::UniquePadId GetId() NN_NOEXCEPT;

    //!< UniquePad の種類を取得します
    system::UniquePadType GetType() NN_NOEXCEPT;

    //!< UniquePad の Bluetooth Address を取得します。
    ::nn::Result GetBluetoothAddress(::nn::bluetooth::Address* pOutAddress) NN_NOEXCEPT;

    //!< UniquePad の接続インターフェースを取得します。
    ::nn::Result GetInterface(system::UniquePadInterface* pOutValue) NN_NOEXCEPT;

    //!< UniquePad のシリアル番号を取得します。
    ::nn::Result GetSerialNumber(system::UniquePadSerialNumber* pOutValue) NN_NOEXCEPT;

    //!< コントローラー番号を取得します
    int GetControllerNumber() NN_NOEXCEPT;

    //!< Usb の接続状態を取得します
    bool IsUsbConnected() NN_NOEXCEPT;

    //!< UniquePad を切断します。
    ::nn::Result Disconnect() NN_NOEXCEPT;

    //!< 定常処理を実施します
    void Sample() NN_NOEXCEPT;

    //!< コントローラーの色情報を更新する
    ::nn::Result UpdateControllerColor(const nn::util::Color4u8Type& mainColor,
                                       const nn::util::Color4u8Type& subColor) NN_NOEXCEPT;

    //!< コントローラーのデザイン情報を更新する
    ::nn::Result UpdateDesignInfo(const nn::util::Color4u8Type& mainColor,
                                  const nn::util::Color4u8Type& subColor,
                                  const nn::util::Color4u8Type& thirdColor,
                                  const nn::util::Color4u8Type& forthColor,
                                  uint8_t variation) NN_NOEXCEPT;

    //!< UniquePad の入力状態を取得します。
    Result GetPadDriverState(PadDriverState* pOutState) NN_NOEXCEPT;

    //!< 6軸センサーの状態を取得します。
    Result GetSixAxisSensorDriverStates(int* pOutCount,
                                        SixAxisSensorDriverState* pOutStates,
                                        int count) NN_NOEXCEPT;

    //!< パケットの受信履歴を取得します。
    Result GetRxPacketHistory(RxPacketHistory* pOutValue) NN_NOEXCEPT;

    //!<  シリアルフラッシュへの操作が完了した際に通知するイベントをバインドします。
    ::nn::Result AcquireSerialFlashEventHandle(::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT;

    //!<  シリアルフラッシュの任意のアドレスから読み込みを行います
    Result ReadSerialFlash(
        const uint32_t address,
        const nn::os::NativeHandle& osHandle,
        bool isManaged,
        int size) NN_NOEXCEPT;

    //!<  シリアルフラッシュの任意のアドレスへの書き込みを行います
    Result WriteSerialFlash(
        const uint32_t address,
        const nn::os::NativeHandle& osHandle,
        bool isManaged,
        int bufferSize,
        int writeSize) NN_NOEXCEPT;

    //!<  シリアルフラッシュへの操作結果を取得します
    Result GetSerialFlashResult() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
