﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AnalogStickCalibrationManager.h"
#include "hid_HandheldManager.h"
#include "hid_UniquePadAbstractedPadHolder.h"

namespace nn { namespace hid { namespace detail {

/**
 * @brief       アナログスティックキャリブレーション用のフラグ定義
 */
struct AnalogStickCalibrationFlags
{
    typedef ::nn::util::BitFlagSet<32, AnalogStickCalibrationFlags>::Flag<0>  IsAnalogStickInReleasePosition;   //!< アナログスティックがリリースポジションにあるかどうか
    typedef ::nn::util::BitFlagSet<32, AnalogStickCalibrationFlags>::Flag<1>  IsAnalogStickInCircumference;     //!< アナログスティックが外周にあるかどうか
};

/**
 * @brief       AnalogStick キャリブレーションのフラグを扱う集合です
 */
typedef ::nn::util::BitFlagSet<32, AnalogStickCalibrationFlags> AnalogStickCalibrationFlagsSet;

class UniquePadAnalogStickDriver final
{
    NN_DISALLOW_COPY(UniquePadAnalogStickDriver);
    NN_DISALLOW_MOVE(UniquePadAnalogStickDriver);

private:
    //!< UniquePad に割り当てられた AbstractedPad を管理するクラス
    UniquePadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Handheld マネージャー
    HandheldManager* m_pHandheldManager;

    //!< AnalogStick の位置
    system::AnalogStickPosition m_Position;

    //!< AnalogStickCalibration マネージャ
    AnalogStickCalibrationManager m_CalibrationManager;

    //!< AnalogStick の Calibration 処理の状況
    system::AnalogStickManualCalibrationStage m_Stage;

    //!< アナログスティックの入力状態
    AnalogStickState m_State;

    //!< フラグの集合
    AnalogStickCalibrationFlagsSet m_Flags;

public:
    UniquePadAnalogStickDriver() NN_NOEXCEPT;
    ~UniquePadAnalogStickDriver() NN_NOEXCEPT;

    //!< Handheld マネージャを設定します。
    void SetHandheldManager(HandheldManager* pManager) NN_NOEXCEPT;

    //!< AbstractedPad を管理するクラスを設定します。
    void SetUniquePadAbstractedPadHolder(UniquePadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< Position を設定します。
    void SetPosition(system::AnalogStickPosition position) NN_NOEXCEPT;

    //!< アナログスティックの手動キャリブレーション処理を開始します。
    ::nn::Result StartAnalogStickManualCalibration() NN_NOEXCEPT;

    //!< 現在のアナログスティックのキャリブレーション処理をやり直します。
    ::nn::Result RetryCurrentAnalogStickManualCalibrationStage() NN_NOEXCEPT;

    //!< 進行中アナログスティックのキャリブレーション処理をキャンセルします。
    ::nn::Result CancelAnalogStickManualCalibration() NN_NOEXCEPT;

    //!< アナログスティックのキャリブレーション値を初期値に戻します。
    ::nn::Result ResetAnalogStickManualCalibration() NN_NOEXCEPT;

    //!< アナログスティックの状態を取得します。
    void GetAnalogStickState(AnalogStickState* pOutValue) NN_NOEXCEPT;

    //!< アナログスティックのキャリブレーションの状態を取得します。
    ::nn::Result GetAnalogStickManualCalibrationStage(system::AnalogStickManualCalibrationStage* pOutValue) NN_NOEXCEPT;

    //!< アナログスティックの状態を取得します。
    bool IsAnalogStickButtonPressed() NN_NOEXCEPT;

    //!< キャリブレーション中のアナログスティックがリリースポジションにある場合は true を返します。
    bool IsAnalogStickInReleasePosition() NN_NOEXCEPT;

    //!< キャリブレーション中のアナログスティックが外周にある場合は true を返します。
    bool IsAnalogStickInCircumference() NN_NOEXCEPT;

    //!< AnalogStick の内部状態の更新を行います。
    void Update() NN_NOEXCEPT;

    //!< AnalogStick の内部状態を初期化します。
    void Reset() NN_NOEXCEPT;

private:
    void ProceedAnalogStickManualCalibration(AnalogStickState& state) NN_NOEXCEPT;

    void ProceedAnalogStickCalibrationUpdate(system::AnalogStickManualCalibrationStage currentStage) NN_NOEXCEPT;

    void InitializeAnalogStickManualCalibration() NN_NOEXCEPT;

    ::nn::Result IsAnalogStickCalibrationSupportedImpl() const NN_NOEXCEPT;

    ::nn::Result ResetAnalogStickManualCalibrationImpl() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
