﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/TargetConfigs/build_Base.h>

#include "hid_Rectangle.h"

namespace nn { namespace hid { namespace detail {

//!< タッチの回転度数の最小値
const int32_t TouchRotationAngleMin = -270;

//!< タッチの回転度数の最大値
const int32_t TouchRotationAngleMax = 270;

//!< タッチの入力変化閾値
const int32_t TouchStateThreshold = 1;

//!< TouchScreen の座標域
const Rectangle TouchScreenRectangle = {
    0,      // x
    0,      // y
    1280,   // width
    720     // height
};

//!< タッチ位置の縮尺を変域に合わせます。
inline TouchState ScaleTouchPosition(
    const TouchState& state, const Rectangle& rect) NN_NOEXCEPT
{
    TouchState value = state;

    if (0 < rect.width)
    {
        value.x = value.x * TouchScreenRectangle.width / rect.width;
    }

    if (0 < rect.height)
    {
        value.y = value.y * TouchScreenRectangle.height / rect.height;
    }

    return value;
}

//!< タッチの各種パラメータを変域内に収めます。
inline TouchState LimitTouch(const TouchState& state) NN_NOEXCEPT
{
    auto Clamp = [](int value, int min, int max) -> int32_t
    {
        return static_cast<int32_t>(::std::min(::std::max(min, value), max));
    };

    const Rectangle rect = {
#if defined(NN_BUILD_CONFIG_SPEC_NX)
        TouchScreenRectangle.x + 15,
        TouchScreenRectangle.y + 15,
        TouchScreenRectangle.width - 30,
        TouchScreenRectangle.height - 30
#else
        TouchScreenRectangle.x,
        TouchScreenRectangle.y,
        TouchScreenRectangle.width,
        TouchScreenRectangle.height
#endif
    };

    TouchState value = {};

    value.deltaTime = state.deltaTime;

    value.attributes = state.attributes;

    value.fingerId = state.fingerId;

    value.x = Clamp(state.x, rect.x, rect.x + rect.width - 1);

    value.y = Clamp(state.y, rect.y, rect.y + rect.height - 1);

    value.diameterX = Clamp(state.diameterX, 0, rect.width);

    value.diameterY = Clamp(state.diameterY, 0, rect.height);

    value.rotationAngle = Clamp(
        state.rotationAngle, TouchRotationAngleMin, TouchRotationAngleMax);

    return value;
}

template <typename T>
inline bool IsChanged(const T& state1, const T& state2, const T& threshold)
{
    T deltaState = state1 - state2;

    if (deltaState < 0)
    {
        deltaState = -deltaState;
    }

    return (deltaState > threshold);
}

//!< TouchState に変化があるか否かを返します
template<size_t N>
inline bool IsTouchScreenStateChanged(
    const TouchScreenState<N>& state1,
    const TouchScreenState<N>& state2) NN_NOEXCEPT
{
    // タッチ数が一致しないときは入力変化があったと見做す
    if (state1.count != state2.count)
    {
        return true;
    }

    // タッチ座標に変化があった時は入力変化があったと見做す
    for (int i = 0; i < state1.count; i++)
    {
        if(IsChanged(state1.touches[i].x,
                     state2.touches[i].x,
                     TouchStateThreshold) ||
           IsChanged(state1.touches[i].y,
                     state2.touches[i].y,
                     TouchStateThreshold))
        {
            return true;
        }
    }

    return false;
}

//!< 自動操作状態を更新します。
inline void UpdateTouchScreenAutoPilotState(
    ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax>* pValue,
    const ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax>& value
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);
    NN_SDK_REQUIRES_MINMAX(pValue->count, 0, TouchStateCountMax);

    int& lhsCount = pValue->count;

    int lhsIndex = -1;

    for (int i = 0; i < lhsCount; ++i)
    {
        pValue->touches[i].attributes.Set<TouchAttribute::End>();
    }

    const int rhsCount = ::std::min(value.count, TouchStateCountMax);

    int rhsIndex = -1;

    for (int i = 0; i < rhsCount; ++i)
    {
        const TouchState& rhsTouch = value.touches[i];

        if (rhsTouch.fingerId < 0)
        {
            break;
        }

        bool found = false;

        for (int j = lhsCount - 1; j > lhsIndex; --j)
        {
            TouchState& lhsTouch = pValue->touches[j];

            if (lhsTouch.fingerId == rhsTouch.fingerId)
            {
                found = true;
                lhsTouch = LimitTouch(rhsTouch);
                lhsTouch.attributes.Reset();
                lhsIndex = j;
                rhsIndex = i;
                break;
            }
        }

        if (!found)
        {
            break;
        }
    }

    for (int i = rhsIndex + 1; i < rhsCount; ++i)
    {
        const TouchState& rhsTouch = value.touches[i];

        if (lhsCount >= TouchStateCountMax || rhsTouch.fingerId < 0)
        {
            break;
        }

        bool found = false;

        for (int j = 0; j < lhsCount; ++j)
        {
            if (pValue->touches[j].fingerId == rhsTouch.fingerId)
            {
                found = true;
                break;
            }
        }

        if (found)
        {
            break;
        }
        else
        {
            TouchState& lhsTouch = pValue->touches[lhsCount];

            lhsTouch = LimitTouch(rhsTouch);
            lhsTouch.attributes.Reset();
            lhsTouch.attributes.Set<TouchAttribute::Start>();

            ++lhsCount;
        }
    }
}

//!< 自動操作状態をフラッシュします。
inline void FlushTouchScreenAutoPilotState(
    ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax>* pValue
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);
    NN_SDK_REQUIRES_MINMAX(pValue->count, 0, TouchStateCountMax);

    int count = 0;

    for (int i = 0; i < pValue->count; ++i)
    {
        TouchState& touch = pValue->touches[i];

        if (touch.attributes.Test<TouchAttribute::End>())
        {
            continue;
        }

        touch.attributes.Reset();

        if (i != count)
        {
            pValue->touches[count] = touch;
        }

        ++count;
    }

    pValue->count = count;

    for (int i = count; i < TouchStateCountMax; ++i)
    {
        pValue->touches[i] = TouchState();
    }
}

}}} // namespace nn::hid::detail
