﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceUserId.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"
#include "hid_TouchScreenImpl.h"
#include "hid_TouchScreenLifo.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< TouchScreen の共有メモリへのアクセスを扱うクラスです。
class TouchScreenSharedMemoryAccessor final : public SharedMemoryAccessor
{
private:
    ::std::atomic<TouchScreenLifo*> m_pLifo;

public:
    TouchScreenSharedMemoryAccessor() NN_NOEXCEPT
        : m_pLifo(nullptr)
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultTouchScreenActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultTouchScreenDeactivationLowerLimitOver());
    }

    virtual ~TouchScreenSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< TouchScreen の LIFO を返します。
    const TouchScreenLifo* GetLifo() const NN_NOEXCEPT
    {
        return m_pLifo;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidServer> pHidServer;
        NN_RESULT_DO(CreateHidServerProxy(&pHidServer));
        NN_RESULT_DO(
            pHidServer->ActivateTouchScreen(GetAppletResourceUserId()));
        m_pLifo = &::nn::util::Get(ptr->touchScreen.lifo);
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateTouchScreen());
        m_pLifo = nullptr;
        NN_RESULT_SUCCESS;
    }
};

//!< TouchScreen の共有メモリアクセッサを返します。
TouchScreenSharedMemoryAccessor& GetTouchScreenSharedMemoryAccessor(
    ) NN_NOEXCEPT
{
    return StaticObject<TouchScreenSharedMemoryAccessor>::Get();
}

} // namespace

::nn::Result InitializeTouchScreen() NN_NOEXCEPT
{
    TouchScreenSharedMemoryAccessor&
        accessor = GetTouchScreenSharedMemoryAccessor();
    NN_RESULT_DO(accessor.SetAppletResourceUserId(GetAppletResourceUserId()));
    NN_RESULT_DO(accessor.Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeTouchScreen() NN_NOEXCEPT
{
    NN_RESULT_DO(GetTouchScreenSharedMemoryAccessor().Deactivate());
    NN_RESULT_SUCCESS;
}

#define NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(n)\
template<>\
::nn::Result GetTouchScreenState(TouchScreenState<n>* pOutValue) NN_NOEXCEPT\
{\
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,\
                           ResultTouchScreenNullStateBuffer());\
    const TouchScreenLifo* pLifo =\
        GetTouchScreenSharedMemoryAccessor().GetLifo(); \
    NN_RESULT_THROW_UNLESS(pLifo != nullptr,\
                           ResultTouchScreenNotInitialized());\
    int count = pLifo->ReadPartially(pOutValue, 1);\
    NN_SDK_REQUIRES_EQUAL(count, 1);\
    NN_UNUSED(count);\
    NN_RESULT_SUCCESS;\
}

#define NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(n)\
template<>\
::nn::Result GetTouchScreenStates(int* pOutCount,\
                                  TouchScreenState<n>* outStates,\
                                  int count) NN_NOEXCEPT\
{\
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);\
    NN_RESULT_THROW_UNLESS(outStates != nullptr,\
                           ResultTouchScreenNullStateBuffer());\
    NN_RESULT_THROW_UNLESS(count >= 0, ResultTouchScreenInvalidStateCount());\
    const TouchScreenLifo* pLifo =\
        GetTouchScreenSharedMemoryAccessor().GetLifo();\
    NN_SDK_REQUIRES(!pLifo->IsEmpty());\
    *pOutCount = pLifo->ReadPartially(outStates, count);\
    NN_RESULT_SUCCESS;\
}

#define NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(n)\
template<>\
::nn::Result SetTouchScreenAutoPilotState(\
    const ::nn::hid::debug::TouchScreenAutoPilotState<n>& value) NN_NOEXCEPT\
{\
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;\
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));\
    const ::nn::sf::InArray<TouchState> touches(\
        value.touches, ::std::max(0, ::std::min(value.count, n)));\
    NN_RESULT_DO(pHidDebugServer->SetTouchScreenAutoPilotState(touches));\
    NN_RESULT_SUCCESS;\
}

::nn::Result UnsetTouchScreenAutoPilotState() NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->UnsetTouchScreenAutoPilotState());
    NN_RESULT_SUCCESS;
}

NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(1)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(2)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(3)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(4)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(5)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(6)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(7)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(8)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(9)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(10)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(11)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(12)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(13)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(14)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(15)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATE(16)

NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(1)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(2)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(3)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(4)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(5)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(6)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(7)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(8)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(9)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(10)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(11)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(12)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(13)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(14)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(15)
NN_HID_DETAIL_DEFINE_GET_TOUCH_SCREEN_STATES(16)

NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(1)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(2)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(3)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(4)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(5)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(6)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(7)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(8)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(9)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(10)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(11)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(12)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(13)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(14)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(15)
NN_HID_DETAIL_DEFINE_SET_TOUCH_SCREEN_AUTO_PILOT_STATE(16)

}}} // namespace nn::hid::detail
