﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <atomic>

namespace nn { namespace hid { namespace detail {

class SynchronizedTimerGroup;

/* Concurrent access validity:
 * - Enable(), Disable() and Clear() can be called from other threads while waiting.
 * - Link() and Unlink() cannot be called from other threads.
 */
class SynchronizedTimer final
{
    NN_DISALLOW_COPY(SynchronizedTimer);
    NN_DISALLOW_MOVE(SynchronizedTimer);

public:
    SynchronizedTimer() NN_NOEXCEPT;
    void Enable(nn::TimeSpan period) NN_NOEXCEPT;
    void Disable() NN_NOEXCEPT;

    void Clear() NN_NOEXCEPT;

    void Link(SynchronizedTimerGroup* group) NN_NOEXCEPT;
    void Unlink() NN_NOEXCEPT;

private:
    friend class SynchronizedTimerGroup;
    SynchronizedTimerGroup* m_Group;
    SynchronizedTimer* m_NextTimer;

    // To handle concurrent access by Clear()/Enable()/Disable()
    nn::os::Mutex m_Mutex;
    nn::TimeSpan m_Period;

    // m_DeadlineNs is the only member accessed concurrently by Enable()/Disable() and by WaitAny()
    std::atomic<int64_t> m_DeadlineNs;
};

class SynchronizedTimerGroup final
{
    NN_DISALLOW_COPY(SynchronizedTimerGroup);
    NN_DISALLOW_MOVE(SynchronizedTimerGroup);

public:
    SynchronizedTimerGroup() NN_NOEXCEPT;
    ~SynchronizedTimerGroup() NN_NOEXCEPT;

    nn::TimeSpan GetTimeout() NN_NOEXCEPT;
    SynchronizedTimer* GetExpiredTimer() NN_NOEXCEPT;

    inline nn::os::MultiWaitHolderType* GetPeriodUpdatedHolder() NN_NOEXCEPT
    {
        return &m_PeriodUpdatedHolder;
    }

private:
    void LinkAssumingTimerLocked(SynchronizedTimer* timer) NN_NOEXCEPT;
    void UnlinkAssumingTimerLocked(SynchronizedTimer* timer) NN_NOEXCEPT;

private:
    friend class SynchronizedTimer;

    SynchronizedTimer* m_Timers;
    nn::os::Tick m_Origin;
    nn::os::Event m_PeriodUpdated;
    nn::os::MultiWaitHolderType m_PeriodUpdatedHolder;

    inline nn::TimeSpan GetNextDeadlineAssumingTimerLocked(SynchronizedTimer* timer) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(0, timer->m_Period.GetNanoSeconds());

        nn::os::Tick now = nn::os::GetSystemTick();
        int64_t periodCount = (now - m_Origin).ToTimeSpan().GetNanoSeconds() / timer->m_Period.GetNanoSeconds();

        return
            m_Origin.ToTimeSpan() +
            nn::TimeSpan::FromNanoSeconds(
                (1 + periodCount) * timer->m_Period.GetNanoSeconds());
    }

    inline void OnPeriodUpdated() NN_NOEXCEPT
    {
        m_PeriodUpdated.Signal();
    }
};

}}}
