﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "hid_SynchronizedTimer.h"
#include <nn/hid/detail/hid_Log.h>
#include <mutex>
#include <limits>

#define NN_HID_DETAIL_SYNCHRONIZED_TIMER_CONFIG_ENABLE_LOG 0

namespace nn { namespace hid { namespace detail {

SynchronizedTimer::SynchronizedTimer() NN_NOEXCEPT:
    m_Group(nullptr),
    m_NextTimer(nullptr),
    m_Mutex(false),
    m_Period(0),
    m_DeadlineNs(0)
{}

void SynchronizedTimer::Enable(nn::TimeSpan period) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_ASSERT_NOT_EQUAL(0, period.GetNanoSeconds());

    // Note: having m_Period be non-zero should not be an issue,
    // but that would allow the user to call Enable() twice in a row,
    // which is counter-intuitive.
    NN_SDK_ASSERT_EQUAL(0, m_Period.GetNanoSeconds());

    m_Period = period;

    if (m_Group != nullptr)
    {
        m_DeadlineNs.store(m_Group->GetNextDeadlineAssumingTimerLocked(this).GetNanoSeconds());
        m_Group->OnPeriodUpdated();
    }
    else
    {
        m_DeadlineNs.store(0);
    }
}

void SynchronizedTimer::Disable() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    m_Period = nn::TimeSpan(0);
    m_DeadlineNs.store(0);

    if (m_Group != nullptr)
    {
        m_Group->OnPeriodUpdated();
    }
}

void SynchronizedTimer::Link(SynchronizedTimerGroup* group) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    group->LinkAssumingTimerLocked(this);
}

void SynchronizedTimer::Unlink() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    m_Group->UnlinkAssumingTimerLocked(this);
}

void SynchronizedTimer::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (m_Group == nullptr)
    {
        return;
    }

    if (m_Period != 0)
    {
        m_DeadlineNs.store(m_Group->GetNextDeadlineAssumingTimerLocked(this).GetNanoSeconds());
    }
    else
    {
        m_DeadlineNs.store(0);
    }
}

SynchronizedTimerGroup::SynchronizedTimerGroup() NN_NOEXCEPT:
    m_Timers(nullptr),
    m_Origin(nn::os::GetSystemTick()),
    m_PeriodUpdated(nn::os::EventClearMode_ManualClear)
{
    nn::os::InitializeMultiWaitHolder(&m_PeriodUpdatedHolder, m_PeriodUpdated.GetBase());
}

SynchronizedTimerGroup::~SynchronizedTimerGroup() NN_NOEXCEPT
{
    nn::os::FinalizeMultiWaitHolder(&m_PeriodUpdatedHolder);
}

nn::TimeSpan SynchronizedTimerGroup::GetTimeout() NN_NOEXCEPT
{
    nn::TimeSpan minTimeout = nn::TimeSpan::FromNanoSeconds(std::numeric_limits<int64_t>::max() / 2);

    if (m_Timers == nullptr)
    {
        return minTimeout;
    }

    SynchronizedTimer* timer = m_Timers;
    nn::TimeSpan now = nn::os::GetSystemTick().ToTimeSpan();
    while (timer != nullptr)
    {
        int64_t deadlineNs = timer->m_DeadlineNs.load();
        if (deadlineNs != 0)
        {
            nn::TimeSpan timeout = nn::TimeSpan::FromNanoSeconds(deadlineNs) - now;
            if (timeout < minTimeout)
            {
                minTimeout = timeout;
            }
        }

        timer = timer->m_NextTimer;
    }

    if (minTimeout >= 0)
    {
        return minTimeout;
    }
    else
    {
        return nn::TimeSpan(0);
    }
}

SynchronizedTimer* SynchronizedTimerGroup::GetExpiredTimer() NN_NOEXCEPT
{
    if (m_Timers == nullptr)
    {
        return nullptr;
    }

    m_PeriodUpdated.Clear();

    SynchronizedTimer* timer = m_Timers;
    nn::TimeSpan now = nn::os::GetSystemTick().ToTimeSpan();
    int timerIndex = 0;
    while (timer != nullptr)
    {
        int64_t deadlineNs = timer->m_DeadlineNs.load();
        if (deadlineNs != 0)
        {
            nn::TimeSpan timeout = nn::TimeSpan::FromNanoSeconds(deadlineNs) - now;
            if (timeout < 0)
            {
#if NN_HID_DETAIL_SYNCHRONIZED_TIMER_CONFIG_ENABLE_LOG
                static int log;
                if (log++%197==0) // 197 is prime
                {
                    std::lock_guard<decltype(timer->m_Mutex)> locker(timer->m_Mutex);

                    NN_DETAIL_HID_INFO("[%lld] Expired timer: %d %p - delay: %lld us period: %lld us\n", now.GetMilliSeconds(), timerIndex, timer, timeout.GetMicroSeconds(), timer->m_Period.GetMicroSeconds());
                }
#endif
                return timer;
            }
        }

        timerIndex++;

        timer = timer->m_NextTimer;
    }

    return nullptr;
}

void SynchronizedTimerGroup::LinkAssumingTimerLocked(SynchronizedTimer* timer) NN_NOEXCEPT
{
    SynchronizedTimer** timerToSet = &m_Timers;

    while (*timerToSet != nullptr)
    {
        timerToSet = &(*timerToSet)->m_NextTimer;
    }
    *timerToSet = timer;

    timer->m_Group = this;

    // Link() must not be called concurrently with Enable()/Disable()/WaitAny()/Clear(),
    // so no issue with loading and storing the atomic not atomically.
    NN_SDK_ASSERT_EQUAL(0, timer->m_DeadlineNs.load());
    if (timer->m_Period != 0)
    {
        timer->m_DeadlineNs.store(GetNextDeadlineAssumingTimerLocked(timer).GetNanoSeconds());
    }
}

void SynchronizedTimerGroup::UnlinkAssumingTimerLocked(SynchronizedTimer* timer) NN_NOEXCEPT
{
    SynchronizedTimer** timerToRemove = &m_Timers;

    while (*timerToRemove != nullptr && *timerToRemove != timer)
    {
        timerToRemove = &(*timerToRemove)->m_NextTimer;
    }

    NN_SDK_ASSERT_NOT_NULL(*timerToRemove);

    *timerToRemove = timer->m_NextTimer;

    timer->m_NextTimer = nullptr;
    timer->m_Group = nullptr;

    timer->m_DeadlineNs.store(0);
}

}}}
