﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_Constant.h>

#include "hid_ConsoleSixAxisSensorCalibrationTypes.h"
#include "hid_SixAxisSensorSanjiroFilter.h"

namespace {

const float Pi2 = nn::util::FloatPi * 2;

const float GravitationalAcceleration = 9.80665f;

const nn::hid::DirectionState UnitDirection = {
    { { { 1.0f, 0.0f, 0.0f } } },
    { { { 0.0f, 1.0f, 0.0f } } },
    { { { 0.0f, 0.0f, 1.0f } } }
};

const nn::util::Float3 ZeroVector3 = {
    { { 0.0f, 0.0f, 0.0f } }
};

const float AngularVelocityConvertCoef = 1.f / Pi2; //!< rad/sec を SDK 提供の単位系に変換する係数

::nn::hid::detail::ConsoleSixAxisSensorCalibrationParameters Convert(
    const nerd::NumericalVector<float, nerd::Imu::gParamCount>& vector) NN_NOEXCEPT
{
    return ::nn::hid::detail::ConsoleSixAxisSensorCalibrationParameters(
        {
            // consoleSixAxisSensorAccelerationBias
            {
                vector[size_t(nerd::Imu::Params::AccBiasX)],
                vector[size_t(nerd::Imu::Params::AccBiasY)],
                vector[size_t(nerd::Imu::Params::AccBiasZ)],
            },
            // consoleSixAxisSensorAngularVelocityBias
            {
                vector[size_t(nerd::Imu::Params::GyroBiasX)],
                vector[size_t(nerd::Imu::Params::GyroBiasY)],
                vector[size_t(nerd::Imu::Params::GyroBiasZ)],
            },
            // consoleSixAxisSensorAccelerationGain
            {
                vector[size_t(nerd::Imu::Params::AccGainXX)],
                vector[size_t(nerd::Imu::Params::AccGainXY)],
                vector[size_t(nerd::Imu::Params::AccGainXZ)],
                vector[size_t(nerd::Imu::Params::AccGainYX)],
                vector[size_t(nerd::Imu::Params::AccGainYY)],
                vector[size_t(nerd::Imu::Params::AccGainYZ)],
                vector[size_t(nerd::Imu::Params::AccGainZX)],
                vector[size_t(nerd::Imu::Params::AccGainZY)],
                vector[size_t(nerd::Imu::Params::AccGainZZ)],
            },
            // consoleSixAxisSensorAngularVelocityGain
            {
                vector[size_t(nerd::Imu::Params::GyroGainXX)],
                vector[size_t(nerd::Imu::Params::GyroGainXY)],
                vector[size_t(nerd::Imu::Params::GyroGainXZ)],
                vector[size_t(nerd::Imu::Params::GyroGainYX)],
                vector[size_t(nerd::Imu::Params::GyroGainYY)],
                vector[size_t(nerd::Imu::Params::GyroGainYZ)],
                vector[size_t(nerd::Imu::Params::GyroGainZX)],
                vector[size_t(nerd::Imu::Params::GyroGainZY)],
                vector[size_t(nerd::Imu::Params::GyroGainZZ)],
            },
            // consoleSixAxisSensorAngularVelocityTimeBias
            {
                0.f,
                0.f,
                0.f,
            },
            // consoleSixAxisSensorAngularAcceleration
            {
                0.f,
                0.f,
                0.f,
                0.f,
                0.f,
                0.f,
                0.f,
                0.f,
                0.f,
            },
        }
    );
}

nerd::NumericalVector<float, nerd::Imu::gParamCount> Convert(
    const ::nn::hid::detail::ConsoleSixAxisSensorCalibrationParameters& parameters) NN_NOEXCEPT
{
    return nerd::NumericalVector<float, nerd::Imu::gParamCount>(
        {
            parameters.consoleSixAxisSensorAngularVelocityBias.x,
            parameters.consoleSixAxisSensorAngularVelocityBias.y,
            parameters.consoleSixAxisSensorAngularVelocityBias.z,

            parameters.consoleSixAxisSensorAngularVelocityGain.xx,
            parameters.consoleSixAxisSensorAngularVelocityGain.xy,
            parameters.consoleSixAxisSensorAngularVelocityGain.xz,

            parameters.consoleSixAxisSensorAngularVelocityGain.yx,
            parameters.consoleSixAxisSensorAngularVelocityGain.yy,
            parameters.consoleSixAxisSensorAngularVelocityGain.yz,

            parameters.consoleSixAxisSensorAngularVelocityGain.zx,
            parameters.consoleSixAxisSensorAngularVelocityGain.zy,
            parameters.consoleSixAxisSensorAngularVelocityGain.zz,

            parameters.consoleSixAxisSensorAccelerationBias.x,
            parameters.consoleSixAxisSensorAccelerationBias.y,
            parameters.consoleSixAxisSensorAccelerationBias.z,

            parameters.consoleSixAxisSensorAccelerationGain.xx,
            parameters.consoleSixAxisSensorAccelerationGain.xy,
            parameters.consoleSixAxisSensorAccelerationGain.xz,

            parameters.consoleSixAxisSensorAccelerationGain.yx,
            parameters.consoleSixAxisSensorAccelerationGain.yy,
            parameters.consoleSixAxisSensorAccelerationGain.yz,

            parameters.consoleSixAxisSensorAccelerationGain.zx,
            parameters.consoleSixAxisSensorAccelerationGain.zy,
            parameters.consoleSixAxisSensorAccelerationGain.zz,
        });
}

} // namespace

namespace nn { namespace hid { namespace detail {

SixAxisSensorSanjiroFilter::SixAxisSensorSanjiroFilter() NN_NOEXCEPT
    : m_SamplingNumber(0)
    , m_CoordinationNumber(0)
    , m_LastUpdateTime(0)
    , m_TickToSec(1.f / nn::os::GetSystemTickFrequency())
    , m_HasVerticality(true)
    , m_Imu()
    , m_Lifo()
    , m_AppletResourceUserId(::nn::applet::AppletResourceUserId::GetInvalidId())
    , m_IsWorkBufferAllocated(false)
    , m_ThresholdStillness(0.2f)
{
    m_Direction = UnitDirection;
    m_Angle = ZeroVector3;
    m_Position = ZeroVector3;
    m_Offset = nn::util::Quaternion(0.0f, 0.0f, 0.0f, 1.0f);

    m_SixAxisSensorAccurateUserCalibrationState.progress = 1.f;
    m_SixAxisSensorAccurateUserCalibrationState.stage = ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_Completed;

    m_SixAxisSensorFusionPolicy.Reset();
}

SixAxisSensorSanjiroFilter::~SixAxisSensorSanjiroFilter() NN_NOEXCEPT
{
    // 何もしない
}

void SixAxisSensorSanjiroFilter::SetAppletResourceUserId(const ::nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    m_AppletResourceUserId = id;
}

::nn::applet::AppletResourceUserId SixAxisSensorSanjiroFilter::GetAppletResourceUserId() NN_NOEXCEPT
{
    return m_AppletResourceUserId;
}

void SixAxisSensorSanjiroFilter::Reset() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsWorkBufferAllocated());

    m_Lifo.Clear();

    ResetState();
}
//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::ResetState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsWorkBufferAllocated());
    m_SamplingNumber = 0;
    m_CurrentTime = 0;
    m_LastUpdateTime = nn::os::Tick(0);
    m_Offset = nn::util::Quaternion(0.0f, 0.0f, 0.0f, 1.0f);

    SetDirection(UnitDirection);
    SetAngle(ZeroVector3);
    SetPosition(ZeroVector3);
}
//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::SetDirection( const nn::hid::DirectionState& direction ) NN_NOEXCEPT
{
    nn::util::Matrix4x3f m(
        nn::util::Vector3f(direction.x),
        nn::util::Vector3f(direction.y),
        nn::util::Vector3f(direction.z),
        nn::util::Vector3f::Zero());
    nn::util::Quaternion global = nn::util::Quaternion::FromRotationMatrix(m);

    // We want global = m_Offset * local
    // so m_Offset= global * local^-1 = global * conjugate(local)
    nerd::Quaternion<float> local = m_Imu.GetRotation();
    nn::util::Quaternion localConj(-local.x, -local.y, -local.z, local.w);
    m_Offset = global * localConj;

    // In that case we are looking for a rotation around z that makes our front matches with the given front
    if (m_HasVerticality)
    {
        auto currentFront = local.TransformVector(nerd::float3(1.0f, 0.0f, 0.0f));
        float theta = atan2(
            -currentFront.y() * direction.x.x + currentFront.x() * direction.x.y,
            currentFront.x() * direction.x.x + currentFront.y() * direction.x.y);
        m_Offset = nn::util::Quaternion::FromAxisAngle( nn::util::Vector3f(0.0f, 0.0f, 1.0f), theta);
    }
}

//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::SetAngle( const nn::util::Float3& angle ) NN_NOEXCEPT
{
    m_Angle = angle;
}
//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::SetPosition( const nn::util::Float3& position ) NN_NOEXCEPT
{
    m_Position = position;
}
//-----------------------------------------------------------------------------
bool SixAxisSensorSanjiroFilter::GetState(SevenSixAxisSensorState* pOutValue,
                                          nn::os::Tick* lastTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(lastTime);
    NN_SDK_REQUIRES(IsWorkBufferAllocated());

    if (m_Lifo.IsEmpty())
    {
        return false;
    }

    m_Lifo.Read(pOutValue, 1);

    if (lastTime != nullptr)
    {
        *lastTime = m_LastUpdateTime;
    }

    return true;
}
//-----------------------------------------------------------------------------
int SixAxisSensorSanjiroFilter::GetStates(SevenSixAxisSensorState* pOutStates,
                                          int count,
                                          nn::os::Tick* lastTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);
    NN_SDK_REQUIRES_NOT_NULL(lastTime);
    NN_SDK_REQUIRES(IsWorkBufferAllocated());

    int num = m_Lifo.Read(pOutStates, count);

    if (lastTime != nullptr)
    {
        *lastTime = m_LastUpdateTime;
    }

    return num;
}

//-----------------------------------------------------------------------------
bool SixAxisSensorSanjiroFilter::GetIsStill() NN_NOEXCEPT
{
    return (m_Imu.GetStillness() > m_ThresholdStillness);
}

//-----------------------------------------------------------------------------
float SixAxisSensorSanjiroFilter::GetVerticalizationError() NN_NOEXCEPT
{
    return m_Imu.GetVerticalizationError();
}

//-----------------------------------------------------------------------------
::nn::util::Float3 SixAxisSensorSanjiroFilter::GetGyroBias() NN_NOEXCEPT
{
    auto calibration = m_Imu.GetCalibration();
    return ::nn::util::Float3{{{
        calibration[size_t(nerd::Imu::Params::GyroBiasX)],
        calibration[size_t(nerd::Imu::Params::GyroBiasY)],
        calibration[size_t(nerd::Imu::Params::GyroBiasZ)],
    }}};
}

//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::Update(const nn::util::Float3 &acceleration,
                                        const nn::util::Float3& angularVelocity,
                                        const nn::TimeSpan& timespan) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsWorkBufferAllocated());
    float dt = float(timespan.GetNanoSeconds() * 1e-9);
    nerd::float3 radPerSec = Pi2 * (nerd::float3 const&)angularVelocity;
    UpdateNerdImpl(GravitationalAcceleration * (nerd::float3 const&)acceleration, radPerSec, dt);
}

//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::Update(const nn::util::Float3 &acceleration,
                                        const nn::util::Float3& angularVelocity,
                                        const nn::os::Tick& time )
{
    NN_SDK_REQUIRES(IsWorkBufferAllocated());
    SevenSixAxisSensorState state;

    state.acceleration = acceleration;
    {
        const auto StableRotationVelocity = m_Imu.GetStableRotationVelocity();
        state.angularVelocity.x = StableRotationVelocity.x() * AngularVelocityConvertCoef;
        state.angularVelocity.y = StableRotationVelocity.y() * AngularVelocityConvertCoef;
        state.angularVelocity.z = StableRotationVelocity.z() * AngularVelocityConvertCoef;
    }

    state.samplingNumber = m_SamplingNumber++;

    if (m_LastUpdateTime.GetInt64Value() == 0) m_LastUpdateTime = time;
    state.timeStamp = ::nn::os::ConvertToTimeSpan(time);
    float dt = float((time - m_LastUpdateTime).GetInt64Value()) * m_TickToSec;
    nerd::float3 radPerSec = Pi2 * (nerd::float3 const&)angularVelocity;

    UpdateNerdImpl(GravitationalAcceleration * (nerd::float3 const&)acceleration, radPerSec, dt);

    auto quaternion = m_Imu.GetRotation().AsVector4();

    ::nn::util::VectorSet(
        &state.rotation,
        quaternion.x(),
        quaternion.y(),
        quaternion.z(),
        quaternion.w()
    );

    state.coordinateNumber = m_CoordinationNumber;
    m_Lifo.Append(state);
    m_LastUpdateTime = time;

    ProceedCalibration();

    SixAxisSensorFusionParameters parameters = {
        m_Imu.GetStillness(),
        m_Imu.GetVerticalizationError()
    };

    m_SixAxisSensorFusionPolicy.Update(parameters);
}

//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::SetStateBuffer(const ::nn::applet::AppletResourceUserId& id, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    NN_UNUSED(id);
    NN_UNUSED(size);

    auto pLifo = new(buffer) SevenSixAxisSensorStateLifo(); // placement new
    pLifo->Clear();

    m_Lifo.Clear();
}

//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::SetWorkBuffer(const ::nn::applet::AppletResourceUserId& id, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_UNUSED(size);

    SetAppletResourceUserId(id);
    m_Imu.Init(*reinterpret_cast<nerd::Imu::WorkingMemory*>(buffer));
    m_IsWorkBufferAllocated = true;
    Reset(); // ワークバッファ設定時に状態クリア
    m_Imu.StartOnlineCalibration();
}

//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::ClearStateBuffer() NN_NOEXCEPT
{
    // Do nothing when work buffer has already been released
    if (IsWorkBufferAllocated() == false)
    {
        return;
    }

    m_Lifo.Clear();
}

void SixAxisSensorSanjiroFilter::ClearWorkBuffer() NN_NOEXCEPT
{
    // Do nothing when work buffer has already been released
    if (IsWorkBufferAllocated() == false)
    {
        return;
    }

    SetAppletResourceUserId(::nn::applet::AppletResourceUserId::GetInvalidId());

    m_Imu.StopOnlineCalibration();
    m_Imu.Finalize();
    m_IsWorkBufferAllocated = false;
}

//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::UpdateNerdImpl(nerd::float3 const& acceleration, nerd::float3 const& angularVelocity, float dt)
{
    NN_SDK_REQUIRES(IsWorkBufferAllocated());
    if (dt == 0)
    {
        return;
    }

    m_CurrentTime+= dt;

    m_Imu.AddSample(acceleration, angularVelocity, dt);
    nn::util::Quaternion q;
    GetQuaternion(&q);

    auto const m = nn::util::Matrix4x3f::MakeRotation(q);
    m.GetAxisX().Get(&m_Direction.x.x, &m_Direction.x.y, &m_Direction.x.z);
    m.GetAxisY().Get(&m_Direction.y.x, &m_Direction.y.y, &m_Direction.y.z);
    m.GetAxisZ().Get(&m_Direction.z.x, &m_Direction.z.y, &m_Direction.z.z);
}

//-----------------------------------------------------------------------------
void SixAxisSensorSanjiroFilter::GetQuaternion(nn::util::Quaternion* pOutQuaternion) const
{
    NN_SDK_REQUIRES_NOT_NULL(pOutQuaternion);
    auto const& rotationQuat = m_Imu.GetRotation();
    *pOutQuaternion = m_Offset*nn::util::Quaternion(rotationQuat.x, rotationQuat.y, rotationQuat.z, rotationQuat.w);
}

void SixAxisSensorSanjiroFilter::ProceedCalibration() NN_NOEXCEPT
{
    if (m_Imu.GetCurrentState() != ::nerd::Imu::ImuState::OfflineCalibration)
    {
        m_SixAxisSensorAccurateUserCalibrationState.stage =
            ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_InitialStage;
        return;
    }

    // Calibration Step
    const auto Info = m_Imu.GetCalibrationInfo();
    m_SixAxisSensorAccurateUserCalibrationState.progress = Info.progress;

    switch (Info.state)
    {
    case nerd::Imu::CalibrationInfo::State::Ready:
        m_SixAxisSensorAccurateUserCalibrationState.stage =
            ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_InitialStage;
        break;
    case nerd::Imu::CalibrationInfo::State::Still:
    case nerd::Imu::CalibrationInfo::State::StillX:
    case nerd::Imu::CalibrationInfo::State::StillY:
    case nerd::Imu::CalibrationInfo::State::StillZ:
        m_SixAxisSensorAccurateUserCalibrationState.stage =
            ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_AtRest;
        break;
    case nerd::Imu::CalibrationInfo::State::TurnX:
        m_SixAxisSensorAccurateUserCalibrationState.stage =
            ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_RotateX;
        break;
    case nerd::Imu::CalibrationInfo::State::TurnY:
        m_SixAxisSensorAccurateUserCalibrationState.stage =
            ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_RotateY;
        break;
    case nerd::Imu::CalibrationInfo::State::TurnZ:
        m_SixAxisSensorAccurateUserCalibrationState.stage =
            ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_RotateZ;
        break;
    case nerd::Imu::CalibrationInfo::State::Done:
        m_SixAxisSensorAccurateUserCalibrationState.stage =
            ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_Completed;
        break;
    default:
        // 何もしない
        break;
    }
}

void SixAxisSensorSanjiroFilter::StartCalibration() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsWorkBufferAllocated());

    m_SixAxisSensorAccurateUserCalibrationState.stage = ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_InitialStage;
    m_SixAxisSensorAccurateUserCalibrationState.progress = 0.f;

    m_Imu.StartOnlineCalibration();
}

void SixAxisSensorSanjiroFilter::StopCalibration() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsWorkBufferAllocated());

    m_SixAxisSensorAccurateUserCalibrationState.progress = 1.f;
    m_SixAxisSensorAccurateUserCalibrationState.stage = ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_Completed;

    m_Imu.StopOnlineCalibration();
}

void SixAxisSensorSanjiroFilter::GetAccurateUserCalibrationState(::nn::hid::system::SixAxisSensorAccurateUserCalibrationState* pOutValue) NN_NOEXCEPT
{
    *pOutValue = m_SixAxisSensorAccurateUserCalibrationState;
}

void SixAxisSensorSanjiroFilter::GetCalibrationParameters(
    ConsoleSixAxisSensorCalibrationParameters* pOutParameters) const NN_NOEXCEPT
{
    nerd::NumericalVector<float, nerd::Imu::gParamCount> parametersVector;
    parametersVector = m_Imu.GetCalibration();
    *pOutParameters = Convert(parametersVector);
}

void SixAxisSensorSanjiroFilter::SetCalibrationParameters(
    const ConsoleSixAxisSensorCalibrationParameters& parameters) NN_NOEXCEPT
{
    nerd::NumericalVector<float, nerd::Imu::gParamCount> parametersVector = Convert(parameters);
    m_Imu.SetCalibration(parametersVector);
}

bool SixAxisSensorSanjiroFilter::IsWorkBufferAllocated() const NN_NOEXCEPT
{
    return m_IsWorkBufferAllocated;
}

void SixAxisSensorSanjiroFilter::SetFilterSetting(const FilterSetting& setting) NN_NOEXCEPT
{
    // Note: FilterSetting consists of parameters which each application can set.
    //       Sensor fusion is incorporated in system side at this moment, so just ignore FilterSetting
    NN_UNUSED(setting);

    if (IsWorkBufferAllocated())
    {
        m_Imu.SetVerticalizationStrength(
            m_SixAxisSensorFusionPolicy.GetVerticalizationStrength()
        );
    }
}

}}} // namespace nn::hid::detail
