﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/util/util_VectorApi.h>

#include "hid_SixAxisSensorProcessor.h"
#include "hid_SixAxisSensorMathUtility.h"

namespace {

/**
 * @brief       ユークリッドノルムを取得します。
 */
float GetEuclideanNorm(const nn::util::Float3& float3Value)
{
    nn::util::Vector3fType vector;
    nn::util::VectorLoad(&vector, float3Value);

    return nn::util::VectorLength(vector);
}

/**
 * @brief       外積計算をします。( float3Value1 × float3Value2 )
 */
void CrossProduct(nn::util::Float3* pOutFloat3Value,
                  const nn::util::Float3& float3Value1,
                  const nn::util::Float3& float3Value2)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFloat3Value);

    nn::util::Vector3fType outVector;
    nn::util::Vector3fType vector1;
    nn::util::Vector3fType vector2;

    nn::util::VectorLoad(&vector1, float3Value1);
    nn::util::VectorLoad(&vector2, float3Value2);

    nn::util::VectorCross(&outVector,
                          vector1,
                          vector2);

    nn::util::VectorStore(pOutFloat3Value, outVector);
}

}

namespace nn { namespace hid { namespace detail {

void InnerProduct(float* pOutValue,
                  const nn::util::Float3& float3Value1,
                  const nn::util::Float3& float3Value2)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    nn::util::Vector3fType vector1;
    nn::util::Vector3fType vector2;

    nn::util::VectorLoad(&vector1, float3Value1);
    nn::util::VectorLoad(&vector2, float3Value2);

    *pOutValue = nn::util::VectorDot(vector1, vector2);
}

float Normalize(nn::util::Float3* pOutFloat3Value)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFloat3Value);

    nn::util::Vector3fType vector;

    nn::util::VectorLoad(&vector, *pOutFloat3Value);

    float lengthSquared = nn::util::VectorNormalize(&vector, vector);

    if(lengthSquared != 0.0f)
    {
        nn::util::VectorStore(pOutFloat3Value, vector);
    }

    return Sqrt(lengthSquared);
}

void NormalizeDirectionState( nn::hid::DirectionState *pOutDir, const float& Threshold )
{
    int count = 0;
    const int IterationCountMax = 100;
    float euclideanNormSum;
    nn::hid::DirectionState vec;

    do {

        Normalize(&pOutDir->x);
        Normalize(&pOutDir->y);
        Normalize(&pOutDir->z);

        // DirectionState は右手系であることを前提とする。
        // 厳密な正規直交系であれば 2 軸の外積結果は残りの 1 軸のベクトルと一致する。
        // 外積計算により得られた各ベクトルの推定値 X^, Y^, Z^ を取得する。
        CrossProduct(&vec.x, pOutDir->y, pOutDir->z); // X^ = Y × Z
        CrossProduct(&vec.y, pOutDir->z, pOutDir->x); // Y^ = Z × X
        CrossProduct(&vec.z, pOutDir->x, pOutDir->y); // Z^ = X × Y

        // euclideanNormSum = |X^| + |Y^| + |Z^|
        // {X^, Y^, Z^} が正規直交系であれば 3軸のユークリッドノルムの和は最大(= 3)になる。
        // Threshold は 3 以下の正値。3に近いほど条件は厳しく pOutDir は正規直交系に近づく
        euclideanNormSum = Normalize( &vec.x );
        euclideanNormSum += Normalize( &vec.y );
        euclideanNormSum += Normalize( &vec.z );

        // 正規化後の推定された系 { X^/|X^|, Y^/|Y^|, Z^/|Z^| } 値と平均を取って更新する
        pOutDir->x.x = ( vec.x.x + pOutDir->x.x ) * 0.5f ;
        pOutDir->x.y = ( vec.x.y + pOutDir->x.y ) * 0.5f ;
        pOutDir->x.z = ( vec.x.z + pOutDir->x.z ) * 0.5f ;

        pOutDir->y.x = ( vec.y.x + pOutDir->y.x ) * 0.5f ;
        pOutDir->y.y = ( vec.y.y + pOutDir->y.y ) * 0.5f ;
        pOutDir->y.z = ( vec.y.z + pOutDir->y.z ) * 0.5f ;

        pOutDir->z.x = ( vec.z.x + pOutDir->z.x ) * 0.5f ;
        pOutDir->z.y = ( vec.z.y + pOutDir->z.y ) * 0.5f ;
        pOutDir->z.z = ( vec.z.z + pOutDir->z.z ) * 0.5f ;

        if(++count > IterationCountMax)
        {
            break;
        }

    } while ( euclideanNormSum < Threshold ) ;
}

void MakeVecDir( nn::hid::DirectionState* pOutDir, const nn::util::Float3& vec1, const nn::util::Float3& vec2 )
{
    // TODO: 数式の理解
    nn::util::Float3  dX;
    nn::util::Float3  dY1;
    nn::util::Float3  dY2;
    float             f1;
    float             f2;
    float             z2y1;
    float             x2z1;
    float             y2x1;
    float             y2z1;
    float             z2x1;
    float             x2y1;

    dX.x = ( z2y1 = vec1.y * vec2.z ) - ( y2z1 = vec1.z * vec2.y ) ;
    dX.y = ( x2z1 = vec1.z * vec2.x ) - ( z2x1 = vec1.x * vec2.z ) ;
    dX.z = ( y2x1 = vec1.x * vec2.y ) - ( x2y1 = vec1.y * vec2.x ) ;

    if ( Normalize(&dX) == 0.0f )
    {
        // 単位行列
        *pOutDir = nn::hid::detail::UnitDirection;
        return;
    }

    CrossProduct( &dY1, vec1, dX );
    CrossProduct( &dY2, vec2, dX );

    pOutDir->x.x =        dX.x * dX.x   + dY2.x * dY1.x + vec2.x * vec1.x ;
    pOutDir->x.y = ( f1 = dX.y * dX.x ) + dY2.y * dY1.x + y2x1 ;
    pOutDir->x.z = ( f2 = dX.z * dX.x ) + dY2.z * dY1.x + z2x1 ;

    pOutDir->y.x =   f1                 + dY2.x * dY1.y + x2y1 ;
    pOutDir->y.y =        dX.y * dX.y   + dY2.y * dY1.y + vec2.y * vec1.y ;
    pOutDir->y.z = ( f1 = dX.z * dX.y ) + dY2.z * dY1.y + z2y1 ;

    pOutDir->z.x =   f2                 + dY2.x * dY1.z + x2z1 ;
    pOutDir->z.y =   f1                 + dY2.y * dY1.z + y2z1 ;
    pOutDir->z.z =        dX.z * dX.z   + dY2.z * dY1.z + vec2.z * vec1.z ;
}

void  MultiplyDirection( nn::hid::DirectionState* pOutDir, const nn::hid::DirectionState& Dir1, const nn::hid::DirectionState& Dir2 )
{
    pOutDir->x.x = Dir1.x.x * Dir2.x.x + Dir1.x.y * Dir2.y.x + Dir1.x.z * Dir2.z.x ;
    pOutDir->x.y = Dir1.x.x * Dir2.x.y + Dir1.x.y * Dir2.y.y + Dir1.x.z * Dir2.z.y ;
    pOutDir->x.z = Dir1.x.x * Dir2.x.z + Dir1.x.y * Dir2.y.z + Dir1.x.z * Dir2.z.z ;

    pOutDir->y.x = Dir1.y.x * Dir2.x.x + Dir1.y.y * Dir2.y.x + Dir1.y.z * Dir2.z.x ;
    pOutDir->y.y = Dir1.y.x * Dir2.x.y + Dir1.y.y * Dir2.y.y + Dir1.y.z * Dir2.z.y ;
    pOutDir->y.z = Dir1.y.x * Dir2.x.z + Dir1.y.y * Dir2.y.z + Dir1.y.z * Dir2.z.z ;

    pOutDir->z.x = Dir1.z.x * Dir2.x.x + Dir1.z.y * Dir2.y.x + Dir1.z.z * Dir2.z.x ;
    pOutDir->z.y = Dir1.z.x * Dir2.x.y + Dir1.z.y * Dir2.y.y + Dir1.z.z * Dir2.z.y ;
    pOutDir->z.z = Dir1.z.x * Dir2.x.z + Dir1.z.y * Dir2.y.z + Dir1.z.z * Dir2.z.z ;
}

float ApproximateDirectionAxis(nn::util::Float3 *pOutVec,
                               const nn::util::Float3& lastVec,
                               const nn::util::Float3& baseVec,
                               const float& revisePower)
{
    nn::util::Float3 vec;
    nn::util::Float3 nrm;
    float            speed;
    float            coefficient;

    // 変位ベクトルを求める
    vec.x = pOutVec->x - lastVec.x;
    vec.y = pOutVec->y - lastVec.y;
    vec.z = pOutVec->z - lastVec.z;
    speed = GetEuclideanNorm(vec);

    // 基本方向から現在方向を見たベクトルを求める
    nrm.x = pOutVec->x - baseVec.x;
    nrm.y = pOutVec->y - baseVec.y;
    nrm.z = pOutVec->z - baseVec.z;

    // 基本方向から見たモーションプラス方向の移動方向
    InnerProduct(&coefficient, nrm, vec);
    if (coefficient >= 0.0f) {
        return (0.0f);    // 静止または遠ざかる時は処理しない(⇔ なす角90度以上)
    }

    // 直交成分を抽出
    InnerProduct(&coefficient, *pOutVec, nrm);
    nrm.x -= coefficient * pOutVec->x;
    nrm.y -= coefficient * pOutVec->y;
    nrm.z -= coefficient * pOutVec->z;

    // 直交成分のノルムが小さければ処理しない
    coefficient = GetEuclideanNorm(nrm);
    if (coefficient < 0.001f)
    {
        return ( 0.0f );
    }
    coefficient = speed * revisePower / coefficient; // 係数は変位ノルムに比例させる

    // 現在方向ベクトルを基本方向に近づける
    pOutVec->x -= coefficient * nrm.x;
    pOutVec->y -= coefficient * nrm.y;
    pOutVec->z -= coefficient * nrm.z;

    return (coefficient);
}

}}} // namespace nn::hid::detail

