﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_HidServer.h"
#include "hid_RingLifoHolder.h"
#include "hid_Session.h"
#include "hid_SixAxisSensorImpl.h"
#include "hid_SixAxisSensorLifo.h"
#include "hid_StaticObject.h"
#include "hid_XpadId.h"
#include "hid_XpadLifo.h"
#include "hid_XpadSessionManager.h"

namespace nn { namespace hid { namespace detail {


namespace {

//!< 6 軸センサーのセッションを扱うためのクラスです。
class SixAxisSensorSession final : public Session
{
    NN_DISALLOW_COPY(SixAxisSensorSession);
    NN_DISALLOW_MOVE(SixAxisSensorSession);

private:
    //!< hid サービスのセッション
    ::nn::sf::SharedPointer<IHidServer> m_Session;

    //!< 6 軸センサーの ID
    BasicXpadId m_XpadId;

    //!< 6 軸センサーの LIFO
    RingLifoHolder<SixAxisSensorLifo> m_LifoHolder;

public:
    explicit SixAxisSensorSession(const BasicXpadId& xpadId) NN_NOEXCEPT
        : m_XpadId(xpadId)
    {
        // 何もしない
    }

    virtual ~SixAxisSensorSession() NN_NOEXCEPT NN_OVERRIDE { /* 何もしない */ }

    //!< Xpad ID の型
    typedef BasicXpadId XpadIdType;

    //!< 6 軸センサーの LIFO を返します。
    const SixAxisSensorLifo* GetLifo() const NN_NOEXCEPT
    {
        return m_LifoHolder.GetLifo();
    }

    void AttachSharedMemoryIfNotMapped() NN_NOEXCEPT
    {
        if(!IsSharedMemoryMapped())
        {
            Activate();
        }
    }

protected:

    virtual ::nn::Result ThrowResultActivationUpperLimitOver(
        ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(ResultSixAxisSensorActivationUpperLimitOver());
    }

    virtual ::nn::Result ThrowResultDeactivationLowerLimitOver(
        ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(ResultSixAxisSensorDeactivationLowerLimitOver());
    }

    virtual ::nn::Result ThrowResultAllocationFailed() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(ResultSixAxisSensorLifoAllocationFailed());
    }

    virtual ::nn::Result CreateProxy() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(CreateHidServerProxy(&m_Session));
    }

    virtual void DestroyProxy() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Session.Reset();
    }

    virtual ::nn::Result ActivateService() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(m_Session->ActivateSixAxisSensor(m_XpadId));
    }

    virtual ::nn::Result DeactivateService() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(m_Session->DeactivateSixAxisSensor(m_XpadId));
    }

    virtual ::nn::Result GetSharedMemoryHandle(::nn::sf::NativeHandle* outValue
                                               ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        NN_RESULT_THROW(m_Session->GetSixAxisSensorLifoHandle(outValue, m_XpadId));
    }

    virtual bool IsSharedMemoryMapped() NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_LifoHolder.GetLifo() != nullptr);
    }

    virtual void AttachSharedMemory(::nn::os::NativeHandle handle,
                                    bool managed) NN_NOEXCEPT NN_OVERRIDE
    {
        m_LifoHolder.Attach(handle, managed);
    }

    virtual void FinalizeSharedMemory() NN_NOEXCEPT NN_OVERRIDE
    {
        m_LifoHolder.Finalize();
    }
};

XpadSessionManager<SixAxisSensorSession>& GetXpadSessionManager() NN_NOEXCEPT
{
    return StaticObject<XpadSessionManager<SixAxisSensorSession>>::Get();
}

} // namespace

::nn::Result GetSixAxisSensorStates(int* outGotCount,
                                    SixAxisSensorState* outStates,
                                    int count,
                                    const BasicXpadId& xpadId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outGotCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultSixAxisSensorNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultSixAxisSensorInvalidStateCount());
    SixAxisSensorSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultBasicXpadPlayerNumberOutOfRange());
    pSession->AttachSharedMemoryIfNotMapped();
    const SixAxisSensorLifo* pLifo = pSession->GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultSixAxisSensorNotInitialized());
    *outGotCount = pLifo->Read(outStates, count);
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
