﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/util/util_BitPack.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/system/hid_Npad.h>

#include "hid_HidServer.h"
#include "hid_BasicXpadImpl.h"
#include "hid_JoyXpadImpl.h"
#include "hid_JoyXpadSixAxisSensorImpl.h"
#include "hid_SixAxisSensorHandle.h"
#include "hid_SixAxisSensorImpl.h"
#include "hid_NpadImpl.h"
#include "hid_NpadId.h"

namespace nn { namespace hid { namespace detail {

SixAxisSensorHandle MakeSixAxisSensorHandle(uint8_t xpadTypeId, int playerNumber, int sixAxisSensorIdx) NN_NOEXCEPT
{
    auto bitPack = ::nn::util::BitPack32();
    bitPack.Set<SixAxisSensorHandleField::XpadTypeValue>(xpadTypeId);
    bitPack.Set<SixAxisSensorHandleField::PlayerNumber>(static_cast<int8_t>(playerNumber));
    bitPack.Set<SixAxisSensorHandleField::SixAxisSensorIdx>(static_cast<int8_t>(sixAxisSensorIdx));

    SixAxisSensorHandle handle = { static_cast<int32_t>(bitPack.storage) };
    return handle;
}

uint8_t GetSixAxisSensorHandleXpadTypeValue(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    ::nn::util::BitPack32 bitPack = { static_cast<unsigned int>(handle._storage) };
    return bitPack.Get<SixAxisSensorHandleField::XpadTypeValue>();
}

uint8_t GetSixAxisSensorHandleNpadStyleIndex(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    return GetSixAxisSensorHandleXpadTypeValue(handle) - NpadSixAxisSensorTypeValueOffset;
}

int GetSixAxisSensorHandlePlayerNumber(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    ::nn::util::BitPack32 bitPack = { static_cast<unsigned int>(handle._storage) };
    return bitPack.Get<SixAxisSensorHandleField::PlayerNumber>();
}

NpadIdType GetSixAxisSensorHandleNpadIdType(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    return static_cast<NpadIdType>(GetSixAxisSensorHandlePlayerNumber(handle));
}

int GetSixAxisSensorHandleSixAxisSensorIdx(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    ::nn::util::BitPack32 bitPack = { static_cast<unsigned int>(handle._storage) };
    return bitPack.Get<SixAxisSensorHandleField::SixAxisSensorIdx>();
}

::nn::Result GetSixAxisSensorHandles(SixAxisSensorHandle* pOutLeftHandle,
                                     SixAxisSensorHandle* pOutRightHandle,
                                     JoyXpadId xpadId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutLeftHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutRightHandle);
    NN_RESULT_DO(detail::AssertValidJoyXpadId(xpadId));
    *pOutLeftHandle = detail::MakeSixAxisSensorHandle(xpadId, SixAxisSensorIdx_Left);
    *pOutRightHandle = detail::MakeSixAxisSensorHandle(xpadId, SixAxisSensorIdx_Right);
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorHandle(SixAxisSensorHandle* pOutHandle,
                                    BasicXpadId xpadId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHandle);
    NN_RESULT_DO(detail::AssertValidXpadId(xpadId));
    *pOutHandle = detail::MakeSixAxisSensorHandle(xpadId, SixAxisSensorIdx_Center);
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorHandle(SixAxisSensorHandle* pOutHandle,
                                    int* pOutCount,
                                    int size,
                                    NpadIdType npadId,
                                    NpadStyleSet style) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutCount);
    NN_ABORT_UNLESS_GREATER_EQUAL(size, 1);
    NN_RESULT_DO(detail::VerifyValidNpadId(npadId));

    if (style.Test<NpadStyleFullKey>() == true)
    {
        *pOutHandle = detail::MakeSixAxisSensorHandle(NpadStyleFullKey::Index + NpadSixAxisSensorTypeValueOffset, static_cast<int>(npadId), SixAxisSensorIdx_Center);
        *pOutCount = 1;
    }
    else if (style.Test<NpadStyleHandheld> () == true)
    {
        *pOutHandle = detail::MakeSixAxisSensorHandle(NpadStyleHandheld::Index + NpadSixAxisSensorTypeValueOffset, static_cast<int>(npadId), SixAxisSensorIdx_Center);
        *pOutCount = 1;
    }
    else if (style.Test<NpadStyleJoyDual>() == true)
    {
        pOutHandle[0] = detail::MakeSixAxisSensorHandle(NpadStyleJoyDual::Index + NpadSixAxisSensorTypeValueOffset, static_cast<int>(npadId), SixAxisSensorIdx_Left);
        *pOutCount = 1;
        if (size > 1)
        {
            pOutHandle[1] = detail::MakeSixAxisSensorHandle(NpadStyleJoyDual::Index + NpadSixAxisSensorTypeValueOffset, static_cast<int>(npadId), SixAxisSensorIdx_Right);
            (*pOutCount)++;
        }
    }
    else if (style.Test<NpadStyleJoyLeft> () == true)
    {
        *pOutHandle = detail::MakeSixAxisSensorHandle(NpadStyleJoyLeft::Index + NpadSixAxisSensorTypeValueOffset, static_cast<int>(npadId), SixAxisSensorIdx_Left);
        *pOutCount = 1;
    }
    else if (style.Test<NpadStyleJoyRight>() == true)
    {
        *pOutHandle = detail::MakeSixAxisSensorHandle(NpadStyleJoyRight::Index + NpadSixAxisSensorTypeValueOffset, static_cast<int>(npadId), SixAxisSensorIdx_Right);
        *pOutCount = 1;
    }
    else if (style.Test<NpadStyleGc>() == true)
    {
        // Gc スタイルの 6軸のハンドルは FullKey を流用
        *pOutHandle = detail::MakeSixAxisSensorHandle(NpadStyleFullKey::Index + NpadSixAxisSensorTypeValueOffset, static_cast<int>(npadId), SixAxisSensorIdx_Center);
        *pOutCount = 1;
    }
    else if (style.Test<system::NpadStyleSystem> () == true)
    {
        *pOutCount = 0;
    }
    else if (style.Test<system::NpadStyleSystemExt>() == true)
    {
        *pOutCount = 0;
    }
    else if (style.Test<NpadStylePalma>() == true)
    {
        // TODO: Palma スタイルは暫定的に 6軸のハンドルは FullKey を流用
        *pOutHandle = detail::MakeSixAxisSensorHandle(NpadStyleFullKey::Index + NpadSixAxisSensorTypeValueOffset, static_cast<int>(npadId), SixAxisSensorIdx_Center);
        *pOutCount = 1;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorState(SixAxisSensorState* outValue,
                                   const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(outValue);

    int outGotCount;
    return GetSixAxisSensorStates(&outGotCount, outValue, 1, handle);
}

::nn::Result GetSixAxisSensorStates(int* outGotCount,
                                    SixAxisSensorState* outStates,
                                    int count,
                                    const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(outGotCount);
    NN_ABORT_UNLESS_NOT_NULL(outStates);

    BasicXpadId basicXpadId;
    JoyXpadId   joyXpadId;
    NpadIdType npadId;
    switch(GetSixAxisSensorHandleXpadTypeValue(handle))
    {
    case XpadType<BasicXpadId>::TypeValue:
        basicXpadId = detail::GetSixAxisSensorHandleXpadId<BasicXpadId>(handle);
        return GetSixAxisSensorStates(outGotCount, outStates, count, basicXpadId); // TORIAEZU
    case XpadType<JoyXpadId>::TypeValue:
        joyXpadId = detail::GetSixAxisSensorHandleXpadId<JoyXpadId>(handle);
        switch(GetSixAxisSensorHandleSixAxisSensorIdx(handle))
        {
        case SixAxisSensorIdx_Left:
            return GetSixAxisSensorLeftStates(outGotCount, outStates, count, joyXpadId); // TORIAEZU
        case SixAxisSensorIdx_Right:
            return GetSixAxisSensorRightStates(outGotCount, outStates, count, joyXpadId); // TORIAEZU
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    default:
        switch(GetSixAxisSensorHandleNpadStyleIndex(handle))
        {
        case NpadStyleFullKey::Index:
            npadId = detail::GetSixAxisSensorHandleNpadIdType(handle);
            return GetSixAxisSensorStates(outGotCount, outStates, count, npadId, NpadStyleFullKey::Mask);

        case NpadStyleHandheld::Index:
            npadId = detail::GetSixAxisSensorHandleNpadIdType(handle);
            return GetSixAxisSensorStates(outGotCount, outStates, count, npadId, NpadStyleHandheld::Mask);

        case NpadStyleJoyDual::Index:
            switch (GetSixAxisSensorHandleSixAxisSensorIdx(handle))
            {
            case SixAxisSensorIdx_Left:
                npadId = detail::GetSixAxisSensorHandleNpadIdType(handle);
                return GetSixAxisSensorLeftStates(outGotCount, outStates, count, npadId, NpadStyleJoyDual::Mask);
            case SixAxisSensorIdx_Right:
                npadId = detail::GetSixAxisSensorHandleNpadIdType(handle);
                return GetSixAxisSensorRightStates(outGotCount, outStates, count, npadId, NpadStyleJoyDual::Mask);
            default:
                NN_UNEXPECTED_DEFAULT;
            }

        case NpadStyleJoyLeft::Index:
            npadId = detail::GetSixAxisSensorHandleNpadIdType(handle);
            return GetSixAxisSensorStates(outGotCount, outStates, count, npadId, NpadStyleJoyLeft::Mask);

        case NpadStyleJoyRight::Index:
            npadId = detail::GetSixAxisSensorHandleNpadIdType(handle);
            return GetSixAxisSensorStates(outGotCount, outStates, count, npadId, NpadStyleJoyRight::Mask);
        case NpadStyleGc::Index:
            npadId = detail::GetSixAxisSensorHandleNpadIdType(handle);
            return GetSixAxisSensorStates(outGotCount, outStates, count, npadId, NpadStyleFullKey::Mask);

        case system::NpadStyleSystem::Index:
        case system::NpadStyleSystemExt::Index:
            NN_RESULT_SUCCESS;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

}}} // namespace nn::hid::detail
