﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       6軸センサーのキャリブレーション処理に特化したクラスの宣言
 */

#pragma once

#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/xcd/xcd_Input.h>

namespace nn { namespace hid { namespace detail {

//!< 6軸センサーのキャリブレーション処理を担うクラス
class SixAxisSensorCalibrator final
{
    NN_DISALLOW_COPY(SixAxisSensorCalibrator);
    NN_DISALLOW_MOVE(SixAxisSensorCalibrator);

private:
    //!< キャリブレーションに用いるパラメータを表す構造体
    struct CalibratorConfig
    {
        ::nn::xcd::SensorState horizontalStateAccelerometerCount;  //!< 水平に置いたときの加速度カウントの理論値
        int32_t horizontalStateAccelerometerCountRange;            //!< 水平に置いたときの加速度カウントの許容範囲
        int32_t angularVelocityCountRange;                         //!< キャリブレーション区間における角速度カウントの許容変動範囲(各軸共通)
        int validSampleCountMax;                                   //!< 有効なセンサー入力数の最大値(キャリブレーション完了と見做す値)
        ::nn::xcd::SensorCalibrationValue calibrationValue;        //!< キャリブレーション値
        ::nn::util::Float3 typicalAccelerometer;                   //!< 加速度センサーの Typical 値
        bool isAtRest;                                             //!< 静止状態か
        bool isHorizontal;                                         //!< 水平状態か
    };

    CalibratorConfig m_Config;                                 //!< 現在の各種設定値

    ::nn::xcd::SixAxisSensorState m_LastSixAxisSensorState;    //!< 前回サンプリングした 6 軸センサー値
    ::nn::util::Float3 m_AngularVelocityCountMax;              //!< キャリブレーション区間における最大角速度カウント値
    ::nn::util::Float3 m_AngularVelocityCountMin;              //!< キャリブレーション区間における最小角速度カウント値
    float m_AccelerometerCountSum[3];                          //!< X/Y/Z 軸における加速度カウント値の総和
    float m_AngularVelocityCountSum[3];                        //!< X/Y/Z 軸における角速度カウント値の総和
    int m_ValidSampleCount;                                    //!< 有効なセンサー入力数

public:
    SixAxisSensorCalibrator()  NN_NOEXCEPT;

    //!< 6軸センサーの生値を注入して内部状態を更新します
    void Update(const ::nn::xcd::SixAxisSensorState& state) NN_NOEXCEPT;

    //!< キャリブレーションが完了しているか否かを返します
    bool IsCalibrated() const NN_NOEXCEPT;

    //!< キャリブレーションのパラメータを設定します
    void SetCalibrationParameters(const ::nn::xcd::SensorState& horizontalStateAccelerometerCount,
                                  const int32_t& horizontalStateAccelerometerCountRange,
                                  const int32_t& angularVelocityCountRange,
                                  const int& validSampleCountMax,
                                  const ::nn::xcd::SensorCalibrationValue& calibrationValue,
                                  const ::nn::util::Float3& typicalAccelerometer) NN_NOEXCEPT;

    //!< キャリブレーション値を取得します
    Result GetUserCalibrationValue(::nn::xcd::SensorCalibrationValue* pOutCalibrationValue) const NN_NOEXCEPT;

    //!< 内部ステートをクリアします。
    void Reset() NN_NOEXCEPT;

    // 静止判定結果を返します。
    bool IsAtRest() const NN_NOEXCEPT;

    // 水平判定結果を返します。
    bool IsHorizontal() const NN_NOEXCEPT;

private:

    //!< 標本サンプルの平均値を取得します
    void GetMeanValues(::nn::util::Float3* pOutAccelerometerCount,
                       ::nn::util::Float3* pOutAngularVelocityCount) const NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
