﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_NpadFullKey.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/hid/hid_NpadJoyDual.h>
#include <nn/hid/hid_NpadJoyLeft.h>
#include <nn/hid/hid_NpadJoyRight.h>

#include <nn/result/result_HandlingUtility.h>

#include "hid_NpadId.h"
#include "hid_SixAxisSensorAppletSetting.h"
#include "hid_SixAxisSensorHandle.h"

namespace {

// Accelerometer
const nn::xcd::AccelerometerFsr DefaultAccelerometerFsr = nn::xcd::AccelerometerFsr_8G;
const nn::hid::AccelerometerPlayMode DefaultAccelerometerMode = nn::hid::AccelerometerPlayMode::AccelerometerPlayMode_Tight;
const float DefaultAccelerometerPlayRadius  = 0.0f;
const float DefaultAccelerometerSensitivity = 1.0f;
const float DefaultAccelerometerRevisePower = 0.030f;
const float DefaultAccelerometerReviseRange = 0.400f;

// Gyroscope
const nn::xcd::GyroscopeFsr DefaultGyroscopeFsr = nn::xcd::GyroscopeFsr_2000dps;
const bool DefaultGyroscopeEnableZeroDrift = true;
const bool DefaultGyroscopeEnableZeroPlay = false;
const float DefaultGyroscopeZeroPlayRadius = 0.0f;
const nn::hid::GyroscopeZeroDriftMode DefaultGyroscopeZeroDriftMode = nn::hid::GyroscopeZeroDriftMode::GyroscopeZeroDriftMode_Standard;
const nn::util::Float3 DefaultGyroscopeMagnification = {{{ 1.0f, 1.0f, 1.0f}}};
const nn::util::Float3 DefaultGyroscopeDirectionMagnification = {{{ 1.0f, 1.0f, 1.0f}}};

// SixAxisSensorProcessor
const nn::hid::DirectionState  DefaultBaseDirection = { {{{1.0f, 0.0f, 0.0f}}},
                                                        {{{0.0f, 1.0f, 0.0f}}},
                                                        {{{0.0f, 0.0f, 1.0f}}} };
const float DefaultBaseDirectionRevisePower = 0.0f;

}// namespace

namespace nn { namespace hid { namespace detail {

void AccelerometerSetting::Reset(bool global) NN_NOEXCEPT
{
    fsr = DefaultAccelerometerFsr;

    ResetPlayMode();
    ResetPlayModeParameters();
    ResetAccelerationReviseParameters();

    if (global)
    {
        ResetShiftCalibrationValue();
    }
}

void AccelerometerSetting::SetAccelerometerFsr(const nn::xcd::AccelerometerFsr& accFsr) NN_NOEXCEPT
{
    fsr = accFsr;
}

bool AccelerometerSetting::IsSixAxisSensorFusionEnabled() const NN_NOEXCEPT
{
    return isReviseEnabled;
}

void AccelerometerSetting::EnableSixAxisSensorFusion(bool enable) NN_NOEXCEPT
{
    isReviseEnabled = enable;
}

void AccelerometerSetting::SetSixAxisSensorFusionParameters(float power,
                                                           float range) NN_NOEXCEPT
{
    revisePower = power;
    reviseRange = range;
}

void AccelerometerSetting::GetSixAxisSensorFusionParameters(float* pOutRevisePower,
                                                           float* pOutReviseRange) const NN_NOEXCEPT
{
    *pOutRevisePower = revisePower;
    *pOutReviseRange = reviseRange;
}

void AccelerometerSetting::ResetAccelerationReviseParameters() NN_NOEXCEPT
{
    isReviseEnabled = true;
    revisePower = DefaultAccelerometerRevisePower;
    reviseRange = DefaultAccelerometerReviseRange;
}

void AccelerometerSetting::SetAccelerometerParameters(float radius,
                                                     float sens) NN_NOEXCEPT
{
    playRadius = radius;
    sensitivity = sens;
}

void AccelerometerSetting::GetAccelerometerParameters(float* pOutPlayRadius,
                                                     float* pOutSensitivity) const NN_NOEXCEPT
{
    *pOutPlayRadius = playRadius;
    *pOutSensitivity = sensitivity;
}

void AccelerometerSetting::ResetPlayModeParameters() NN_NOEXCEPT
{
    playRadius = DefaultAccelerometerPlayRadius;
    sensitivity = DefaultAccelerometerSensitivity;
}

void AccelerometerSetting::SetAccelerometerPlayMode(nn::hid::AccelerometerPlayMode mode) NN_NOEXCEPT
{
    playMode = mode;
}

void AccelerometerSetting::GetAccelerometerPlayMode(nn::hid::AccelerometerPlayMode* pOutMode) const NN_NOEXCEPT
{
    *pOutMode = playMode;
}

void AccelerometerSetting::ResetPlayMode() NN_NOEXCEPT
{
    playMode = DefaultAccelerometerMode;
}

void AccelerometerSetting::SetShiftCalibrationValue(float setOrigin, float setSensitivity) NN_NOEXCEPT
{
    shiftOrigin = setOrigin;
    shiftSensitivity = setSensitivity;
}

void AccelerometerSetting::GetShiftCalibrationValue(float* pOutOrigin, float* pOutSensitivity) const NN_NOEXCEPT
{
    *pOutOrigin = shiftOrigin;
    *pOutSensitivity = shiftSensitivity;
}

void AccelerometerSetting::ResetShiftCalibrationValue() NN_NOEXCEPT
{
    SetShiftCalibrationValue(0.f, 1.f);
}

void GyroscopeSetting::Reset(bool global) NN_NOEXCEPT
{
    fsr = DefaultGyroscopeFsr;
    enableZeroDrift = DefaultGyroscopeEnableZeroDrift;
    enableZeroPlay = DefaultGyroscopeEnableZeroPlay;
    ResetZeroPlayParameters();
    ResetZeroDriftMode();
    magnification = DefaultGyroscopeMagnification;
    directionMagnification = DefaultGyroscopeDirectionMagnification;

    if(global)
    {
        ResetShiftCalibrationValue();
    }
}

void GyroscopeSetting::SetGyroscopeFsr(const nn::xcd::GyroscopeFsr& gyroFsr) NN_NOEXCEPT
{
    fsr = gyroFsr;
}

void GyroscopeSetting::EnableZeroDrift(bool enable) NN_NOEXCEPT
{
    enableZeroDrift = enable;
}

bool GyroscopeSetting::IsZeroDriftEnabled() const NN_NOEXCEPT
{
    return enableZeroDrift;
}

void GyroscopeSetting::SetGyroscopeZeroDriftMode(nn::hid::GyroscopeZeroDriftMode mode) NN_NOEXCEPT
{
    zeroDriftMode = mode;
}

void GyroscopeSetting::GetGyroscopeZeroDriftMode(nn::hid::GyroscopeZeroDriftMode* pOutMode) const NN_NOEXCEPT
{
    *pOutMode = zeroDriftMode;
}

void GyroscopeSetting::ResetZeroDriftMode() NN_NOEXCEPT
{
    zeroDriftMode = DefaultGyroscopeZeroDriftMode;
}

void GyroscopeSetting::EnableZeroPlay(bool enable) NN_NOEXCEPT
{
    enableZeroPlay = enable;
}

bool GyroscopeSetting::IsZeroPlayEnabled() const NN_NOEXCEPT
{
    return enableZeroPlay;
}

void GyroscopeSetting::SetZeroPlayParameters(const float& radius) NN_NOEXCEPT
{
    zeroPlayRadius = radius;
}

void GyroscopeSetting::GetZeroPlayParameters(float* pOutRadius) const NN_NOEXCEPT
{
    *pOutRadius = zeroPlayRadius;
}

void GyroscopeSetting::ResetZeroPlayParameters() NN_NOEXCEPT
{
    zeroPlayRadius = DefaultGyroscopeZeroPlayRadius;
}

void GyroscopeSetting::SetMagnification(const nn::util::Float3& mag) NN_NOEXCEPT
{
    magnification = mag;
}

void GyroscopeSetting::SetDirectionMagnification(const nn::util::Float3& directionMag) NN_NOEXCEPT
{
    directionMagnification = directionMag;
}

void GyroscopeSetting::SetShiftCalibrationValue(float origin, float sensitivity) NN_NOEXCEPT
{
    shiftOrigin = origin;
    shiftSensitivity = sensitivity;
}

void GyroscopeSetting::GetShiftCalibrationValue(float* pOutOrigin, float* pOutSensitivity) const NN_NOEXCEPT
{
    *pOutOrigin = shiftOrigin;
    *pOutSensitivity = shiftSensitivity;
}

void GyroscopeSetting::ResetShiftCalibrationValue() NN_NOEXCEPT
{
    SetShiftCalibrationValue(0.f, 1.f);
}

void SixAxisSensorProcessorSetting::EnableSampling(bool enable) NN_NOEXCEPT
{
    isSamplingEnabled = enable;
}

bool SixAxisSensorProcessorSetting::IsSamplingEnabled() const NN_NOEXCEPT
{
    return isSamplingEnabled;
}

void SixAxisSensorProcessorSetting::Reset(bool global) NN_NOEXCEPT
{
    NN_UNUSED(global);
    ResetBaseDirection();
    EnableSampling(false);
}

void SixAxisSensorProcessorSetting::EnableBaseDirection(bool enable) NN_NOEXCEPT
{
    isBaseDirReviseEnabled = enable;
}

bool SixAxisSensorProcessorSetting::IsBaseDirectionEnabled() const NN_NOEXCEPT
{
    return isBaseDirReviseEnabled;
}

void SixAxisSensorProcessorSetting::SetBaseDirection(const nn::hid::DirectionState& direction,
                                                    const float& power) NN_NOEXCEPT
{
    baseDirection = direction;
    baseDirectionRevisePower = power;
}

void SixAxisSensorProcessorSetting::GetBaseDirection(nn::hid::DirectionState* pOutDirection,
                                                    float* pOutPower) const NN_NOEXCEPT
{
    *pOutDirection = baseDirection;
    *pOutPower = baseDirectionRevisePower;
}

void SixAxisSensorProcessorSetting::ResetBaseDirection() NN_NOEXCEPT
{
    isBaseDirReviseEnabled = false;
    baseDirectionRevisePower = DefaultBaseDirectionRevisePower;
    baseDirection = DefaultBaseDirection;
}

void SixAxisSensorSetting::Reset(bool global) NN_NOEXCEPT
{
    accelerometerSetting.Reset(global);
    gyroscopeSetting.Reset(global);
    sixAxisSensorProcessorSetting.Reset(global);
}

void SixAxisSensorAppletSetting::Reset(bool global) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSettings[] = {
        fullKey,
        handheld,
        joyDualLeft,
        joyDualRight,
        joyLeft,
        joyRight,
        dummyStyle,
    };

    for (auto pSetting : pSettings)
    {
        for (int i = 0; i < NpadIdCountMax; i++)
        {
            pSetting[i].accelerometerSetting.Reset(global);
            pSetting[i].gyroscopeSetting.Reset(global);
            pSetting[i].sixAxisSensorProcessorSetting.Reset(global);
        }
    }
}

SixAxisSensorSetting* SixAxisSensorAppletSetting::GetSixAxisSensorSetting(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    auto npadId = GetSixAxisSensorHandleNpadIdType(handle);
    auto npadStyleIndex = GetSixAxisSensorHandleNpadStyleIndex(handle);
    auto index = GetIndexFromNpadIdType(npadId);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::VerifyValidNpadId(npadId));

    switch (npadStyleIndex)
    {
    case NpadStyleFullKey::Index:
        return &fullKey[index];
    case NpadStyleHandheld::Index:
        return &handheld[index];
    case NpadStyleJoyDual::Index:
        switch (GetSixAxisSensorHandleSixAxisSensorIdx(handle))
        {
        case SixAxisSensorIdx_Left:
            return &joyDualLeft[index];
        case SixAxisSensorIdx_Right:
            return &joyDualRight[index];
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    case NpadStyleJoyLeft::Index:
        return &joyLeft[index];
    case NpadStyleJoyRight::Index:
        return &joyRight[index];
    default:
        return &dummyStyle[index]; // NpadStyleSystem, NpadStyleSystemExt のケース
    }
}

}}} // namespace nn::hid::detail
