﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_SharedMemoryFormat.h"
#include "hid_SharedMemoryHolder.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 他プロセスに許可されるメモリアクセス権
const ::nn::os::MemoryPermission OtherPermission =
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    ::nn::os::MemoryPermission_ReadOnly;
#else
    ::nn::os::MemoryPermission_ReadWrite;
#endif

} // namespace

SharedMemoryHolder::SharedMemoryHolder() NN_NOEXCEPT
    : m_IsOwner(true)
    , m_IsCreated(false)
    , m_IsMapped(false)
    , m_SharedMemory()
    , m_Address(nullptr)
{
}

SharedMemoryHolder::SharedMemoryHolder(SharedMemoryHolder&& other) NN_NOEXCEPT
    : m_IsOwner(other.m_IsOwner)
    , m_IsCreated(other.m_IsCreated)
    , m_IsMapped(other.m_IsMapped)
    , m_SharedMemory(other.m_SharedMemory)
    , m_Address(other.m_Address)
{
    other.m_IsOwner = false;
}

bool SharedMemoryHolder::IsMapped() const NN_NOEXCEPT
{
    return m_IsMapped;
}

::nn::Result SharedMemoryHolder::Initialize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsOwner);

    NN_RESULT_DO(
        ::nn::os::CreateSharedMemory(&m_SharedMemory,
                                     sizeof(SharedMemoryFormat),
                                     ::nn::os::MemoryPermission_ReadWrite,
                                     OtherPermission));

    ::nn::os::MapSharedMemory(&m_SharedMemory,
                              ::nn::os::MemoryPermission_ReadWrite);

    m_IsCreated = true;

    m_IsMapped = true;

    m_Address = reinterpret_cast<SharedMemoryFormat*>(
        ::nn::os::GetSharedMemoryAddress(&m_SharedMemory));

    NN_RESULT_SUCCESS;
}

void SharedMemoryHolder::Attach(::nn::os::NativeHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsOwner);

    ::nn::os::AttachSharedMemory(&m_SharedMemory,
                                 sizeof(SharedMemoryFormat),
                                 handle,
                                 false);

    ::nn::os::MapSharedMemory(&m_SharedMemory, OtherPermission);

    m_IsMapped = true;

    m_Address = reinterpret_cast<SharedMemoryFormat*>(
        ::nn::os::GetSharedMemoryAddress(&m_SharedMemory));
}

void SharedMemoryHolder::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsOwner);

    if (m_Address != nullptr)
    {
        ::nn::os::UnmapSharedMemory(&m_SharedMemory);
    }

    m_IsCreated = false;

    m_IsMapped = false;

    m_Address = nullptr;

    ::nn::os::DestroySharedMemory(&m_SharedMemory);
}

::nn::os::NativeHandle SharedMemoryHolder::GetSharedMemoryHandle(
    ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsOwner);

    return ::nn::os::GetSharedMemoryHandle(&m_SharedMemory);
}

SharedMemoryFormat* SharedMemoryHolder::GetAddress() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsOwner);

    return m_Address;
}

}}} // namespace nn::hid::detail
