﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_BitTypes.h>
#include <nn/nn_Macro.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AudioControlLifo.h"
#include "hid_CaptureButtonLifo.h"
#include "hid_ConsoleSixAxisSensorLifo.h"
#include "hid_DebugPadLifo.h"
#include "hid_GestureLifo.h"
#include "hid_HomeButtonLifo.h"
#include "hid_InputDetectorLifo.h"
#include "hid_KeyboardLifo.h"
#include "hid_MouseLifo.h"
#include "hid_NpadInternalState.h"
#include "hid_SleepButtonLifo.h"
#include "hid_TouchScreenLifo.h"
#include "hid_XpadLifo.h"

namespace nn { namespace hid { namespace detail {

//!< DebugPad の共有メモリのフォーマットを表す構造体です。
struct DebugPadSharedMemoryFormat final
{
    //!< DebugPad のリング LIFO
    ::nn::util::TypedStorage<DebugPadLifo,
                      sizeof(DebugPadLifo),
                  NN_ALIGNOF(DebugPadLifo)> lifo;

    ::nn::Bit8 margin[1024 - sizeof(DebugPadLifo)];
};

NN_STATIC_ASSERT(sizeof(DebugPadSharedMemoryFormat) == 1024);

//!< TouchScreen の共有メモリのフォーマットを表す構造体です。
struct TouchScreenSharedMemoryFormat final
{
    //!< TouchScreen のリング LIFO
    ::nn::util::TypedStorage<TouchScreenLifo,
                      sizeof(TouchScreenLifo),
                  NN_ALIGNOF(TouchScreenLifo)> lifo;

    ::nn::Bit8 margin[12 * 1024 - sizeof(TouchScreenLifo)];
};

NN_STATIC_ASSERT(sizeof(TouchScreenSharedMemoryFormat) == 12 * 1024);

//!< Mouse の共有メモリのフォーマットを表す構造体です。
struct MouseSharedMemoryFormat final
{
    //!< Mouse のリング LIFO
    ::nn::util::TypedStorage<MouseLifo,
                      sizeof(MouseLifo),
                  NN_ALIGNOF(MouseLifo)> lifo;

    ::nn::Bit8 margin[1024 - sizeof(MouseLifo)];
};

NN_STATIC_ASSERT(sizeof(MouseSharedMemoryFormat) == 1024);

//!< Keyboard の共有メモリのフォーマットを表す構造体です。
struct KeyboardSharedMemoryFormat final
{
    //!< Keyboard のリング LIFO
    ::nn::util::TypedStorage<KeyboardLifo,
                      sizeof(KeyboardLifo),
                  NN_ALIGNOF(KeyboardLifo)> lifo;

    ::nn::Bit8 margin[1024 - sizeof(KeyboardLifo)];
};

NN_STATIC_ASSERT(sizeof(KeyboardSharedMemoryFormat) == 1024);

//!< 基本的な構成を持つ Xpad の共有メモリのエントリを表す構造体です。
struct BasicXpadSharedMemoryEntry final
{
    //!< 基本的な構成を持つ Xpad のリング LIFO
    ::nn::util::TypedStorage<BasicXpadLifo,
                      sizeof(BasicXpadLifo),
                  NN_ALIGNOF(BasicXpadLifo)> lifo;

    ::nn::Bit8 margin[1024 - sizeof(BasicXpadLifo)];
};

NN_STATIC_ASSERT(sizeof(BasicXpadSharedMemoryEntry) == 1024);

//!< 基本的な構成を持つ Xpad の共有メモリのフォーマットを表す構造体です。
struct BasicXpadSharedMemoryFormat final
{
    //!< 基本的な構成を持つ Xpad の共有メモリのエントリ
    BasicXpadSharedMemoryEntry entries[4];
};

NN_STATIC_ASSERT(sizeof(BasicXpadSharedMemoryFormat) == 4 * 1024);

//!< ホームボタンの共有メモリのフォーマットを表す構造体です。
struct HomeButtonSharedMemoryFormat final
{
    //!< ホームボタンのリング LIFO
    ::nn::util::TypedStorage<HomeButtonLifo,
                      sizeof(HomeButtonLifo),
                  NN_ALIGNOF(HomeButtonLifo)> lifo;

    ::nn::Bit8 margin[512 - sizeof(HomeButtonLifo)];
};

NN_STATIC_ASSERT(sizeof(HomeButtonSharedMemoryFormat) == 512);

//!< スリープボタンの共有メモリのフォーマットを表す構造体です。
struct SleepButtonSharedMemoryFormat final
{
    //!< スリープボタンのリング LIFO
    ::nn::util::TypedStorage<SleepButtonLifo,
                      sizeof(SleepButtonLifo),
                  NN_ALIGNOF(SleepButtonLifo)> lifo;

    ::nn::Bit8 margin[512 - sizeof(SleepButtonLifo)];
};

NN_STATIC_ASSERT(sizeof(SleepButtonSharedMemoryFormat) == 512);

//!< 撮影ボタンの共有メモリのフォーマットを表す構造体です。
struct CaptureButtonSharedMemoryFormat final
{
    //!< 撮影ボタンのリング LIFO
    ::nn::util::TypedStorage<CaptureButtonLifo,
                      sizeof(CaptureButtonLifo),
                  NN_ALIGNOF(CaptureButtonLifo)> lifo;

    ::nn::Bit8 margin[512 - sizeof(CaptureButtonLifo)];
};

NN_STATIC_ASSERT(sizeof(CaptureButtonSharedMemoryFormat) == 512);

//!< Gesture の共有メモリのフォーマットを表す構造体です。
struct GestureSharedMemoryFormat final
{
    //!< Gesture のリング LIFO
    ::nn::util::TypedStorage<GestureLifo,
                      sizeof(GestureLifo),
                  NN_ALIGNOF(GestureLifo)> lifo;

    ::nn::Bit8 margin[2 * 1024 - sizeof(GestureLifo)];
};

NN_STATIC_ASSERT(sizeof(GestureSharedMemoryFormat) == 2 * 1024);

//!< InputDetector の共有メモリのエントリを表す構造体です。
struct InputDetectorSharedMemoryEntry final
{
    //!< InputDetector のリング LIFO
    ::nn::util::TypedStorage<InputDetectorLifo,
        sizeof(InputDetectorLifo),
        NN_ALIGNOF(InputDetectorLifo)> lifo;

    ::nn::Bit8 margin[128 - sizeof(InputDetectorLifo)];
};

NN_STATIC_ASSERT(sizeof(InputDetectorSharedMemoryEntry) == 128);

//!< Npad の共有メモリのエントリを表す構造体です。
struct NpadSharedMemoryEntry final
{
    //!< Npad のリング LIFO
    ::nn::util::TypedStorage<NpadInternalState,
                      sizeof(NpadInternalState),
                  NN_ALIGNOF(NpadInternalState)> internalState;

    ::nn::Bit8 margin[20 * 1024 - sizeof(NpadInternalState)];
};

NN_STATIC_ASSERT(sizeof(NpadSharedMemoryEntry) == 20 * 1024);

//!< Npad の共有メモリのフォーマットを表す構造体です。
struct NpadSharedMemoryFormat final
{
    //!< Npad の共有メモリのエントリ
    NpadSharedMemoryEntry entries[10];
};

NN_STATIC_ASSERT(sizeof(NpadSharedMemoryFormat) == 20 * 10 * 1024);


//!< InputDetector の共有メモリのフォーマットを表す構造体です。
struct InputDetectorSharedMemoryFormat final
{
    //!< InputDetector の共有メモリのエントリ
    InputDetectorSharedMemoryEntry entries[::nn::hid::system::InputSourceIdCountMax];
};

//!< 未使用領域 (旧 UniquePad 領域)
struct NonUsedSpace01 final
{
    ::nn::Bit8 margin[1024 * 16];
};

NN_STATIC_ASSERT(sizeof(NonUsedSpace01) == 1024 * 16);

//!< ConsoleSixAxisSensor の共有メモリのフォーマットを表す構造体です。
struct ConsoleSixAxisSensorSharedMemoryFormat final
{
    //!< ConsoleSixAxisSensor のリング LIFO
    ::nn::util::TypedStorage<ConsoleSixAxisSensorInternalState,
                      sizeof(ConsoleSixAxisSensorInternalState),
                  NN_ALIGNOF(ConsoleSixAxisSensorInternalState)> internalState;

    ::nn::Bit8 margin[1024 - sizeof(ConsoleSixAxisSensorInternalState)];
};

NN_STATIC_ASSERT(sizeof(ConsoleSixAxisSensorSharedMemoryFormat) == 1024);

//!< 未使用領域 (旧 ConsoleSixAxisSensor 領域)
struct NonUsedSpace02 final
{
    ::nn::Bit8 margin[1024 * 5];
};

NN_STATIC_ASSERT(sizeof(NonUsedSpace02) == 1024 * 5);

//!< オーディオコントロールの共有メモリのフォーマットを表す構造体です。
struct AudioControlSharedMemoryFormat final
{
    //!< オーディオコントロールのリング LIFO
    ::nn::util::TypedStorage<AudioControlLifo,
                      sizeof(AudioControlLifo),
                  NN_ALIGNOF(AudioControlLifo)> lifo;

    ::nn::Bit8 margin[512 - sizeof(AudioControlLifo)];
};

NN_STATIC_ASSERT(sizeof(AudioControlSharedMemoryFormat) == 512);

//!< 共有メモリのフォーマットを表す構造体です。
struct SharedMemoryFormat final
{
    DebugPadSharedMemoryFormat debugPad;
    TouchScreenSharedMemoryFormat touchScreen;
    MouseSharedMemoryFormat mouse;
    KeyboardSharedMemoryFormat keyboard;
    BasicXpadSharedMemoryFormat basicXpad;
    HomeButtonSharedMemoryFormat homeButton;
    SleepButtonSharedMemoryFormat sleepButton;
    CaptureButtonSharedMemoryFormat captureButton;
    InputDetectorSharedMemoryFormat inputDetector;
    NonUsedSpace01 nonUsed01;
    NpadSharedMemoryFormat npad;
    GestureSharedMemoryFormat gesture;
    ConsoleSixAxisSensorSharedMemoryFormat consoleSixAxisSensor;
    NonUsedSpace02 nonUsed02;
    AudioControlSharedMemoryFormat audioControl;

    ::nn::Bit8 margin[
        64 * ::nn::os::MemoryPageSize
            - sizeof(DebugPadSharedMemoryFormat)
            - sizeof(TouchScreenSharedMemoryFormat)
            - sizeof(MouseSharedMemoryFormat)
            - sizeof(KeyboardSharedMemoryFormat)
            - sizeof(BasicXpadSharedMemoryFormat)
            - sizeof(HomeButtonSharedMemoryFormat)
            - sizeof(SleepButtonSharedMemoryFormat)
            - sizeof(CaptureButtonSharedMemoryFormat)
            - sizeof(InputDetectorSharedMemoryFormat)
            - sizeof(NonUsedSpace01)
            - sizeof(NpadSharedMemoryFormat)
            - sizeof(GestureSharedMemoryFormat)
            - sizeof(ConsoleSixAxisSensorSharedMemoryFormat)
            - sizeof(NonUsedSpace02)
            - sizeof(AudioControlSharedMemoryFormat)
    ];
};

NN_STATIC_ASSERT(sizeof(SharedMemoryFormat) == 64 * ::nn::os::MemoryPageSize);

//!< 共有メモリのフォーマットを初期化します。
void InitializeSharedMemoryFormat(SharedMemoryFormat* ptr) NN_NOEXCEPT;

//!< 共有メモリのフォーマットを破棄します。
void FinalizeSharedMemoryFormat(SharedMemoryFormat* ptr) NN_NOEXCEPT;

}}} // namespace nn::hid::detail
