﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_CaptureButtonLifo.h"
#include "hid_ConsoleSixAxisSensorLifo.h"
#include "hid_DebugPadLifo.h"
#include "hid_GestureLifo.h"
#include "hid_HomeButtonLifo.h"
#include "hid_InputDetectorLifo.h"
#include "hid_MouseLifo.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SleepButtonLifo.h"
#include "hid_TouchScreenLifo.h"
#include "hid_XpadLifo.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< DebugPad の共有メモリのフォーマットを初期化します。
void InitializeDebugPadSharedMemoryFormat(
    DebugPadSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    new(&::nn::util::Get(ptr->lifo)) DebugPadLifo();
}

//!< DebugPad の共有メモリのフォーマットを破棄します。
void FinalizeDebugPadSharedMemoryFormat(
    DebugPadSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    ::nn::util::Get(ptr->lifo).~DebugPadLifo();
}

//!< TouchScreen の共有メモリのフォーマットを初期化します。
void InitializeTouchScreenSharedMemoryFormat(
    TouchScreenSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    new(&::nn::util::Get(ptr->lifo)) TouchScreenLifo();
}

//!< TouchScreen の共有メモリのフォーマットを破棄します。
void FinalizeTouchScreenSharedMemoryFormat(
    TouchScreenSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    ::nn::util::Get(ptr->lifo).~TouchScreenLifo();
}

//!< Mouse の共有メモリのフォーマットを初期化します。
void InitializeMouseSharedMemoryFormat(MouseSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    new(&::nn::util::Get(ptr->lifo)) MouseLifo();
}

//!< Mouse の共有メモリのフォーマットを破棄します。
void FinalizeMouseSharedMemoryFormat(MouseSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    ::nn::util::Get(ptr->lifo).~MouseLifo();
}

//!< Keyboard の共有メモリのフォーマットを初期化します。
void InitializeKeyboardSharedMemoryFormat(
    KeyboardSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    new(&::nn::util::Get(ptr->lifo)) KeyboardLifo();
}

//!< Keyboard の共有メモリのフォーマットを破棄します。
void FinalizeKeyboardSharedMemoryFormat(
    KeyboardSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    ::nn::util::Get(ptr->lifo).~KeyboardLifo();
}

//!< 基本的な構成を持つ Xpad の共有メモリのフォーマットを初期化します。
void InitializeBasicXpadSharedMemoryFormat(
    BasicXpadSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    for (BasicXpadSharedMemoryEntry& entry : ptr->entries)
    {
        new(&::nn::util::Get(entry.lifo)) BasicXpadLifo();
    }
}

//!< 基本的な構成を持つ Xpad の共有メモリのフォーマットを破棄します。
void FinalizeBasicXpadSharedMemoryFormat(
    BasicXpadSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    for (BasicXpadSharedMemoryEntry& entry : ptr->entries)
    {
        ::nn::util::Get(entry.lifo).~BasicXpadLifo();
    }
}

//!< Npad の共有メモリのフォーマットを初期化します。
void InitializeNpadSharedMemoryFormat(
    NpadSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    for (NpadSharedMemoryEntry& entry : ptr->entries)
    {
        new(&::nn::util::Get(entry.internalState)) NpadInternalState();
    }
}

//!< Npad の共有メモリのフォーマットを破棄します。
void FinalizeNpadSharedMemoryFormat(
    NpadSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    for (NpadSharedMemoryEntry& entry : ptr->entries)
    {
        ::nn::util::Get(entry.internalState).~NpadInternalState();
    }
}

//!< ホームボタンの共有メモリのフォーマットを初期化します。
void InitializeHomeButtonSharedMemoryFormat(
    HomeButtonSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    new(&::nn::util::Get(ptr->lifo)) HomeButtonLifo();
}

//!< ホームボタンの共有メモリのフォーマットを破棄します。
void FinalizeHomeButtonSharedMemoryFormat(
    HomeButtonSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    ::nn::util::Get(ptr->lifo).~HomeButtonLifo();
}

//!< スリープボタンの共有メモリのフォーマットを初期化します。
void InitializeSleepButtonSharedMemoryFormat(
    SleepButtonSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    new(&::nn::util::Get(ptr->lifo)) SleepButtonLifo();
}

//!< スリープボタンの共有メモリのフォーマットを破棄します。
void FinalizeSleepButtonSharedMemoryFormat(
    SleepButtonSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    ::nn::util::Get(ptr->lifo).~SleepButtonLifo();
}

//!< 撮影ボタンの共有メモリのフォーマットを初期化します。
void InitializeCaptureButtonSharedMemoryFormat(
    CaptureButtonSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    new(&::nn::util::Get(ptr->lifo)) CaptureButtonLifo();
}

//!< 撮影ボタンの共有メモリのフォーマットを破棄します。
void FinalizeCaptureButtonSharedMemoryFormat(
    CaptureButtonSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    ::nn::util::Get(ptr->lifo).~CaptureButtonLifo();
}

//!< Gesture の共有メモリのフォーマットを初期化します。
void InitializeGestureSharedMemoryFormat(
    GestureSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    new(&::nn::util::Get(ptr->lifo)) GestureLifo();
}

//!< Gesture の共有メモリのフォーマットを破棄します。
void FinalizeGestureSharedMemoryFormat(
    GestureSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    ::nn::util::Get(ptr->lifo).~GestureLifo();
}

//!< InputDetector の共有メモリのフォーマットを初期化します。
void InitializeInputDetectorSharedMemoryFormat(
    InputDetectorSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    for (InputDetectorSharedMemoryEntry& entry : ptr->entries)
    {
        new(&::nn::util::Get(entry.lifo)) InputDetectorLifo();
    }
}

//!< InputDetector の共有メモリのフォーマットを破棄します。
void FinalizeInputDetectorSharedMemoryFormat(
    InputDetectorSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    for (InputDetectorSharedMemoryEntry& entry : ptr->entries)
    {
        ::nn::util::Get(entry.lifo).~InputDetectorLifo();
    }
}

//!< ConsoleSixAxisSensor の共有メモリのフォーマットを初期化します。
void InitializeConsoleSixAxisSensorSharedMemoryFormat(
    ConsoleSixAxisSensorSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    new(&::nn::util::Get(ptr->internalState)) ConsoleSixAxisSensorInternalState();
}

//!< ConsoleSixAxisSensor の共有メモリのフォーマットを破棄します。
void FinalizeConsoleSixAxisSensorSharedMemoryFormat(
    ConsoleSixAxisSensorSharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    ::nn::util::Get(ptr->internalState).~ConsoleSixAxisSensorInternalState();
}

} // namespace

void InitializeSharedMemoryFormat(SharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    InitializeDebugPadSharedMemoryFormat(&ptr->debugPad);
    InitializeTouchScreenSharedMemoryFormat(&ptr->touchScreen);
    InitializeMouseSharedMemoryFormat(&ptr->mouse);
    InitializeKeyboardSharedMemoryFormat(&ptr->keyboard);
    InitializeBasicXpadSharedMemoryFormat(&ptr->basicXpad);
    InitializeNpadSharedMemoryFormat(&ptr->npad);
    InitializeHomeButtonSharedMemoryFormat(&ptr->homeButton);
    InitializeSleepButtonSharedMemoryFormat(&ptr->sleepButton);
    InitializeCaptureButtonSharedMemoryFormat(&ptr->captureButton);
    InitializeGestureSharedMemoryFormat(&ptr->gesture);
    InitializeInputDetectorSharedMemoryFormat(&ptr->inputDetector);
    InitializeConsoleSixAxisSensorSharedMemoryFormat(&ptr->consoleSixAxisSensor);
}

void FinalizeSharedMemoryFormat(SharedMemoryFormat* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    FinalizeDebugPadSharedMemoryFormat(&ptr->debugPad);
    FinalizeTouchScreenSharedMemoryFormat(&ptr->touchScreen);
    FinalizeMouseSharedMemoryFormat(&ptr->mouse);
    FinalizeKeyboardSharedMemoryFormat(&ptr->keyboard);
    FinalizeBasicXpadSharedMemoryFormat(&ptr->basicXpad);
    FinalizeNpadSharedMemoryFormat(&ptr->npad);
    FinalizeHomeButtonSharedMemoryFormat(&ptr->homeButton);
    FinalizeSleepButtonSharedMemoryFormat(&ptr->sleepButton);
    FinalizeCaptureButtonSharedMemoryFormat(&ptr->captureButton);
    FinalizeGestureSharedMemoryFormat(&ptr->gesture);
    FinalizeInputDetectorSharedMemoryFormat(&ptr->inputDetector);
    FinalizeConsoleSixAxisSensorSharedMemoryFormat(&ptr->consoleSixAxisSensor);
}

}}} // namespace nn::hid::detail
