﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/os/os_Mutex.h>

#include "hid_ActivationCount.h"
#include "hid_SharedMemoryFormat.h"

namespace nn { namespace hid { namespace detail {

//!< 共有メモリへのアクセスを扱うクラスです。
class SharedMemoryAccessor
{
    NN_DISALLOW_COPY(SharedMemoryAccessor);
    NN_DISALLOW_MOVE(SharedMemoryAccessor);

private:
    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< ミューテックス
    ::nn::os::Mutex m_Mutex;

    //!< アクティブ化された回数が上限を超えたことを表す Result 値
    ::nn::Result m_ResultActivationUpperLimitOver;

    //!< アクティブ化された回数が下限を超えたことを表す Result 値
    ::nn::Result m_ResultDeactivationLowerLimitOver;

public:
    SharedMemoryAccessor() NN_NOEXCEPT;

    virtual ~SharedMemoryAccessor() NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID を設定します。
    ::nn::Result SetAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 共有メモリアクセッサをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< 共有メモリアクセッサを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

protected:
    //!< アクティブ化された回数が上限を超えたことを表す Result 値を設定します。
    void SetResultActivationUpperLimitOver(::nn::Result value) NN_NOEXCEPT;

    //!< アクティブ化された回数が下限を超えたことを表す Result 値を設定します。
    void SetResultDeactivationLowerLimitOver(::nn::Result value) NN_NOEXCEPT;

protected:
    //!< 共有メモリをアタッチします。
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT = 0;

    //!< 共有メモリをデタッチします。
    virtual ::nn::Result Detach() NN_NOEXCEPT = 0;
};

}}} // namespace nn::hid::detail
